/* File: cave.c */

/*
 * Copyright (c) 1997 Ben Harrison, James E. Wilson, Robert A. Koeneke
 *
 * This software may be copied and distributed for educational, research,
 * and not for profit purposes provided that this copyright and statement
 * are included in all such copies.  Other copyrights may also apply.
 */

#include "angband.h"



/*
 * Approximate Distance between two points.
 *
 * When either the X or Y component dwarfs the other component,
 * this function is almost perfect, and otherwise, it tends to
 * over-estimate about one grid per fifteen grids of distance.
 *
 * Algorithm: hypot(dy,dx) = max(dy,dx) + min(dy,dx) / 2
 */
sint distance(int y1, int x1, int y2, int x2)
{
	int ay, ax;

	/* Find the absolute y/x distance components */
	ay = (y1 > y2) ? (y1 - y2) : (y2 - y1);
	ax = (x1 > x2) ? (x1 - x2) : (x2 - x1);

	/* Hack -- approximate the distance */
	return ((ay > ax) ? (ay + (ax>>1)) : (ax + (ay>>1)));
}


/*
 * A simple, fast, integer-based line-of-sight algorithm.  By Joseph Hall,
 * 4116 Brewster Drive, Raleigh NC 27606.  Email to jnh@ecemwl.ncsu.edu.
 *
 * This function returns TRUE if a "line of sight" can be traced from the
 * center of the grid (x1,y1) to the center of the grid (x2,y2), with all
 * of the grids along this path (except for the endpoints) being non-wall
 * grids.  Actually, the "chess knight move" situation is handled by some
 * special case code which allows the grid diagonally next to the player
 * to be obstructed, because this yields better gameplay semantics.  This
 * algorithm is totally reflexive, except for "knight move" situations.
 *
 * Because this function uses (short) ints for all calculations, overflow
 * may occur if dx and dy exceed 90.
 *
 * Once all the degenerate cases are eliminated, we determine the "slope"
 * ("m"), and we use special "fixed point" mathematics in which we use a
 * special "fractional component" for one of the two location components
 * ("qy" or "qx"), which, along with the slope itself, are "scaled" by a
 * scale factor equal to "abs(dy*dx*2)" to keep the math simple.  Then we
 * simply travel from start to finish along the longer axis, starting at
 * the border between the first and second tiles (where the y offset is
 * thus half the slope), using slope and the fractional component to see
 * when motion along the shorter axis is necessary.  Since we assume that
 * vision is not blocked by "brushing" the corner of any grid, we must do
 * some special checks to avoid testing grids which are "brushed" but not
 * actually "entered".
 *
 * Angband three different "line of sight" type concepts, including this
 * function (which is used almost nowhere), the "project()" method (which
 * is used for determining the paths of projectables and spells and such),
 * and the "update_view()" concept (which is used to determine which grids
 * are "viewable" by the player, which is used for many things, such as
 * determining which grids are illuminated by the player's torch, and which
 * grids and monsters can be "seen" by the player, etc).
 */
bool los(int y1, int x1, int y2, int x2)
{
	/* Delta */
	int dx, dy;

	/* Absolute */
	int ax, ay;

	/* Signs */
	int sx, sy;

	/* Fractions */
	int qx, qy;

	/* Scanners */
	int tx, ty;

	/* Scale factors */
	int f1, f2;

	/* Slope, or 1/Slope, of LOS */
	int m;


	/* Extract the offset */
	dy = y2 - y1;
	dx = x2 - x1;

	/* Extract the absolute offset */
	ay = ABS(dy);
	ax = ABS(dx);


	/* Handle adjacent (or identical) grids */
	if ((ax < 2) && (ay < 2)) return (TRUE);


	/* Directly South/North */
	if (!dx)
	{
		/* South -- check for walls */
		if (dy > 0)
		{
			for (ty = y1 + 1; ty < y2; ty++)
			{
				if (!cave_floor_bold(ty, x1)) return (FALSE);
			}
		}

		/* North -- check for walls */
		else
		{
			for (ty = y1 - 1; ty > y2; ty--)
			{
				if (!cave_floor_bold(ty, x1)) return (FALSE);
			}
		}

		/* Assume los */
		return (TRUE);
	}

	/* Directly East/West */
	if (!dy)
	{
		/* East -- check for walls */
		if (dx > 0)
		{
			for (tx = x1 + 1; tx < x2; tx++)
			{
				if (!cave_floor_bold(y1, tx)) return (FALSE);
			}
		}

		/* West -- check for walls */
		else
		{
			for (tx = x1 - 1; tx > x2; tx--)
			{
				if (!cave_floor_bold(y1, tx)) return (FALSE);
			}
		}

		/* Assume los */
		return (TRUE);
	}


	/* Extract some signs */
	sx = (dx < 0) ? -1 : 1;
	sy = (dy < 0) ? -1 : 1;


	/* Vertical "knights" */
	if (ax == 1)
	{
		if (ay == 2)
		{
			if (cave_floor_bold(y1 + sy, x1)) return (TRUE);
		}
	}

	/* Horizontal "knights" */
	else if (ay == 1)
	{
		if (ax == 2)
		{
			if (cave_floor_bold(y1, x1 + sx)) return (TRUE);
		}
	}


	/* Calculate scale factor div 2 */
	f2 = (ax * ay);

	/* Calculate scale factor */
	f1 = f2 << 1;


	/* Travel horizontally */
	if (ax >= ay)
	{
		/* Let m = dy / dx * 2 * (dy * dx) = 2 * dy * dy */
		qy = ay * ay;
		m = qy << 1;

		tx = x1 + sx;

		/* Consider the special case where slope == 1. */
		if (qy == f2)
		{
			ty = y1 + sy;
			qy -= f1;
		}
		else
		{
			ty = y1;
		}

		/* Note (below) the case (qy == f2), where */
		/* the LOS exactly meets the corner of a tile. */
		while (x2 - tx)
		{
			if (!cave_floor_bold(ty, tx)) return (FALSE);

			qy += m;

			if (qy < f2)
			{
				tx += sx;
			}
			else if (qy > f2)
			{
				ty += sy;
				if (!cave_floor_bold(ty, tx)) return (FALSE);
				qy -= f1;
				tx += sx;
			}
			else
			{
				ty += sy;
				qy -= f1;
				tx += sx;
			}
		}
	}

	/* Travel vertically */
	else
	{
		/* Let m = dx / dy * 2 * (dx * dy) = 2 * dx * dx */
		qx = ax * ax;
		m = qx << 1;

		ty = y1 + sy;

		if (qx == f2)
		{
			tx = x1 + sx;
			qx -= f1;
		}
		else
		{
			tx = x1;
		}

		/* Note (below) the case (qx == f2), where */
		/* the LOS exactly meets the corner of a tile. */
		while (y2 - ty)
		{
			if (!cave_floor_bold(ty, tx)) return (FALSE);

			qx += m;

			if (qx < f2)
			{
				ty += sy;
			}
			else if (qx > f2)
			{
				tx += sx;
				if (!cave_floor_bold(ty, tx)) return (FALSE);
				qx -= f1;
				ty += sy;
			}
			else
			{
				tx += sx;
				qx -= f1;
				ty += sy;
			}
		}
	}

	/* Assume los */
	return (TRUE);
}




/*
 * Returns true if the player's grid is dark
 */
bool no_lite(void)
{
	int py = p_ptr->py;
	int px = p_ptr->px;

	return (!player_can_see_bold(py, px));
}




/*
 * Determine if a given location may be "destroyed"
 *
 * Used by destruction spells, and for placing stairs, etc.
 */
bool cave_valid_bold(int y, int x)
{
	s16b this_o_idx, next_o_idx = 0;


	/* Forbid perma-grids */
	if (cave_perma_bold(y, x)) return (FALSE);

	/* Check objects */
	for (this_o_idx = cave_o_idx[y][x]; this_o_idx; this_o_idx = next_o_idx)
	{
		object_type *o_ptr;

		/* Acquire object */
		o_ptr = &o_list[this_o_idx];

		/* Acquire next object */
		next_o_idx = o_ptr->next_o_idx;

		/* Forbid artifact grids */
		if (artifact_p(o_ptr)) return (FALSE);
	}

	/* Accept */
	return (TRUE);
}





/*
 * Hack -- encode attr/char pair
 */
#define PICT(A,C) ((((u16b)(A)) << 8) | ((byte)(C)))

/*
 * Hack -- decode attr/char pair
 */
#define PICT_A(P) ((byte)((P) >> 8))
#define PICT_C(P) ((char)((byte)(P)))


/*
 * Hack -- Legal monster codes
 */
static const char image_monster_hack[] = \
"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";

/*
 * Hack -- Hallucinatory monster
 */
static u16b image_monster(void)
{
	byte a;
	char c;

	/* Random symbol from set above (not including final nul) */
	c = image_monster_hack[rand_int(sizeof(image_monster_hack) - 1)];

	/* Random color */
	a = randint(15);

	/* Encode */
	return (PICT(a,c));
}


/*
 * Hack -- Legal object codes
 */
static const char image_object_hack[] = \
"?/|\\\"!$()_-=[]{},~"; /* " */

/*
 * Hack -- Hallucinatory object
 */
static u16b image_object(void)
{
	byte a;
	char c;

	/* Random symbol from set above (not including final nul) */
	c = image_object_hack[rand_int(sizeof(image_object_hack) - 1)];

	/* Random color */
	a = randint(15);

	/* Encode */
	return (PICT(a,c));
}


/*
 * Hack -- Random hallucination
 */
static u16b image_random(void)
{
	/* Normally, assume monsters */
	if (rand_int(100) < 75)
	{
		return (image_monster());
	}

	/* Otherwise, assume objects */
	else
	{
		return (image_object());
	}
}



/*
 * Extract the attr/char to display at the given (legal) map location
 *
 * Note that this function, since it is called by "lite_spot()" which
 * is called by "update_view()", is a major efficiency concern.
 *
 * Basically, we examine each "layer" of the world (terrain, objects,
 * monsters/players), from the bottom up, extracting a new attr/char
 * if necessary at each layer, and defaulting to "darkness".  This is
 * not the fastest method, but it is very simple, and it is about as
 * fast as it could be for grids which contain no "marked" objects or
 * "visible" monsters.
 *
 * We apply the effects of hallucination during each layer.  Objects will
 * always appear as random "objects", monsters will always appear as random
 * "monsters", and normal grids occasionally appear as random "monsters" or
 * "objects", but note that these random "monsters" and "objects" are really
 * just "colored ascii symbols" (which may look silly on some machines).
 *
 * The hallucination functions avoid taking any pointers to local variables
 * because some compilers refuse to use registers for any local variables
 * whose address is taken anywhere in the function.
 *
 * As an optimization, we can handle the "player" grid as a special case.
 *
 * Note that the memorization of "objects" and "monsters" is not related
 * to the memorization of "terrain".  This allows the player to memorize
 * the terrain of a grid without memorizing any objects in that grid, and
 * to detect monsters without detecting anything about the terrain of the
 * grid containing the monster.
 *
 * The fact that all interesting "objects" and "terrain features" are
 * memorized as soon as they become visible for the first time means
 * that we only have to check the "CAVE_SEEN" flag for "boring" grids.
 *
 * Note that bizarre things must be done when the "attr" and/or "char"
 * codes have the "high-bit" set, since these values are used to encode
 * various "special" pictures in some versions, and certain situations,
 * such as "multi-hued" or "clear" monsters, cause the attr/char codes
 * to be "scrambled" in various ways.
 *
 * Note that the "zero" entry in the feature/object/monster arrays are
 * used to provide "special" attr/char codes, with "monster zero" being
 * used for the player attr/char, "object zero" being used for the "stack"
 * attr/char, and "feature zero" being used for the "nothing" attr/char.
 *
 * Note that eventually we may want to use the "&" symbol for embedded
 * treasure, and use the "*" symbol to indicate multiple objects, but
 * currently, we simply use the attr/char of the first "marked" object
 * in the stack, if any, and so "object zero" is unused.  XXX XXX XXX
 *
 * Note the assumption that doing "x_ptr = &x_info[x]" plus a few of
 * "x_ptr->xxx", is quicker than "x_info[x].xxx", even if "x" is a fixed
 * constant.  If this is incorrect then a lot of code should be changed.
 *
 *
 * Some comments on the "terrain" layer...
 *
 * Note that "boring" grids (floors, invisible traps, and any illegal grids)
 * are very different from "interesting" grids (all other terrain features),
 * and the two types of grids are handled completely separately.  The most
 * important distinction is that "boring" grids may or may not be memorized
 * when they are first encountered, and so we must use the "CAVE_SEEN" flag
 * to see if they are "see-able".
 *
 *
 * Some comments on the "terrain" layer (boring grids)...
 *
 * Note that "boring" grids are always drawn using the picture for "empty
 * floors", which is stored in "f_info[FEAT_FLOOR]".  Sometimes, special
 * lighting effects may cause this picture to be modified.
 *
 * Note that "invisible traps" are always displayes exactly like "empty
 * floors", which prevents various forms of "cheating", with no loss of
 * efficiency.  There are still a few ways to "guess" where traps may be
 * located, for example, objects will never fall into a grid containing
 * an invisible trap.  XXX XXX
 *
 * To determine if a "boring" grid should be displayed, we simply check to
 * see if it is either memorized ("CAVE_MARK"), or currently "see-able" by
 * the player ("CAVE_SEEN").  Note that "CAVE_SEEN" is now maintained by the
 * "update_view()" function.
 *
 * Note the "special lighting effects" which can be activated for "boring"
 * grids using the "view_special_lite" option, causing certain such grids
 * to be displayed using special colors (if they are normally "white").
 * If the grid is "see-able" by the player, we will use the normal "white"
 * (except that, if the "view_yellow_lite" option is set, and the grid
 * is *only* "see-able" because of the player's torch, then we will use
 * "yellow"), else if the player is "blind", we will use "dark gray",
 * else if the grid is not "illuminated", we will use "dark gray", else
 * if the "view_bright_lite" option is set, we will use "slate" (gray),
 * else we will use the normal "white".
 *
 *
 * Some comments on the "terrain" layer (non-boring grids)...
 *
 * Note the use of the "mimic" field in the "terrain feature" processing,
 * which allows any feature to "pretend" to be another feature.  This is
 * used to "hide" secret doors, and to make all "doors" appear the same,
 * and all "walls" appear the same, and "hidden" treasure stay hidden.
 * Note that it is possible to use this field to make a feature "look"
 * like a floor, but the "view_special_lite" flag only affects actual
 * "boring" grids.
 *
 * Since "interesting" grids are always memorized as soon as they become
 * "see-able" by the player ("CAVE_SEEN"), such a grid only needs to be
 * displayed if it is memorized ("CAVE_MARK").  Most "interesting" grids
 * are in fact non-memorized, non-see-able, wall grids, so the fact that
 * we do not have to check the "CAVE_SEEN" flag adds some efficiency, at
 * the cost of *forcing* the memorization of all "interesting" grids when
 * they are first seen.  Since the "CAVE_SEEN" flag is now maintained by
 * the "update_view()" function, this efficiency is not as significant as
 * it was in previous versions, and could perhaps be removed.
 *
 * Note the "special lighting effects" which can be activated for "wall"
 * grids using the "view_granite_lite" option, causing certain such grids
 * to be displayed using special colors (if they are normally "white").
 * If the grid is "see-able" by the player, we will use the normal "white"
 * else if the player is "blind", we will use "dark gray", else if the
 * "view_bright_lite" option is set, we will use "slate" (gray), else we
 * will use the normal "white".
 *
 * Note that "wall" grids are more complicated than "boring" grids, due to
 * the fact that "CAVE_GLOW" for a "wall" grid means that the grid *might*
 * be glowing, depending on where the player is standing in relation to the
 * wall.  In particular, the wall of an illuminated room should look just
 * like any other (dark) wall unless the player is actually inside the room.
 *
 * Thus, we do not support as many visual special effects for "wall" grids
 * as we do for "boring" grids, since many of them would give the player
 * information about the "CAVE_GLOW" flag of the wall grid, in particular,
 * it would allow the player to notice the walls of illuminated rooms from
 * a dark hallway that happened to run beside the room.
 *
 *
 * Some comments on the "object" layer...
 *
 * Currently, we do nothing with multi-hued objects, because there are
 * not any.  If there were, they would have to set "shimmer_objects"
 * when they were created, and then new "shimmer" code in "dungeon.c"
 * would have to be created handle the "shimmer" effect, and the code
 * in "cave.c" would have to be updated to create the shimmer effect.
 * This did not seem worth the effort.  XXX XXX
 *
 *
 * Some comments on the "monster"/"player" layer...
 *
 * Note that monsters can have some "special" flags, including "ATTR_MULTI",
 * which means their color changes, and "ATTR_CLEAR", which means they take
 * the color of whatever is under them, and "CHAR_CLEAR", which means that
 * they take the symbol of whatever is under them.  Technically, the flag
 * "CHAR_MULTI" is supposed to indicate that a monster looks strange when
 * examined, but this flag is currently ignored.  All of these flags are
 * ignored if the "avoid_other" option is set, since checking for these
 * conditions is expensive (and annoying) on some systems.
 *
 * Normally, players could be handled just like monsters, except that the
 * concept of the "torch lite" of others player would add complications.
 * For efficiency, however, we handle the (only) player first, since the
 * "player" symbol always "pre-empts" any other facts about the grid.
 *
 * The "hidden_player" efficiency option, which only makes sense with a
 * single player, allows the player symbol to be hidden while running.
 */
void map_info(int y, int x, byte *ap, char *cp)
{
	byte a;
	char c;

	byte feat;
	byte info;

	feature_type *f_ptr;

	s16b this_o_idx, next_o_idx = 0;

	s16b m_idx;

	bool image = p_ptr->image;


	/* Monster/Player */
	m_idx = cave_m_idx[y][x];

#ifdef MAP_INFO_MULTIPLE_PLAYERS

	/* Handle "player" grids below */

#else /* MAP_INFO_MULTIPLE_PLAYERS */

	/* Handle "player" */
	if ((m_idx < 0) && !(p_ptr->running && hidden_player))
	{
		monster_race *r_ptr = &r_info[0];

		/* Get the "player" attr */
		a = r_ptr->x_attr;

		/* Get the "player" char */
		c = r_ptr->x_char;

		/* Result */
		(*ap) = a;
		(*cp) = c;

		/* Done */
		return;
	}

#endif /* MAP_INFO_MULTIPLE_PLAYERS */


	/* Feature */
	feat = cave_feat[y][x];

	/* Cave flags */
	info = cave_info[y][x];

	/* Hack -- rare random hallucination on non-outer walls */
	if (image && (!rand_int(256)) && (feat < FEAT_PERM_SOLID))
	{
		int i = image_random();
		a = PICT_A(i);
		c = PICT_C(i);
	}

	/* Boring grids (floors, etc) */
	else if (feat <= FEAT_INVIS)
	{
		/* Memorized (or seen) floor */
		if ((info & (CAVE_MARK)) ||
		    (info & (CAVE_SEEN)))
		{
			/* Access floor */
			f_ptr = &f_info[FEAT_FLOOR];

			/* Normal char */
			c = f_ptr->z_char;

			/* Normal attr */
			a = f_ptr->z_attr;

			/* Special lighting effects */
			if (view_special_lite && (a == TERM_WHITE))
			{
				/* Handle "seen" grids */
				if (info & (CAVE_SEEN))
				{
					/* Only lit by "torch" lite */
					if (view_yellow_lite && !(info & (CAVE_GLOW)))
					{
						/* Use "yellow" */
						a = TERM_YELLOW;
					}
				}

				/* Handle "blind" */
				else if (p_ptr->blind)
				{
					/* Use "dark gray" */
					a = TERM_L_DARK;
				}

				/* Handle "dark" grids */
				else if (!(info & (CAVE_GLOW)))
				{
					/* Use "dark gray" */
					a = TERM_L_DARK;
				}

				/* Handle "view_bright_lite" */
				else if (view_bright_lite)
				{
					/* Use "gray" */
					a = TERM_SLATE;
				}
			}
		}

		/* Unknown */
		else
		{
			/* Access darkness */
			f_ptr = &f_info[FEAT_NONE];

			/* Normal attr */
			a = f_ptr->z_attr;

			/* Normal char */
			c = f_ptr->z_char;
		}
	}

	/* Interesting grids (non-floors) */
	else
	{
		/* Memorized grids */
		if (info & (CAVE_MARK))
		{
			/* Apply "mimic" field */
			feat = f_info[feat].mimic;

			/* Access feature */
			f_ptr = &f_info[feat];

			/* Normal char */
			c = f_ptr->z_char;

			/* Normal attr */
			a = f_ptr->z_attr;

			/* Special lighting effects (walls only) */
			if (view_granite_lite && (a == TERM_WHITE) &&
			    (feat >= FEAT_SECRET))
			{
				/* Handle "seen" grids */
				if (info & (CAVE_SEEN))
				{
					/* Use "white" */
				}

				/* Handle "blind" */
				else if (p_ptr->blind)
				{
					/* Use "dark gray" */
					a = TERM_L_DARK;
				}

				/* Handle "view_bright_lite" */
				else if (view_bright_lite)
				{
					/* Use "gray" */
					a = TERM_SLATE;
				}
			}
		}

		/* Unknown */
		else
		{
			/* Access darkness */
			f_ptr = &f_info[FEAT_NONE];

			/* Normal attr */
			a = f_ptr->z_attr;

			/* Normal char */
			c = f_ptr->z_char;
		}
	}


	/* Objects */
	for (this_o_idx = cave_o_idx[y][x]; this_o_idx; this_o_idx = next_o_idx)
	{
		object_type *o_ptr;

		/* Acquire object */
		o_ptr = &o_list[this_o_idx];

		/* Acquire next object */
		next_o_idx = o_ptr->next_o_idx;

		/* Memorized objects */
		if (o_ptr->marked)
		{
			/* Hack -- object hallucination */
			if (image)
			{
				int i = image_object();
				a = PICT_A(i);
				c = PICT_C(i);
			}

			/* Normal */
			else
			{
				/* Normal char */
				c = object_char(o_ptr);

				/* Normal attr */
				a = object_attr(o_ptr);
			}

			/* Done */
			break;
		}
	}


	/* Monsters */
	if (m_idx > 0)
	{
		monster_type *m_ptr = &m_list[m_idx];

		/* Visible monster */
		if (m_ptr->ml)
		{
			monster_race *r_ptr = &r_info[m_ptr->r_idx];

			byte da;
			char dc;

			/* Desired attr */
			da = r_ptr->x_attr;

			/* Desired char */
			dc = r_ptr->x_char;

			/* Hack -- monster hallucination */
			if (image)
			{
				int i = image_monster();
				a = PICT_A(i);
				c = PICT_C(i);
			}

			/* Ignore weird codes */
			else if (avoid_other)
			{
				/* Use char */
				c = dc;

				/* Use attr */
				a = da;
			}

			/* Special attr/char codes */
			else if ((da & 0x80) && (dc & 0x80))
			{
				/* Use char */
				c = dc;

				/* Use attr */
				a = da;
			}

			/* Multi-hued monster */
			else if (r_ptr->flags1 & (RF1_ATTR_MULTI))
			{
				/* Normal char */
				c = dc;

				/* Multi-hued attr */
				a = randint(15);
			}

			/* Normal monster (not "clear" in any way) */
			else if (!(r_ptr->flags1 & (RF1_ATTR_CLEAR | RF1_CHAR_CLEAR)))
			{
				/* Use char */
				c = dc;

				/* Use attr */
				a = da;
			}

			/* Hack -- Bizarre grid under monster */
			else if ((a & 0x80) || (c & 0x80))
			{
				/* Use char */
				c = dc;

				/* Use attr */
				a = da;
			}

			/* Normal char, Clear attr, monster */
			else if (!(r_ptr->flags1 & (RF1_CHAR_CLEAR)))
			{
				/* Normal char */
				c = dc;
			}

			/* Normal attr, Clear char, monster */
			else if (!(r_ptr->flags1 & (RF1_ATTR_CLEAR)))
			{
				/* Normal attr */
				a = da;
			}
		}
	}

#ifdef MAP_INFO_MULTIPLE_PLAYERS

	/* Players */
	else if (m_idx < 0)
	{
		monster_race *r_ptr = &r_info[0];

		/* Get the "player" attr */
		a = r_ptr->x_attr;

		/* Get the "player" char */
		c = r_ptr->x_char;
	}

#endif


	/* Result */
	(*ap) = a;
	(*cp) = c;
}



/*
 * Move the cursor to a given map location
 */
void move_cursor_relative(int y, int x)
{
	unsigned ky, kx;
	unsigned vy, vx;

	/* Location relative to panel */
	ky = (unsigned)(y - p_ptr->wy);

	/* Verify location */
	if (ky >= (unsigned)(SCREEN_HGT)) return;

	/* Location relative to panel */
	kx = (unsigned)(x - p_ptr->wx);

	/* Verify location */
	if (kx >= (unsigned)(SCREEN_WID)) return;

	/* Location in window */
	vy = ky + ROW_MAP;

	/* Verify location */
	if (vy >= (unsigned)(Term->hgt)) return;

	/* Location in window */
	vx = kx + COL_MAP;

	/* Verify location */
	if (vx >= (unsigned)(Term->wid)) return;

	/* Go there */
	Term_gotoxy(vx, vy);
}



/*
 * Display an attr/char pair at the given map location
 *
 * Note the inline use of "panel_contains()" for efficiency.
 *
 * Note the use of "Term_queue_char()" for efficiency.
 */
void print_rel(char c, byte a, int y, int x)
{
	unsigned ky, kx;
	unsigned vy, vx;

	/* Location relative to panel */
	ky = (unsigned)(y - p_ptr->wy);

	/* Verify location */
	if (ky >= (unsigned)(SCREEN_HGT)) return;

	/* Location relative to panel */
	kx = (unsigned)(x - p_ptr->wx);

	/* Verify location */
	if (kx >= (unsigned)(SCREEN_WID)) return;

	/* Location in window */
	vy = ky + ROW_MAP;

	/* Verify location */
	if (vy >= (unsigned)(Term->hgt)) return;

	/* Location in window */
	vx = kx + COL_MAP;

	/* Verify location */
	if (vx >= (unsigned)(Term->wid)) return;

	/* Hack -- Queue it */
	Term_queue_char(vx, vy, a, c);
}




/*
 * Memorize interesting viewable object/features in the given grid
 *
 * This function should only be called on "legal" grids.
 *
 * This function will memorize the object and/or feature in the given grid,
 * if they are (1) see-able and (2) interesting.  Note that all objects are
 * interesting, all terrain features except floors (and invisible traps) are
 * interesting, and floors (and invisible traps) are interesting sometimes
 * (depending on various options involving the illumination of floor grids).
 *
 * The automatic memorization of all objects and non-floor terrain features
 * as soon as they are displayed allows incredible amounts of optimization
 * in various places, especially "map_info()" and this function itself.
 *
 * Note that the memorization of objects is completely separate from the
 * memorization of terrain features, preventing annoying floor memorization
 * when a detected object is picked up from a dark floor, and object
 * memorization when an object is dropped into a floor grid which is
 * memorized but out-of-sight.
 *
 * This function should be called every time the "memorization" of a grid
 * (or the object in a grid) is called into question, such as when an object
 * is created in a grid, when a terrain feature "changes" from "floor" to
 * "non-floor", and when any grid becomes "see-able" for any reason.
 *
 * This function is called primarily from the "update_view()" function, for
 * each grid which becomes newly "see-able".
 */
void note_spot(int y, int x)
{
	byte info;

	s16b this_o_idx, next_o_idx = 0;


	/* Get cave info */
	info = cave_info[y][x];

	/* Require "seen" flag */
	if (!(info & (CAVE_SEEN))) return;


	/* Hack -- memorize objects */
	for (this_o_idx = cave_o_idx[y][x]; this_o_idx; this_o_idx = next_o_idx)
	{
		object_type *o_ptr = &o_list[this_o_idx];

		/* Acquire next object */
		next_o_idx = o_ptr->next_o_idx;

		/* Memorize objects */
		o_ptr->marked = TRUE;
	}


	/* Hack -- memorize grids */
	if (!(info & (CAVE_MARK)))
	{
		/* Memorize some "boring" grids */
		if (cave_feat[y][x] <= FEAT_INVIS)
		{
			/* Option -- memorize certain floors */
			if (((info & (CAVE_GLOW)) && view_perma_grids) ||
			    view_torch_grids)
			{
				/* Memorize */
				cave_info[y][x] |= (CAVE_MARK);
			}
		}

		/* Memorize all "interesting" grids */
		else
		{
			/* Memorize */
			cave_info[y][x] |= (CAVE_MARK);
		}
	}
}


/*
 * Redraw (on the screen) a given map location
 *
 * This function should only be called on "legal" grids.
 *
 * Note the inline use of "print_rel()" for efficiency.
 */
void lite_spot(int y, int x)
{
	byte a;
	char c;

	unsigned ky, kx;
	unsigned vy, vx;

	/* Location relative to panel */
	ky = (unsigned)(y - p_ptr->wy);

	/* Verify location */
	if (ky >= (unsigned)(SCREEN_HGT)) return;

	/* Location relative to panel */
	kx = (unsigned)(x - p_ptr->wx);

	/* Verify location */
	if (kx >= (unsigned)(SCREEN_WID)) return;

	/* Location in window */
	vy = ky + ROW_MAP;

	/* Verify location XXX XXX XXX */
	if (vy >= (unsigned)(Term->hgt)) return;

	/* Location in window */
	vx = kx + COL_MAP;

	/* Verify location XXX XXX XXX */
	if (vx >= (unsigned)(Term->wid)) return;

	/* Hack -- redraw the grid */
	map_info(y, x, &a, &c);

	/* Hack -- Queue it */
	Term_queue_char(vx, vy, a, c);
}



/*
 * Redraw (on the screen) the current map panel
 *
 * Note the inline use of "lite_spot()" for efficiency.
 */
void prt_map(void)
{
	byte a;
	char c;

	int y, x;
	int vy, vx;
	int ty, tx;

	/* Assume screen */
	ty = ROW_MAP + SCREEN_HGT;
	tx = COL_MAP + SCREEN_WID;

	/* Handle reduced windows XXX XXX */
	if (ty > Term->hgt) ty = Term->hgt;
	if (tx > Term->wid) tx = Term->wid;

	/* Dump the map */
	for (y = p_ptr->wy, vy = ROW_MAP; vy < ty; vy++, y++)
	{
		for (x = p_ptr->wx, vx = COL_MAP; vx < tx; vx++, x++)
		{
			/* Determine what is there */
			map_info(y, x, &a, &c);

			/* Hack -- Queue it */
			Term_queue_char(vx, vy, a, c);
		}
	}
}





/*
 * Display highest priority object in the RATIO by RATIO area
 */
#define	RATIO 3

/*
 * Display the entire map
 */
#define MAP_HGT (DUNGEON_HGT / RATIO)
#define MAP_WID (DUNGEON_WID / RATIO)

/*
 * Hack -- priority array (see below)
 *
 * Note that all "walls" always look like "secret doors" (see "map_info()").
 */
static byte priority_table[][2] =
{
	/* Dark */
	{ FEAT_NONE, 2 },

	/* Floors */
	{ FEAT_FLOOR, 5 },

	/* Walls */
	{ FEAT_SECRET, 10 },

	/* Quartz */
	{ FEAT_QUARTZ, 11 },

	/* Magma */
	{ FEAT_MAGMA, 12 },

	/* Rubble */
	{ FEAT_RUBBLE, 13 },

	/* Open doors */
	{ FEAT_OPEN, 15 },
	{ FEAT_BROKEN, 15 },

	/* Closed doors */
	{ FEAT_DOOR_HEAD + 0x00, 17 },

	/* Hidden gold */
	{ FEAT_QUARTZ_K, 19 },
	{ FEAT_MAGMA_K, 19 },

	/* Stairs */
	{ FEAT_LESS, 25 },
	{ FEAT_MORE, 25 },

	/* End */
	{ 0, 0 }
};


/*
 * Hack -- a priority function (see below)
 */
static byte priority(byte a, char c)
{
	int i, p0, p1;

	feature_type *f_ptr;

	/* Scan the table */
	for (i = 0; TRUE; i++)
	{
		/* Priority level */
		p1 = priority_table[i][1];

		/* End of table */
		if (!p1) break;

		/* Feature index */
		p0 = priority_table[i][0];

		/* Access the feature */
		f_ptr = &f_info[p0];

		/* Check character and attribute, accept matches */
		if ((f_ptr->z_char == c) && (f_ptr->z_attr == a)) return (p1);
	}

	/* Default */
	return (20);
}


/*
 * Display a "small-scale" map of the dungeon in the active Term
 *
 * Note that this function must "disable" the special lighting
 * effects so that the "priority" function will work.
 *
 * Note the use of a specialized "priority" function to allow this
 * function to work with any graphic attr/char mappings, and the
 * attempts to optimize this function where possible.
 */
void display_map(int *cy, int *cx)
{
	int py = p_ptr->py;
	int px = p_ptr->px;

	int i, j, x, y;

	byte ta;
	char tc;

	byte tp;

	byte ma[MAP_HGT + 2][MAP_WID + 2];
	char mc[MAP_HGT + 2][MAP_WID + 2];

	byte mp[MAP_HGT + 2][MAP_WID + 2];

	bool old_view_special_lite;
	bool old_view_granite_lite;


	/* Save lighting effects */
	old_view_special_lite = view_special_lite;
	old_view_granite_lite = view_granite_lite;

	/* Disable lighting effects */
	view_special_lite = FALSE;
	view_granite_lite = FALSE;


	/* Clear the chars and attributes */
	for (y = 0; y < MAP_HGT+2; ++y)
	{
		for (x = 0; x < MAP_WID+2; ++x)
		{
			/* Nothing here */
			ma[y][x] = TERM_WHITE;
			mc[y][x] = ' ';

			/* No priority */
			mp[y][x] = 0;
		}
	}

	/* Fill in the map */
	for (i = 0; i < DUNGEON_WID; ++i)
	{
		for (j = 0; j < DUNGEON_HGT; ++j)
		{
			/* Location */
			x = i / RATIO + 1;
			y = j / RATIO + 1;

			/* Extract the current attr/char at that map location */
			map_info(j, i, &ta, &tc);

			/* Extract the priority of that attr/char */
			tp = priority(ta, tc);

			/* Save "best" */
			if (mp[y][x] < tp)
			{
				/* Save the char */
				mc[y][x] = tc;

				/* Save the attr */
				ma[y][x] = ta;

				/* Save priority */
				mp[y][x] = tp;
			}
		}
	}


	/* Corners */
	x = MAP_WID + 1;
	y = MAP_HGT + 1;

	/* Draw the corners */
	mc[0][0] = mc[0][x] = mc[y][0] = mc[y][x] = '+';

	/* Draw the horizontal edges */
	for (x = 1; x <= MAP_WID; x++) mc[0][x] = mc[y][x] = '-';

	/* Draw the vertical edges */
	for (y = 1; y <= MAP_HGT; y++) mc[y][0] = mc[y][x] = '|';


	/* Display each map line in order */
	for (y = 0; y < MAP_HGT+2; ++y)
	{
		/* Start a new line */
		Term_gotoxy(0, y);

		/* Display the line */
		for (x = 0; x < MAP_WID+2; ++x)
		{
			ta = ma[y][x];
			tc = mc[y][x];

			/* Add the character */
			Term_addch(ta, tc);
		}
	}


	/* Player location */
	(*cy) = py / RATIO + 1;
	(*cx) = px / RATIO + 1;


	/* Restore lighting effects */
	view_special_lite = old_view_special_lite;
	view_granite_lite = old_view_granite_lite;
}


/*
 * Display a "small-scale" map of the dungeon.
 *
 * Note that the "player" is always displayed on the map.
 */
void do_cmd_view_map(void)
{
	int cy, cx;

	/* Enter "icky" mode */
	character_icky = TRUE;

	/* Save the screen */
	Term_save();

	/* Note */
	prt("Please wait...", 0, 0);

	/* Flush */
	Term_fresh();

	/* Clear the screen */
	Term_clear();

	/* Display the map */
	display_map(&cy, &cx);

	/* Wait for it */
	put_str("Hit any key to continue", 23, 23);

	/* Hilite the player */
	move_cursor(cy, cx);

	/* Get any key */
	inkey();

	/* Restore the screen */
	Term_load();

	/* Leave "icky" mode */
	character_icky = FALSE;
}



/*
 * Some comments on the dungeon related data structures and functions...
 *
 * Angband is primarily a dungeon exploration game, and it should come as
 * no surprise that the internal representation of the dungeon has evolved
 * over time in much the same way as the game itself, to provide semantic
 * changes to the game itself, to make the code simpler to understand, and
 * to make the executable itself faster or more efficient in various ways.
 *
 * There are a variety of dungeon related data structures, and associated
 * functions, which store information about the dungeon, and provide methods
 * by which this information can be accessed or modified.
 *
 * Some of this information applies to the dungeon as a whole, such as the
 * list of unique monsters which are still alive.  Some of this information
 * only applies to the current dungeon level, such as the current depth, or
 * the list of monsters currently inhabiting the level.  And some of the
 * information only applies to a single grid of the current dungeon level,
 * such as whether the grid is illuminated, or whether the grid contains a
 * monster, or whether the grid can be seen by the player.  If Angband was
 * to be turned into a multi-player game, some of the information currently
 * associated with the dungeon should really be associated with the player,
 * such as whether a given grid is viewable by a given player.
 *
 * One of the major bottlenecks in ancient versions of Angband was in the
 * calculation of "line of sight" from the player to various grids, such
 * as those containing monsters, using the relatively expensive "los()"
 * function.  This was such a nasty bottleneck that a lot of silly things
 * were done to reduce the dependancy on "line of sight", for example, you
 * could not "see" any grids in a lit room until you actually entered the
 * room, at which point every grid in the room became "illuminated" and
 * all of the grids in the room were "memorized" forever.  Other major
 * bottlenecks involved the determination of whether a grid was lit by the
 * player's torch, and whether a grid blocked the player's line of sight.
 * These bottlenecks led to the development of special new functions to
 * optimize issues involved with "line of sight" and "torch lit grids".
 * These optimizations led to entirely new additions to the game, such as
 * the ability to display the player's entire field of view using different
 * colors than were used for the "memorized" portions of the dungeon, and
 * the ability to memorize dark floor grids, but to indicate by the way in
 * which they are displayed that they are not actually illuminated.  And
 * of course many of them simply made the game itself faster or more fun.
 * Also, over time, the definition of "line of sight" has been relaxed to
 * allow the player to see a wider "field of view", which is slightly more
 * realistic, and only slightly more expensive to maintain.
 *
 * Currently, a lot of the information about the dungeon is stored in ways
 * that make it very efficient to access or modify the information, while
 * still attempting to be relatively conservative about memory usage, even
 * if this means that some information is stored in multiple places, or in
 * ways which require the use of special code idioms.  For example, each
 * monster record in the monster array contains the location of the monster,
 * and each cave grid has an index into the monster array, or a zero if no
 * monster is in the grid.  This allows the monster code to efficiently see
 * where the monster is located, while allowing the dungeon code to quickly
 * determine not only if a monster is present in a given grid, but also to
 * find out which monster.  The extra space used to store the information
 * twice is inconsequential compared to the speed increase.
 *
 * Some of the information about the dungeon is used by functions which can
 * constitute the "critical efficiency path" of the game itself, and so the
 * way in which they are stored and accessed has been optimized in order to
 * optimize the game itself.  For example, the "update_view()" function was
 * originally created to speed up the game itself (when the player was not
 * running), but then it took on extra responsibility as the provider of the
 * new "special effects lighting code", and became one of the most important
 * bottlenecks when the player was running.  So many rounds of optimization
 * were performed on both the function itself, and the data structures which
 * it uses, resulting eventually in a function which not only made the game
 * faster than before, but which was responsible for even more calculations
 * (including the determination of which grids are "viewable" by the player,
 * which grids are illuminated by the player's torch, and which grids can be
 * "seen" in some way by the player), as well as for providing the guts of
 * the special effects lighting code, and for the efficient redisplay of any
 * grids whose visual representation may have changed.
 *
 * Several pieces of information about each cave grid are stored in various
 * two dimensional arrays, with one unit of information for each grid in the
 * dungeon.  Some of these arrays have been intentionally expanded by a small
 * factor to make the two dimensional array accesses faster by allowing the
 * use of shifting instead of multiplication.
 *
 * Several pieces of information about each cave grid are stored in the
 * "cave_info" array, which is a special two dimensional array of bytes,
 * one for each cave grid, each containing eight separate "flags" which
 * describe some property of the cave grid.  These flags can be checked and
 * modified extremely quickly, especially when special idioms are used to
 * force the compiler to keep a local register pointing to the base of the
 * array.  Special location offset macros can be used to minimize the number
 * of computations which must be performed at runtime.  Note that using a
 * byte for each flag set may be slightly more efficient than using a larger
 * unit, so if another flag (or two) is needed later, and it must be fast,
 * then the two existing flags which do not have to be fast should be moved
 * out into some other data structure and the new flags should take their
 * place.  This may require a few minor changes in the savefile code.
 *
 * The "CAVE_ROOM" flag is saved in the savefile and is used to determine
 * which grids are part of "rooms", and thus which grids are affected by
 * "illumination" spells.  This flag does not have to be very fast.
 *
 * The "CAVE_ICKY" flag is saved in the savefile and is used to determine
 * which grids are part of "vaults", and thus which grids cannot serve as
 * the destinations of player teleportation.  This flag does not have to
 * be very fast.
 *
 * The "CAVE_MARK" flag is saved in the savefile and is used to determine
 * which grids have been "memorized" by the player.  This flag is used by
 * the "map_info()" function to determine if a grid should be displayed.
 * This flag is used in a few other places to determine if the player can
 * "know" about a given grid.  This flag must be very fast. 
 *
 * The "CAVE_GLOW" flag is saved in the savefile and is used to determine
 * which grids are "permanently illuminated".  This flag is used by the
 * "update_view()" function to help determine which viewable flags may
 * be "seen" by the player.  This flag is used by the "map_info" function
 * to determine if a grid is only lit by the player's torch.  This flag
 * has special semantics for wall grids (see "update_view()").  This flag
 * must be very fast.
 *
 * The "CAVE_WALL" flag is used to determine which grids block the player's
 * line of sight.  This flag is used by the "update_view()" function to
 * determine which grids block line of sight, and to help determine which
 * grids can be "seen" by the player.  This flag must be very fast.
 *
 * The "CAVE_VIEW" flag is used to determine which grids are currently in
 * line of sight of the player.  This flag is set by (and used by) the
 * "update_view()" function.  This flag is used by any code which needs to
 * know if the player can "view" a given grid.  This flag is used by the
 * "map_info()" function for some optional special lighting effects.  The
 * "player_has_los_bold()" macro wraps an abstraction around this flag, but
 * certain code idioms are much more efficient.  This flag is used to check
 * if a modification to a terrain feature might affect the player's field of
 * view.  This flag is used to see if certain monsters are "visible" to the
 * player.  This flag is used to allow any monster in the player's field of
 * view to "sense" the presence of the player.  This flag must be very fast.
 *
 * The "CAVE_SEEN" flag is used to determine which grids are currently in
 * line of sight of the player and also illuminated in some way.  This flag
 * is set by the "update_view()" function, using computations based on the
 * "CAVE_VIEW" and "CAVE_WALL" and "CAVE_GLOW" flags of various grids.  This
 * flag is used by any code which needs to know if the player can "see" a
 * given grid.  This flag is used by the "map_info()" function both to see
 * if a given "boring" grid can be seen by the player, and for some optional
 * special lighting effects.  The "player_can_see_bold()" macro wraps an
 * abstraction around this flag, but certain code idioms are much more
 * efficient.  This flag is used to see if certain monsters are "visible" to
 * the player.  This flag is never set for a grid unless "CAVE_VIEW" is also
 * set for the grid.  Whenever the "CAVE_WALL" or "CAVE_GLOW" flag changes
 * for a grid which has the "CAVE_VIEW" flag set, the "CAVE_SEEN" flag must
 * be recalculated.  The simplest way to do this is to call "forget_view()"
 * and "update_view()" whenever the "CAVE_WALL" or "CAVE_GLOW" flags change
 * for a grid which has "CAVE_VIEW" set.  This flag must be very fast.
 *
 * The "CAVE_TEMP" flag is used for a variety of temporary purposes.  This
 * flag is used to determine if the "CAVE_SEEN" flag for a grid has changed
 * during the "update_view()" function.  This flag is used to "spread" light
 * or darkness through a room.  This flag is used by the "monster flow code".
 * This flag must always be cleared by any code which sets it, often, this
 * can be optimized by the use of the special "temp_g", "temp_y", "temp_x"
 * arrays (and the special "temp_n" global).  This flag must be very fast.
 *
 * Note that the "CAVE_MARK" flag is used for many reasons, some of which
 * are strictly for optimization purposes.  The "CAVE_MARK" flag means that
 * even if the player cannot "see" the grid, he "knows" about the terrain in
 * that grid.  This is used to "memorize" grids when they are first "seen" by
 * the player, and to allow certain grids to be "detected" by certain magic.
 * Note that most grids are always memorized when they are first "seen", but
 * "boring" grids (floor grids) are only memorized if the "view_torch_grids"
 * option is set, or if the "view_perma_grids" option is set, and the grid
 * in question has the "CAVE_GLOW" flag set.
 *
 * Objects are "memorized" in a different way, using a special "marked" flag
 * on the object itself, which is set when an object is observed or detected.
 * This allows objects to be "memorized" independant of the terrain features.
 *
 * The "update_view()" function is an extremely important function.  It is
 * called only when the player moves, significant terrain changes, or the
 * player's blindness or torch radius changes.  Note that when the player
 * is resting, or performing any repeated actions (like digging, disarming,
 * farming, etc), there is no need to call the "update_view()" function, so
 * even if it was not very efficient, this would really only matter when the
 * player was "running" through the dungeon.  It sets the "CAVE_VIEW" flag
 * on every cave grid in the player's field of view, and maintains an array
 * of all such grids in the global "view_g" array.  It also checks the torch
 * radius of the player, and sets the "CAVE_SEEN" flag for every grid which
 * is in the "field of view" of the player and which is also "illuminated",
 * either by the players torch (if any) or by any permanent light source.
 * It could use and help maintain information about multiple light sources,
 * which would be helpful in a multi-player version of Angband.
 *
 * The "update_view()" function maintains the special "view_g" array, which
 * contains exactly those grids which have the "CAVE_VIEW" flag set.  This
 * array is used by "update_view()" to (only) memorize grids which become
 * newly "seen", and to (only) redraw grids whose "seen" value changes, which
 * allows the use of some interesting (and very efficient) "special lighting
 * effects".  In addition, this array could be used elsewhere to quickly scan
 * through all the grids which are in the player's field of view.
 *
 * Note that the "update_view()" function allows, among other things, a room
 * to be "partially" seen as the player approaches it, with a growing cone
 * of floor appearing as the player gets closer to the door.  Also, by not
 * turning on the "memorize perma-lit grids" option, the player will only
 * "see" those floor grids which are actually in line of sight.  And best
 * of all, you can now activate the special lighting effects to indicate
 * which grids are actually in the player's field of view by using dimmer
 * colors for grids which are not in the player's field of view, and/or to
 * indicate which grids are illuminated only by the player's torch by using
 * the color yellow for those grids.
 *
 * The old "update_view()" algorithm uses the special "CAVE_EASY" flag as a
 * temporary internal flag to mark those grids which are not only in view,
 * but which are also "easily" in line of sight of the player.  This flag
 * is actually just the "CAVE_SEEN" flag, and the "update_view()" function
 * makes sure to clear it for all old "CAVE_SEEN" grids, and then use it in
 * the algorithm as "CAVE_EASY", and then clear it for all "CAVE_EASY" grids,
 * and then reset it as appropriate for all new "CAVE_SEEN" grids.  This is
 * kind of messy, but it works.  The old algorithm may disappear eventually.
 *
 * The new "update_view()" algorithm uses a faster and more mathematically
 * correct algorithm, assisted by a large machine generated static array, to
 * determine the "CAVE_VIEW" and "CAVE_SEEN" flags simultaneously.  See below.
 *
 * It seems as though slight modifications to the "update_view()" functions
 * would allow us to determine "reverse" line-of-sight as well as "normal"
 * line-of-sight", which would allow monsters to have a more "correct" way
 * to determine if they can "see" the player, since right now, they "cheat"
 * somewhat and assume that if the player has "line of sight" to them, then
 * they can "pretend" that they have "line of sight" to the player.  But if
 * such a change was attempted, the monsters would actually start to exhibit
 * some undesirable behavior, such as "freezing" near the entrances to long
 * hallways containing the player, and code would have to be added to make
 * the monsters move around even if the player was not detectable, and to
 * "remember" where the player was last seen, to avoid looking stupid.
 *
 * Note that the "CAVE_GLOW" flag means that a grid is permanently lit in
 * some way.  However, for the player to "see" the grid, as determined by
 * the "CAVE_SEEN" flag, the player must not be blind, the grid must have
 * the "CAVE_VIEW" flag set, and if the grid is a "wall" grid, and it is
 * not lit by the player's torch, then it must touch a grid which does not
 * have the "CAVE_WALL" flag set, but which does have both the "CAVE_GLOW"
 * and "CAVE_VIEW" flags set.  This last part about wall grids is induced
 * by the semantics of "CAVE_GLOW" as applied to wall grids, and checking
 * the technical requirements can be very expensive, especially since the
 * grid may be touching some "illegal" grids.  Luckily, it is more or less
 * correct to restrict the "touching" grids from the eight "possible" grids
 * to the (at most) three grids which are touching the grid, and which are
 * closer to the player than the grid itself, which eliminates more than
 * half of the work, including all of the potentially "illegal" grids, if
 * at most one of the three grids is a "diagonal" grid.  In addition, in
 * almost every situation, it is possible to ignore the "CAVE_VIEW" flag
 * on these three "touching" grids, for a variety of technical reasons.
 * Finally, note that in most situations, it is only necessary to check
 * a single "touching" grid, in fact, the grid which is strictly closest
 * to the player of all the touching grids, and in fact, it is normally
 * only necessary to check the "CAVE_GLOW" flag of that grid, again, for
 * various technical reasons.  However, one of the situations which does
 * not work with this last reduction is the very common one in which the
 * player approaches an illuminated room from a dark hallway, in which the
 * two wall grids which form the "entrance" to the room would not be marked
 * as "CAVE_SEEN", since of the three "touching" grids nearer to the player
 * than each wall grid, only the farthest of these grids is itself marked
 * "CAVE_GLOW". 
 *
 *
 * Here are some pictures of the legal "light source" radius values, in
 * which the numbers indicate the "order" in which the grids could have
 * been calculated, if desired.  Note that the code will work with larger
 * radiuses, though currently yields such a radius, and the game would
 * become slower in some situations if it did.
 *
 *       Rad=0     Rad=1      Rad=2        Rad=3
 *      No-Lite  Torch,etc   Lantern     Artifacts
 *    
 *                                          333
 *                             333         43334
 *                  212       32123       3321233
 *         @        1@1       31@13       331@133
 *                  212       32123       3321233
 *                             333         43334
 *                                          333
 *
 *
 * Here is an illustration of the two different "update_view()" algorithms,
 * in which the grids marked "%" are pillars, and the grids marked "?" are
 * not in line of sight of the player.
 *
 *
 *                    Sample situation
 *
 *                  #####################
 *                  ############.%.%.%.%#
 *                  #...@..#####........#
 *                  #............%.%.%.%#
 *                  #......#####........#
 *                  ############........#
 *                  #####################
 *
 *
 *          New Algorithm             Old Algorithm
 *
 *      ########?????????????    ########?????????????
 *      #...@..#?????????????    #...@..#?????????????
 *      #...........?????????    #.........???????????
 *      #......#####.....????    #......####??????????
 *      ########?????????...#    ########?????????????
 *
 *      ########?????????????    ########?????????????
 *      #.@....#?????????????    #.@....#?????????????
 *      #............%???????    #...........?????????
 *      #......#####........?    #......#####?????????
 *      ########??????????..#    ########?????????????
 *
 *      ########?????????????    ########?????%???????
 *      #......#####........#    #......#####..???????
 *      #.@..........%???????    #.@..........%???????
 *      #......#####........#    #......#####..???????
 *      ########?????????????    ########?????????????
 *
 *      ########??????????..#    ########?????????????
 *      #......#####........?    #......#####?????????
 *      #............%???????    #...........?????????
 *      #.@....#?????????????    #.@....#?????????????
 *      ########?????????????    ########?????????????
 *
 *      ########?????????%???    ########?????????????
 *      #......#####.....????    #......####??????????
 *      #...........?????????    #.........???????????
 *      #...@..#?????????????    #...@..#?????????????
 *      ########?????????????    ########?????????????
 */




/*
 * Forget the "CAVE_VIEW" grids, redrawing as needed
 */
void forget_view(void)
{
	int i, g;

	int fast_view_n = view_n;
	u16b *fast_view_g = view_g;

	byte *fast_cave_info = &cave_info[0][0];


	/* None to forget */
	if (!fast_view_n) return;

	/* Clear them all */
	for (i = 0; i < fast_view_n; i++)
	{
		int y, x;

		/* Grid */
		g = fast_view_g[i];

		/* Location */
		y = GRID_Y(g);
		x = GRID_X(g);

		/* Clear "CAVE_VIEW" and "CAVE_SEEN" flags */
		fast_cave_info[g] &= ~(CAVE_VIEW | CAVE_SEEN);

		/* Clear "CAVE_LITE" flag */
		/* fast_cave_info[g] &= ~(CAVE_LITE); */

		/* Redraw */
		lite_spot(y, x);
	}

	/* None left */
	fast_view_n = 0;


	/* Save 'view_n' */
	view_n = fast_view_n;
}



#if defined(UPDATE_VIEW_NEW)


/*
 * This is the new version of the "update_view()" algorithm.
 *
 * See the "update_view()" function (below) for more details.
 */


/* XXX XXX XXX Begin machine generated code XXX XXX XXX */


/* The 'vinfo_type' structure */

typedef struct vinfo_type vinfo_type;

struct vinfo_type
{
	s16b grid_0;
	s16b grid_1;
	s16b grid_2;
	s16b grid_3;
	s16b grid_4;
	s16b grid_5;
	s16b grid_6;
	s16b grid_7;

	u32b bits_3;
	u32b bits_2;
	u32b bits_1;
	u32b bits_0;

	vinfo_type *next_0;
	vinfo_type *next_1;

	byte y;
	byte x;
	byte d;
	byte r;
};


#define VINFO_MAX_GRIDS 161

static vinfo_type vinfo[VINFO_MAX_GRIDS] =
{

	/* vinfo[0] */
	{
		GRID(+0,+0), GRID(+0,+0), GRID(+0,+0), GRID(+0,+0),
		GRID(+0,+0), GRID(+0,+0), GRID(+0,+0), GRID(+0,+0),
		0x00000000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[2], &vinfo[2], 0, 0, 0, 0
	},

	/* vinfo[1] */
	{
		GRID(+0,+1), GRID(+1,+0), GRID(+1,+0), GRID(+0,-1),
		GRID(+0,-1), GRID(-1,+0), GRID(-1,+0), GRID(+0,+1),
		0x1FFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF,
		&vinfo[3], &vinfo[4], 0, 1, 1, 1
	},

	/* vinfo[2] */
	{
		GRID(+1,+1), GRID(+1,+1), GRID(+1,-1), GRID(+1,-1),
		GRID(-1,-1), GRID(-1,-1), GRID(-1,+1), GRID(-1,+1),
		0x3FFFFFFF, 0xFFFFFFFF, 0xFF800000, 0x00000000,
		&vinfo[5], &vinfo[5], 1, 1, 1, 1
	},

	/* vinfo[3] */
	{
		GRID(+0,+2), GRID(+2,+0), GRID(+2,+0), GRID(+0,-2),
		GRID(+0,-2), GRID(-2,+0), GRID(-2,+0), GRID(+0,+2),
		0x00000000, 0x00000000, 0x003FFFFF, 0xFFFFFFFF,
		&vinfo[6], &vinfo[7], 0, 2, 2, 2
	},

	/* vinfo[4] */
	{
		GRID(+1,+2), GRID(+2,+1), GRID(+2,-1), GRID(+1,-2),
		GRID(-1,-2), GRID(-2,-1), GRID(-2,+1), GRID(-1,+2),
		0x1FFFFFFF, 0xFFFFFFFF, 0xFFFFFFF0, 0x00000000,
		&vinfo[7], &vinfo[8], 1, 2, 2, 0
	},

	/* vinfo[5] */
	{
		GRID(+2,+2), GRID(+2,+2), GRID(+2,-2), GRID(+2,-2),
		GRID(-2,-2), GRID(-2,-2), GRID(-2,+2), GRID(-2,+2),
		0x3FFFFFFF, 0xFF000000, 0x00000000, 0x00000000,
		&vinfo[9], &vinfo[9], 2, 2, 3, 2
	},

	/* vinfo[6] */
	{
		GRID(+0,+3), GRID(+3,+0), GRID(+3,+0), GRID(+0,-3),
		GRID(+0,-3), GRID(-3,+0), GRID(-3,+0), GRID(+0,+3),
		0x00000000, 0x00000000, 0x00000007, 0xFFFFFFFF,
		&vinfo[10], &vinfo[11], 0, 3, 3, 3
	},

	/* vinfo[7] */
	{
		GRID(+1,+3), GRID(+3,+1), GRID(+3,-1), GRID(+1,-3),
		GRID(-1,-3), GRID(-3,-1), GRID(-3,+1), GRID(-1,+3),
		0x00000000, 0x007FFFFF, 0xFFFFFFFF, 0xF8000000,
		&vinfo[11], &vinfo[12], 1, 3, 3, 0
	},

	/* vinfo[8] */
	{
		GRID(+2,+3), GRID(+3,+2), GRID(+3,-2), GRID(+2,-3),
		GRID(-2,-3), GRID(-3,-2), GRID(-3,+2), GRID(-2,+3),
		0x1FFFFFFF, 0xFFFFFFF8, 0x00000000, 0x00000000,
		&vinfo[12], &vinfo[13], 2, 3, 4, 0
	},

	/* vinfo[9] */
	{
		GRID(+3,+3), GRID(+3,+3), GRID(+3,-3), GRID(+3,-3),
		GRID(-3,-3), GRID(-3,-3), GRID(-3,+3), GRID(-3,+3),
		0x3FFFFFE0, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[14], &vinfo[14], 3, 3, 4, 3
	},

	/* vinfo[10] */
	{
		GRID(+0,+4), GRID(+4,+0), GRID(+4,+0), GRID(+0,-4),
		GRID(+0,-4), GRID(-4,+0), GRID(-4,+0), GRID(+0,+4),
		0x00000000, 0x00000000, 0x00000000, 0x03FFFFFF,
		&vinfo[15], &vinfo[16], 0, 4, 4, 4
	},

	/* vinfo[11] */
	{
		GRID(+1,+4), GRID(+4,+1), GRID(+4,-1), GRID(+1,-4),
		GRID(-1,-4), GRID(-4,-1), GRID(-4,+1), GRID(-1,+4),
		0x00000000, 0x00000003, 0xFFFFFFFF, 0xFFC00000,
		&vinfo[16], &vinfo[17], 1, 4, 4, 0
	},

	/* vinfo[12] */
	{
		GRID(+2,+4), GRID(+4,+2), GRID(+4,-2), GRID(+2,-4),
		GRID(-2,-4), GRID(-4,-2), GRID(-4,+2), GRID(-2,+4),
		0x0000000F, 0xFFFFFFFF, 0xFF800000, 0x00000000,
		&vinfo[17], &vinfo[18], 2, 4, 5, 0
	},

	/* vinfo[13] */
	{
		GRID(+3,+4), GRID(+4,+3), GRID(+4,-3), GRID(+3,-4),
		GRID(-3,-4), GRID(-4,-3), GRID(-4,+3), GRID(-3,+4),
		0x1FFFFFFF, 0xFFF80000, 0x00000000, 0x00000000,
		&vinfo[18], &vinfo[19], 3, 4, 5, 0
	},

	/* vinfo[14] */
	{
		GRID(+4,+4), GRID(+4,+4), GRID(+4,-4), GRID(+4,-4),
		GRID(-4,-4), GRID(-4,-4), GRID(-4,+4), GRID(-4,+4),
		0x3FFFF000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[20], &vinfo[20], 4, 4, 6, 4
	},

	/* vinfo[15] */
	{
		GRID(+0,+5), GRID(+5,+0), GRID(+5,+0), GRID(+0,-5),
		GRID(+0,-5), GRID(-5,+0), GRID(-5,+0), GRID(+0,+5),
		0x00000000, 0x00000000, 0x00000000, 0x001FFFFF,
		&vinfo[21], &vinfo[22], 0, 5, 5, 5
	},

	/* vinfo[16] */
	{
		GRID(+1,+5), GRID(+5,+1), GRID(+5,-1), GRID(+1,-5),
		GRID(-1,-5), GRID(-5,-1), GRID(-5,+1), GRID(-1,+5),
		0x00000000, 0x00000000, 0x003FFFFF, 0xFFF80000,
		&vinfo[22], &vinfo[23], 1, 5, 5, 0
	},

	/* vinfo[17] */
	{
		GRID(+2,+5), GRID(+5,+2), GRID(+5,-2), GRID(+2,-5),
		GRID(-2,-5), GRID(-5,-2), GRID(-5,+2), GRID(-2,+5),
		0x00000000, 0x0003FFFF, 0xFFFF8000, 0x00000000,
		&vinfo[23], &vinfo[24], 2, 5, 6, 0
	},

	/* vinfo[18] */
	{
		GRID(+3,+5), GRID(+5,+3), GRID(+5,-3), GRID(+3,-5),
		GRID(-3,-5), GRID(-5,-3), GRID(-5,+3), GRID(-3,+5),
		0x000007FF, 0xFFFFFFC0, 0x00000000, 0x00000000,
		&vinfo[24], &vinfo[25], 3, 5, 6, 0
	},

	/* vinfo[19] */
	{
		GRID(+4,+5), GRID(+5,+4), GRID(+5,-4), GRID(+4,-5),
		GRID(-4,-5), GRID(-5,-4), GRID(-5,+4), GRID(-4,+5),
		0x1FFFFFFF, 0xF0000000, 0x00000000, 0x00000000,
		&vinfo[25], &vinfo[26], 4, 5, 7, 0
	},

	/* vinfo[20] */
	{
		GRID(+5,+5), GRID(+5,+5), GRID(+5,-5), GRID(+5,-5),
		GRID(-5,-5), GRID(-5,-5), GRID(-5,+5), GRID(-5,+5),
		0x3FFF0000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[27], &vinfo[27], 5, 5, 7, 5
	},

	/* vinfo[21] */
	{
		GRID(+0,+6), GRID(+6,+0), GRID(+6,+0), GRID(+0,-6),
		GRID(+0,-6), GRID(-6,+0), GRID(-6,+0), GRID(+0,+6),
		0x00000000, 0x00000000, 0x00000000, 0x0003FFFF,
		&vinfo[28], &vinfo[29], 0, 6, 6, 6
	},

	/* vinfo[22] */
	{
		GRID(+1,+6), GRID(+6,+1), GRID(+6,-1), GRID(+1,-6),
		GRID(-1,-6), GRID(-6,-1), GRID(-6,+1), GRID(-1,+6),
		0x00000000, 0x00000000, 0x00003FFF, 0xFFFF0000,
		&vinfo[29], &vinfo[30], 1, 6, 6, 0
	},

	/* vinfo[23] */
	{
		GRID(+2,+6), GRID(+6,+2), GRID(+6,-2), GRID(+2,-6),
		GRID(-2,-6), GRID(-6,-2), GRID(-6,+2), GRID(-2,+6),
		0x00000000, 0x0000001F, 0xFFFFFF00, 0x00000000,
		&vinfo[30], &vinfo[31], 2, 6, 7, 0
	},

	/* vinfo[24] */
	{
		GRID(+3,+6), GRID(+6,+3), GRID(+6,-3), GRID(+3,-6),
		GRID(-3,-6), GRID(-6,-3), GRID(-6,+3), GRID(-3,+6),
		0x00000000, 0x07FFFFFF, 0xE0000000, 0x00000000,
		&vinfo[31], &vinfo[32], 3, 6, 7, 0
	},

	/* vinfo[25] */
	{
		GRID(+4,+6), GRID(+6,+4), GRID(+6,-4), GRID(+4,-6),
		GRID(-4,-6), GRID(-6,-4), GRID(-6,+4), GRID(-4,+6),
		0x00007FFF, 0xFFFE0000, 0x00000000, 0x00000000,
		&vinfo[32], &vinfo[33], 4, 6, 8, 0
	},

	/* vinfo[26] */
	{
		GRID(+5,+6), GRID(+6,+5), GRID(+6,-5), GRID(+5,-6),
		GRID(-5,-6), GRID(-6,-5), GRID(-6,+5), GRID(-5,+6),
		0x1FFFFFF8, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[33], &vinfo[34], 5, 6, 8, 0
	},

	/* vinfo[27] */
	{
		GRID(+6,+6), GRID(+6,+6), GRID(+6,-6), GRID(+6,-6),
		GRID(-6,-6), GRID(-6,-6), GRID(-6,+6), GRID(-6,+6),
		0x3FF80000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[35], &vinfo[35], 6, 6, 9, 6
	},

	/* vinfo[28] */
	{
		GRID(+0,+7), GRID(+7,+0), GRID(+7,+0), GRID(+0,-7),
		GRID(+0,-7), GRID(-7,+0), GRID(-7,+0), GRID(+0,+7),
		0x00000000, 0x00000000, 0x00000000, 0x00007FFF,
		&vinfo[36], &vinfo[37], 0, 7, 7, 7
	},

	/* vinfo[29] */
	{
		GRID(+1,+7), GRID(+7,+1), GRID(+7,-1), GRID(+1,-7),
		GRID(-1,-7), GRID(-7,-1), GRID(-7,+1), GRID(-1,+7),
		0x00000000, 0x00000000, 0x0000007F, 0xFFFFC000,
		&vinfo[37], &vinfo[38], 1, 7, 7, 0
	},

	/* vinfo[30] */
	{
		GRID(+2,+7), GRID(+7,+2), GRID(+7,-2), GRID(+2,-7),
		GRID(-2,-7), GRID(-7,-2), GRID(-7,+2), GRID(-2,+7),
		0x00000000, 0x00000000, 0x0FFFFFF0, 0x00000000,
		&vinfo[38], &vinfo[39], 2, 7, 8, 0
	},

	/* vinfo[31] */
	{
		GRID(+3,+7), GRID(+7,+3), GRID(+7,-3), GRID(+3,-7),
		GRID(-3,-7), GRID(-7,-3), GRID(-7,+3), GRID(-3,+7),
		0x00000000, 0x0000FFFF, 0xFF800000, 0x00000000,
		&vinfo[39], &vinfo[40], 3, 7, 8, 0
	},

	/* vinfo[32] */
	{
		GRID(+4,+7), GRID(+7,+4), GRID(+7,-4), GRID(+4,-7),
		GRID(-4,-7), GRID(-7,-4), GRID(-7,+4), GRID(-4,+7),
		0x00000003, 0xFFFFFF80, 0x00000000, 0x00000000,
		&vinfo[40], &vinfo[41], 4, 7, 9, 0
	},

	/* vinfo[33] */
	{
		GRID(+5,+7), GRID(+7,+5), GRID(+7,-5), GRID(+5,-7),
		GRID(-5,-7), GRID(-7,-5), GRID(-7,+5), GRID(-5,+7),
		0x0003FFFF, 0xFF000000, 0x00000000, 0x00000000,
		&vinfo[41], &vinfo[42], 5, 7, 9, 0
	},

	/* vinfo[34] */
	{
		GRID(+6,+7), GRID(+7,+6), GRID(+7,-6), GRID(+6,-7),
		GRID(-6,-7), GRID(-7,-6), GRID(-7,+6), GRID(-6,+7),
		0x1FFFFF80, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[42], &vinfo[43], 6, 7, 10, 0
	},

	/* vinfo[35] */
	{
		GRID(+7,+7), GRID(+7,+7), GRID(+7,-7), GRID(+7,-7),
		GRID(-7,-7), GRID(-7,-7), GRID(-7,+7), GRID(-7,+7),
		0x3FC00000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[44], &vinfo[44], 7, 7, 10, 7
	},

	/* vinfo[36] */
	{
		GRID(+0,+8), GRID(+8,+0), GRID(+8,+0), GRID(+0,-8),
		GRID(+0,-8), GRID(-8,+0), GRID(-8,+0), GRID(+0,+8),
		0x00000000, 0x00000000, 0x00000000, 0x00001FFF,
		&vinfo[45], &vinfo[46], 0, 8, 8, 8
	},

	/* vinfo[37] */
	{
		GRID(+1,+8), GRID(+8,+1), GRID(+8,-1), GRID(+1,-8),
		GRID(-1,-8), GRID(-8,-1), GRID(-8,+1), GRID(-1,+8),
		0x00000000, 0x00000000, 0x00000007, 0xFFFFE000,
		&vinfo[46], &vinfo[47], 1, 8, 8, 0
	},

	/* vinfo[38] */
	{
		GRID(+2,+8), GRID(+8,+2), GRID(+8,-2), GRID(+2,-8),
		GRID(-2,-8), GRID(-8,-2), GRID(-8,+2), GRID(-2,+8),
		0x00000000, 0x00000000, 0x003FFFFF, 0x00000000,
		&vinfo[47], &vinfo[48], 2, 8, 9, 0
	},

	/* vinfo[39] */
	{
		GRID(+3,+8), GRID(+8,+3), GRID(+8,-3), GRID(+3,-8),
		GRID(-3,-8), GRID(-8,-3), GRID(-8,+3), GRID(-3,+8),
		0x00000000, 0x0000003F, 0xFFFC0000, 0x00000000,
		&vinfo[48], &vinfo[49], 3, 8, 9, 0
	},

	/* vinfo[40] */
	{
		GRID(+4,+8), GRID(+8,+4), GRID(+8,-4), GRID(+4,-8),
		GRID(-4,-8), GRID(-8,-4), GRID(-8,+4), GRID(-4,+8),
		0x00000000, 0x007FFFFE, 0x00000000, 0x00000000,
		&vinfo[49], &vinfo[50], 4, 8, 10, 0
	},

	/* vinfo[41] */
	{
		GRID(+5,+8), GRID(+8,+5), GRID(+8,-5), GRID(+5,-8),
		GRID(-5,-8), GRID(-8,-5), GRID(-8,+5), GRID(-5,+8),
		0x0000003F, 0xFFFF0000, 0x00000000, 0x00000000,
		&vinfo[50], &vinfo[51], 5, 8, 10, 0
	},

	/* vinfo[42] */
	{
		GRID(+6,+8), GRID(+8,+6), GRID(+8,-6), GRID(+6,-8),
		GRID(-6,-8), GRID(-8,-6), GRID(-8,+6), GRID(-6,+8),
		0x001FFFFF, 0xE0000000, 0x00000000, 0x00000000,
		&vinfo[51], &vinfo[52], 6, 8, 11, 0
	},

	/* vinfo[43] */
	{
		GRID(+7,+8), GRID(+8,+7), GRID(+8,-7), GRID(+7,-8),
		GRID(-7,-8), GRID(-8,-7), GRID(-8,+7), GRID(-7,+8),
		0x1FFFF800, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[52], &vinfo[53], 7, 8, 11, 0
	},

	/* vinfo[44] */
	{
		GRID(+8,+8), GRID(+8,+8), GRID(+8,-8), GRID(+8,-8),
		GRID(-8,-8), GRID(-8,-8), GRID(-8,+8), GRID(-8,+8),
		0x3F800000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[54], &vinfo[54], 8, 8, 12, 8
	},

	/* vinfo[45] */
	{
		GRID(+0,+9), GRID(+9,+0), GRID(+9,+0), GRID(+0,-9),
		GRID(+0,-9), GRID(-9,+0), GRID(-9,+0), GRID(+0,+9),
		0x00000000, 0x00000000, 0x00000000, 0x00000FFF,
		&vinfo[55], &vinfo[56], 0, 9, 9, 9
	},

	/* vinfo[46] */
	{
		GRID(+1,+9), GRID(+9,+1), GRID(+9,-1), GRID(+1,-9),
		GRID(-1,-9), GRID(-9,-1), GRID(-9,+1), GRID(-1,+9),
		0x00000000, 0x00000000, 0x00000000, 0x7FFFF000,
		&vinfo[56], &vinfo[57], 1, 9, 9, 0
	},

	/* vinfo[47] */
	{
		GRID(+2,+9), GRID(+9,+2), GRID(+9,-2), GRID(+2,-9),
		GRID(-2,-9), GRID(-9,-2), GRID(-9,+2), GRID(-2,+9),
		0x00000000, 0x00000000, 0x0001FFFF, 0xE0000000,
		&vinfo[57], &vinfo[58], 2, 9, 10, 0
	},

	/* vinfo[48] */
	{
		GRID(+3,+9), GRID(+9,+3), GRID(+9,-3), GRID(+3,-9),
		GRID(-3,-9), GRID(-9,-3), GRID(-9,+3), GRID(-3,+9),
		0x00000000, 0x00000000, 0xFFFFC000, 0x00000000,
		&vinfo[58], &vinfo[59], 3, 9, 10, 0
	},

	/* vinfo[49] */
	{
		GRID(+4,+9), GRID(+9,+4), GRID(+9,-4), GRID(+4,-9),
		GRID(-4,-9), GRID(-9,-4), GRID(-9,+4), GRID(-4,+9),
		0x00000000, 0x00007FFF, 0xFC000000, 0x00000000,
		&vinfo[59], &vinfo[60], 4, 9, 11, 0
	},

	/* vinfo[50] */
	{
		GRID(+5,+9), GRID(+9,+5), GRID(+9,-5), GRID(+5,-9),
		GRID(-5,-9), GRID(-9,-5), GRID(-9,+5), GRID(-5,+9),
		0x00000000, 0x0FFFFF00, 0x00000000, 0x00000000,
		&vinfo[60], &vinfo[61], 5, 9, 11, 0
	},

	/* vinfo[51] */
	{
		GRID(+6,+9), GRID(+9,+6), GRID(+9,-6), GRID(+6,-9),
		GRID(-6,-9), GRID(-9,-6), GRID(-9,+6), GRID(-6,+9),
		0x000003FF, 0xFFC00000, 0x00000000, 0x00000000,
		&vinfo[61], &vinfo[62], 6, 9, 12, 0
	},

	/* vinfo[52] */
	{
		GRID(+7,+9), GRID(+9,+7), GRID(+9,-7), GRID(+7,-9),
		GRID(-7,-9), GRID(-9,-7), GRID(-9,+7), GRID(-7,+9),
		0x003FFFFC, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[62], &vinfo[63], 7, 9, 12, 0
	},

	/* vinfo[53] */
	{
		GRID(+8,+9), GRID(+9,+8), GRID(+9,-8), GRID(+8,-9),
		GRID(-8,-9), GRID(-9,-8), GRID(-9,+8), GRID(-8,+9),
		0x1FFFE000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[63], &vinfo[64], 8, 9, 13, 0
	},

	/* vinfo[54] */
	{
		GRID(+9,+9), GRID(+9,+9), GRID(+9,-9), GRID(+9,-9),
		GRID(-9,-9), GRID(-9,-9), GRID(-9,+9), GRID(-9,+9),
		0x3F000000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[65], &vinfo[65], 9, 9, 13, 9
	},

	/* vinfo[55] */
	{
		GRID(+0,+10), GRID(+10,+0), GRID(+10,+0), GRID(+0,-10),
		GRID(+0,-10), GRID(-10,+0), GRID(-10,+0), GRID(+0,+10),
		0x00000000, 0x00000000, 0x00000000, 0x000007FF,
		&vinfo[66], &vinfo[67], 0, 10, 10, 10
	},

	/* vinfo[56] */
	{
		GRID(+1,+10), GRID(+10,+1), GRID(+10,-1), GRID(+1,-10),
		GRID(-1,-10), GRID(-10,-1), GRID(-10,+1), GRID(-1,+10),
		0x00000000, 0x00000000, 0x00000000, 0x0FFFF800,
		&vinfo[67], &vinfo[68], 1, 10, 10, 0
	},

	/* vinfo[57] */
	{
		GRID(+2,+10), GRID(+10,+2), GRID(+10,-2), GRID(+2,-10),
		GRID(-2,-10), GRID(-10,-2), GRID(-10,+2), GRID(-2,+10),
		0x00000000, 0x00000000, 0x00001FFF, 0xF8000000,
		&vinfo[68], &vinfo[69], 2, 10, 11, 0
	},

	/* vinfo[58] */
	{
		GRID(+3,+10), GRID(+10,+3), GRID(+10,-3), GRID(+3,-10),
		GRID(-3,-10), GRID(-10,-3), GRID(-10,+3), GRID(-3,+10),
		0x00000000, 0x00000000, 0x01FFFE00, 0x00000000,
		&vinfo[69], &vinfo[70], 3, 10, 11, 0
	},

	/* vinfo[59] */
	{
		GRID(+4,+10), GRID(+10,+4), GRID(+10,-4), GRID(+4,-10),
		GRID(-4,-10), GRID(-10,-4), GRID(-10,+4), GRID(-4,+10),
		0x00000000, 0x0000007F, 0xFF800000, 0x00000000,
		&vinfo[70], &vinfo[71], 4, 10, 12, 0
	},

	/* vinfo[60] */
	{
		GRID(+5,+10), GRID(+10,+5), GRID(+10,-5), GRID(+5,-10),
		GRID(-5,-10), GRID(-10,-5), GRID(-10,+5), GRID(-5,+10),
		0x00000000, 0x001FFFF8, 0x00000000, 0x00000000,
		&vinfo[71], &vinfo[72], 5, 10, 12, 0
	},

	/* vinfo[61] */
	{
		GRID(+6,+10), GRID(+10,+6), GRID(+10,-6), GRID(+6,-10),
		GRID(-6,-10), GRID(-10,-6), GRID(-10,+6), GRID(-6,+10),
		0x00000001, 0xFFFF8000, 0x00000000, 0x00000000,
		&vinfo[72], &vinfo[73], 6, 10, 13, 0
	},

	/* vinfo[62] */
	{
		GRID(+7,+10), GRID(+10,+7), GRID(+10,-7), GRID(+7,-10),
		GRID(-7,-10), GRID(-10,-7), GRID(-10,+7), GRID(-7,+10),
		0x00000FFF, 0xFC000000, 0x00000000, 0x00000000,
		&vinfo[73], &vinfo[74], 7, 10, 13, 0
	},

	/* vinfo[63] */
	{
		GRID(+8,+10), GRID(+10,+8), GRID(+10,-8), GRID(+8,-10),
		GRID(-8,-10), GRID(-10,-8), GRID(-10,+8), GRID(-8,+10),
		0x007FFFE0, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[74], &vinfo[75], 8, 10, 14, 0
	},

	/* vinfo[64] */
	{
		GRID(+9,+10), GRID(+10,+9), GRID(+10,-9), GRID(+9,-10),
		GRID(-9,-10), GRID(-10,-9), GRID(-10,+9), GRID(-9,+10),
		0x1FFF8000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[75], &vinfo[76], 9, 10, 14, 0
	},

	/* vinfo[65] */
	{
		GRID(+10,+10), GRID(+10,+10), GRID(+10,-10), GRID(+10,-10),
		GRID(-10,-10), GRID(-10,-10), GRID(-10,+10), GRID(-10,+10),
		0x3E000000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[77], &vinfo[77], 10, 10, 15, 10
	},

	/* vinfo[66] */
	{
		GRID(+0,+11), GRID(+11,+0), GRID(+11,+0), GRID(+0,-11),
		GRID(+0,-11), GRID(-11,+0), GRID(-11,+0), GRID(+0,+11),
		0x00000000, 0x00000000, 0x00000000, 0x000003FF,
		&vinfo[78], &vinfo[79], 0, 11, 11, 11
	},

	/* vinfo[67] */
	{
		GRID(+1,+11), GRID(+11,+1), GRID(+11,-1), GRID(+1,-11),
		GRID(-1,-11), GRID(-11,-1), GRID(-11,+1), GRID(-1,+11),
		0x00000000, 0x00000000, 0x00000000, 0x03FFFC00,
		&vinfo[79], &vinfo[80], 1, 11, 11, 0
	},

	/* vinfo[68] */
	{
		GRID(+2,+11), GRID(+11,+2), GRID(+11,-2), GRID(+2,-11),
		GRID(-2,-11), GRID(-11,-2), GRID(-11,+2), GRID(-2,+11),
		0x00000000, 0x00000000, 0x000000FF, 0xFE000000,
		&vinfo[80], &vinfo[81], 2, 11, 12, 0
	},

	/* vinfo[69] */
	{
		GRID(+3,+11), GRID(+11,+3), GRID(+11,-3), GRID(+3,-11),
		GRID(-3,-11), GRID(-11,-3), GRID(-11,+3), GRID(-3,+11),
		0x00000000, 0x00000000, 0x003FFFC0, 0x00000000,
		&vinfo[81], &vinfo[82], 3, 11, 12, 0
	},

	/* vinfo[70] */
	{
		GRID(+4,+11), GRID(+11,+4), GRID(+11,-4), GRID(+4,-11),
		GRID(-4,-11), GRID(-11,-4), GRID(-11,+4), GRID(-4,+11),
		0x00000000, 0x00000003, 0xFFF00000, 0x00000000,
		&vinfo[82], &vinfo[83], 4, 11, 13, 0
	},

	/* vinfo[71] */
	{
		GRID(+5,+11), GRID(+11,+5), GRID(+11,-5), GRID(+5,-11),
		GRID(-5,-11), GRID(-11,-5), GRID(-11,+5), GRID(-5,+11),
		0x00000000, 0x00003FFF, 0xC0000000, 0x00000000,
		&vinfo[83], &vinfo[84], 5, 11, 13, 0
	},

	/* vinfo[72] */
	{
		GRID(+6,+11), GRID(+11,+6), GRID(+11,-6), GRID(+6,-11),
		GRID(-6,-11), GRID(-11,-6), GRID(-11,+6), GRID(-6,+11),
		0x00000000, 0x01FFFE00, 0x00000000, 0x00000000,
		&vinfo[84], &vinfo[85], 6, 11, 14, 0
	},

	/* vinfo[73] */
	{
		GRID(+7,+11), GRID(+11,+7), GRID(+11,-7), GRID(+7,-11),
		GRID(-7,-11), GRID(-11,-7), GRID(-11,+7), GRID(-7,+11),
		0x0000000F, 0xFFF00000, 0x00000000, 0x00000000,
		&vinfo[85], &vinfo[86], 7, 11, 14, 0
	},

	/* vinfo[74] */
	{
		GRID(+8,+11), GRID(+11,+8), GRID(+11,-8), GRID(+8,-11),
		GRID(-8,-11), GRID(-11,-8), GRID(-11,+8), GRID(-8,+11),
		0x00003FFF, 0xC0000000, 0x00000000, 0x00000000,
		&vinfo[86], &vinfo[87], 8, 11, 15, 0
	},

	/* vinfo[75] */
	{
		GRID(+9,+11), GRID(+11,+9), GRID(+11,-9), GRID(+9,-11),
		GRID(-9,-11), GRID(-11,-9), GRID(-11,+9), GRID(-9,+11),
		0x00FFFF00, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[87], &vinfo[88], 9, 11, 15, 0
	},

	/* vinfo[76] */
	{
		GRID(+10,+11), GRID(+11,+10), GRID(+11,-10), GRID(+10,-11),
		GRID(-10,-11), GRID(-11,-10), GRID(-11,+10), GRID(-10,+11),
		0x1FFE0000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[88], &vinfo[89], 10, 11, 16, 0
	},

	/* vinfo[77] */
	{
		GRID(+11,+11), GRID(+11,+11), GRID(+11,-11), GRID(+11,-11),
		GRID(-11,-11), GRID(-11,-11), GRID(-11,+11), GRID(-11,+11),
		0x3C000000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[90], &vinfo[90], 11, 11, 16, 11
	},

	/* vinfo[78] */
	{
		GRID(+0,+12), GRID(+12,+0), GRID(+12,+0), GRID(+0,-12),
		GRID(+0,-12), GRID(-12,+0), GRID(-12,+0), GRID(+0,+12),
		0x00000000, 0x00000000, 0x00000000, 0x000001FF,
		&vinfo[91], &vinfo[92], 0, 12, 12, 12
	},

	/* vinfo[79] */
	{
		GRID(+1,+12), GRID(+12,+1), GRID(+12,-1), GRID(+1,-12),
		GRID(-1,-12), GRID(-12,-1), GRID(-12,+1), GRID(-1,+12),
		0x00000000, 0x00000000, 0x00000000, 0x00FFFE00,
		&vinfo[92], &vinfo[93], 1, 12, 12, 0
	},

	/* vinfo[80] */
	{
		GRID(+2,+12), GRID(+12,+2), GRID(+12,-2), GRID(+2,-12),
		GRID(-2,-12), GRID(-12,-2), GRID(-12,+2), GRID(-2,+12),
		0x00000000, 0x00000000, 0x0000001F, 0xFF800000,
		&vinfo[93], &vinfo[94], 2, 12, 13, 0
	},

	/* vinfo[81] */
	{
		GRID(+3,+12), GRID(+12,+3), GRID(+12,-3), GRID(+3,-12),
		GRID(-3,-12), GRID(-12,-3), GRID(-12,+3), GRID(-3,+12),
		0x00000000, 0x00000000, 0x0007FFF0, 0x00000000,
		&vinfo[94], &vinfo[95], 3, 12, 13, 0
	},

	/* vinfo[82] */
	{
		GRID(+4,+12), GRID(+12,+4), GRID(+12,-4), GRID(+4,-12),
		GRID(-4,-12), GRID(-12,-4), GRID(-12,+4), GRID(-4,+12),
		0x00000000, 0x00000000, 0x1FFF0000, 0x00000000,
		&vinfo[95], &vinfo[96], 4, 12, 14, 0
	},

	/* vinfo[83] */
	{
		GRID(+5,+12), GRID(+12,+5), GRID(+12,-5), GRID(+5,-12),
		GRID(-5,-12), GRID(-12,-5), GRID(-12,+5), GRID(-5,+12),
		0x00000000, 0x000000FF, 0xFE000000, 0x00000000,
		&vinfo[96], &vinfo[97], 5, 12, 14, 0
	},

	/* vinfo[84] */
	{
		GRID(+6,+12), GRID(+12,+6), GRID(+12,-6), GRID(+6,-12),
		GRID(-6,-12), GRID(-12,-6), GRID(-12,+6), GRID(-6,+12),
		0x00000000, 0x0007FFF0, 0x00000000, 0x00000000,
		&vinfo[97], &vinfo[98], 6, 12, 15, 0
	},

	/* vinfo[85] */
	{
		GRID(+7,+12), GRID(+12,+7), GRID(+12,-7), GRID(+7,-12),
		GRID(-7,-12), GRID(-12,-7), GRID(-12,+7), GRID(-7,+12),
		0x00000000, 0x1FFFC000, 0x00000000, 0x00000000,
		&vinfo[98], &vinfo[99], 7, 12, 15, 0
	},

	/* vinfo[86] */
	{
		GRID(+8,+12), GRID(+12,+8), GRID(+12,-8), GRID(+8,-12),
		GRID(-8,-12), GRID(-12,-8), GRID(-12,+8), GRID(-8,+12),
		0x0000007F, 0xFF000000, 0x00000000, 0x00000000,
		&vinfo[99], &vinfo[100], 8, 12, 16, 0
	},

	/* vinfo[87] */
	{
		GRID(+9,+12), GRID(+12,+9), GRID(+12,-9), GRID(+9,-12),
		GRID(-9,-12), GRID(-12,-9), GRID(-12,+9), GRID(-9,+12),
		0x0000FFFE, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[100], &vinfo[101], 9, 12, 16, 0
	},

	/* vinfo[88] */
	{
		GRID(+10,+12), GRID(+12,+10), GRID(+12,-10), GRID(+10,-12),
		GRID(-10,-12), GRID(-12,-10), GRID(-12,+10), GRID(-10,+12),
		0x01FFFC00, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[101], &vinfo[102], 10, 12, 17, 0
	},

	/* vinfo[89] */
	{
		GRID(+11,+12), GRID(+12,+11), GRID(+12,-11), GRID(+11,-12),
		GRID(-11,-12), GRID(-12,-11), GRID(-12,+11), GRID(-11,+12),
		0x1FFC0000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[102], &vinfo[103], 11, 12, 17, 0
	},

	/* vinfo[90] */
	{
		GRID(+12,+12), GRID(+12,+12), GRID(+12,-12), GRID(+12,-12),
		GRID(-12,-12), GRID(-12,-12), GRID(-12,+12), GRID(-12,+12),
		0x38000000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[104], &vinfo[104], 12, 12, 18, 12
	},

	/* vinfo[91] */
	{
		GRID(+0,+13), GRID(+13,+0), GRID(+13,+0), GRID(+0,-13),
		GRID(+0,-13), GRID(-13,+0), GRID(-13,+0), GRID(+0,+13),
		0x00000000, 0x00000000, 0x00000000, 0x000000FF,
		&vinfo[105], &vinfo[106], 0, 13, 13, 13
	},

	/* vinfo[92] */
	{
		GRID(+1,+13), GRID(+13,+1), GRID(+13,-1), GRID(+1,-13),
		GRID(-1,-13), GRID(-13,-1), GRID(-13,+1), GRID(-1,+13),
		0x00000000, 0x00000000, 0x00000000, 0x003FFF00,
		&vinfo[106], &vinfo[107], 1, 13, 13, 0
	},

	/* vinfo[93] */
	{
		GRID(+2,+13), GRID(+13,+2), GRID(+13,-2), GRID(+2,-13),
		GRID(-2,-13), GRID(-13,-2), GRID(-13,+2), GRID(-2,+13),
		0x00000000, 0x00000000, 0x00000007, 0xFFC00000,
		&vinfo[107], &vinfo[108], 2, 13, 14, 0
	},

	/* vinfo[94] */
	{
		GRID(+3,+13), GRID(+13,+3), GRID(+13,-3), GRID(+3,-13),
		GRID(-3,-13), GRID(-13,-3), GRID(-13,+3), GRID(-3,+13),
		0x00000000, 0x00000000, 0x00007FFC, 0x00000000,
		&vinfo[108], &vinfo[109], 3, 13, 14, 0
	},

	/* vinfo[95] */
	{
		GRID(+4,+13), GRID(+13,+4), GRID(+13,-4), GRID(+4,-13),
		GRID(-4,-13), GRID(-13,-4), GRID(-13,+4), GRID(-4,+13),
		0x00000000, 0x00000000, 0x00FFE000, 0x00000000,
		&vinfo[109], &vinfo[110], 4, 13, 15, 0
	},

	/* vinfo[96] */
	{
		GRID(+5,+13), GRID(+13,+5), GRID(+13,-5), GRID(+5,-13),
		GRID(-5,-13), GRID(-13,-5), GRID(-13,+5), GRID(-5,+13),
		0x00000000, 0x00000007, 0xFF800000, 0x00000000,
		&vinfo[110], &vinfo[111], 5, 13, 15, 0
	},

	/* vinfo[97] */
	{
		GRID(+6,+13), GRID(+13,+6), GRID(+13,-6), GRID(+6,-13),
		GRID(-6,-13), GRID(-13,-6), GRID(-13,+6), GRID(-6,+13),
		0x00000000, 0x00001FFF, 0x00000000, 0x00000000,
		&vinfo[111], &vinfo[112], 6, 13, 16, 0
	},

	/* vinfo[98] */
	{
		GRID(+7,+13), GRID(+13,+7), GRID(+13,-7), GRID(+7,-13),
		GRID(-7,-13), GRID(-13,-7), GRID(-13,+7), GRID(-7,+13),
		0x00000000, 0x007FFC00, 0x00000000, 0x00000000,
		&vinfo[112], &vinfo[113], 7, 13, 16, 0
	},

	/* vinfo[99] */
	{
		GRID(+8,+13), GRID(+13,+8), GRID(+13,-8), GRID(+8,-13),
		GRID(-8,-13), GRID(-13,-8), GRID(-13,+8), GRID(-8,+13),
		0x00000000, 0xFFF80000, 0x00000000, 0x00000000,
		&vinfo[113], &vinfo[114], 8, 13, 17, 0
	},

	/* vinfo[100] */
	{
		GRID(+9,+13), GRID(+13,+9), GRID(+13,-9), GRID(+9,-13),
		GRID(-9,-13), GRID(-13,-9), GRID(-13,+9), GRID(-9,+13),
		0x000001FF, 0xF8000000, 0x00000000, 0x00000000,
		&vinfo[114], &vinfo[115], 9, 13, 17, 0
	},

	/* vinfo[101] */
	{
		GRID(+10,+13), GRID(+13,+10), GRID(+13,-10), GRID(+10,-13),
		GRID(-10,-13), GRID(-13,-10), GRID(-13,+10), GRID(-10,+13),
		0x0001FFF0, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[115], &vinfo[116], 10, 13, 18, 0
	},

	/* vinfo[102] */
	{
		GRID(+11,+13), GRID(+13,+11), GRID(+13,-11), GRID(+11,-13),
		GRID(-11,-13), GRID(-13,-11), GRID(-13,+11), GRID(-11,+13),
		0x03FFF000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[116], &vinfo[117], 11, 13, 18, 0
	},

	/* vinfo[103] */
	{
		GRID(+12,+13), GRID(+13,+12), GRID(+13,-12), GRID(+12,-13),
		GRID(-12,-13), GRID(-13,-12), GRID(-13,+12), GRID(-12,+13),
		0x1FF00000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[117], &vinfo[118], 12, 13, 19, 0
	},

	/* vinfo[104] */
	{
		GRID(+13,+13), GRID(+13,+13), GRID(+13,-13), GRID(+13,-13),
		GRID(-13,-13), GRID(-13,-13), GRID(-13,+13), GRID(-13,+13),
		0x30000000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[0], &vinfo[0], 13, 13, 19, 13
	},

	/* vinfo[105] */
	{
		GRID(+0,+14), GRID(+14,+0), GRID(+14,+0), GRID(+0,-14),
		GRID(+0,-14), GRID(-14,+0), GRID(-14,+0), GRID(+0,+14),
		0x00000000, 0x00000000, 0x00000000, 0x0000007F,
		&vinfo[119], &vinfo[120], 0, 14, 14, 14
	},

	/* vinfo[106] */
	{
		GRID(+1,+14), GRID(+14,+1), GRID(+14,-1), GRID(+1,-14),
		GRID(-1,-14), GRID(-14,-1), GRID(-14,+1), GRID(-1,+14),
		0x00000000, 0x00000000, 0x00000000, 0x001FFF80,
		&vinfo[120], &vinfo[121], 1, 14, 14, 0
	},

	/* vinfo[107] */
	{
		GRID(+2,+14), GRID(+14,+2), GRID(+14,-2), GRID(+2,-14),
		GRID(-2,-14), GRID(-14,-2), GRID(-14,+2), GRID(-2,+14),
		0x00000000, 0x00000000, 0x00000001, 0xFFE00000,
		&vinfo[121], &vinfo[122], 2, 14, 15, 0
	},

	/* vinfo[108] */
	{
		GRID(+3,+14), GRID(+14,+3), GRID(+14,-3), GRID(+3,-14),
		GRID(-3,-14), GRID(-14,-3), GRID(-14,+3), GRID(-3,+14),
		0x00000000, 0x00000000, 0x00000FFF, 0x80000000,
		&vinfo[122], &vinfo[123], 3, 14, 15, 0
	},

	/* vinfo[109] */
	{
		GRID(+4,+14), GRID(+14,+4), GRID(+14,-4), GRID(+4,-14),
		GRID(-4,-14), GRID(-14,-4), GRID(-14,+4), GRID(-4,+14),
		0x00000000, 0x00000000, 0x003FFC00, 0x00000000,
		&vinfo[123], &vinfo[124], 4, 14, 16, 0
	},

	/* vinfo[110] */
	{
		GRID(+5,+14), GRID(+14,+5), GRID(+14,-5), GRID(+5,-14),
		GRID(-5,-14), GRID(-14,-5), GRID(-14,+5), GRID(-5,+14),
		0x00000000, 0x00000000, 0x7FE00000, 0x00000000,
		&vinfo[124], &vinfo[125], 5, 14, 16, 0
	},

	/* vinfo[111] */
	{
		GRID(+6,+14), GRID(+14,+6), GRID(+14,-6), GRID(+6,-14),
		GRID(-6,-14), GRID(-14,-6), GRID(-14,+6), GRID(-6,+14),
		0x00000000, 0x000001FF, 0xF0000000, 0x00000000,
		&vinfo[125], &vinfo[126], 6, 14, 17, 0
	},

	/* vinfo[112] */
	{
		GRID(+7,+14), GRID(+14,+7), GRID(+14,-7), GRID(+7,-14),
		GRID(-7,-14), GRID(-14,-7), GRID(-14,+7), GRID(-7,+14),
		0x00000000, 0x0003FFE0, 0x00000000, 0x00000000,
		&vinfo[126], &vinfo[127], 7, 14, 17, 0
	},

	/* vinfo[113] */
	{
		GRID(+8,+14), GRID(+14,+8), GRID(+14,-8), GRID(+8,-14),
		GRID(-8,-14), GRID(-14,-8), GRID(-14,+8), GRID(-8,+14),
		0x00000000, 0x03FFE000, 0x00000000, 0x00000000,
		&vinfo[127], &vinfo[128], 8, 14, 18, 0
	},

	/* vinfo[114] */
	{
		GRID(+9,+14), GRID(+14,+9), GRID(+14,-9), GRID(+9,-14),
		GRID(-9,-14), GRID(-14,-9), GRID(-14,+9), GRID(-9,+14),
		0x00000007, 0xFF800000, 0x00000000, 0x00000000,
		&vinfo[128], &vinfo[129], 9, 14, 18, 0
	},

	/* vinfo[115] */
	{
		GRID(+10,+14), GRID(+14,+10), GRID(+14,-10), GRID(+10,-14),
		GRID(-10,-14), GRID(-14,-10), GRID(-14,+10), GRID(-10,+14),
		0x000007FF, 0x80000000, 0x00000000, 0x00000000,
		&vinfo[129], &vinfo[130], 10, 14, 19, 0
	},

	/* vinfo[116] */
	{
		GRID(+11,+14), GRID(+14,+11), GRID(+14,-11), GRID(+11,-14),
		GRID(-11,-14), GRID(-14,-11), GRID(-14,+11), GRID(-11,+14),
		0x0007FFC0, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[130], &vinfo[0], 11, 14, 19, 0
	},

	/* vinfo[117] */
	{
		GRID(+12,+14), GRID(+14,+12), GRID(+14,-12), GRID(+12,-14),
		GRID(-12,-14), GRID(-14,-12), GRID(-14,+12), GRID(-12,+14),
		0x07FFC000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[0], &vinfo[0], 12, 14, 20, 0
	},

	/* vinfo[118] */
	{
		GRID(+13,+14), GRID(+14,+13), GRID(+14,-13), GRID(+13,-14),
		GRID(-13,-14), GRID(-14,-13), GRID(-14,+13), GRID(-13,+14),
		0x1FE00000, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[0], &vinfo[0], 13, 14, 20, 0
	},

	/* vinfo[119] */
	{
		GRID(+0,+15), GRID(+15,+0), GRID(+15,+0), GRID(+0,-15),
		GRID(+0,-15), GRID(-15,+0), GRID(-15,+0), GRID(+0,+15),
		0x00000000, 0x00000000, 0x00000000, 0x0000003F,
		&vinfo[131], &vinfo[132], 0, 15, 15, 15
	},

	/* vinfo[120] */
	{
		GRID(+1,+15), GRID(+15,+1), GRID(+15,-1), GRID(+1,-15),
		GRID(-1,-15), GRID(-15,-1), GRID(-15,+1), GRID(-1,+15),
		0x00000000, 0x00000000, 0x00000000, 0x000FFFC0,
		&vinfo[132], &vinfo[133], 1, 15, 15, 0
	},

	/* vinfo[121] */
	{
		GRID(+2,+15), GRID(+15,+2), GRID(+15,-2), GRID(+2,-15),
		GRID(-2,-15), GRID(-15,-2), GRID(-15,+2), GRID(-2,+15),
		0x00000000, 0x00000000, 0x00000000, 0x3FF00000,
		&vinfo[133], &vinfo[134], 2, 15, 16, 0
	},

	/* vinfo[122] */
	{
		GRID(+3,+15), GRID(+15,+3), GRID(+15,-3), GRID(+3,-15),
		GRID(-3,-15), GRID(-15,-3), GRID(-15,+3), GRID(-3,+15),
		0x00000000, 0x00000000, 0x000001FF, 0xC0000000,
		&vinfo[134], &vinfo[135], 3, 15, 16, 0
	},

	/* vinfo[123] */
	{
		GRID(+4,+15), GRID(+15,+4), GRID(+15,-4), GRID(+4,-15),
		GRID(-4,-15), GRID(-15,-4), GRID(-15,+4), GRID(-4,+15),
		0x00000000, 0x00000000, 0x000FFF80, 0x00000000,
		&vinfo[135], &vinfo[136], 4, 15, 17, 0
	},

	/* vinfo[124] */
	{
		GRID(+5,+15), GRID(+15,+5), GRID(+15,-5), GRID(+5,-15),
		GRID(-5,-15), GRID(-15,-5), GRID(-15,+5), GRID(-5,+15),
		0x00000000, 0x00000000, 0x07FE0000, 0x00000000,
		&vinfo[136], &vinfo[137], 5, 15, 17, 0
	},

	/* vinfo[125] */
	{
		GRID(+6,+15), GRID(+15,+6), GRID(+15,-6), GRID(+6,-15),
		GRID(-6,-15), GRID(-15,-6), GRID(-15,+6), GRID(-6,+15),
		0x00000000, 0x0000000F, 0xFF000000, 0x00000000,
		&vinfo[137], &vinfo[138], 6, 15, 18, 0
	},

	/* vinfo[126] */
	{
		GRID(+7,+15), GRID(+15,+7), GRID(+15,-7), GRID(+7,-15),
		GRID(-7,-15), GRID(-15,-7), GRID(-15,+7), GRID(-7,+15),
		0x00000000, 0x00000FFC, 0x00000000, 0x00000000,
		&vinfo[138], &vinfo[139], 7, 15, 18, 0
	},

	/* vinfo[127] */
	{
		GRID(+8,+15), GRID(+15,+8), GRID(+15,-8), GRID(+8,-15),
		GRID(-8,-15), GRID(-15,-8), GRID(-15,+8), GRID(-8,+15),
		0x00000000, 0x003FF800, 0x00000000, 0x00000000,
		&vinfo[139], &vinfo[140], 8, 15, 19, 0
	},

	/* vinfo[128] */
	{
		GRID(+9,+15), GRID(+15,+9), GRID(+15,-9), GRID(+9,-15),
		GRID(-9,-15), GRID(-15,-9), GRID(-15,+9), GRID(-9,+15),
		0x00000000, 0x3FFC0000, 0x00000000, 0x00000000,
		&vinfo[140], &vinfo[0], 9, 15, 19, 0
	},

	/* vinfo[129] */
	{
		GRID(+10,+15), GRID(+15,+10), GRID(+15,-10), GRID(+10,-15),
		GRID(-10,-15), GRID(-15,-10), GRID(-15,+10), GRID(-10,+15),
		0x0000001F, 0xFE000000, 0x00000000, 0x00000000,
		&vinfo[0], &vinfo[0], 10, 15, 20, 0
	},

	/* vinfo[130] */
	{
		GRID(+11,+15), GRID(+15,+11), GRID(+15,-11), GRID(+11,-15),
		GRID(-11,-15), GRID(-15,-11), GRID(-15,+11), GRID(-11,+15),
		0x00001FFF, 0x00000000, 0x00000000, 0x00000000,
		&vinfo[0], &vinfo[0], 11, 15, 20, 0
	},

	/* vinfo[131] */
	{
		GRID(+0,+16), GRID(+16,+0), GRID(+16,+0), GRID(+0,-16),
		GRID(+0,-16), GRID(-16,+0), GRID(-16,+0), GRID(+0,+16),
		0x00000000, 0x00000000, 0x00000000, 0x0000001F,
		&vinfo[141], &vinfo[142], 0, 16, 16, 16
	},

	/* vinfo[132] */
	{
		GRID(+1,+16), GRID(+16,+1), GRID(+16,-1), GRID(+1,-16),
		GRID(-1,-16), GRID(-16,-1), GRID(-16,+1), GRID(-1,+16),
		0x00000000, 0x00000000, 0x00000000, 0x0007FFE0,
		&vinfo[142], &vinfo[143], 1, 16, 16, 0
	},

	/* vinfo[133] */
	{
		GRID(+2,+16), GRID(+16,+2), GRID(+16,-2), GRID(+2,-16),
		GRID(-2,-16), GRID(-16,-2), GRID(-16,+2), GRID(-2,+16),
		0x00000000, 0x00000000, 0x00000000, 0x1FF80000,
		&vinfo[143], &vinfo[144], 2, 16, 17, 0
	},

	/* vinfo[134] */
	{
		GRID(+3,+16), GRID(+16,+3), GRID(+16,-3), GRID(+3,-16),
		GRID(-3,-16), GRID(-16,-3), GRID(-16,+3), GRID(-3,+16),
		0x00000000, 0x00000000, 0x0000003F, 0xF0000000,
		&vinfo[144], &vinfo[145], 3, 16, 17, 0
	},

	/* vinfo[135] */
	{
		GRID(+4,+16), GRID(+16,+4), GRID(+16,-4), GRID(+4,-16),
		GRID(-4,-16), GRID(-16,-4), GRID(-16,+4), GRID(-4,+16),
		0x00000000, 0x00000000, 0x0000FFE0, 0x00000000,
		&vinfo[145], &vinfo[146], 4, 16, 18, 0
	},

	/* vinfo[136] */
	{
		GRID(+5,+16), GRID(+16,+5), GRID(+16,-5), GRID(+5,-16),
		GRID(-5,-16), GRID(-16,-5), GRID(-16,+5), GRID(-5,+16),
		0x00000000, 0x00000000, 0x007F8000, 0x00000000,
		&vinfo[146], &vinfo[147], 5, 16, 18, 0
	},

	/* vinfo[137] */
	{
		GRID(+6,+16), GRID(+16,+6), GRID(+16,-6), GRID(+6,-16),
		GRID(-6,-16), GRID(-16,-6), GRID(-16,+6), GRID(-6,+16),
		0x00000000, 0x00000001, 0xFF800000, 0x00000000,
		&vinfo[147], &vinfo[148], 6, 16, 19, 0
	},

	/* vinfo[138] */
	{
		GRID(+7,+16), GRID(+16,+7), GRID(+16,-7), GRID(+7,-16),
		GRID(-7,-16), GRID(-16,-7), GRID(-16,+7), GRID(-7,+16),
		0x00000000, 0x000003FF, 0x80000000, 0x00000000,
		&vinfo[148], &vinfo[0], 7, 16, 19, 0
	},

	/* vinfo[139] */
	{
		GRID(+8,+16), GRID(+16,+8), GRID(+16,-8), GRID(+8,-16),
		GRID(-8,-16), GRID(-16,-8), GRID(-16,+8), GRID(-8,+16),
		0x00000000, 0x0001FFC0, 0x00000000, 0x00000000,
		&vinfo[0], &vinfo[0], 8, 16, 20, 0
	},

	/* vinfo[140] */
	{
		GRID(+9,+16), GRID(+16,+9), GRID(+16,-9), GRID(+9,-16),
		GRID(-9,-16), GRID(-16,-9), GRID(-16,+9), GRID(-9,+16),
		0x00000000, 0x00FFF000, 0x00000000, 0x00000000,
		&vinfo[0], &vinfo[0], 9, 16, 20, 0
	},

	/* vinfo[141] */
	{
		GRID(+0,+17), GRID(+17,+0), GRID(+17,+0), GRID(+0,-17),
		GRID(+0,-17), GRID(-17,+0), GRID(-17,+0), GRID(+0,+17),
		0x00000000, 0x00000000, 0x00000000, 0x0000000F,
		&vinfo[149], &vinfo[150], 0, 17, 17, 17
	},

	/* vinfo[142] */
	{
		GRID(+1,+17), GRID(+17,+1), GRID(+17,-1), GRID(+1,-17),
		GRID(-1,-17), GRID(-17,-1), GRID(-17,+1), GRID(-1,+17),
		0x00000000, 0x00000000, 0x00000000, 0x0003FFF0,
		&vinfo[150], &vinfo[151], 1, 17, 17, 0
	},

	/* vinfo[143] */
	{
		GRID(+2,+17), GRID(+17,+2), GRID(+17,-2), GRID(+2,-17),
		GRID(-2,-17), GRID(-17,-2), GRID(-17,+2), GRID(-2,+17),
		0x00000000, 0x00000000, 0x00000000, 0x07FC0000,
		&vinfo[151], &vinfo[152], 2, 17, 18, 0
	},

	/* vinfo[144] */
	{
		GRID(+3,+17), GRID(+17,+3), GRID(+17,-3), GRID(+3,-17),
		GRID(-3,-17), GRID(-17,-3), GRID(-17,+3), GRID(-3,+17),
		0x00000000, 0x00000000, 0x0000000F, 0xF8000000,
		&vinfo[152], &vinfo[153], 3, 17, 18, 0
	},

	/* vinfo[145] */
	{
		GRID(+4,+17), GRID(+17,+4), GRID(+17,-4), GRID(+4,-17),
		GRID(-4,-17), GRID(-17,-4), GRID(-17,+4), GRID(-4,+17),
		0x00000000, 0x00000000, 0x00003FF0, 0x00000000,
		&vinfo[153], &vinfo[154], 4, 17, 19, 0
	},

	/* vinfo[146] */
	{
		GRID(+5,+17), GRID(+17,+5), GRID(+17,-5), GRID(+5,-17),
		GRID(-5,-17), GRID(-17,-5), GRID(-17,+5), GRID(-5,+17),
		0x00000000, 0x00000000, 0x003FF000, 0x00000000,
		&vinfo[154], &vinfo[0], 5, 17, 19, 0
	},

	/* vinfo[147] */
	{
		GRID(+6,+17), GRID(+17,+6), GRID(+17,-6), GRID(+6,-17),
		GRID(-6,-17), GRID(-17,-6), GRID(-17,+6), GRID(-6,+17),
		0x00000000, 0x00000000, 0x3FC00000, 0x00000000,
		&vinfo[0], &vinfo[0], 6, 17, 20, 0
	},

	/* vinfo[148] */
	{
		GRID(+7,+17), GRID(+17,+7), GRID(+17,-7), GRID(+7,-17),
		GRID(-7,-17), GRID(-17,-7), GRID(-17,+7), GRID(-7,+17),
		0x00000000, 0x0000001F, 0xF8000000, 0x00000000,
		&vinfo[0], &vinfo[0], 7, 17, 20, 0
	},

	/* vinfo[149] */
	{
		GRID(+0,+18), GRID(+18,+0), GRID(+18,+0), GRID(+0,-18),
		GRID(+0,-18), GRID(-18,+0), GRID(-18,+0), GRID(+0,+18),
		0x00000000, 0x00000000, 0x00000000, 0x00000007,
		&vinfo[155], &vinfo[156], 0, 18, 18, 18
	},

	/* vinfo[150] */
	{
		GRID(+1,+18), GRID(+18,+1), GRID(+18,-1), GRID(+1,-18),
		GRID(-1,-18), GRID(-18,-1), GRID(-18,+1), GRID(-1,+18),
		0x00000000, 0x00000000, 0x00000000, 0x0001FFF8,
		&vinfo[156], &vinfo[157], 1, 18, 18, 0
	},

	/* vinfo[151] */
	{
		GRID(+2,+18), GRID(+18,+2), GRID(+18,-2), GRID(+2,-18),
		GRID(-2,-18), GRID(-18,-2), GRID(-18,+2), GRID(-2,+18),
		0x00000000, 0x00000000, 0x00000000, 0x03FE0000,
		&vinfo[157], &vinfo[158], 2, 18, 19, 0
	},

	/* vinfo[152] */
	{
		GRID(+3,+18), GRID(+18,+3), GRID(+18,-3), GRID(+3,-18),
		GRID(-3,-18), GRID(-18,-3), GRID(-18,+3), GRID(-3,+18),
		0x00000000, 0x00000000, 0x00000007, 0xFC000000,
		&vinfo[158], &vinfo[0], 3, 18, 19, 0
	},

	/* vinfo[153] */
	{
		GRID(+4,+18), GRID(+18,+4), GRID(+18,-4), GRID(+4,-18),
		GRID(-4,-18), GRID(-18,-4), GRID(-18,+4), GRID(-4,+18),
		0x00000000, 0x00000000, 0x000007F8, 0x00000000,
		&vinfo[0], &vinfo[0], 4, 18, 20, 0
	},

	/* vinfo[154] */
	{
		GRID(+5,+18), GRID(+18,+5), GRID(+18,-5), GRID(+5,-18),
		GRID(-5,-18), GRID(-18,-5), GRID(-18,+5), GRID(-5,+18),
		0x00000000, 0x00000000, 0x001FF800, 0x00000000,
		&vinfo[0], &vinfo[0], 5, 18, 20, 0
	},

	/* vinfo[155] */
	{
		GRID(+0,+19), GRID(+19,+0), GRID(+19,+0), GRID(+0,-19),
		GRID(+0,-19), GRID(-19,+0), GRID(-19,+0), GRID(+0,+19),
		0x00000000, 0x00000000, 0x00000000, 0x00000003,
		&vinfo[159], &vinfo[160], 0, 19, 19, 19
	},

	/* vinfo[156] */
	{
		GRID(+1,+19), GRID(+19,+1), GRID(+19,-1), GRID(+1,-19),
		GRID(-1,-19), GRID(-19,-1), GRID(-19,+1), GRID(-1,+19),
		0x00000000, 0x00000000, 0x00000000, 0x0000FFFC,
		&vinfo[160], &vinfo[0], 1, 19, 19, 0
	},

	/* vinfo[157] */
	{
		GRID(+2,+19), GRID(+19,+2), GRID(+19,-2), GRID(+2,-19),
		GRID(-2,-19), GRID(-19,-2), GRID(-19,+2), GRID(-2,+19),
		0x00000000, 0x00000000, 0x00000000, 0x01FF0000,
		&vinfo[0], &vinfo[0], 2, 19, 20, 0
	},

	/* vinfo[158] */
	{
		GRID(+3,+19), GRID(+19,+3), GRID(+19,-3), GRID(+3,-19),
		GRID(-3,-19), GRID(-19,-3), GRID(-19,+3), GRID(-3,+19),
		0x00000000, 0x00000000, 0x00000003, 0xFF000000,
		&vinfo[0], &vinfo[0], 3, 19, 20, 0
	},

	/* vinfo[159] */
	{
		GRID(+0,+20), GRID(+20,+0), GRID(+20,+0), GRID(+0,-20),
		GRID(+0,-20), GRID(-20,+0), GRID(-20,+0), GRID(+0,+20),
		0x00000000, 0x00000000, 0x00000000, 0x00000001,
		&vinfo[0], &vinfo[0], 0, 20, 20, 20
	},

	/* vinfo[160] */
	{
		GRID(+1,+20), GRID(+20,+1), GRID(+20,-1), GRID(+1,-20),
		GRID(-1,-20), GRID(-20,-1), GRID(-20,+1), GRID(-1,+20),
		0x00000000, 0x00000000, 0x00000000, 0x00007FFE,
		&vinfo[0], &vinfo[0], 1, 20, 20, 0
	},

};


#define VINFO_MAX_SLOPES 126

/* Bit :     Slope   Grids */
/* --- :     -----   ----- */
/*   0 :      2439      21 */
/*   1 :      2564      21 */
/*   2 :      2702      21 */
/*   3 :      2857      21 */
/*   4 :      3030      21 */
/*   5 :      3225      21 */
/*   6 :      3448      21 */
/*   7 :      3703      21 */
/*   8 :      4000      21 */
/*   9 :      4347      21 */
/*  10 :      4761      21 */
/*  11 :      5263      21 */
/*  12 :      5882      21 */
/*  13 :      6666      21 */
/*  14 :      7317      22 */
/*  15 :      7692      20 */
/*  16 :      8108      21 */
/*  17 :      8571      21 */
/*  18 :      9090      20 */
/*  19 :      9677      21 */
/*  20 :     10344      21 */
/*  21 :     11111      20 */
/*  22 :     12000      21 */
/*  23 :     12820      22 */
/*  24 :     13043      22 */
/*  25 :     13513      22 */
/*  26 :     14285      20 */
/*  27 :     15151      22 */
/*  28 :     15789      22 */
/*  29 :     16129      22 */
/*  30 :     17241      22 */
/*  31 :     17647      22 */
/*  32 :     17948      23 */
/*  33 :     18518      22 */
/*  34 :     18918      22 */
/*  35 :     20000      19 */
/*  36 :     21212      22 */
/*  37 :     21739      22 */
/*  38 :     22580      22 */
/*  39 :     23076      22 */
/*  40 :     23809      22 */
/*  41 :     24137      22 */
/*  42 :     24324      23 */
/*  43 :     25714      23 */
/*  44 :     25925      23 */
/*  45 :     26315      23 */
/*  46 :     27272      22 */
/*  47 :     28000      23 */
/*  48 :     29032      23 */
/*  49 :     29411      23 */
/*  50 :     29729      24 */
/*  51 :     30434      23 */
/*  52 :     31034      23 */
/*  53 :     31428      23 */
/*  54 :     33333      18 */
/*  55 :     35483      23 */
/*  56 :     36000      23 */
/*  57 :     36842      23 */
/*  58 :     37142      24 */
/*  59 :     37931      24 */
/*  60 :     38461      24 */
/*  61 :     39130      24 */
/*  62 :     39393      24 */
/*  63 :     40740      24 */
/*  64 :     41176      24 */
/*  65 :     41935      24 */
/*  66 :     42857      23 */
/*  67 :     44000      24 */
/*  68 :     44827      24 */
/*  69 :     45454      23 */
/*  70 :     46666      24 */
/*  71 :     47368      24 */
/*  72 :     47826      24 */
/*  73 :     48148      24 */
/*  74 :     48387      24 */
/*  75 :     51515      25 */
/*  76 :     51724      25 */
/*  77 :     52000      25 */
/*  78 :     52380      25 */
/*  79 :     52941      25 */
/*  80 :     53846      25 */
/*  81 :     54838      25 */
/*  82 :     55555      24 */
/*  83 :     56521      25 */
/*  84 :     57575      26 */
/*  85 :     57894      25 */
/*  86 :     58620      25 */
/*  87 :     60000      23 */
/*  88 :     61290      25 */
/*  89 :     61904      25 */
/*  90 :     62962      25 */
/*  91 :     63636      25 */
/*  92 :     64705      25 */
/*  93 :     65217      25 */
/*  94 :     65517      25 */
/*  95 :     67741      26 */
/*  96 :     68000      26 */
/*  97 :     68421      26 */
/*  98 :     69230      26 */
/*  99 :     70370      26 */
/* 100 :     71428      25 */
/* 101 :     72413      26 */
/* 102 :     73333      26 */
/* 103 :     73913      26 */
/* 104 :     74193      27 */
/* 105 :     76000      26 */
/* 106 :     76470      26 */
/* 107 :     77777      25 */
/* 108 :     78947      26 */
/* 109 :     79310      26 */
/* 110 :     80952      26 */
/* 111 :     81818      26 */
/* 112 :     82608      26 */
/* 113 :     84000      26 */
/* 114 :     84615      26 */
/* 115 :     85185      26 */
/* 116 :     86206      27 */
/* 117 :     86666      27 */
/* 118 :     88235      27 */
/* 119 :     89473      27 */
/* 120 :     90476      27 */
/* 121 :     91304      27 */
/* 122 :     92000      27 */
/* 123 :     92592      27 */
/* 124 :     93103      28 */
/* 125 :    100000      13 */

#define VINFO_BITS_3 0x3FFFFFFF
#define VINFO_BITS_2 0xFFFFFFFF
#define VINFO_BITS_1 0xFFFFFFFF
#define VINFO_BITS_0 0xFFFFFFFF


/* XXX XXX XXX End machine generated code XXX XXX XXX */




/*
 * Calculate the complete field of view using a new algorithm
 *
 * If "view_g" and "temp_g" were global pointers to arrays of grids, as
 * opposed to actual arrays of grids, then we could be more efficient by
 * using "pointer swapping".
 *
 * Note the following idiom, which is used in the function below.
 * This idiom processes each "octant" of the field of view, in a
 * clockwise manner, starting with the east strip, south side,
 * and for each octant, allows a simple calculation to set "g"
 * equal to the proper grids, relative to "pg", in the octant.
 *
 *   for (o2 = 0; o2 < 16; o2 += 2)
 *   ...
 *         g = pg + *((s16b*)(((byte*)(p))+o2));
 *   ...
 *
 *
 * Normally, vision along the major axes is more likely than vision
 * along the diagonal axes, so we check the bits corresponding to
 * the lines of sight near the major axes first.
 *
 * We use the "temp_g" array (and the "CAVE_TEMP" flag) to keep track of
 * which grids were previously marked "CAVE_SEEN", since only those grids
 * whose "CAVE_SEEN" value changes during this routine must be redrawn.
 *
 * This function is now responsible for maintaining the "CAVE_SEEN"
 * flags as well as the "CAVE_VIEW" flags, which is good, because
 * the only grids which normally need to be memorized and/or redrawn
 * are the ones whose "CAVE_SEEN" flag changes during this routine.
 *
 * Basically, this function divides the "octagon of view" into octants of
 * grids (where grids on the main axes and diagonal axes are "shared" by
 * two octants), and processes each octant one at a time, processing each
 * octant one grid at a time, processing only those grids which "might" be
 * viewable, and setting the "CAVE_VIEW" flag for each grid for which there
 * is an (unobstructed) line of sight from the center of the player grid to
 * any internal point in the grid (and collecting these "CAVE_VIEW" grids
 * into the "view_g" array), and setting the "CAVE_SEEN" flag for the grid
 * if, in addition, the grid is "illuminated" in some way.
 *
 * This function relies on a theorem (suggested and proven by Mat Hostetter)
 * which states that in each octant of a field of view, a given grid will
 * be "intersected" by one or more unobstructed "lines of sight" from the
 * center of the player grid if and only if it is "intersected" by at least
 * one such unobstructed "line of sight" which passes directly through some
 * corner of some grid in the octant which is not shared by any other octant.
 * The proof is based on the fact that there are at least three significant
 * lines of sight involving any non-shared grid in any octant, one which
 * intersects the grid and passes though the corner of the grid closest to
 * the player, and two which "brush" the grid, passing through the "outer"
 * corners of the grid, and that any line of sight which intersects a grid
 * without passing through the corner of a grid in the octant can be "slid"
 * slowly towards the corner of the grid closest to the player, until it
 * either reaches it or until it brushes the corner of another grid which
 * is closer to the player, and in either case, the existanc of a suitable
 * line of sight is thus demonstrated.
 *
 * It turns out that in each octant of the radius 20 "octagon of view",
 * there are 161 grids (with 128 not shared by any other octant), and there
 * are exactly 126 distinct "lines of sight" passing from the center of the
 * player grid through any corner of any non-shared grid in the octant.  To
 * determine if a grid is "viewable" by the player, therefore, you need to
 * simply show that one of these 126 lines of sight intersects the grid but
 * does not intersect any wall grid closer to the player.  So we simply use
 * a bit vector with 126 bits to represent the set of interesting lines of
 * sight which have not yet been obstructed by wall grids, and then we scan
 * all the grids in the octant, moving outwards from the player grid.  For
 * each grid, if any of the lines of sight which intersect that grid have not
 * yet been obstructed, then the grid is viewable.  Furthermore, if the grid
 * is a wall grid, then all of the lines of sight which intersect the grid
 * should be marked as obstructed for future reference.  Also, we only need
 * to check those grids for whom at least one of the "parents" was a viewable
 * non-wall grid, where the parents include the two grids touching the grid
 * but closer to the player grid (one adjacent, and one diagonal).  For the
 * bit vector, we simply use 4 32-bit integers.  All of the static values
 * which are needed by this function are stored in the large "vinfo" array
 * (above), which is machine generated by another program.  XXX XXX XXX
 *
 * Hack -- The queue must be able to hold more than VINFO_MAX_GRIDS grids
 * because the grids at the edge of the field of view use "grid zero" as
 * their children, and the queue must be able to hold several of these
 * special grids.  Because the actual number of required grids is bizarre,
 * we simply allocate twice as many as we would normally need.  XXX XXX XXX
 */
void update_view(void)
{
	int py = p_ptr->py;
	int px = p_ptr->px;

	int pg = GRID(py,px);

	int i, g, o2;

	int radius;

	int fast_view_n = view_n;
	u16b *fast_view_g = view_g;

	int fast_temp_n = 0;
	u16b *fast_temp_g = temp_g;

	byte *fast_cave_info = &cave_info[0][0];

	byte info;


	/*** Step 0 -- Begin ***/

	/* Save the old "view" grids for later */
	for (i = 0; i < fast_view_n; i++)
	{
		/* Grid */
		g = fast_view_g[i];

		/* Get grid info */
		info = fast_cave_info[g];

		/* Save "CAVE_SEEN" grids */
		if (info & (CAVE_SEEN))
		{
			/* Set "CAVE_TEMP" flag */
			info |= (CAVE_TEMP);

			/* Save grid for later */
			fast_temp_g[fast_temp_n++] = g;
		}

		/* Clear "CAVE_VIEW" and "CAVE_SEEN" flags */
		info &= ~(CAVE_VIEW | CAVE_SEEN);

		/* Clear "CAVE_LITE" flag */
		/* info &= ~(CAVE_LITE); */

		/* Save cave info */
		fast_cave_info[g] = info;
	}

	/* Reset the "view" array */
	fast_view_n = 0;

	/* Extract "radius" value */
	radius = p_ptr->cur_lite;

	/* Handle real light */
	if (radius > 0) ++radius;


	/*** Step 1 -- player grid ***/

	/* Player grid */
	g = pg;

	/* Get grid info */
	info = fast_cave_info[g];

	/* Assume viewable */
	info |= (CAVE_VIEW);

	/* Torch-lit grid */
	if (0 < radius)
	{
		/* Mark as "CAVE_SEEN" */
		info |= (CAVE_SEEN);

		/* Mark as "CAVE_LITE" */
		/* info |= (CAVE_LITE); */
	}

	/* Perma-lit grid */
	else if (info & (CAVE_GLOW))
	{
		/* Mark as "CAVE_SEEN" */
		info |= (CAVE_SEEN);
	}

	/* Save cave info */
	fast_cave_info[g] = info;

	/* Save in array */
	fast_view_g[fast_view_n++] = g;


	/*** Step 2 -- octants ***/

	/* Scan each octant */
	for (o2 = 0; o2 < 16; o2 += 2)
	{
		vinfo_type *p;

		/* Last added */
		vinfo_type *last = &vinfo[0];

		/* Grid queue */
		int queue_head = 0;
		int queue_tail = 0;
		vinfo_type *queue[VINFO_MAX_GRIDS*2];

		/* Slope bit vector */
		u32b bits0 = VINFO_BITS_0;
		u32b bits1 = VINFO_BITS_1;
		u32b bits2 = VINFO_BITS_2;
		u32b bits3 = VINFO_BITS_3;

		/* Reset queue */
		queue_head = queue_tail = 0;

		/* Initial grids */
		queue[queue_tail++] = &vinfo[1];
		queue[queue_tail++] = &vinfo[2];

		/* Process queue */
		while (queue_head < queue_tail)
		{
			/* Dequeue next grid */
			p = queue[queue_head++];

			/* Check bits */
			if ((bits0 & (p->bits_0)) ||
			    (bits1 & (p->bits_1)) ||
			    (bits2 & (p->bits_2)) ||
			    (bits3 & (p->bits_3)))
			{
				/* Extract grid value XXX XXX XXX */
				g = pg + *((s16b*)(((byte*)(p))+o2));

				/* Get grid info */
				info = fast_cave_info[g];

				/* Handle wall */
				if (info & (CAVE_WALL))
				{
					/* Clear bits */
					bits0 &= ~(p->bits_0);
					bits1 &= ~(p->bits_1);
					bits2 &= ~(p->bits_2);
					bits3 &= ~(p->bits_3);

					/* Newly viewable wall */
					if (!(info & (CAVE_VIEW)))
					{
						/* Mark as viewable */
						info |= (CAVE_VIEW);

						/* Torch-lit grids */
						if (p->d < radius)
						{
							/* Mark as "CAVE_SEEN" */
							info |= (CAVE_SEEN);

							/* Mark as "CAVE_LITE" */
							/* info |= (CAVE_LITE); */
						}

						/* Perma-lit grids */
						else if (info & (CAVE_GLOW))
						{
							int y = GRID_Y(g);
							int x = GRID_X(g);

							/* Hack -- move towards player */
							int yy = (y < py) ? (y + 1) : (y > py) ? (y - 1) : y;
							int xx = (x < px) ? (x + 1) : (x > px) ? (x - 1) : x;

#ifdef UPDATE_VIEW_COMPLEX_WALL_ILLUMINATION

							/* Check for "complex" illumination */
							if ((!(cave_info[yy][xx] & (CAVE_WALL)) &&
							      (cave_info[yy][xx] & (CAVE_GLOW))) ||
							    (!(cave_info[y][xx] & (CAVE_WALL)) &&
							      (cave_info[y][xx] & (CAVE_GLOW))) ||
							    (!(cave_info[yy][x] & (CAVE_WALL)) &&
							      (cave_info[yy][x] & (CAVE_GLOW))))
							{
								/* Mark as seen */
								info |= (CAVE_SEEN);
							}

#else /* UPDATE_VIEW_COMPLEX_WALL_ILLUMINATION */

							/* Check for "simple" illumination */
							if (cave_info[yy][xx] & (CAVE_GLOW))
							{
								/* Mark as seen */
								info |= (CAVE_SEEN);
							}

#endif /* UPDATE_VIEW_COMPLEX_WALL_ILLUMINATION */

						}

						/* Save cave info */
						fast_cave_info[g] = info;

						/* Save in array */
						fast_view_g[fast_view_n++] = g;
					}
				}

				/* Handle non-wall */
				else
				{
					/* Enqueue child */
					if (last != p->next_0)
					{
						queue[queue_tail++] = last = p->next_0;
					}

					/* Enqueue child */
					if (last != p->next_1)
					{
						queue[queue_tail++] = last = p->next_1;
					}

					/* Newly viewable non-wall */
					if (!(info & (CAVE_VIEW)))
					{
						/* Mark as "viewable" */
						info |= (CAVE_VIEW);

						/* Torch-lit grids */
						if (p->d < radius)
						{
							/* Mark as "CAVE_SEEN" */
							info |= (CAVE_SEEN);

							/* Mark as "CAVE_LITE" */
							/* info |= (CAVE_LITE); */
						}

						/* Perma-lit grids */
						else if (info & (CAVE_GLOW))
						{
							/* Mark as "CAVE_SEEN" */
							info |= (CAVE_SEEN);
						}

						/* Save cave info */
						fast_cave_info[g] = info;

						/* Save in array */
						fast_view_g[fast_view_n++] = g;
					}
				}
			}
		}
	}


	/*** Step 3 -- Complete the algorithm ***/

	/* Handle blindness */
	if (p_ptr->blind)
	{
		/* Process "new" grids */
		for (i = 0; i < fast_view_n; i++)
		{
			/* Grid */
			g = fast_view_g[i];

			/* Grid cannot be "CAVE_SEEN" */
			fast_cave_info[g] &= ~(CAVE_SEEN);
		}
	}

	/* Process "new" grids */
	for (i = 0; i < fast_view_n; i++)
	{
		/* Grid */
		g = fast_view_g[i];

		/* Get grid info */
		info = fast_cave_info[g];

		/* Was not "CAVE_SEEN", is now "CAVE_SEEN" */
		if ((info & (CAVE_SEEN)) && !(info & (CAVE_TEMP)))
		{
			int y, x;

			/* Location */
			y = GRID_Y(g);
			x = GRID_X(g);

			/* Note */
			note_spot(y, x);

			/* Redraw */
			lite_spot(y, x);
		}
	}

	/* Process "old" grids */
	for (i = 0; i < fast_temp_n; i++)
	{
		/* Grid */
		g = fast_temp_g[i];

		/* Get grid info */
		info = fast_cave_info[g];

		/* Clear "CAVE_TEMP" flag */
		info &= ~(CAVE_TEMP);

		/* Save cave info */
		fast_cave_info[g] = info;

		/* Was "CAVE_SEEN", is now not "CAVE_SEEN" */
		if (!(info & (CAVE_SEEN)))
		{
			int y, x;

			/* Location */
			y = GRID_Y(g);
			x = GRID_X(g);

			/* Redraw */
			lite_spot(y, x);
		}
	}


	/* Save 'view_n' */
	view_n = fast_view_n;
}


#else /* UPDATE_VIEW_NEW */


/*
 * This is the old version of the "update_view()" algorithm.
 *
 * See the "update_view()" function (below) for more details.
 */



/*
 * Adapted from the "los()" function above.
 *
 * Returns TRUE if a line of sight can be traced from the current player
 * grid "pg" (y1,x1) to a grid "g" (y2,x2), assuming that this function
 * is being called (correctly) from the "update_view()" function.
 *
 * We assume, as is done elsewhere, that the "mathematical corner" of a
 * non-floor tile does not block line of sight.
 *
 * We assume that the distance from (x1,y1) to (x2,y2) is less than 90 so
 * we can use short ints for computations.
 *
 * We assume that (ABS(y2-y1) > 0) and (ABS(x2-x1) > 0) so we can ignore
 * the degenerate cases of infinite and undefined slopes.
 *
 * We assume that (ABS(y2-y1) != ABS(x2-x1)) so we can ignore the special
 * cases of slope equals +1 or -1.
 *
 * We assume that (ABS(y2-y1) > 2) and ABS(x2-x1) > 2) so we can ignore the
 * bizarre cases of "knight move" grids.
 *
 * All of these assumptions are either defined by, or provided by, the
 * "update_view()" and "update_view_aux()" functions, and nobody else
 * is allowed to call this function.
 *
 * Once all the degenerate cases are eliminated, we use "frac" mod "full",
 * where "full = abs(dx * dy * 2)", a scaling factor, to perform efficient
 * fixed point integer arithmetic.  Each grid is divided into "full" units,
 * from "-half" to "+half", where "half = full / 2".
 *
 * We begin at the center of the player grid, and travel towards the center
 * of the destination grid, along the longer axis, starting at the border
 * between the player grid and the grid which is directly adjacent to that
 * grid in the direction of the destination, where the y offset = slope/2,
 * taking into account the scale factor.  We stop as soon as we hit a wall
 * grid (having determined that line of sight does not exist), or when we
 * reach the destination grid (having determined that line of sight does
 * exist).  This function is more expensive than the "projectable()" code.
 *
 * Note the special handling of the common case where (ABS(y2-y1) == 1)
 * or (ABS(x2-x1) == 1), which often occurs as the player is approaching
 * a hallway while misaligned by a single grid.  In this case, instead of
 * checking line of sight from the player to the grid, we check line of
 * sight from the grid to the player, stopping as soon as we hit a wall,
 * or wrap to the player axis (at which point the outcome will be clear).
 * This method is normally faster than the default method, since if line
 * of sight exists, we will only check half as many grids as usual, and
 * if it does not, the blockage will normally (?) be no closer to the
 * player than to the destination grid.
 */
static bool update_view_los(int pg, int g)
{
	/* Delta */
	int dy, dx;

	/* Absolute */
	int ay, ax;

	/* Offsets */
	int sy, sx;

	/* Grid */
	int tg;

	/* Fractions */
	int frac;

	/* Scale factors */
	int full, half;

	/* Slope */
	int m;


	byte *fast_cave_info = &cave_info[0][0];

	byte info;


	/* Offset */
	dy = GRID_Y(g) - GRID_Y(pg);
	dx = GRID_X(g) - GRID_X(pg);

	/* Analyze "dy" */
	if (dy < 0)
	{
		ay = -dy;
		sy = GRID(-1,+0);
	}
	else
	{
		ay = dy;
		sy = GRID(+1,+0);
	}

	/* Analyze "dx" */
	if (dx < 0)
	{
		ax = -dx;
		sx = GRID(+0,-1);
	}
	else
	{
		ax = dx;
		sx = GRID(+0,+1);
	}

	/* Number of "units" in one "half" grid */
	half = (ay * ax);

	/* Number of "units" in one "full" grid */
	full = half << 1;

	/* Travel horizontally */
	if (ax >= ay)
	{
		/* Start at tile edge */
		frac = ay * ay;

		/* Let m = ((dy/dx) * full) = (dy * dy * 2) = (frac * 2) */
		m = frac << 1;

		/* Handle the most common case backwards */
		if (ay <= 1)
		{
			/* Start */
			tg = g - sx;

			/* Get grid info */
			info = fast_cave_info[tg];

			/* Retreat */
			while (1)
			{
				/* Wall grid blocks line of sight */
				if (info & CAVE_WALL) return (FALSE);

				/* Retreat (Y) part 1 */
				frac += m;

				/* Vertical change */
				if (frac >= half)
				{
					/* Retreat (Y) part 2 */
					tg -= sy;

					/* Cross into a new grid */
					if (frac > half)
					{
						/* Get grid info */
						info = fast_cave_info[tg];

						/* Wall grid blocks line of sight */
						if (info & CAVE_WALL) return (FALSE);
					}

					/* Retreat (Y) part 3 */
					/* frac -= full; */

					/* Retreat (X) */
					tg -= sx;

					/* Get grid info */
					info = fast_cave_info[tg];

					/* If not viewable floor grid, block line of sight */
					if (!(info & (CAVE_VIEW))) return (FALSE);

					/* Allow line of sight */
					return (TRUE);
				}

				/* Retreat (X) */
				tg -= sx;
			}
		}

		/* Handle normal cases */
		else
		{
			/* Start */
			tg = pg + sx;

			/* Get grid info */
			info = fast_cave_info[tg];

			/* Advance */
			while (tg != g)
			{
				/* Wall grid blocks line of sight */
				if (info & CAVE_WALL) return (FALSE);

				/* Advance (Y) part 1 */
				frac += m;

				/* Vertical change */
				if (frac >= half)
				{
					/* Advance (Y) part 2 */
					tg += sy;

					/* Cross into a new grid */
					if (frac > half)
					{
						/* Get grid info */
						info = fast_cave_info[tg];

						/* Wall grid blocks line of sight */
						if (info & CAVE_WALL) return (FALSE);
					}

					/* Advance (Y) part 3 */
					frac -= full;
				}

				/* Advance (X) */
				tg += sx;

				/* Get grid info */
				info = fast_cave_info[tg];
			}
		}
	}

	/* Travel vertically */
	else
	{
		/* Start at tile edge */
		frac = ax * ax;

		/* Let m = ((dx/dy) * full) = (dx * dx * 2) = (frac * 2) */
		m = frac << 1;

		/* Handle the most common case backwards */
		if (ax <= 1)
		{
			/* Start */
			tg = g - sy;

			/* Get grid info */
			info = fast_cave_info[tg];

			/* Retreat */
			while (1)
			{
				/* Wall grid blocks line of sight */
				if (info & CAVE_WALL) return (FALSE);

				/* Retreat (X) part 1 */
				frac += m;

				/* Horizontal change */
				if (frac >= half)
				{
					/* Retreat (X) part 2 */
					tg -= sx;

					/* Cross into a new grid */
					if (frac > half)
					{
						/* Get grid info */
						info = fast_cave_info[tg];

						/* Wall grid blocks line of sight */
						if (info & CAVE_WALL) return (FALSE);
					}

					/* Retreat (X) part 3 */
					/* frac -= full; */

					/* Retreat (Y) */
					tg -= sy;

					/* Get grid info */
					info = fast_cave_info[tg];

					/* If not viewable floor grid, block line of sight */
					if (!(info & (CAVE_VIEW))) return (FALSE);

					/* Allow line of sight */
					return (TRUE);
				}

				/* Retreat (Y) */
				tg -= sy;

				/* Get grid info */
				info = fast_cave_info[tg];
			}
		}

		/* Handle normal cases */
		else
		{
			/* Start */
			tg = pg + sy;

			/* Get grid info */
			info = fast_cave_info[tg];

			/* Advance */
			while (tg != g)
			{
				/* Wall grid blocks line of sight */
				if (info & CAVE_WALL) return (FALSE);

				/* Advance (X) part 1 */
				frac += m;

				/* Horizontal change */
				if (frac >= half)
				{
					/* Advance (X) part 2 */
					tg += sx;
				
					/* Cross into a new grid */
					if (frac > half)
					{
						/* Get grid info */
						info = fast_cave_info[tg];

						/* Wall grid blocks line of sight */
						if (info & CAVE_WALL) return (FALSE);
					}

					/* Advance (X) part 3 */
					frac -= full;
				}

				/* Advance (Y) */
				tg += sy;

				/* Get grid info */
				info = fast_cave_info[tg];
			}
		}
	}


	/* Assume line of sight */
	return (TRUE);
}



/*
 * Hack -- Use "CAVE_SEEN" flag for "CAVE_EASY" property
 */
#define CAVE_EASY CAVE_SEEN


/*
 * Simplify the "update_view()" function
 *
 * This macro processes a "strip" of the current "octant of view".
 *
 * For each grid "g" in the strip, this macro determines the "viewability"
 * of a grid "g", from the point of view of the player, based on properties
 * of the two grids "g1" and "g2", which touch the grid "g" and which are
 * located between the player and grid "g", with grid "g1" being "diagonally
 * adjacent" to "g", and grid "g2" being "directly adjacent" to "g".
 *
 * We set the variable "block" to TRUE if vision will be "blocked" by the
 * grid "g", and FALSE otherwise, and then we use this flag to help avoid
 * unnecessary (or illegal) computations.
 *
 * This macro is never called for grids which are on the main axes, or on
 * the main diagonal axes from the player.
 *
 * This macro is only called on the "knight move" grids from the player if
 * the grid "g1" (which is adjacent to the player) does not block line of
 * sight, in which case this macro will never send the grid "g" to the
 * "update_view_los()" function.
 *
 * Note the following idiom:
 *
 *   info = fast_cave_info[g];
 *   // ... Calculations involving "info" ...
 *   info |= (CAVE_VIEW | CAVE_EASY);
 *   fast_cave_info[g] = info;
 *   fast_view_g[fast_view_n++] = g;
 *
 * This allows us to set the "CAVE_VIEW" (and "CAVE_EASY") flags for the
 * grid "g", while adding the grid to the array of viewable grids.  This
 * idiom is never called for illegal grids, or for grids which have been
 * placed into the array already, and the array never gets full, so the
 * idiom does not have to perform any bounds checking.
 *
 * It looks like the references to "block" and "QVAR" could be removed,
 * but only if some kind of top level check was added to prevent starting
 * on an "illegal" grid, and if no references to "g" were made until it
 * was verified that "g1" and/or "g2" was a viewable floor grid (ensuring
 * that "g" is not "illegal").  This might or might not be a good idea,
 * since the "QVAR" code allows us, in some situations, to bypass a lot
 * of expensive checks.
 *
 * Assumes existance of local 'fast_cave_info', 'fast_view_g', and
 * 'fast_view_n'.  XXX XXX XXX
 */
#define update_view_strip(QVAR,PG,START,OFFSET,N,NPZ) \
\
	do \
	{ \
	if (n < (QVAR)) \
	{ \
		int dpn;\
\
		int limit; \
\
		int g; \
\
		int info, info1, info2; \
\
		/* Start */ \
		g = (PG) + (START) * (N); \
\
		/* Start info */ \
		info = fast_cave_info[g]; \
\
		/* Default limit for the next strip */ \
		limit = (N); \
\
		/* Scan */ \
		for (dpn = (N)+1; dpn <= (NPZ); dpn++) \
		{ \
			byte block; \
			bool bv1, bv2; \
\
			/* Advance */ \
			g += (OFFSET); \
\
			/* Info */ \
			info2 = info; \
			info1 = fast_cave_info[g - (START)]; \
			info = fast_cave_info[g]; \
\
			/* Assume blockage */ \
			block = CAVE_WALL; \
\
			/* Determine if grid "g1" is a "viewable floor grid" */ \
			bv1 = (!(info1 & (CAVE_WALL)) && (info1 & (CAVE_VIEW))); \
\
			/* Determine if grid "g2" is a "viewable floor grid" */ \
			bv2 = (!(info2 & (CAVE_WALL)) && (info2 & (CAVE_VIEW))); \
\
			/* Require at least one viewable floor grid */ \
			if (bv1 || bv2) \
			{ \
				/* Extract the blockage flag */ \
				block = (info & CAVE_WALL); \
\
				/* Both "viewable" yields "viewable" */ \
				if (bv1 && bv2) \
				{ \
					/* Both "easily" yields "easily" */ \
					if ((info1 & (CAVE_EASY)) && (info2 & (CAVE_EASY))) \
					{ \
						/* Easily viewable */ \
						info |= (CAVE_VIEW | CAVE_EASY); \
						fast_cave_info[g] = info; \
						fast_view_g[fast_view_n++] = g; \
					} \
\
					/* Otherwise just "viewable" */ \
					else \
					{ \
						/* Viewable */ \
						info |= (CAVE_VIEW); \
						fast_cave_info[g] = info; \
						fast_view_g[fast_view_n++] = g; \
					} \
				} \
\
				/* Diagonal "easily" yields "viewable" */ \
				else if (bv1 && (info1 & (CAVE_EASY))) \
				{ \
					/* Viewable */ \
					info |= (CAVE_VIEW); \
					fast_cave_info[g] = info; \
					fast_view_g[fast_view_n++] = g; \
				} \
\
				/* Hack -- Assume line of sight to walls */ \
				else if (block) \
				{ \
					/* Viewable */ \
					info |= (CAVE_VIEW); \
					fast_cave_info[g] = info; \
					fast_view_g[fast_view_n++] = g; \
				} \
\
				/* Check line of sight */ \
				else if (update_view_los((PG), g)) \
				{ \
					/* Viewable */ \
					info |= (CAVE_VIEW); \
					fast_cave_info[g] = info; \
					fast_view_g[fast_view_n++] = g; \
				} \
\
				/* No line of sight */ \
				else \
				{ \
					/* Assume blockage */ \
					block = CAVE_WALL; \
				} \
			} \
\
			/* Handle "blockage" */ \
			if (block) \
			{ \
				if (dpn >= (QVAR)) break; \
			} \
\
			/* Track most distant "non-blockage" */ \
			else \
			{ \
				limit = dpn; \
			} \
		} \
\
		/* Save limit for the next strip */ \
		(QVAR) = limit + 1; \
	} \
\
	} while (0)






/*
 * Update the field of view, memorizing and redrawing (only) as needed
 *
 * If "view_g" and "temp_g" were global pointers to arrays of grids, as
 * opposed to actual arrays of grids, then we could be more efficient by
 * using "pointer swapping".
 *
 * We use the "temp_g" array (and the "CAVE_TEMP" flag) to keep track of
 * which grids were previously marked "CAVE_SEEN", since only those grids
 * whose "CAVE_SEEN" value changes during this routine must be redrawn.
 *
 * Algorithm summary:
 *
 *  1: Process the player
 *  1a: The player is always (easily) viewable
 *  2: Process the diagonals
 *  2a: The diagonals are (easily) viewable up to the first wall
 *  2b: But never go more than 14/20 of the "full" distance
 *  3: Process the main axes
 *  3a: The main axes are (easily) viewable up to the first wall
 *  3b: But never go more than the "full" distance
 *  4: Process sequential "strips" in each of the eight octants
 *  4a: Each strip runs along the previous strip
 *  4b: The main axes are "previous" to the first strip
 *  4c: Process both "sides" of each "direction" of each strip
 *  4c1: Each side aborts as soon as possible
 *  4c2: Each side tells the next strip how far it has to check
 *  4c3: Determine the (easily) viewable flag for each grid
 *
 * Note that the octant processing involves some pretty interesting
 * observations involving when a grid might possibly be viewable from
 * a given grid, and on the order in which the strips are processed.
 *
 * Note the use of the mathematical facts shown below, which derive
 * from the fact that (1 < 1.4 < sqrt(2) < 1.5), and that the length of the
 * hypotenuse of a right triangle is primarily determined by the length
 * of the longest side, when one side is small, and is strictly less
 * than one-and-a-half times as long as the longest side when both of
 * the sides are large.
 *
 *   if (manhatten(dy,dx) < R) then (hypot(dy,dx) < R)
 *   if (manhatten(dy,dx) > R*3/2) then (hypot(dy,dx) > R)
 *
 *   hypot(dy,dx) is approximated by (dx+dy+MAX(dx,dy)) / 2
 *
 * These observations are important because the calculation of the actual
 * value of "hypot(dx,dy)" is extremely expensive, involving square roots,
 * while for small values (up to about 20 or so), the approximations above
 * are correct to within an error of at most one or two grids.
 *
 * Observe the use of "MAX_SIGHT" (20) and "MAX_SIGHT * 7 / 10" (14) in the
 * code below, in terms of the way in which they are used to maintain the
 * "octagon" of view, which, as mentioned above, is somewhat smaller that
 * the "circle" of view, and a lot cheaper to calculate.
 *
 * Note that by only processing the "octagon of view" instead of the more
 * traditional "circle of view", we restrict the maximal field of view to
 * a mere 1149 grids (the full "square of view" contains 1681 grids).
 *
 * The bizarre "qse","qsw","qne","qnw","qes","qen","qws","qwn" variables
 * are very important.  They indicate, for each "octant" (major axis and
 * then minor direction), the maximal distance which may be traveled along
 * strips in that octant.  They start out as the maximum viewable distance
 * along the major axis.  Then, for example, consider the "qes" variable.
 * While moving down the east-bound strip just to the south of the last
 * such strip, as soon as we get to a grid which does not allow "viewing"
 * to pass, if all previous related strips (including the primary "south"
 * axis) have terminated at or before the same point, then we can stop,
 * and reset the "max distance" to ourself, or to be more precise, to
 * the grid (if any) following the last "viewable" grid encountered in this
 * strip.  Note that if no such grids were encountered, then we can reset
 * "qes" to some value (like zero) which causes all subsequent possible
 * strips in this octant to be ignored.
 *
 * In the picture below, the grids marked ":" are not relevant, the grids
 * marked "1"-"5" are the "diagonal" grids which delineate each strip, the
 * grids marked "#" are walls, the grids marked "x" are not even checked,
 * the grids marked "." are (viewable) floor grids.
 *
 *   :::::::::::::::::
 *   :@...........#xxx   Strip 0, we check 12 - 0 grids, and set qes to 12
 *   ::1.........##xxx   Strip 1, we check 12 - 1 grids, and set qes to 11
 *   :::2...........#x   Strip 2, we check 14 - 2 grids, and set qes to 14
 *   ::::3###########x   Strip 3, we check 14 - 3 grids, and set qes to 4
 *   :::::4xxxxxxxxxxx   Strip 4, we check 0 grids, and set qes to 5
 *   ::::::5xxxxxxxxxx   Strip 5, we check 0 grids, and set qes to 6 (etc)
 *
 * Note the use of the huge inline macro "update_view_strip", which is
 * instantiated eight times inside "update_view()", for efficiency.
 *
 * The algorithm is very fast, since it checks the "obvious" grids very
 * quickly, and only has to call "update_view_los()" on the questionable
 * cases, and then it can perform some relatively efficient line of sight
 * computations.  Note that very few grids which are not in line of sight
 * are checked at all, and the ones which are usually get ruled out by the
 * more efficient checks, not by the expensive "update_view_los()" function.
 *
 * The worst "possible" case for this algorithm is when the player is running
 * through a dungeon level in which there are only a few wall grids remaining,
 * scattered in a pathalogical manner throughout the dungeon level.  In this
 * case, each of the 1149 grids in the theoretical "octagon of view" will be
 * checked once, most of them getting an "instant" rating, but with many of
 * them requiring an expensive call to "update_view_los()".
 *
 * The worst "normal" case for this algorithm is when the player is running
 * through the middle of the town, since the town contains large open regions
 * and a few scattered weirdly placed obstructions.  In this case, more than
 * half of the grids in the theoretical "octagon of view" are viewable by the
 * player.
 *
 * The best "normal" case for this algorithm is when the player is running
 * down a normal stretch of corridor with closed doors at both ends.  In
 * this case, the algorithm makes exactly one check for each viewable grid,
 * and makes no calls whatsoever to "update_view_los()".  If the corridor
 * opens into a room, as the player approaches the room, the algorithm gets
 * slightly less efficient, making a few calls to "update_view_los()", and
 * as the player enters the room, and walks around misaligned by one or two
 * grids from each corridor which enters the room, the algorithm is forced
 * to make quite a few calls to "update_view_los()", sometimes as many as
 * ten (or more) times.  The "update_view_los()" function itself has been
 * optimized for the common case in which the player is exactly one grid
 * misaligned from a corridor, but even this case requires computation.
 *
 * The pictures below shows the player and one quandrant (two octants) of the
 * viewable region (with MAX_RANGE defined as 10).  The first picture shows
 * the grid processing order.  The pure diagonals and main axes are processed
 * first, up to the appropriate distance, and then the sequential strips are
 * processed (one in each octant), also up to the appropriate distance, or
 * until the strip-termination grid is reached.
 *
 * For each grid "g", we consider the two grids "g1" and "g2" which touch
 * the grid "g" and which stand between "g" and the player, with "g1" being
 * "diagonally" in the direction of the player, and "g2" being the same
 * distance from the main axis as the grid "g".  In the second picture, we
 * show three "g","g1","g2" sets (shown as "g","a","b").
 *
 * The remaining pictures shows the effects of some common placements of
 * wall grids (shown as "#"), and the grids which do not even have to be
 * checked (shown as "x").  Note that wall grids in "strange" locations
 * (as in the third picture) do not allow us to bypass any grid checking,
 * since the walls behind them may be viewable by some other means, but
 * wall grids in "normal" locations (as in the fourth picture) allow us
 * to skip quite a bit of checking, including some entire strips, and as
 * an interesting corollary, any "illegal" grids.
 *
 *    @0000000000    @0000000000    @#xxxxxxxxx    @0000000000
 *    00111111111    00111a11111    00xxxxxxxxx    00111111111
 *    0102222222.    01022bg222.    010xxxxxxx.    0102222222.
 *    0120333333.    0120333333.    0120xxxxxx.    0120333333.
 *    012304444..    012304444..    01230xxxx..    #12304444..
 *    012340555..    012340555..    012340xxx..    x####0555..
 *    01234506...    01234a06...    #123450x...    xxxxxx06...
 *    0123456....    01234bg....    x1#3456....    xxxxxxx....
 *    012345.....    012345.....    x12345.....    xxxxxx.....
 *    0123.......    0ba3.......    x123.......    xxxx.......
 *    01.........    0g.........    x1.........    xx.........
 */



void update_view(void)
{
	int py = p_ptr->py;
	int px = p_ptr->px;

	int pg = GRID(py,px);

	int ay, ax, ad;

	int i, d, g, n;

	int qse, qsw, qne, qnw;
	int qes, qen, qws, qwn;

	int radius;

	int fast_view_n = view_n;
	u16b *fast_view_g = view_g;

	int fast_temp_n = 0;
	u16b *fast_temp_g = temp_g;

	byte *fast_cave_info = &cave_info[0][0];

	byte info;


	/*** Step 0 -- Begin ***/

	/* Save the old "view" grids for later */
	for (i = 0; i < fast_view_n; i++)
	{
		/* Grid */
		g = fast_view_g[i];

		/* Get grid info */
		info = fast_cave_info[g];

		/* Save "CAVE_SEEN" grids */
		if (info & (CAVE_SEEN))
		{
			/* Set "CAVE_TEMP" flag */
			info |= (CAVE_TEMP);

			/* Save grid for later */
			fast_temp_g[fast_temp_n++] = g;
		}

		/* Clear "CAVE_VIEW" and "CAVE_SEEN" flags */
		info &= ~(CAVE_VIEW | CAVE_SEEN);

		/* Clear "CAVE_LITE" flag */
		/* info &= ~(CAVE_LITE); */

		/* Save grid info */
		fast_cave_info[g] = info;
	}

	/* Reset the "view" array */
	fast_view_n = 0;


	/*** Step 1 -- adjacent grids ***/

	/* Player grid */
	g = pg;

	/* Get grid info */
	info = fast_cave_info[g];

	/* Player grid is easily viewable */
	info |= (CAVE_VIEW | CAVE_EASY);
	fast_cave_info[g] = info;
	fast_view_g[fast_view_n++] = g;


	/*** Step 2 -- Major Diagonals ***/

	/* Scan south-east */
	g = pg + GRID(+1,+1);
	for (d = 1; d < (MAX_SIGHT * 7 / 10); d++)
	{
		/* Get grid info */
		info = fast_cave_info[g];

		/* Grid is easily viewable */
		info |= (CAVE_VIEW | CAVE_EASY);
		fast_cave_info[g] = info;
		fast_view_g[fast_view_n++] = g;

		/* Stop on walls */
		if (info & CAVE_WALL) break;

		/* Advance */
		g += GRID(+1,+1);
	}

	/* Scan south-west */
	g = pg + GRID(+1,-1);
	for (d = 1; d < (MAX_SIGHT * 7 / 10); d++)
	{
		/* Get grid info */
		info = fast_cave_info[g];

		/* Grid is easily viewable */
		info |= (CAVE_VIEW | CAVE_EASY);
		fast_cave_info[g] = info;
		fast_view_g[fast_view_n++] = g;

		/* Stop on walls */
		if (info & CAVE_WALL) break;

		/* Advance */
		g += GRID(+1,-1);
	}

	/* Scan north-east */
	g = pg + GRID(-1,+1);
	for (d = 1; d < (MAX_SIGHT * 7 / 10); d++)
	{
		/* Get grid info */
		info = fast_cave_info[g];

		/* Grid is easily viewable */
		info |= (CAVE_VIEW | CAVE_EASY);
		fast_cave_info[g] = info;
		fast_view_g[fast_view_n++] = g;

		/* Stop on walls */
		if (info & CAVE_WALL) break;

		/* Advance */
		g += GRID(-1,+1);
	}

	/* Scan north-west */
	g = pg + GRID(-1,-1);
	for (d = 1; d < (MAX_SIGHT * 7 / 10); d++)
	{
		/* Get grid info */
		info = fast_cave_info[g];

		/* Grid is easily viewable */
		info |= (CAVE_VIEW | CAVE_EASY);
		fast_cave_info[g] = info;
		fast_view_g[fast_view_n++] = g;

		/* Stop on walls */
		if (info & CAVE_WALL) break;

		/* Advance */
		g += GRID(-1,-1);
	}


	/*** Step 3 -- major axes ***/

	/* Scan south */
	g = pg + GRID(+1,+0);
	for (d = 1; d <= MAX_SIGHT; d++)
	{
		/* Get grid info */
		info = fast_cave_info[g];

		/* Grid is easily viewable */
		info |= (CAVE_VIEW | CAVE_EASY);
		fast_cave_info[g] = info;
		fast_view_g[fast_view_n++] = g;

		/* Stop on walls */
		if (info & CAVE_WALL) break;

		/* Advance */
		g += GRID(+1,+0);
	}

	/* Initialize the "south strips" */
	qse = qsw = d;

	/* Scan north */
	g = pg + GRID(-1,+0);
	for (d = 1; d <= MAX_SIGHT; d++)
	{
		/* Get grid info */
		info = fast_cave_info[g];

		/* Grid is easily viewable */
		info |= (CAVE_VIEW | CAVE_EASY);
		fast_cave_info[g] = info;
		fast_view_g[fast_view_n++] = g;

		/* Stop on walls */
		if (info & CAVE_WALL) break;

		/* Advance */
		g += GRID(-1,+0);
	}

	/* Initialize the "north strips" */
	qne = qnw = d;

	/* Scan east */
	g = pg + GRID(+0,+1);
	for (d = 1; d <= MAX_SIGHT; d++)
	{
		/* Get grid info */
		info = fast_cave_info[g];

		/* Grid is easily viewable */
		info |= (CAVE_VIEW | CAVE_EASY);
		fast_cave_info[g] = info;
		fast_view_g[fast_view_n++] = g;

		/* Stop on walls */
		if (info & CAVE_WALL) break;

		/* Advance */
		g += GRID(+0,+1);
	}

	/* Initialize the "east strips" */
	qes = qen = d;

	/* Scan west */
	g = pg + GRID(+0,-1);
	for (d = 1; d <= MAX_SIGHT; d++)
	{
		/* Get grid info */
		info = fast_cave_info[g];

		/* Grid is easily viewable */
		info |= (CAVE_VIEW | CAVE_EASY);
		fast_cave_info[g] = info;
		fast_view_g[fast_view_n++] = g;

		/* Stop on walls */
		if (info & CAVE_WALL) break;

		/* Advance */
		g += GRID(+0,-1);
	}

	/* Initialize the "west strips" */
	qws = qwn = d;


	/*** Step 4 -- Process each "octant" as "strips" ***/

	/* Now check each "strip" */
	for (n = 1; n < (MAX_SIGHT * 7 / 10); n++)
	{
		int npz;

		/* Octagonal limit */
		npz = MAX_SIGHT - (n >> 1);

		/* East strip, South side */
		update_view_strip(qes, pg, GRID(+1,+1), GRID(+0,+1), n, npz);

		/* South strip, East side */
		update_view_strip(qse, pg, GRID(+1,+1), GRID(+1,+0), n, npz);

		/* South strip, West side */
		update_view_strip(qsw, pg, GRID(+1,-1), GRID(+1,+0), n, npz);

		/* West strip, South side */
		update_view_strip(qws, pg, GRID(+1,-1), GRID(+0,-1), n, npz);

		/* West strip, North side */
		update_view_strip(qwn, pg, GRID(-1,-1), GRID(+0,-1), n, npz);

		/* North strip, West side */
		update_view_strip(qnw, pg, GRID(-1,-1), GRID(-1,+0), n, npz);

		/* North strip, East side */
		update_view_strip(qne, pg, GRID(-1,+1), GRID(-1,+0), n, npz);

		/* East strip, North side */
		update_view_strip(qen, pg, GRID(-1,+1), GRID(+0,+1), n, npz);
	}


	/*** Step 5 -- Complete the algorithm ***/

	/* Extract "radius" value */
	radius = p_ptr->cur_lite;

	/* Handle real light */
	if (radius > 0) ++radius;

	/* Process "new" grids */
	for (i = 0; i < fast_view_n; i++)
	{
		int y, x;

		/* Grid */
		g = fast_view_g[i];

		/* Location */
		y = GRID_Y(g);
		x = GRID_X(g);

		/* Get grid info */
		info = fast_cave_info[g];

		/* Clear "CAVE_EASY" flag */
		info &= ~(CAVE_EASY);

		/* Inline 'ad = distance(py,px,y,x)' */
		ay = (y > py) ? (y - py) : (py - y);
		ax = (x > px) ? (x - px) : (px - x);
		ad = (ay > ax) ? (ay + (ax>>1)) : (ax + (ay>>1));

		/* Check the light radius */
		if (ad < radius)
		{
			/* Mark as "CAVE_SEEN" */
			info |= (CAVE_SEEN);

			/* Mark as "CAVE_LITE" */
			/* info |= (CAVE_LITE); */
		}

		/* Check for "perma-lite" of the grid */
		else if (info & (CAVE_GLOW))
		{
			/* Floors are simple */
			if (!(info & (CAVE_WALL)))
			{
				/* Mark as "CAVE_SEEN" */
				info |= (CAVE_SEEN);
			}

			/* Walls are complicated */
			else
			{
				/* Hack -- move towards player */
				int yy = (y < py) ? (y + 1) : (y > py) ? (y - 1) : y;
				int xx = (x < px) ? (x + 1) : (x > px) ? (x - 1) : x;

#ifdef UPDATE_VIEW_COMPLEX_WALL_ILLUMINATION

				/* Check for "complex" illumination */
				if ((!(cave_info[yy][xx] & (CAVE_WALL)) &&
				      (cave_info[yy][xx] & (CAVE_GLOW))) ||
				    (!(cave_info[y][xx] & (CAVE_WALL)) &&
				      (cave_info[y][xx] & (CAVE_GLOW))) ||
				    (!(cave_info[yy][x] & (CAVE_WALL)) &&
				      (cave_info[yy][x] & (CAVE_GLOW))))
				{
					/* Mark as "CAVE_SEEN" */
					info |= (CAVE_SEEN);
				}

#else /* UPDATE_VIEW_COMPLEX_WALL_ILLUMINATION */

				/* Check for "simple" illumination */
				if (cave_info[yy][xx] & (CAVE_GLOW))
				{
					/* Mark as "CAVE_SEEN" */
					info |= (CAVE_SEEN);
				}

#endif /* UPDATE_VIEW_COMPLEX_WALL_ILLUMINATION */

			}
		}

		/* Save grid info */
		fast_cave_info[g] = info;
	}

	/* Handle blindness */
	if (p_ptr->blind)
	{
		/* Process "new" grids */
		for (i = 0; i < fast_view_n; i++)
		{
			/* Grid */
			g = fast_view_g[i];

			/* Grid cannot be "CAVE_SEEN" */
			fast_cave_info[g] &= ~(CAVE_SEEN);
		}
	}

	/* Process "new" grids */
	for (i = 0; i < fast_view_n; i++)
	{
		/* Grid */
		g = fast_view_g[i];

		/* Get grid info */
		info = fast_cave_info[g];

		/* Was not "CAVE_SEEN", is now "CAVE_SEEN" */
		if ((info & (CAVE_SEEN)) && !(info & (CAVE_TEMP)))
		{
			int y, x;

			/* Location */
			y = GRID_Y(g);
			x = GRID_X(g);

			/* Note */
			note_spot(y, x);

			/* Redraw */
			lite_spot(y, x);
		}
	}

	/* Process "old" grids */
	for (i = 0; i < fast_temp_n; i++)
	{
		/* Grid */
		g = fast_temp_g[i];

		/* Get grid info */
		info = fast_cave_info[g];

		/* Clear "CAVE_TEMP" flag */
		info &= ~(CAVE_TEMP);

		/* Save cave info */
		fast_cave_info[g] = info;

		/* Was "CAVE_SEEN", is now not "CAVE_SEEN" */
		if (!(info & (CAVE_SEEN)))
		{
			int y, x;

			/* Location */
			y = GRID_Y(g);
			x = GRID_X(g);

			/* Redraw */
			lite_spot(y, x);
		}
	}


	/* Save 'view_n' */
	view_n = fast_view_n;
}

#endif /* defined(UPDATE_VIEW_NEW) */



#ifdef MONSTER_FLOW

/*
 * Hack -- provide some "speed" for the "flow" code
 * This entry is the "current index" for the "when" field
 * Note that a "when" value of "zero" means "not used".
 *
 * Note that the "cost" indexes from 1 to 127 are for
 * "old" data, and from 128 to 255 are for "new" data.
 *
 * This means that as long as the player does not "teleport",
 * then any monster up to 128 + MONSTER_FLOW_DEPTH will be
 * able to track down the player, and in general, will be
 * able to track down either the player or a position recently
 * occupied by the player.
 */
static int flow_n = 0;

#endif /* MONSTER_FLOW */



/*
 * Hack -- forget the "flow" information
 */
void forget_flow(void)
{

#ifdef MONSTER_FLOW

	int x, y;

	/* Nothing to forget */
	if (!flow_n) return;

	/* Check the entire dungeon */
	for (y = 0; y < DUNGEON_HGT; y++)
	{
		for (x = 0; x < DUNGEON_WID; x++)
		{
			/* Forget the old data */
			cave_cost[y][x] = 0;
			cave_when[y][x] = 0;
		}
	}

	/* Start over */
	flow_n = 0;

#endif

}


/*
 * Hack -- fill in the "cost" field of every grid that the player can
 * "reach" with the number of steps needed to reach that grid.  This
 * also yields the "distance" of the player from every grid.
 *
 * In addition, mark the "when" of the grids that can reach the player
 * with the incremented value of "flow_n".
 *
 * Hack -- use the "temp_y" and "temp_x" arrays as a "circular queue"
 * of cave grids.
 *
 * We do not need a priority queue because the cost from grid to grid
 * is always "one" (even along diagonals) and we process them in order.
 */
void update_flow(void)
{

#ifdef MONSTER_FLOW

	int py = p_ptr->py;
	int px = p_ptr->px;

	int ty, tx;

	int y, x;

	int n, d;

	int flow_tail = 0;
	int flow_head = 0;


	/* Hack -- disabled */
	if (!flow_by_sound) return;

	/* Paranoia -- make sure the array is empty */
	if (temp_n) return;


	/*** Cycle timestamp ***/

	/* Cycle the old entries (once per 128 updates) */
	if (flow_n == 255)
	{
		/* Rotate the time-stamps */
		for (y = 0; y < DUNGEON_HGT; y++)
		{
			for (x = 0; x < DUNGEON_WID; x++)
			{
				int w;
				w = cave_when[y][x];
				cave_when[y][x] = (w > 128) ? (w - 128) : 0;
			}
		}

		/* Restart */
		flow_n = 127;
	}

	/* Start a new flow (never use "zero") */
	flow_n++;


	/*** Player Grid ***/

	/* Save the time-stamp */
	cave_when[py][px] = flow_n;

	/* Save the flow cost */
	cave_cost[py][px] = 0;

	/* Enqueue that entry */
	temp_y[flow_head] = py;
	temp_x[flow_head] = px;

	/* Advance the queue */
	++flow_tail;


	/*** Process Queue ***/

	/* Now process the queue */
	while (flow_head != flow_tail)
	{
		/* Extract the next entry */
		ty = temp_y[flow_head];
		tx = temp_x[flow_head];

		/* Forget that entry (with wrap) */
		if (++flow_head == TEMP_MAX) flow_head = 0;

		/* Add the "children" */
		for (d = 0; d < 8; d++)
		{
			int old_head = flow_tail;

			/* Child location */
			y = ty + ddy_ddd[d];
			x = tx + ddx_ddd[d];
			n = cave_cost[y][x] + 1;

			/* Ignore "pre-stamped" entries */
			if (cave_when[y][x] == flow_n) continue;

			/* Ignore "walls" and "rubble" */
			if (cave_feat[y][x] >= FEAT_RUBBLE) continue;

			/* Save the time-stamp */
			cave_when[y][x] = flow_n;

			/* Save the flow cost */
			cave_cost[y][x] = n;

			/* Hack -- limit flow depth */
			if (n == MONSTER_FLOW_DEPTH) continue;

			/* Enqueue that entry */
			temp_y[flow_tail] = y;
			temp_x[flow_tail] = x;

			/* Advance the queue */
			if (++flow_tail == TEMP_MAX) flow_tail = 0;

			/* Hack -- Overflow by forgetting new entry */
			if (flow_tail == flow_head) flow_tail = old_head;
		}
	}

#endif

}




/*
 * Map the current panel (plus some) ala "magic mapping"
 *
 * We must never attempt to map the outer dungeon walls, or we
 * might induce illegal cave grid references.
 */
void map_area(void)
{
	int i, x, y, y1, y2, x1, x2;


	/* Pick an area to map */
	y1 = p_ptr->wy - randint(10);
	y2 = p_ptr->wy+SCREEN_HGT + randint(10);
	x1 = p_ptr->wx - randint(20);
	x2 = p_ptr->wx+SCREEN_WID + randint(20);

	/* Efficiency -- shrink to fit legal bounds */
	if (y1 < 1) y1 = 1;
	if (y2 > DUNGEON_HGT-1) y2 = DUNGEON_HGT-1;
	if (x1 < 1) x1 = 1;
	if (x2 > DUNGEON_WID-1) x2 = DUNGEON_WID-1;

	/* Scan that area */
	for (y = y1; y < y2; y++)
	{
		for (x = x1; x < x2; x++)
		{
			/* All non-walls are "checked" */
			if (cave_feat[y][x] < FEAT_SECRET)
			{
				/* Memorize normal features */
				if (cave_feat[y][x] > FEAT_INVIS)
				{
					/* Memorize the object */
					cave_info[y][x] |= (CAVE_MARK);
				}

				/* Memorize known walls */
				for (i = 0; i < 8; i++)
				{
					int yy = y + ddy_ddd[i];
					int xx = x + ddx_ddd[i];

					/* Memorize walls (etc) */
					if (cave_feat[yy][xx] >= FEAT_SECRET)
					{
						/* Memorize the walls */
						cave_info[yy][xx] |= (CAVE_MARK);
					}
				}
			}
		}
	}

	/* Redraw map */
	p_ptr->redraw |= (PR_MAP);

	/* Window stuff */
	p_ptr->window |= (PW_OVERHEAD);
}



/*
 * Light up the dungeon using "claravoyance"
 *
 * This function "illuminates" every grid in the dungeon, memorizes all
 * "objects", memorizes all grids as with magic mapping, and, under the
 * standard option settings (view_perma_grids but not view_torch_grids)
 * memorizes all floor grids too.
 *
 * Note that if "view_perma_grids" is not set, we do not memorize floor
 * grids, since this would defeat the purpose of "view_perma_grids", not
 * that anyone seems to play without this option.
 *
 * Note that if "view_torch_grids" is set, we do not memorize floor grids,
 * since this would prevent the use of "view_torch_grids" as a method to
 * keep track of what grids have been observed directly.
 */
void wiz_lite(void)
{
	int i, y, x;


	/* Memorize objects */
	for (i = 1; i < o_max; i++)
	{
		object_type *o_ptr = &o_list[i];

		/* Skip dead objects */
		if (!o_ptr->k_idx) continue;

		/* Skip held objects */
		if (o_ptr->held_m_idx) continue;

		/* Memorize */
		o_ptr->marked = TRUE;
	}

	/* Scan all normal grids */
	for (y = 1; y < DUNGEON_HGT-1; y++)
	{
		/* Scan all normal grids */
		for (x = 1; x < DUNGEON_WID-1; x++)
		{
			/* Process all non-walls */
			if (cave_feat[y][x] < FEAT_SECRET)
			{
				/* Scan all neighbors */
				for (i = 0; i < 9; i++)
				{
					int yy = y + ddy_ddd[i];
					int xx = x + ddx_ddd[i];

					/* Perma-lite the grid */
					cave_info[yy][xx] |= (CAVE_GLOW);

					/* Memorize normal features */
					if (cave_feat[yy][xx] > FEAT_INVIS)
					{
						/* Memorize the grid */
						cave_info[yy][xx] |= (CAVE_MARK);
					}

					/* Normally, memorize floors (see above) */
					if (view_perma_grids && !view_torch_grids)
					{
						/* Memorize the grid */
						cave_info[yy][xx] |= (CAVE_MARK);
					}
				}
			}
		}
	}

	/* Fully update the visuals */
	p_ptr->update |= (PU_FORGET_VIEW | PU_UPDATE_VIEW | PU_MONSTERS);

	/* Redraw map */
	p_ptr->redraw |= (PR_MAP);

	/* Window stuff */
	p_ptr->window |= (PW_OVERHEAD);
}


/*
 * Forget the dungeon map (ala "Thinking of Maud...").
 */
void wiz_dark(void)
{
	int i, y, x;

	
	/* Forget every grid */
	for (y = 0; y < DUNGEON_HGT; y++)
	{
		for (x = 0; x < DUNGEON_WID; x++)
		{
			/* Process the grid */
			cave_info[y][x] &= ~(CAVE_MARK);
		}
	}

	/* Forget all objects */
	for (i = 1; i < o_max; i++)
	{
		object_type *o_ptr = &o_list[i];

		/* Skip dead objects */
		if (!o_ptr->k_idx) continue;

		/* Skip held objects */
		if (o_ptr->held_m_idx) continue;

		/* Forget the object */
		o_ptr->marked = FALSE;
	}

	/* Fully update the visuals */
	p_ptr->update |= (PU_FORGET_VIEW | PU_UPDATE_VIEW | PU_MONSTERS);

	/* Redraw map */
	p_ptr->redraw |= (PR_MAP);

	/* Window stuff */
	p_ptr->window |= (PW_OVERHEAD);
}



/*
 * Light or Darken the town
 */
void town_illuminate(bool daytime)
{
	int y, x, i;


	/* Apply light or darkness */
	for (y = 0; y < DUNGEON_HGT; y++)
	{
		for (x = 0; x < DUNGEON_WID; x++)
		{
			/* Interesting grids */
			if (cave_feat[y][x] > FEAT_INVIS)
			{
				/* Illuminate the grid */
				cave_info[y][x] |= (CAVE_GLOW);

				/* Memorize the grid */
				cave_info[y][x] |= (CAVE_MARK);
			}

			/* Boring grids (light) */
			else if (daytime)
			{
				/* Illuminate the grid */
				cave_info[y][x] |= (CAVE_GLOW);

				/* Hack -- Memorize grids */
				if (view_perma_grids)
				{
					cave_info[y][x] |= (CAVE_MARK);
				}
			}

			/* Boring grids (dark) */
			else
			{
				/* Darken the grid */
				cave_info[y][x] &= ~(CAVE_GLOW);

				/* Hack -- Forget grids */
				if (view_perma_grids)
				{
					cave_info[y][x] &= ~(CAVE_MARK);
				}
			}
		}
	}


	/* Handle shop doorways */
	for (y = 0; y < DUNGEON_HGT; y++)
	{
		for (x = 0; x < DUNGEON_WID; x++)
		{
			/* Track shop doorways */
			if ((cave_feat[y][x] >= FEAT_SHOP_HEAD) &&
			    (cave_feat[y][x] <= FEAT_SHOP_TAIL))
			{
				for (i = 0; i < 8; i++)
				{
					int yy = y + ddy_ddd[i];
					int xx = x + ddx_ddd[i];

					/* Illuminate the grid */
					cave_info[yy][xx] |= (CAVE_GLOW);
	
					/* Hack -- Memorize grids */
					if (view_perma_grids)
					{
						cave_info[yy][xx] |= (CAVE_MARK);
					}
				}
			}
		}
	}


	/* Fully update the visuals */
	p_ptr->update |= (PU_FORGET_VIEW | PU_UPDATE_VIEW | PU_MONSTERS);

	/* Redraw map */
	p_ptr->redraw |= (PR_MAP);

	/* Window stuff */
	p_ptr->window |= (PW_OVERHEAD);
}



/*
 * Change the "feat" flag for a grid, and notice/redraw the grid
 */
void cave_set_feat(int y, int x, int feat)
{
	/* Change the feature */
	cave_feat[y][x] = feat;

	/* Handle "wall/door" grids */
	if (feat >= FEAT_DOOR_HEAD)
	{
		cave_info[y][x] |= (CAVE_WALL);
	}

	/* Handle "floor"/etc grids */
	else
	{
		cave_info[y][x] &= ~(CAVE_WALL);
	}

	/* Notice/Redraw */
	if (character_dungeon)
	{
		/* Notice */
		note_spot(y, x);

		/* Redraw */
		lite_spot(y, x);
	}
}



/*
 * Determine the path taken by a projection.
 *
 * The projection will always start from the grid (y1,x1), and will travel
 * towards the grid (y2,x2), touching one grid per unit of distance along
 * the major axis, and stopping when it enters the destination grid or a
 * wall grid, or has travelled the maximum legal distance of "range".
 *
 * Note that "distance" in this function (as in the "update_view()" code)
 * is defined as "MAX(dy,dx) + MIN(dy,dx)/2", which means that the player
 * actually has an "octagon of projection" not a "circle of projection".
 *
 * The path grids are saved into the grid array pointed to by "gp", and
 * there should be room for at least "range" grids in "gp".  Note that
 * due to the way in which distance is calculated, this function normally
 * uses fewer than "range" grids for the projection path, so the result
 * of this function should never be compared directly to "range".  Note
 * that the initial grid (y1,x1) is never saved into the grid array, not
 * even if the initial grid is also the final grid.  XXX XXX XXX
 *
 * The "flg" flags can be used to modify the behavior of this function.
 *
 * In particular, the "PROJECT_STOP" and "PROJECT_THRU" flags have the same
 * semantics as they do for the "project" function, namely, that the path
 * will stop as soon as it hits a monster, or that the path will continue
 * through the destination grid, respectively.
 *
 * The "PROJECT_JUMP" flag, which for the "project()" function means to
 * start at a special grid (which makes no sense in this function), means
 * that the path should be "angled" slightly if needed to avoid any wall
 * grids, allowing the player to "target" any grid which is in "view".
 * This flag is non-trivial and has not yet been implemented, but could
 * perhaps make use of the "vinfo" array (above).  XXX XXX XXX
 *
 * This function returns the number of grids (if any) in the path.  This
 * function will return zero if and only if (y1,x1) and (y2,x2) are equal.
 *
 * This algorithm is similar to, but slightly different from, the one used
 * by "update_view_los()", and very different from the one used by "los()".
 */
sint project_path(u16b *gp, int range, int y1, int x1, int y2, int x2, int flg)
{
	int y, x;

	int n = 0;
	int k = 0;

	/* Absolute */
	int ay, ax;

	/* Offsets */
	int sy, sx;

	/* Fractions */
	int frac;

	/* Scale factors */
	int full, half;

	/* Slope */
	int m;


	/* No path necessary (or allowed) */
	if ((x1 == x2) && (y1 == y2)) return (0);


	/* Analyze "dy" */
	if (y2 < y1)
	{
		ay = (y1 - y2);
		sy = -1;
	}
	else
	{
		ay = (y2 - y1);
		sy = 1;
	}

	/* Analyze "dx" */
	if (x2 < x1)
	{
		ax = (x1 - x2);
		sx = -1;
	}
	else
	{
		ax = (x2 - x1);
		sx = 1;
	}


	/* Number of "units" in one "half" grid */
	half = (ay * ax);

	/* Number of "units" in one "full" grid */
	full = half << 1;


	/* Vertical */
	if (ay > ax)
	{
		/* Start at tile edge */
		frac = ax * ax;

		/* Let m = ((dx/dy) * full) = (dx * dx * 2) = (frac * 2) */
		m = frac << 1;

		/* Start */
		y = y1 + sy;
		x = x1;

		/* Create the projection path */
		while (1)
		{
			/* Save grid */
			gp[n++] = GRID(y,x);

			/* Hack -- Check maximum range */
			if ((n + (k >> 1)) >= range) break;

			/* Sometimes stop at destination grid */
			if (!(flg & (PROJECT_THRU)))
			{
				if ((x == x2) && (y == y2)) break;
			}

			/* Always stop at non-initial wall grids */
			if ((n > 0) && !cave_floor_bold(y, x)) break;

			/* Sometimes stop at non-initial monsters/players */
			if (flg & (PROJECT_STOP))
			{
				if ((n > 0) && (cave_m_idx[y][x] != 0)) break;
			}

			/* Slant */
			if (m)
			{
				/* Advance (X) part 1 */
				frac += m;

				/* Horizontal change */
				if (frac >= half)
				{
					/* Advance (X) part 2 */
					x += sx;

					/* Advance (X) part 3 */
					frac -= full;

					/* Track distance */
					k++;
				}
			}

			/* Advance (Y) */
			y += sy;
		}
	}

	/* Horizontal */
	else if (ax > ay)
	{
		/* Start at tile edge */
		frac = ay * ay;

		/* Let m = ((dy/dx) * full) = (dy * dy * 2) = (frac * 2) */
		m = frac << 1;

		/* Start */
		y = y1;
		x = x1 + sx;

		/* Create the projection path */
		while (1)
		{
			/* Save grid */
			gp[n++] = GRID(y,x);

			/* Hack -- Check maximum range */
			if ((n + (k >> 1)) >= range) break;

			/* Sometimes stop at destination grid */
			if (!(flg & (PROJECT_THRU)))
			{
				if ((x == x2) && (y == y2)) break;
			}

			/* Always stop at non-initial wall grids */
			if ((n > 0) && !cave_floor_bold(y, x)) break;

			/* Sometimes stop at non-initial monsters/players */
			if (flg & (PROJECT_STOP))
			{
				if ((n > 0) && (cave_m_idx[y][x] != 0)) break;
			}

			/* Slant */
			if (m)
			{
				/* Advance (Y) part 1 */
				frac += m;

				/* Vertical change */
				if (frac >= half)
				{
					/* Advance (Y) part 2 */
					y += sy;

					/* Advance (Y) part 3 */
					frac -= full;

					/* Track distance */
					k++;
				}
			}

			/* Advance (X) */
			x += sx;
		}
	}

	/* Diagonal */
	else
	{
		/* Start */
		y = y1 + sy;
		x = x1 + sx;

		/* Create the projection path */
		while (1)
		{
			/* Save grid */
			gp[n++] = GRID(y,x);

			/* Hack -- Check maximum range */
			if ((n + (n >> 1)) >= range) break;

			/* Sometimes stop at destination grid */
			if (!(flg & (PROJECT_THRU)))
			{
				if ((x == x2) && (y == y2)) break;
			}

			/* Always stop at non-initial wall grids */
			if ((n > 0) && !cave_floor_bold(y, x)) break;

			/* Sometimes stop at non-initial monsters/players */
			if (flg & (PROJECT_STOP))
			{
				if ((n > 0) && (cave_m_idx[y][x] != 0)) break;
			}

			/* Advance (Y) */
			y += sy;

			/* Advance (X) */
			x += sx;
		}
	}


	/* Length */
	return (n);
}


/*
 * Determine if a bolt spell cast from (y1,x1) to (y2,x2) will arrive
 * at the final destination, assuming that no monster gets in the way,
 * using the "project_path()" function to check the projection path.
 *
 * Note that no grid is ever "projectable()" from itself.
 *
 * This function is used to determine if the player can (easily) target
 * a given grid, and if a monster can target the player.
 */
bool projectable(int y1, int x1, int y2, int x2)
{
	int y, x;

	s16b grid_n = 0;
	s16b grid_g[512];

	/* Check the projection path */
	grid_n = project_path(grid_g, MAX_RANGE, y1, x1, y2, x2, 0);

	/* No grid is ever projectable from itself */
	if (!grid_n) return (FALSE);

	/* Final grid */
	y = GRID_Y(grid_g[grid_n-1]);
	x = GRID_X(grid_g[grid_n-1]);

	/* May not end in a wall grid */
	if (!cave_floor_bold(y, x)) return (FALSE);

	/* May not end in an unrequested grid */
	if ((y != y2) || (x != x2)) return (FALSE);

	/* Assume okay */
	return (TRUE);
}



/*
 * Standard "find me a location" function
 *
 * Obtains a legal location within the given distance of the initial
 * location, and with "los()" from the source to destination location.
 *
 * This function is often called from inside a loop which searches for
 * locations while increasing the "d" distance.
 *
 * Currently the "m" parameter is unused.
 */
void scatter(int *yp, int *xp, int y, int x, int d, int m)
{
	int nx, ny;

	/* Unused */
	m = m;

	/* Pick a location */
	while (TRUE)
	{
		/* Pick a new location */
		ny = rand_spread(y, d);
		nx = rand_spread(x, d);

		/* Ignore annoying locations */
		if (!in_bounds_fully(y, x)) continue;

		/* Ignore "excessively distant" locations */
		if ((d > 1) && (distance(y, x, ny, nx) > d)) continue;

		/* Require "line of sight" */
		if (los(y, x, ny, nx)) break;
	}

	/* Save the location */
	(*yp) = ny;
	(*xp) = nx;
}






/*
 * Track a new monster
 */
void health_track(int m_idx)
{
	/* Track a new guy */
	p_ptr->health_who = m_idx;

	/* Redraw (later) */
	p_ptr->redraw |= (PR_HEALTH);
}



/*
 * Hack -- track the given monster race
 */
void monster_race_track(int r_idx)
{
	/* Save this monster ID */
	p_ptr->monster_race_idx = r_idx;

	/* Window stuff */
	p_ptr->window |= (PW_MONSTER);
}



/*
 * Hack -- track the given object kind
 */
void object_kind_track(int k_idx)
{
	/* Save this object ID */
	p_ptr->object_kind_idx = k_idx;

	/* Window stuff */
	p_ptr->window |= (PW_OBJECT);
}



/*
 * Something has happened to disturb the player.
 *
 * The first arg indicates a major disturbance, which affects search.
 *
 * The second arg is currently unused, but could induce output flush.
 *
 * All disturbance cancels repeated commands, resting, and running.
 */
void disturb(int stop_search, int unused_flag)
{
	/* Cancel auto-commands */
	/* p_ptr->command_new = 0; */

	/* Cancel repeated commands */
	if (p_ptr->command_rep)
	{
		/* Cancel */
		p_ptr->command_rep = 0;

		/* Redraw the state (later) */
		p_ptr->redraw |= (PR_STATE);
	}

	/* Cancel Resting */
	if (p_ptr->resting)
	{
		/* Cancel */
		p_ptr->resting = 0;

		/* Redraw the state (later) */
		p_ptr->redraw |= (PR_STATE);
	}

	/* Cancel running */
	if (p_ptr->running)
	{
		/* Cancel */
		p_ptr->running = 0;

		/* Calculate torch radius */
		p_ptr->update |= (PU_TORCH);

		/* Redraw the player */
		if (hidden_player)
		{
			int py = p_ptr->py;
			int px = p_ptr->px;

			/* Redraw player */
			lite_spot(py, px);
		}
	}

	/* Cancel searching if requested */
	if (stop_search && p_ptr->searching)
	{
		/* Cancel */
		p_ptr->searching = FALSE;

		/* Recalculate bonuses */
		p_ptr->update |= (PU_BONUS);

		/* Redraw the state */
		p_ptr->redraw |= (PR_STATE);
	}

	/* Flush the input if requested */
	if (flush_disturb) flush();
}




/*
 * Hack -- Check if a level is a "quest" level
 */
bool is_quest(int level)
{
	int i;

	/* Town is never a quest */
	if (!level) return (FALSE);

	/* Check quests */
	for (i = 0; i < MAX_Q_IDX; i++)
	{
		/* Check for quest */
		if (q_list[i].level == level) return (TRUE);
	}

	/* Nope */
	return (FALSE);
}







