/* irq_driver.c : interrupt driver for the PMS frame grabber
   it simply handles the interrupts that trigger the read op
   after this read it is disabled until the next write op
   partially adopted from David Etherton
 */

/* Kernel includes */

#include <linux/errno.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/fs.h>
#include <linux/major.h>

#include <asm/segment.h>
#include <asm/io.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/signal.h>
#include <linux/module.h>
#include <linux/ioport.h>
#include <linux/mman.h>
#include <linux/malloc.h>
#include <linux/string.h>

#define dprintk(x)	printk x

/*
 * NB. we must include the kernel idenfication string in to install the module.
 */
#include <linux/version.h>
char kernel_version[] = UTS_RELEASE;

#ifndef FALSE
#define FALSE	0
#define TRUE	1
#endif

#define FG_MAJOR    31 /* nice and high */


/* if your version of insmod is recent enough, you can reassign these
   variable appropriately from the command line. */

static int ProMovieSource = 1; /* use 0 for composite, 1 for S-video */

static int ProMovieInUse = FALSE;
#include "mvv_parms.h" 

static inline void ProMovieWrite(unsigned char port,unsigned char value) {
  outw(port | (value << 8), ProMoviePort);
}

static inline int ProMovieRead(unsigned char port) {
  outb_p(port, ProMoviePort);
  return inw(ProMoviePort) >> 8;
}

static void ProMovieSetIRQ(int irq) {
  /* set irq */
  int temp = inw(ProMoviePort);
  outw(0x882E,ProMoviePort);
  outb(temp,ProMoviePort);
  outb((inb(ProMoviePort + 2) & 0xF0) | irq, ProMoviePort + 2);
  if (ProMovieSource == 1) /* select video source , must be first here !!! */
    outw(0x312E,ProMoviePort); 
  else 
    outw(0x302E,ProMoviePort);
}

static void ProMovieIRQEnable(void) {
  ProMovieWrite(0x01,ProMovieRead(0x01) | 0x01);
}

static void ProMovieIRQDisable(void) {
  ProMovieWrite(0x01,ProMovieRead(0x01) & 0xFC);
}

static void ProMovieIRQClear(void) {
  int index = inw(ProMoviePort) & 0x7F;
  ProMovieWrite(0x00,0xFF);
  outb(index,ProMoviePort);
}

/*
 * The driver.
 */

static struct wait_queue *fg_wait_queue = 0;

#define STATIC
int request = 0;
int ir_counter = 0;
STATIC void fg_interrupt(int dummy, struct pt_regs *ps) {
  ir_counter++;
  if (request) {
    request = 0;
    wake_up_interruptible(&fg_wait_queue);
  }
  ProMovieIRQClear();
}

/* returns only number of occurred interrupts since last read */

STATIC int fg_read(struct inode *node, struct file *file,char *buf,int count) {
  if (file->f_flags & O_NONBLOCK) return -EWOULDBLOCK;
  request = 1;   
  /* on/off setting gets jumping frames */
  /*  outw(0x0110,ProMoviePort);  single capture on */  
  interruptible_sleep_on(&fg_wait_queue);
  /* outw(0x0010,ProMoviePort);  capture off */
  if (current->signal & ~current->blocked) return -EINTR;
  memcpy_tofs(buf, &ir_counter, 4); /* return number of counted irqs */
  ir_counter = 0;
  return count;
}

STATIC int fg_open(struct inode* ino, struct file* filep) {
  int error;

  if (ProMovieInUse) return -EBUSY;

  /* we don't try to grab the interrupt until the device
     is actually opened to avoid hogging it any longer
     than necessary. */

 switch (ProMovieIRQ) { 
    /* these are the only ones recommended by MV */
  case 2: case 3: case 5: case 7: case 10:
  case 11: case 12: case 15: 
    break;
  default:
    return -EINVAL;
  }

  error = request_irq(ProMovieIRQ, fg_interrupt, 0, "fgrabber");

  if (!error) {
    ProMovieSetIRQ(ProMovieIRQ);   
    ProMovieIRQEnable();
    ProMovieInUse = TRUE;
  }
  
  return error;
}

STATIC void fg_close(struct inode* ino, struct file* filep) {
  ProMovieIRQDisable();
  free_irq(ProMovieIRQ);
  /* disable IRQ and map out ram */
  /* ProMovieConfig(0); */
  ProMovieInUse = FALSE;
}

static struct file_operations fg_fops = {
  NULL,		/* fg_lseek */
  fg_read,
  NULL,         /* fg_write */
  NULL,		/* fg_readdir */
  NULL,		/* fg_select */
  NULL,	        /* fg_ioctl */
  NULL,  	/* fg_mmap */
  fg_open,
  fg_close,
  /* don't care about anything after this... */
};


/*
 * And now the modules code and kernel interface.
 */

#ifdef __cplusplus
extern "C" {
#endif

extern int printk(const char* fmt, ...);

int init_module (void) {
  static char signon[] = "ProMovie 0.3 Copyright 1994, David Etherton: ";
  if (register_chrdev(FG_MAJOR, "fg", &fg_fops)) {
    printk("%s register_chrdev failed.\n",signon);
    return -EIO;
  }
  else 
    printk("%s\nUsing port 0x%x, irq %d\n",signon, ProMoviePort, ProMovieIRQ);
  return 0;
}

void cleanup_module(void)
{
  if (ProMovieInUse)
    printk("ProMovie: device busy.\n");
  else if (unregister_chrdev(FG_MAJOR, "fg") != 0)
    printk("ProMovie: unregister_chrdev failed\n");
  else {
    /* turn off port decoding... */
    outb(0xB8, 0x9A01);
  }
}

#ifdef __cplusplus
}
#endif

