// view.c : functions for display an image under X-window
//  a part of PMS-grabber package
// the "GNU Public Lincense" policy applies to all programs of this package
// (c) Wolfgang Koehler, wolf@first.gmd.de, Dec. 1994
//     Kiefernring 15
//     14478 Potsdam, Germany
//     0331-863238

#include "window.h"
#include <X11/Xutil.h>
#include "ximage.h"
#include "evaluation.h"
#include "view.h"
#include "grabber.h"

#include <fcntl.h>
#include <unistd.h>

view_window *grab_wi;
unsigned short *cap_buf = NULL;
XImage *ximage = NULL;
mapped_image *act_image = 0;

int biWidth, biHeight;
int tcap = 0, tdraw = 0; // delta-t between two grabs: capture & draw

// handles interrupt wait of frame grabber with installed irq_driver
class irq_handle {
  int fid;
  int nthr;
public: 
  irq_handle() {
    fid = open("/dev/fg", 0);
    // printf("fid = %d\n",fid);
    nthr = 0;
  }
  ~irq_handle() {
    if (fid > 0) close(fid);
  }
  void wait_irq() {
    if (fid > 0)  { // simply waits until read is performed
      int ni; read(fid,&ni,4); // read number of irqs between two read 
      if (ni > 1) nthr++;
      if (nthr > 20) { 
	int max_fr = 100/(tcap+tdraw); nthr = 0; 
	printf("warning : frame rate too high, set to max %d %d\n",max_fr,ni);
      }
    } 
  }
};

// capture one frame, transform with coltab and draw into grab_wi
void show_capt() {
  if (cap_buf == NULL) return;

  static irq_handle fg_irq;  // not global (init time too early) !!
  fg_irq.wait_irq();
  Bool rgb555 = ! True_Color; // in PseudoColor use only cap_mode 555 !!
  fast_capture(cap_buf,biWidth,biHeight, rgb555);
 
  tcap = comp_time();  // 320x240: 0.1, 500x400 : 0.17 sec
  map_and_redraw();
}    

#include <sys/times.h>
// computes the time since last call to this function
int comp_time() {
  struct tms tt; 
  unsigned tx = times(&tt);  // returns time since boot in 1/100 sec ticks
  static unsigned tl = 0;
  int delt = tx - tl; 
  tl = tx;
  return delt;
}

// uses coltab555 to directly map a rgb555-buffer into a XImage
void map_buf555(unsigned short *buf, int n) {
  unsigned char *imb =  (unsigned char *) ximage->data;
  if (imb == NULL) return;  
  int i; 
  for (i=0; i < n; i++) *imb++ = coltab555[ *buf++ ];
}

void map_and_redraw() {
  int n = biWidth*biHeight;
  if( ! True_Color) map_buf555(cap_buf, n);   
  // full color transformation : 320x240 : 0.10 sec
  grab_wi->redraw();    // 320x240 
  tdraw = comp_time(); // wird erst beim naechsten Bild dargestellt
}

int sharpness() {
  int xm = biWidth/2, ym = biHeight/2; 
  int x,y,x0 = xm-50, x1 = xm+50, y0 = ym-50, y1 = ym+50; 
  int s = 0;
  for (y = y0; y < y1; y++) for (x = x0; x < x1; x++) {
    short (*image) [biWidth] = (short (*)[biWidth]) cap_buf;
    short dx,dy,z; 
    dx = image[y][x] - image[y][x-1];
    dy = image[y][x] - image[y-1][x];
    z = SQR(b555(dx)) + SQR(g555(dx)) + SQR(r555(dx)) + 
	SQR(b555(dy)) + SQR(g555(dy)) + SQR(r555(dy));
    s+= z;
  }
  return (s);
}

// ****** class "view_window" ***** 
// size is determined from globals bi...
view_window::view_window(window &parent, int x, int y, info_window *infw) :
  window(parent,biWidth,biHeight,x,y,0), infw(infw) { 
  // info window at bottom
  selection_mask |= ButtonPressMask | ButtonReleaseMask | PointerMotionMask;
  Cursor grab_cursor = XCreateFontCursor(display,24); // XC_circle
  XDefineCursor(display,Win,grab_cursor);
  redraw_chain = NULL;
  create_buffers(biWidth, biHeight);
}

// info about cursor point coloring
void view_window::Motion_CB(XMotionEvent ev) {
  short c,b,r,g;  // first part : values from capture buffer  
  // printf("%d %d,",ev.x,ev.y); fflush(stdout);
  c = cap_buf[ev.x + ev.y*biWidth]; 
  b = b565(c);
  g = g565(c);
  r = r565(c);
  short lum = r+g+b; // luminance value

  int rf = (100*r)/31, gf = (100*g)/63, bf = (100*b)/31; // procent values
  sprintf(infw->info,"(%3d,%3d) c=(%2d,%2d,%2d) %3d (%2d,%2d,%2d)",
	  ev.x,ev.y,r,g,b,lum,rf,gf,bf);

  infw->redraw();
}

void view_window::redraw() { 
  if (ximage == NULL) return;
  int ww = MIN(width,ximage->width),
      wh = MIN(height,ximage->height);

  // use the macro from ximage.h
  act_image->PutImage(Win,0,0,ww,wh);
 
  char * format = (width < 300) ? "%3d %3d %.2f %.2f %d": 
                                    "w = %d h = %d tc = %.2f td = %.2f %d"; 
  int sharp = 0; // = sharpness();
  sprintf(infw->info,format,ww,wh,0.01*tcap,0.01*tdraw,sharp);
  infw->redraw();
  if (redraw_chain) (*redraw_chain)();
}

void view_window::Expose_CB(XExposeEvent ev) {
   XPutImage(display,Win,gc_copy,ximage,ev.x,ev.y,ev.x,ev.y,
             ev.width,ev.height); 
} 

// create a new object of mapped_image of requested size, 
// delete the old one (if present)
// set the globals "cap_buf" and "ximage", "act_image"
void create_buffers(int w, int h) {
  if (act_image) delete act_image;
  act_image = new mapped_image(w,h, True_Color);
  cap_buf = act_image->buf_16;
  ximage = act_image->ximage;
}  

// adjusting the size of the main window to fit an image ww*hh
void adjust_size(int ww, int hh) {
  window *mw = grab_wi->mainw; // not yet consistent
  mw->resize(ww,hh+40);
  XResizeWindow(display,mw->Win,ww,hh+40); 
}

