/*
 * scud-busters solution
 * V. Khera	29-OCT-1991
 *
 * compile with gcc
 */

#include <stdio.h>

#define MAX_PTS 101	/* most number of points in kingdom */
#define MAX_KING 20	/* most number of kingdoms */

typedef struct point {
	int x,y;
} point_t;

typedef struct kingdom {
	int size;	/* number of points in convex hull */
	double area;	/* total area of this polygon */
	point_t list[MAX_PTS+1]; /* list of points in convex hull */
} kingdom_t;


static kingdom_t kingdoms[MAX_KING];
static int num_kingdoms = 0;

/* return the absolute value of an integer */
static inline int abs(int x) { return x >= 0 ? x : -x; }

/* 
 * return a value between 0.0 and 360.0 which is suitable for sorting
 * points based on the angle the line segment p1,p2 makes with the
 * positive horizontal. the value returned is not that angle, but has
 * the same ordering properties.
 */
static const double theta(point_t p1, point_t p2)
{
	int dx, dy, ax, ay;
	double t;

	dx = p2.x-p1.x; ax = abs(dx);
	dy = p2.y-p1.y; ay = abs(dy);

	if ((dx == 0) && (dy == 0))
		t = 0.0;
	else
		t = (double)dy / (ax+ay);

	if (dx < 0)
		t = 2.0 - t;
	else if (dy < 0)
		t = 4.0 + t;

	return (t * 90.0);
} /* theta */


/* 
 * calculate the convex hull of a given set of points.  the number of 
 * points defining the convex hull is the return value.  the convex 
 * hull is built in-place in the array hull[].  the array passed must 
 * have room for one additional point at the end.  num_points is the
 * number of points in the array passed in.  the convex hull list generated
 * is in counterclockwise order and is closed.
 */
int convexHull(point_t hull[], int num_points)
{
	int i, min, num;
	double minangle, v;
	point_t t;

	/* find the element with min y coordinate */
	min = 0;
	for (i = 1; i < num_points; ++i) {
		if (hull[i].y < hull[min].y) min = i;
	}

	num = 0;	/* no points in convex hull yet */
	hull[num_points] = hull[min];
	minangle = 0.0;

	do {
		t = hull[num]; hull[num] = hull[min]; hull[min] = t;
		min = num_points;
		v = minangle;
		minangle = 360.0;
		for (i = num+1; i <= num_points; ++i)
			if ((theta(hull[num],hull[i]) > v) &&
			    (theta(hull[num],hull[i]) < minangle)) {
				min = i;
				minangle = theta(hull[num],hull[min]);
			}
		++num;
	} while ( min != num_points);
	hull[num++] = hull[0];	/* close up polygon */
	return num;
} /* convexHull */


/* 
 * read in a set of points from which to generate the convex hull. the 
 * parameter indicates how many points are specified, one point per line.
 * the array is assumed to have enough space.  reads from stdin.
 */
void readPoints(point_t p[], int num)
{
	while (num--) {
		scanf("%d %d\n", &p[num].x, &p[num].y);
	}
} /* readPoints */


/* 
 * calculate the area of a polygon given a list of points describing 
 * the polygon.  the points may be listed in either clockwise or 
 * counterclockwise order.  num is the number of points specified
 * in the list.  if the polygon array is given in clockwise order,
 * the sign of the area will be negative.  the last point in the polygon
 * must be the same as the first.
 */
double polyArea(point_t p[], int num)
{
	double a = 0.0;
	int i;

	for (i = 1; i < num; ++i)
		a += (double)p[i-1].x * p[i].y - (double)p[i].x * p[i-1].y;

	return a/2;
} /* polyArea */


/* 
 * determine if the path from p0 to p1 to p2 is clockwise or 
 * counterclockwise.  returns true if counterclockwise, false otherwise.
 */
int ccw(point_t p0, point_t p1, point_t p2)
{
	point_t list[4];

	list[0] = list[3] = p0;
	list[1] = p1;
	list[2] = p2;

	return (polyArea(list, 4) > 0);
}

/* 
 * determine if the given point is inside the specified convex 
 * polygon. the convex polygon must be listed in counterclockwise 
 * order.  num indicates the number of points.  returns 1 if inside,
 * otherwise returns 0.
 */
int inPolygon(point_t p, point_t poly[], int num)
{
	int i;

	for (i = 1; i < num; ++i) {
		if (!ccw(poly[i-1],poly[i],p)) return 0;
	}
	return 1;
} /* inPolygon */ 


int main(void)
{
	int howMany, i;
	double blackout = 0.0;
	point_t p;

	scanf("%d\n", &howMany);

	/* 
	 * read in the kingdom descriptions and calculate the convex 
	 * hull and area associated with each.  this is all we store.
	 */
	while (howMany >= 0) {
		readPoints(kingdoms[num_kingdoms].list, howMany);
		kingdoms[num_kingdoms].size = 
			convexHull(kingdoms[num_kingdoms].list, howMany);
		kingdoms[num_kingdoms].area =
			polyArea(kingdoms[num_kingdoms].list,
				 kingdoms[num_kingdoms].size);
#ifdef DEBUG
	printf("size = %d\n",kingdoms[num_kingdoms].size);
	for (howMany = 0; howMany < kingdoms[num_kingdoms].size; ++howMany) {
		printf("(%d, %d)\n", kingdoms[num_kingdoms].list[howMany].x,
				     kingdoms[num_kingdoms].list[howMany].y);

	}
	printf("area: %.2lf\n", kingdoms[num_kingdoms].area);
	putchar('\n');
#endif /* DEBUG */
		++num_kingdoms;
		scanf("%d\n", &howMany);
	}

	/* 
	 * now read in the missile attacks and calculate which kingdom 
	 * it fell in, if any.  then add the area of that kingdom to 
	 * the total.
	 */
	while (scanf("%d %d\n", &p.x, &p.y) != EOF) {
		for (i = 0; i < num_kingdoms; ++i) {
			if (inPolygon(p, kingdoms[i].list, kingdoms[i].size)) {
				blackout += kingdoms[i].area;
				kingdoms[i].area = 0.0; /* not counted again */
#ifdef DEBUG
				printf("Hit! (%d,%d)\n",p.x,p.y);
#endif
			}
#ifdef DEBUG
			else printf("Miss! (%d,%d)\n",p.x,p.y);
#endif

		}
#ifdef DEBUG
		putchar('\n');
#endif
	}

	printf ("%.2lf\n",blackout);
	return 0;
} /* main */


