/*
 * $Id: ipxripd.c,v 1.1 1995/01/05 09:25:54 root Exp $
 *
 * Main program
 */
#include "sap.h"
#include <ulimit.h>

IPXRouteEnt *net_table;
SAPInfo *sap_table;

int ripsock;
extern int sapsock;
/*
 * Read IPX network file
 */
IPXRouteEnt *read_ipxnet_file(char *cmd) {
  FILE *fp = fopen(IPXNET_FILE,"r");
  char *args[LSTRLEN],lbuf[LSTRLEN];
  IPXRouteEnt *p=NULL;

  if (fp == NULL) sysMsg(MSG_FATAL,ERR_ERRNO,"%s-%s",cmd,IPXNET_FILE);
  /*
   * Read net lines one at a time
   */
  while (getline(lbuf,LSTRLEN,fp)) {
    int argc=parse(lbuf,args,lbuf);
    if (argc) {
      if (argc >= 3)
	p = add_local_route(args[0],asc2frame(args[1]),gethex(args[2]),p);
    }
  }
  fclose(fp);
  dump_ipxtab(p);
  return p;
}

void down_ipxnet_file(char *cmd) {
  FILE *fp = fopen(IPXNET_FILE,"r");
  char *args[LSTRLEN],lbuf[LSTRLEN];

  if (fp == NULL) sysMsg(MSG_FATAL,ERR_ERRNO,"%s-%s",cmd,IPXNET_FILE);
  /*
   * Read net lines one at a time
   */
  while (getline(lbuf,LSTRLEN,fp)) {
    int argc=parse(lbuf,args,lbuf);
    if (argc) {
      if (argc >= 3)
	down_local_route(args[0],asc2frame(args[1]),gethex(args[2]));
    }
  }
  fclose(fp);
}

IPXRouteEnt *rtes=NULL;

void main_loop() {
  int maxfd, i, sendrip=FALSE;
  fd_set readfds;
  struct timeval tmo = { 30, 0} ;

  ripsock = opensock(IPXS_RIP,IPXT_RIP);
  sapsock = opensock(IPXS_SAP,IPXT_SAP);

  maxfd = (ripsock > sapsock ? ripsock : sapsock)+1; 
  for (;;) {
    FD_ZERO(&readfds);
    FD_SET(ripsock,&readfds);
    FD_SET(sapsock,&readfds);
    
    do {
      i = select(maxfd,&readfds,NULL,NULL,&tmo);
    } while (i == -1 && errno == EINTR);
    
    if (i < 0) sysMsg(MSG_FATAL,ERR_ERRNO,"select");
    if (i) {
      if (FD_ISSET(ripsock,&readfds)) {
	rtes = read_rip(ripsock,rtes);
      }
      if (FD_ISSET(sapsock,&readfds)) {
	read_sap(sapsock,rtes);
      }
    }
    if (!tmo.tv_sec) {
      rtes = age_rip(rtes);
      age_sap();
      sendrip = !sendrip;
      if (sendrip) {
	/* Broadcast my RIP table */
	broadcast_rip(net_table);
	broadcast_rip(rtes);
      }
      else {
	broadcast_sap(sap_table);
      }
      tmo.tv_sec = 30;
      tmo.tv_usec = 0;
    }
  }
}

/*
 * Main program
 */
int verbose;

/*
 * You should bring the router down using a 'kill -HUP pid' or kill -TERM pid',
 * this will cause the router to broadcast to neighboring routers that
 * it is down.  Note that a Ctrl-Alt-Del and /sbin/shutdown send TERM signals
 * to all processes so we are safe that way.
 */
void router_down()
{
  sysMsg(MSG_WARN, ERR_OK, "DOWN command for the IPX router...");
  maxhop_rip(rtes);
  maxhop_rip(net_table);
  maxhop_sap();
  broadcast_rip(rtes);
  broadcast_rip(net_table);
  broadcast_sap(sap_table);
  down_ipxnet_file("down");
  sysMsg(MSG_WARN, ERR_OK, "The IPX router has shut down properly");
  exit(0);
}

void main(int argc,char **argv) {
  verbose = argc > 1;

#ifndef DEBUG
  if (!fork()) {
    int i;
    for (i=0;i<32;close(i++));
  }
  else exit(0);
#endif



  net_table = read_ipxnet_file(argv[0]);	/* Read local routes */
  dump_ipxtab(net_table);
  sap_table = NULL;	/* No services by default */

  broadcast_rip(net_table);

  /*
   * It is probably easier for a person to remember to do a HUP rather
   * than a TERM so I'll trap HUPs as being the shutdown command.
   * However, we are lucky in the fact that Linux sends TERM signals
   * to all processes during shutdown, so I'll trap those too.
   */
  signal(SIGTERM, router_down);
  signal(SIGHUP, router_down);

  /* Ok.  We begin listening here */
  alarm(2);
  signal(SIGALRM,general_request);
  main_loop();
}

void general_request(int sig) {
  unsigned char pkt[IPX_MTU];
  IPXRipEnt *p = (IPXRipEnt *) (&pkt[2]);
  IPXRouteEnt *nets = net_table;
  struct sockaddr_ipx sk,tsk;
  int tsock; /* tsock and tsk are the local socket and addresses */

  pkt[0] = 0;
  pkt[1] = IPX_REQ;
  p->network = IPX_GEN_REQ;
  p->hops = p->ticks = 0;

  sk.sipx_family = AF_IPX;	/* Set IPX dest socket */
  memset(sk.sipx_node,0xFF,sizeof sk.sipx_node);

  sk.sipx_port = htons(IPXS_RIP);
  sk.sipx_type = IPXT_RIP;

  while (nets) {
    tsock = socket(AF_IPX, SOCK_DGRAM, 0);
    tsk.sipx_network = htonl(nets->network);
    tsk.sipx_family = AF_IPX;
    tsk.sipx_port = htons(IPXS_RIP);
    tsk.sipx_type = IPXT_RIP;
    memset(&tsk.sipx_node, 0, sizeof(tsk.sipx_node));
    if (bind(tsock, (struct sockaddr *)&tsk, sizeof(tsk)) < 0) {
      sysMsg(MSG_FATAL, ERR_ERRNO,
        "Binding local RIP sock on network %lX, remember to set up a local net?");
      exit(1);
    }
    sk.sipx_family = AF_IPX;
    sk.sipx_network = htonl(nets->network);
    sk.sipx_port = htons(IPXS_RIP);
    sk.sipx_type = IPXT_RIP;
    memset(&sk.sipx_node, 0xFF, sizeof(sk.sipx_node));
    if (connect(tsock, (struct sockaddr *)&sk, sizeof(sk)) < 0) {
      sysMsg(MSG_FATAL,ERR_ERRNO,"Connecting to RIP destination is periodic broadcast");
      exit(1);
    }
    write(tsock, pkt, 6);
    close(tsock);
    nets = nets->next;
  } 

  pkt[0] = 0;
  pkt[1] = 1;
  pkt[2] = 0xFF;
  pkt[3] = 0xFF;
  for (nets = net_table ; nets ; nets = nets->next) {
    tsock = socket(AF_IPX, SOCK_DGRAM, 0);
    tsk.sipx_network = htonl(nets->network);
    tsk.sipx_family = AF_IPX;
    tsk.sipx_port = htons(IPXS_SAP);
    tsk.sipx_type = IPXT_SAP;
    memset(&tsk.sipx_node, 0, sizeof(tsk.sipx_node));
    if (bind(tsock, (struct sockaddr *)&tsk, sizeof(tsk)) < 0) {
      sysMsg(MSG_FATAL, ERR_ERRNO,
        "Binding local SAP sock on network %lX, remember to set up a local net?");
      exit(1);
    }

    sk.sipx_family = AF_IPX;
    sk.sipx_network = htonl(nets->network);
    sk.sipx_port = htons(IPXS_SAP);
    sk.sipx_type = IPXT_SAP;
    memset(&sk.sipx_node, 0xFF, sizeof(sk.sipx_node));
    if (connect(tsock, (struct sockaddr *)&sk, sizeof(sk)) < 0) {
      sysMsg(MSG_FATAL,ERR_ERRNO,"Connecting to RIP destination is periodic broadcast");
      exit(1);
    }

    if (write(tsock, pkt, 4) < 0) {
      sysMsg(MSG_FATAL,ERR_ERRNO,"Broadcasting periodic SAP");
      exit(1);
    }
    close(tsock);
  } 
}
