/*
**	BANJOPLAY -- Compose banjo part to fit chords
**	banjotune(key, stones, chan);			"tune" the banjo
**		int key;		0-11, 0=>C, 11=>B
**		int stones[12];		scale of piece, booleans
**		int chan;		0-15, MIDI channel
**	banjoplay(start, stop, otones, ntones, vel);	"play" the banjo
**		long start;		beginning time, MPU clocks
**		long stop;		ending time, MPU clocks
**		int otones[12];		notes in current chord, booleans
**		int ntones[12];		notes in next chord, booleans
**					ntones[0] < 0 => end
**		int vel;		MIDI key velocity for notes
**	psl 2/89
*/
#include	<stdio.h>
#include	"midi.h"

#define	WHOLE		(MPU_CLOCK_PERIOD<<1)
#define	SIXTEENTH	(MPU_CLOCK_PERIOD>>3)
#define	DBG		if(Debug)fprintf
#define	DBG2		if(Debug>1)fprintf
#define	PATLEN	8		/* # of sixteenths / pattern */
#define	MAXREACH	4	/* how big your hands are */
#define	STRINGS	5
#define	FIRST	0		/* index of first string */
#define	SECOND	1
#define	THIRD	2
#define	FOURTH	3
#define	FIFTH	4		/* index of fifth string */
#define	DIGITS	3
#define	THUMB	0
#define	INDEX	1
#define	MIDDLE	2
#define	OPEN	0		/* fret of open string */
#define	T	1		/* masks for Rhpat[] & Fi[] */
#define	I	2
#define	M	4
#define	NRHP	3		/* how many to try */
#define	PITCHOF(S,F)	(Tuning[S]+(F))
#define	RANDR(L,H)	((L)+(rand()/3)%((H)-(L)+1))
#define	ABS(A)		((A)<0?-(A):(A))
#define	MIN(A,B)	((A)<(B)?(A):(B))
#define	MAX(A,B)	((A)>(B)?(A):(B))
#define	PR50		(rand()&010)		/* 50-50 fcn */

static struct	fistr	{	/* finger info */
	int	mask;		/* Rhpat mask */
	int	lostr;		/* lowest (numbered) string it can reach */
	int	histr;		/* highest (numbered) string it can reach */
	int	bstr;		/* best (favorite) string */
} Fi[DIGITS] = {
/*	code	lowest	highest	favorite	*/
	{ T,	THIRD,	FIFTH,	FIFTH, },	/* thumb (numbers are -1) */
	{ I,	SECOND,	FOURTH,	THIRD, },	/* index */
	{ M,	FIRST,	THIRD,	FIRST, },	/* middle */
};
static	int	Rhpat[][PATLEN] = {
	T,   I,   M,   T,   I,   M,   T,   M,		/* forward */
	T,   M,   I,   T,   M,   I,   T,   M,		/* backward */
	T,   I,   M,   T,   M,   I,   T,   M,		/* backup */
	T,   I,   M,   T,   T,   I,   M,   T,		/* dbld thumb */
	T,   I,   M,   T,   M,   T,   M,   I,		/* ? */
	T,   M,   T,   M,   T,   M,   T,   M,		/* flat-pick */
	I,   M,   I,   M,   T,   I,   M,   T,		/* foggymtn */
	T,   I,   T,   M,   T,   I,   T,   M,		/* double thumb */
/****
	T,   I,   T,   M,   T,   0,   T|M, 0,		/* John Hickman */
/****
	I,   0,   T|M, 0,   I,   0,   T|M, 0,		/* bum-chang */
};
typedef	struct	bseqstr {
	int	start;		/* note start time in MPU clocks */
	int	keys[5];	/* notes (relative to lowest root of key) */
} BSEQ;
static	BSEQ	End0[]	= {
	{ 0,	0,-1,-1,-1,-1, },
	{ 120,	4,-1,-1,-1,-1, },
	{ 150,	7,-1,-1,-1,-1, },
	{ 180,	9,-1,-1,-1,-1, },
	{ 240,	12,-1,-1,-1,-1, },
	{ 360,	4,7,12,-1,-1, },
	{ 390,	-1,-1,-1,-1,-1, },
	{ 480,	-1,-1,-1,-1,-1, },
};
static	BSEQ	End1[]	= {
	{ 0,	12,-1,-1,-1,-1, },
	{ 120,	9,-1,-1,-1,-1, },
	{ 150,	7,-1,-1,-1,-1, },
	{ 180,	4,-1,-1,-1,-1, },
	{ 240,	0,-1,-1,-1,-1, },
	{ 360,	4,7,12,-1,-1, },
	{ 390,	-1,-1,-1,-1,-1, },
	{ 480,	-1,-1,-1,-1,-1, },
};
static	BSEQ	End2[]	= {
	{ 0,	3,-1,-1,-1,-1, },
	{ 30,	4,-1,-1,-1,-1, },
	{ 60,	9,-1,-1,-1,-1, },
	{ 90,	12,-1,-1,-1,-1, },
	{ 120,	7,-1,-1,-1,-1, },
	{ 150,	4,-1,-1,-1,-1, },
	{ 180,	9,-1,-1,-1,-1, },
	{ 210,	12,-1,-1,-1,-1, },
	{ 240,	7,-1,-1,-1,-1, },
	{ 270,	3,-1,-1,-1,-1, },
	{ 300,	2,-1,-1,-1,-1, },
	{ 360,	0,4,7,-1,-1, },
	{ 480,	-1,-1,-1,-1,-1, },
};
static	BSEQ	End3[]	= {
	0,	3,4,9,-1,-1,
	90,	3,4,9,-1,-1,
	180,	3,4,9,-1,-1,
	270,	12,-1,-1,-1,-1,
	360,	0,4,7,-1,-1,
	480,	-1,-1,-1,-1,-1,
};
static	BSEQ	End4[]	= {
	60,	15,16,19,-1,-1,
	90,	14,-1,-1,-1,-1,
	150,	16,19,-1,-1,-1,
	180,	9,-1,-1,-1,-1,
	240,	12,-1,-1,-1,-1,
	360,	19,24,-1,-1,-1,
	480,	-1,-1,-1,-1,-1,
};
static	BSEQ	End5[]	= {
	0,	0,-1,-1,-1,-1,
	60,	12,-1,-1,-1,-1,
	120,	10,-1,-1,-1,-1,
	135,	9,-1,-1,-1,-1,
	150,	7,-1,-1,-1,-1,
	180,	3,-1,-1,-1,-1,
	195,	2,-1,-1,-1,-1,
	210,	0,-1,-1,-1,-1,
	240,	0,-1,-1,-1,-1,
	360,	0,4,7,12,-1,
	480,	-1,-1,-1,-1,
};

static	struct	endstr	{
	int	len;		/* length in MPU clocks */
	int	lo, hi;		/* range of key values */
	int	first;		/* key to use for distance calc */
	BSEQ	*seq;		/* BSEQ str with notes */
} Bends[]	= {
	{ 480,   0, 12, 0, End0, },
	{ 480,   0, 12, 12, End1, },
	{ 480,   0, 12, 3, End2, },
	{ 480,   0, 12, 9, End3, },
	{ 480,   9, 24, 16, End4, },
	{ 480,   0, 12, 12, End5, },
	{ 0, },
};
#define	NRHPAT	(sizeof Rhpat / sizeof Rhpat[0])

typedef	struct	stepstr	{
	int	rhd;			/* rh digits for this step */
	int	rhs[DIGITS];		/* rh strings for each digit */
	int	lhf[DIGITS];		/* lh frets for each digit's string */
	int	key[DIGITS];		/* pitch for each digit's string */
} STEP;

typedef	struct	patstr	{
	int	lhloc;
	int	pat;
	STEP	step[PATLEN];		/* step info for each step */
} PSEQ;

static	PSEQ	Last	= {
	1, 0,
};

static	int	Nrhpat	= NRHPAT;	/* number of patterns we can use */
static	int	Tuning[STRINGS]	= {	/* How the banjo is tuned */
	62, 59, 55, 50, 67,	/* open G-Major */
};
static	int	Tunes[12][STRINGS] = {	/* tunings for each key */
	62, 59, 55, 50, 67,	/* C (open G) */
	63, 60, 56, 51, 68,	/* C# (open G up 1) */
	62, 57, 54, 50, 66,	/* D (open D) */
	63, 58, 55, 51, 67,	/* D# (open D up 1) */
	64, 59, 56, 52, 68,	/* E (open D up 2) */
	65, 60, 57, 53, 69,	/* F (open D up 3) */
	66, 61, 58, 54, 70,	/* F# (open D up 4) */
	62, 59, 55, 50, 67,	/* G (open G) */
	63, 60, 56, 51, 68,	/* G# (open G up 1) */
	64, 61, 57, 52, 69,	/* A (open G up 2) */
	65, 62, 58, 53, 70,	/* A# (open G up 3) */
	66, 63, 59, 54, 71,	/* B (open G up 4) */
};
static	int	Beatval[]	= { 2,0,1,0,2,0,1,0, };
static	int	Chan;		/* lead instrument channel */
static	int	Key;
static	int	Stones[12];

/* from accl.c (or whatever provides main()) */
extern	int	Debug;		/* debugging output */

void
banjotune(key, stones, chan)
int	stones[12];
{
	register int i, pt;

	Key = key;
	Chan = chan;
	for (i = 12; --i >= 0; Stones[i] = (stones[i]? 1 : 0));
	for (pt = PATLEN; --pt >= 0; ) {
	    for (i = DIGITS; --i >= THUMB; ) {
		Last.step[pt].key[i] = 0;
		Last.step[pt].rhs[i] = -1;
	    }
	}
	for (i = STRINGS; --i >= 0; Tuning[i] = Tunes[Key % 12][i]);
}

void
banjoplay(start, stop, otones, ntones, vel, ofp)
long	start, stop;
int	otones[12], ntones[12];
FILE	*ofp;
{
	int tstart, tstop, t;
	int pt, p, t0, nrhp, i, lhloc, maxlhloc, pat;
	int v, bv, tries;
	int tones[PATLEN][12];
	STEP *cstep, *pstep;
	PSEQ try, btry;

	tstart = start / SIXTEENTH;
	tstop = stop / SIXTEENTH;
DBG(stderr, "banjoplay(%d, %d, otones, ntones, %d, ofp) tstart,tstop = %d,%d\n", start, stop, vel, tstart, tstop);
	if (ntones[0] < 0) {
	    for (pt = PATLEN; --pt >= 0; ) {
		for (i = DIGITS; --i >= THUMB; )
		    if (v = Last.step[pt].key[i])
			break;
		if (v)
		    break;
	    }
	    ending(start, stop, vel, v = v? v : (60 + Key), ofp);
	    return;
	}
	for (t = tstart; t < tstop; t++) {
	    pt = t % PATLEN;
DBG(stderr, "t = %d, pt = %d\n", t, pt);
	    for (p = 12; --p >= 0; )
		tones[pt][p] = 4 * otones[p] + 2 * Stones[p] + ntones[p];
	    if (pt != PATLEN - 1)
		continue;
	    t0 = t - pt;
DBG(stderr, "t0 = %d\n", t0);
	    bv= 0;
	    for (nrhp = 0; nrhp < NRHP; nrhp++) { /* try right-hand patterns */
		pat = nrhp? (rand() % Nrhpat) : Last.pat;
		try.pat = pat;
		for (i = 0; i < PATLEN; i++)
		    try.step[i].rhd = Rhpat[pat][i];
		i = Last.lhloc;		/* try all left-hand locations */
		maxlhloc = i + (i < 9? 3 : (18 - i) / 3);
		for (lhloc = (i < 4)? 1 : (i - 3); lhloc < maxlhloc; lhloc++) {
		    try.lhloc = lhloc;
		    for (tries = 0; tries < 2; tries++) {
			for (pt = 0; pt < PATLEN; pt++ ) {
			    cstep = &try.step[pt];
			    pstep = pt? &try.step[pt-1] : &Last.step[PATLEN-1];
			    pickstring(cstep, pstep);
			    pickfret(lhloc, cstep, tones[pt], Beatval[pt]);
			}
			v = eval(&try, &Last, tones);
DBG(stderr, "pat %d = %d, lhloc = %d, eval = %d\n", nrhp, pat, lhloc, v);
			if (v > bv) {
			    btry = try;
			    bv = v;
			}
		    }
		}
	    }
DBG(stderr, "output pat=%d lhloc=%d\n", btry.pat, btry.lhloc);
	    output(t0, &btry, vel, ofp);
	    Last = btry;
	}
}

pickstring(cstep, pstep)	/* randomly associate strings with digits */
STEP	*cstep, *pstep;		/* avoid overlaps */
{
	register int lo, hi;

	if (cstep->rhd & M) {
	    lo = FIRST;
	    hi = Fi[MIDDLE].histr;
	    if (pstep->rhs[INDEX] >= 0)
		hi = MIN(hi, pstep->rhs[INDEX] - 1);
	    else if (pstep->rhs[THUMB] >= 0)
		hi = MIN(hi, pstep->rhs[THUMB] - 1);
	    cstep->rhs[MIDDLE] = RANDR(lo, hi);
	} else
	    cstep->rhs[MIDDLE] = -1;
	if (cstep->rhd & I) {
	    lo = Fi[INDEX].lostr;
	    if (pstep->rhs[MIDDLE] >= 0)
		lo = MAX(lo, pstep->rhs[MIDDLE] + 1);
	    if (cstep->rhs[MIDDLE] >= 0)
		lo = MAX(lo, cstep->rhs[MIDDLE] + 1);
	    hi = Fi[INDEX].histr;
	    if (pstep->rhs[THUMB] >= 0)
		hi = MIN(hi, pstep->rhs[THUMB] - 1);
	    cstep->rhs[INDEX] = RANDR(lo, hi);
	} else
	    cstep->rhs[INDEX] = -1;
	if (cstep->rhd & T) {
	    lo = Fi[THUMB].lostr;
	    if (pstep->rhs[INDEX] >= 0)
		lo = MAX(lo, pstep->rhs[INDEX] + 1);
	    else if (pstep->rhs[MIDDLE] >= 0)
		lo = MAX(lo, pstep->rhs[MIDDLE] + 1);
	    if (cstep->rhs[INDEX] >= 0)
		lo = MAX(lo, cstep->rhs[INDEX] + 1);
	    else if (cstep->rhs[MIDDLE] >= 0)
		lo = MAX(lo, cstep->rhs[MIDDLE] + 1);
	    hi = FIFTH;
	    cstep->rhs[THUMB] = RANDR(lo, hi);
	} else
	    cstep->rhs[THUMB] = -1;
}

/* randomly pick frets for played strings */
pickfret(lhloc, cstep, tones, beatval)
STEP	*cstep;
int	tones[12];
{
	register int s, d, n, w, f;
	int fr[32], sp[32];

	for (d = DIGITS; --d >= THUMB; ) {
	    if ((s = cstep->rhs[d]) >= 0) {
		n = 0;
		sp[OPEN] = PITCHOF(s, OPEN);
		for (w = tones[sp[OPEN] % 12]; --w >= beatval; fr[n++] = OPEN);
		for (f = lhloc + MAXREACH; --f >= lhloc; ) {
		    if (s == FIFTH && f <= 5)
			continue;
		    sp[f] = PITCHOF(s, (s == FIFTH? f - 5 : f));
		    for (w = tones[sp[f] % 12]; --w >= beatval; fr[n++] = f);
		}
		f = n? fr[rand() % n] : OPEN;
		cstep->lhf[d] = f;
		cstep->key[d] = sp[f];
	    } else
		cstep->lhf[d] = cstep->key[d] = 0;
	}
}

output(t0, tp, vel, ofp)
PSEQ	*tp;
FILE	*ofp;
{
	u_char mbuf[8];
	int pt, d;
	MCMD m;

	/* output the best */
	m.len = 3;
	m.cmd = mbuf;
	m.cmd[0] = CH_KEY_ON | Chan;
	for (pt = 0; pt < PATLEN; pt++) {
	    m.when = (t0 + pt) * SIXTEENTH;
	    m.cmd[2] = vel;
	    for (d = DIGITS; --d >= THUMB; ) {
		if (tp->step[pt].rhs[d] >= 0) {
		    m.cmd[1] = tp->step[pt].key[d];
		    putmcmd(ofp, &m);
DBG(stderr, "%5d: %s\n", m.when, key2name(m.cmd[1]));
		}
	    }
	    m.when += SIXTEENTH;
	    m.cmd[2] = 0;
	    for (d = DIGITS; --d >= THUMB; ) {
		if (tp->step[pt].rhs[d] >= 0) {
		    m.cmd[1] = tp->step[pt].key[d];
		    putmcmd(ofp, &m);
		}
	    }
	}
}

eval(ctp, ptp, tones)	/* evaluate the PATLEN steps in *ptp */
PSEQ	*ctp, *ptp;
int	tones[][12];
{
	register int v, pt, d, p, i, d2, p2, dp;
	STEP *cstep, *pstep, *ppstep;

	/* chord/scale spectrum */
	v = 0;
	for (pt = 0; pt < PATLEN; pt++)
	    for (d = DIGITS; --d >= THUMB; )
		if (p = ctp->step[pt].key[d])
		    v += 1 + (Beatval[pt] + 1) * tones[pt][p % 12];
	/* resolution */
	cstep = &ptp->step[PATLEN - 1];
	for (pt = 0; pt < PATLEN; pt++) {
	    pstep = cstep;
	    cstep = &ctp->step[pt];
	    for (d = DIGITS; --d >= THUMB; ) {
		if (p = pstep[pt].key[d]) {
		    i = tones[pt][p % 12];
		    if (i < tones[pt][(p + 1) % 12]) {
			for (d2 = DIGITS; --d2 >= THUMB; )
			    if (cstep[pt].key[d2] == p + 1)
				v += (Beatval[pt] + 1) * 4;
		    }
		    if (i < tones[pt][(p - 1) % 12]) {
			for (d2 = DIGITS; --d2 >= THUMB; )
			    if (cstep[pt].key[d2] == p - 1)
				v += (Beatval[pt] + 1) * 4;
		    }
		}
	    }
	}
	/* motion */
	pstep = &ptp->step[PATLEN - 2];
	cstep = &ptp->step[PATLEN - 1];
	for (pt = 0; pt < PATLEN; pt++) {
	    ppstep = pstep;
	    pstep = cstep;
	    cstep = &ctp->step[pt];
	    for (d = DIGITS; --d >= THUMB; ) {
		if (p = ctp->step[pt].key[d]) {
		    for (d2 = DIGITS; --d2 >= THUMB; ) {
			if (p2 = pstep[pt].key[d2]) {
			    dp = p - p2;
			    dp = ABS(dp);
			    if (dp == 0)
				v += 2;
			    else if (dp < 3)
				v += 6;
			    else if (dp < 7)
				v += 3;
			    else if (dp < 12)
				v += 1;
			}
			if (p2 = ppstep[pt].key[d2]) {
			    dp = p - p2;
			    dp = ABS(dp);
			    if (dp == 0)
				v += 4;
			    else if (dp < 3)
				v += 5;
			    else if (dp < 7)
				v += 3;
			    else if (dp < 12)
				v += 1;
			}
		    }
		}
	    }
	}
	return(v);
}

ending(start, stop, vel, lastkey, ofp)
long	start, stop;
FILE	*ofp;
{
	u_char mbuf[8];
	int n, i;
	int seq, bseq, dist, bdist, len, blen, k[5];
	BSEQ *bsp;
	MCMD m;

	m.len = 3;
	m.cmd = mbuf;
	m.cmd[0] = CH_KEY_ON | Chan;
	while (stop - start >= WHOLE) {
	    bdist = 999;
	    blen = 0;
	    for (seq = 0; Bends[seq].len; seq++) {
		len = Bends[seq].len;
		if (len > stop - start || len < blen)
		    continue;
		n = 12 * ((61 - Key - Bends[seq].lo) / 12) + Key;
		dist = lastkey - (n + Bends[seq].first);
		dist = dist < 0? -dist : dist;
		if (len > blen
		 || dist < bdist
		 || (dist == bdist && PR50)) {
		    bseq = seq;
		    bdist = dist;
		    blen = len;
		}
	    }
	    n = 12 * ((61 - Key - Bends[bseq].lo) / 12) + Key; /* trans */
	    for (i = 5; --i >= 0; k[i] = -1);
	    for (bsp = Bends[bseq].seq; ; bsp++) {
		m.when = start + bsp->start;
		m.cmd[2] = 0;
		for (i = 0; i < 5; i++) {
		    if (k[i] >= 0) {
			m.cmd[1] = n + k[i];
			putmcmd(ofp, &m);
		    }
		}
		if (m.when >= start + blen)
		    break;
		m.cmd[2] = vel;
		for (i = 0; i < 5; i++) {
		    k[i] = bsp->keys[i];
		    if (k[i] >= 0) {
			m.cmd[1] = n + k[i];
			putmcmd(ofp, &m);
			lastkey = m.cmd[1];
		    }
		}
	    }
	    start += blen;
	}
	return;
}
