/*	BELLSCORE -- Bellscore control panel, psl 2/89 */

#include	<sys/signal.h>
#include	<sys/wait.h>
#include	<suntool/sunview.h>
#include	<suntool/panel.h>
#include	<midi.h>
#include	<stdio.h>
#include	<pwd.h>

#define	LOGFILE		"/u/psl/MIDI/BELLSCORE/bs.log"
#define	FONTPATH	"/usr/lib/fonts/fixedwidthfonts/screen.r.14"
#define	PGV(I)		panel_get_value(I)
#define	PSV(I,V)	panel_set_value(I,V)
#define	IMGI(N)		((Menu_item)menu_get(Inst,MENU_NTH_ITEM,N))
#define	IMGS(N)		((char *)menu_get(IMGI(N),MENU_STRING))
#define	OPTIM(M)	((int)PGV(Opts)&M)
#define	LOCKED(M)	((int)PGV(Lock)&M)

#define	PSLUID	257

#define	BARLEN	(2*MPU_CLOCK_PERIOD)
#define	ACCA	"acca"
#define	ACCL	"accl"
#define	CHMAP	"chmap"
#define	INST	"inst"
#define	MCLEAN	"mpuclean"
#define	MERGE	"merge"
#define	MKCC	"mkcc"
#define	PLAY	"play"
#define	MAXSAV	16
#define	MSFLEN	24
#define	MAXNDP	4
#define	METKEY	0x4b

/* channel numbers in the BELLSCORE files (also used as initial settings) */
#define	BCHAN	2
#define	CCHAN	3
#define	DCHAN	10	/* others used, too (see dpchn[]) */

#define	LCHAN0	4	/* used by most */
#define	LCHAN1	5	/* used by some */
#define	LCHAN2	6	/* used by some */

/* Dbug, Lock, & Opts toggle masks */
#define	M_MKCC		0x0001
#define	M_ACCA		0x0002
#define	M_ACCL		0x0004
#define	M_BELLSCORE	0x0008

#define	DBG	if(Debug)fprintf
struct	stylstr	{
	char	*name;		/* for programs */
	int	barlen;		/* bar length (MPU clocks) */
	int	units;		/* length quantum (MPU clocks) */
	int	key;		/* current key */
	int	cnti;		/* current count-in */
	char	leng[10];	/* current length */
	int	tempo;		/* current tempo */
	char	*lname;		/* for people (files & button labels)*/
	char	*desc;		/* description (for menu) */
	int	binst;		/* bass inst */
	int	bchan;		/* bass chan */
	int	bvelo;		/* bass velocity */
	int	cinst;		/* chording instrument */
	int	cchan;		/* chord chan */
	int	cvelo;		/* chord velocity */
	int	dpart;		/* drum part */
	int	dchan;		/* drum chan */
	int	dvelo;		/* drum velocity */
	int	dpchn[4];	/* drum part channels */
	int	linst[3];	/* lead instruments */
	int	lchan[3];	/* lead chans */
	int	lvelo[3];	/* lead velocities */
	char	lseed[16];	/* lead seed */
	int	lener;		/* lead energy */
	int	lpred;		/* lead predictability */
} S[32]	= {
	{ "bebop", 480,	960,	10, 0, "30:00", 192,
	    "BEBOP", "Bebop Jazz",
	    89,BCHAN,99,
	    2,CCHAN,64,
	    0,DCHAN,64, { DCHAN, },
	    42,0,0, LCHAN0,0,0, 40,0,0, "0",60,25 },
	{ "grass", 480,	480,	7, 0, "28:00", 137,
	    "BLUEGRASS", "Bluegrass Banjo",
	    93,BCHAN,80,
	    101,CCHAN,64,
	    0,DCHAN,64, { 0, },
	    105,0,0, LCHAN0,0,0, 84,0,0, "0",50,50 },
	{ "boogi", 480,	960,	0, 0, "36:00", 174,
	    "BOOGIE", "Boogie-Woogie",
	    88,BCHAN,64,
	    1,CCHAN,76,
	    0,DCHAN,64, { DCHAN, },
	    1,0,0, LCHAN0,0,0, 64,0,0, "0",50,50 },
	{ "class", 480,	960,	0, 0, "28:00", 120,
	    "CLASSICAL", "Alberti Bass",
	    17,BCHAN,64,
	    33,CCHAN,64,
	    0,DCHAN,64, { 0, },
	    17,0,0, LCHAN0,0,0, 80,0,0, "0",50,50 },
	{ "count", 480,	960,	0, 0, "32:00", 105,
	    "country", "Country-style",
	    0,BCHAN,64,
	    0,CCHAN,64,
	    0,DCHAN,64, { DCHAN, },
	    0,0,0, LCHAN0,0,0, 80,0,0, "0",50,50 },
	{ "eerie", 480,	960,	0, 0, "32:00", 105,
	    "eerie", "Spooky Stuff",
	    0,BCHAN,64,
	    0,CCHAN,64,
	    0,DCHAN,64, { DCHAN, },
	    0,0,0, LCHAN0,0,0, 80,0,0, "0",50,50 },
	{ "funka", 480,	960,	0, 0, "32:00", 105,
	    "funkadelic", "Funk-O-Rama",
	    0,BCHAN,64,
	    0,CCHAN,64,
	    0,DCHAN,64, { DCHAN, },
	    0,0,0, LCHAN0,0,0, 80,0,0, "0",50,50 },
	{ "fusio", 480,	960,	0, 0, "32:00", 105,
	    "fusion", "Rock Fusion",
	    0,BCHAN,64,
	    0,CCHAN,64,
	    0,DCHAN,64, { DCHAN, },
	    0,0,0, LCHAN0,0,0, 80,0,0, "0",50,50 },
	{ "march", 480,	960,	8, 0, "24:00", 112,
	    "MARCH", "Strident March",
	    48,BCHAN,64,
	    38,CCHAN,64,
	    0,DCHAN,64, { DCHAN, },
	    41,0,0, LCHAN0,0,0, 80,0,0, "0",50,50 },
	{ "mozar", 360,	360,	0, 0, "30:00", 96,
	    "MOZART", "Mozart Waltz",
	    17,BCHAN,64,
	    33,CCHAN,64,
	    0,DCHAN,64, { 0, },
	    17,0,0, LCHAN0,0,0, 80,0,0, "0",50,50 },
	{ "newag", 480,	960,	0, 0, "32:00", 105,
	    "new age", "New Age Wallpaper",
	    0,BCHAN,64,
	    0,CCHAN,64,
	    0,DCHAN,64, { DCHAN, },
	    0,0,0, LCHAN0,0,0, 80,0,0, "0",50,50 },
	{ "regga", 480,	960,	0, 0, "32:00", 105,
	    "reggae", "Rasta Man",
	    0,BCHAN,64,
	    0,CCHAN,64,
	    0,DCHAN,64, { DCHAN, },
	    0,0,0, LCHAN0,0,0, 80,0,0, "0",50,50 },
	{ "rhyth", 480,	960,	0, 0, "32:00", 105,
	    "r & b", "Rhythm & Blues",
	    0,BCHAN,64,
	    0,CCHAN,64,
	    0,DCHAN,64, { DCHAN, },
	    0,0,0, LCHAN0,0,0, 80,0,0, "0",50,50 },
	{ "samba", 480,	960,	4, 0, "48:00", 120,
	    "SAMBA", "Latin Samba",
	    87,BCHAN,64,
	    7,CCHAN,64,
	    0,DCHAN,64, { 10, 11, 12, 0, },
	    70,0,0, LCHAN0,0,0, 99,0,0, "0",50,67 },
	{ "seque", 480,	120,	0, 0, "10:00", 168,
	    "SEQUENCE", "5-note Wallpaper",
	    87,BCHAN,80,
	    69,CCHAN,64,
	    0,DCHAN,64, { 10, 11, 12, 0, },
	    72,0,0, LCHAN0,0,0, 64,0,0, "0",50,50 },
	{ "swing", 480,	960,	10, 0, "60:00", 128,
	    "SWING", "Swing Combo Jazz",
	    89,BCHAN,64,
	    2,CCHAN,64,
	    0,DCHAN,64, { DCHAN, },
	    60,0,0, LCHAN0,0,0, 80,0,0, "0",60,67 },
	{ "tango", 480,	960,	10, 0, "64:00", 120,
	    "tango", "Tango",
	    0,BCHAN,64,
	    0,CCHAN,64,
	    0,DCHAN,64, { DCHAN, },
	    0,0,0, LCHAN0,0,0, 80,0,0, "0",50,50 },
	{ "toner", 360,	360,	0, 0, "10:00", 150,
	    "TONE ROW", "12-tone Sequences",
	    89,BCHAN,64,
	    1,CCHAN,64,
	    0,DCHAN,64, { DCHAN, },
	    70,24,1, LCHAN0,LCHAN1,LCHAN2, 72,80,80, "0",50,50 },
#ifndef	LINT
	{ (char *) 0, },
#endif
};

struct	stylstr	*Sp;	/* current style */

#ifndef	LINT
static	short Bsicon_bits[] = {
/* Format_version=1, Width=64, Height=64, Depth=1, Valid_bits_per_item=16
 */
	0xFFFF,0xFFFF,0xFFFF,0xFFFF,0x8000,0x0000,0x0000,0x0001,
	0x8000,0x0000,0x0000,0x0001,0x8000,0x0000,0x0000,0x0001,
	0x8000,0x0000,0x0000,0x0001,0x87FF,0xFFFF,0xFFFF,0xFFC1,
	0x8400,0x0000,0x0000,0x0041,0x84EE,0xEEEE,0xEEEE,0xEE41,
	0x84AE,0xAEAE,0xAEAE,0xAE41,0x84EE,0xEEEE,0xEEEE,0xEE41,
	0x8400,0x0000,0x0000,0x0041,0x87FF,0xFFFF,0xFFFF,0xFFC1,
	0x8000,0x0000,0x0000,0x0001,0x8000,0x0000,0x0000,0x0001,
	0x8000,0x0000,0x0000,0x0001,0x8000,0x0000,0x0000,0x0001,
	0x8000,0x0000,0x0040,0x0001,0x8000,0x0000,0x0040,0x0001,
	0x9FFF,0xFFFF,0xFFFF,0xFFF9,0x9401,0x0000,0x2040,0x0011,
	0x9401,0x0000,0x2040,0x0011,0x9401,0x0002,0x2040,0x0011,
	0x9401,0x0002,0x2040,0x0011,0x9401,0x0002,0x2040,0x0011,
	0x9FFF,0xFFFF,0xFFFF,0xFFF9,0x9401,0x0002,0x23C0,0x0011,
	0x9401,0x0002,0x26C0,0x0011,0x9401,0x0202,0x2440,0x0011,
	0x940F,0x0202,0x2FE1,0xF811,0x941F,0x0202,0x23C1,0xF811,
	0x9FFF,0xFFFF,0xFFFF,0xFFF9,0x943F,0x821E,0x2040,0x0011,
	0x940E,0x023E,0x2040,0x0011,0x9400,0x023E,0x2040,0x0011,
	0x9400,0x027F,0x23C0,0x0011,0x9400,0x021C,0x26C0,0x0011,
	0x9FFF,0xFFFF,0xFFFF,0xFFF9,0x9400,0x1E00,0x2FE0,0x0011,
	0x9400,0x3600,0x23C0,0x0011,0x9400,0x2200,0x2040,0x0011,
	0x9400,0x7F00,0x23C0,0x0011,0x9400,0x1C00,0x26C0,0x0011,
	0x9FFF,0xFFFF,0xFFFF,0xFFF9,0x8000,0x0000,0x0FE0,0x0001,
	0x8000,0x0000,0x0380,0x0001,0x8000,0x0000,0x0000,0x0001,
	0x8000,0x0000,0x0000,0x0001,0x8000,0x0000,0x0000,0x0001,
	0x8000,0x0000,0x0000,0x0001,0x838F,0xA081,0xC71C,0xF3E1,
	0x8248,0x2082,0x28A2,0x8A01,0x8248,0x2082,0x0822,0x8A01,
	0x8248,0x2081,0x0822,0x8A01,0x83CF,0x2080,0x8822,0xF3C1,
	0x8228,0x2080,0x4822,0xA201,0x8228,0x2080,0x2822,0x9201,
	0x8228,0x2082,0x2822,0x9201,0x8228,0x2082,0x28A2,0x9201,
	0x83CF,0xBEF9,0xC71C,0x8BE1,0x8000,0x0000,0x0000,0x0001,
	0x8000,0x0000,0x0000,0x0001,0x8000,0x0000,0x0000,0x0001,
	0x8000,0x0000,0x0000,0x0001,0xFFFF,0xFFFF,0xFFFF,0xFFFF
};
mpr_static(Bsicon_pixrect, 64, 64, 1, Bsicon_bits);
#else	LINT
Pixrect	Bsicon_pixrect;
#endif	LINT

char	*Tmpfile	= "/tmp/bellscore";
char	Sfile[MAXSAV][MSFLEN];	/* names of saved files */
char	Sysbuf[512];		/* for sys() calls */
int	Debug	= 0;		/* for bellscore trace output */
int	Nsaved	= 0;		/* number saved */
int	Accaok	= FALSE;	/* is /tmp/bellscore.acc up to date? */
int	Acclok	= FALSE;	/* is /tmp/bellscore.lead up to date? */
int	Mkccok	= FALSE;	/* is /tmp/bellscore.cc up to date? */
int	Xprtsho	= FALSE;	/* panel of experts */
int	Basssho	= FALSE;	/* acca bass parameters panel */
int	Chrdsho	= FALSE;	/* acca chord parameters panel */
int	Drumsho	= FALSE;	/* acca drum parameters panel */
int	Leadsho	= FALSE;	/* accl parameters panel */
int	Playpid	= 0;		/* >0 ==> something is playing */
int	Syschan	= 10;		/* D110 system channel */
int	Ipatch	= 64;		/* initial D110 patch */
Pixfont	*Fontp;			/* the font we use (almost everywhere) */
int	Fontwidth;		/* the width of chars in *Fontp */
Frame	Display;			/* whole display's frame */
Panel	Controls;		/* control panel */
Panel_item	Styl;		/* choice; musical "style" */
Panel_item	Tmpo;		/* slider; approx. tempo */
Panel_item	Leng;		/* text; duration of piece */
Panel_item	Stop;		/* button; stop playing */
Panel_item	Play;		/* button; compose & play */
Panel_item	Save;		/* button; save last composition */
Panel_item	Repl;		/* button; replay */
Panel_item	Xprt;		/* button; show/hide panel of experts */
Panel_item	Quit;		/* button; exit */
Frame	Xframe;			/* frame for Xpanel */
Panel	Xpanel;			/* panel of experts */
Panel_item	Lead;		/* button; show/hide lead parameters panel */
Panel_item	Bass;		/* button; show/hide bass parameters panel */
Panel_item	Chrd;		/* button; show/hide chord parameters panel */
Panel_item	Drum;		/* button; show/hide drum parameters panel */
Panel_item	Revo;		/* button; revoice saved compositions */
Panel_item	Key;		/* cycle; key */
Panel_item	Cnti;		/* cycle; count-in */
Panel_item	Opts;		/* toggle; choose optimization */
Panel_item	Lock;		/* toggle; lock current cc, acc, or lead */
Panel_item	Dbug;		/* toggle; choose debugging */
Menu	Inst;			/* all the choices */
Frame	Lframe;			/* frame for Lpanel */
Panel	Lpanel;			/* accl parameters panel */
Panel_item	Linst[3];	/* button; choose lead instruments */
Panel_item	Linam[3];	/* message; lead instrument names */
Panel_item	Lchan[3];	/* text; lead channels */
Panel_item	Lvelo[3];	/* slider; lead velocities */
Panel_item	Lener;		/* slider; lead energy */
Panel_item	Lpred;		/* slider; lead predictability */
Panel_item	Lseed;		/* text; lead seed */
Frame	Bframe;			/* frame for Bpanel */
Panel	Bpanel;			/* acca bass parameters panel */
Panel_item	Bchan;		/* text; bass channel */
Panel_item	Binst;		/* button; choose bass instrument */
Panel_item	Binam;		/* message; bass instrument name */
Panel_item	Bvelo;		/* slider; bass velocity */
Frame	Cframe;			/* frame for Cpanel */
Panel	Cpanel;			/* acca chord parameters panel */
Panel_item	Cchan;		/* text; chord channel */
Panel_item	Cinst;		/* button; choose chord instrument */
Panel_item	Cinam;		/* message; chord instrument name */
Panel_item	Cvelo;		/* slider; chording velocity */
Frame	Dframe;			/* frame for Dpanel */
Panel	Dpanel;			/* acca drum parameters panel */
Panel_item	Dpart;		/* cycle; drum part to use */
Panel_item	Dchan;		/* text; drum channel */
Panel_item	Dvelo;		/* slider; drum velocity */

char	*namegen();
FILE	*nopen();
extern	char	*key2pc(), *strcopy();

main(argc, argv)
char	*argv[];
{
	setbuf(stderr, 0);
	log("exec");
	while (--argc > 0) {
	    if (argv[argc][0] == '-') {
		switch (argv[argc][1]) {
		default:
		    goto syntax;
		}
	    } else {
syntax:
		fprintf(stderr, "Usage: %s\n", argv[0]);
		exit(2);
	    }
	}
	if (!(Fontp = pf_open(FONTPATH)))
	    Fontp = pf_default();
	Fontwidth = Fontp->pf_defaultsize.x;
	if (fork() > 0) {
	    sleep(1);
	    printf("\nWelcome to Bellscore!\n");
	    printf("Please be patient while I wake up the musicians...\n\n");
	    sleep(5);
	    exit(0);
	}
	for (Sp = S; Sp->name && strcmp(Sp->name, "class"); Sp++);
	sprintf(Sysbuf, "rm -f %s.mpu;touch /tmp/mpu0lock >/dev/null", Tmpfile);
	system(Sysbuf);
	miscinits();				/* 1 time SunView inits */
	window_main_loop(Display);
	cleanup(0);
	/*NOTREACHED*/
}

miscinits()	/* one-time SunView inits */
{
	int i, x, y;
	struct singlecolor fgc, bgc;
	Rect tmp;
	Pixwin *dpw;
	void winevent(), cntlproc(), butproc();
	Panel_setting textproc();

	tmp.r_left = 400;
	tmp.r_top = 64;
	tmp.r_width = 600;
	tmp.r_height = 600;
	Display = window_create(NULL, FRAME,
	    FRAME_LABEL,		" -= BELLSCORE MAIN CONTROL PANEL =-",
	    FRAME_ICON,
		icon_create(ICON_IMAGE, &Bsicon_pixrect, 0),
	    WIN_FONT,			Fontp,
	    FRAME_EMBOLDEN_LABEL,	TRUE,
	    WIN_EVENT_PROC,		winevent,
	    FRAME_OPEN_RECT,		&tmp,
	0);
	dpw = (Pixwin *) window_get(Display, WIN_PIXWIN);
	if (dpw->pw_pixrect->pr_depth > 1) {
	    fgc.red = 100;	bgc.red = 240;
	    fgc.green = 0;	bgc.green = 240;
	    fgc.blue = 100;	bgc.blue = 210;
	    window_set(Display,
		FRAME_FOREGROUND_COLOR,	&fgc,
		FRAME_BACKGROUND_COLOR,	&bgc,
		FRAME_INHERIT_COLORS,	TRUE,
	    0);
	}
	Inst = menu_create(
	    MENU_NCOLS,			8,
	    MENU_FONT,			Fontp,
	    MENU_STRINGS,
	    "Acou Piano 1", "Acou Piano 2", "Acou Piano 3", "Honky-Tonk",
	    "Elec Piano 1", "Elec Piano 2", "Elec Piano 3", "Elec Piano 4",
	    "Elec Organ 1", "Elec Organ 2", "Elec Organ 3", "Elec Organ 4",
	    "Pipe Organ 1", "Pipe Organ 2", "Pipe Organ 3", "Accordion",
	    "Harpsi 1", "Harpsi 2", "Harpsi 3", "Clav 1",
	    "Clav 2", "Clav 3", "Celesta 1", "Celesta 2",
	    "Violin 1", "Violin 2", "Cello 1", "Cello 2",
	    "Contrabass", "Pizzicato", "Harp 1", "Harp 2",
	    "Strings 1", "Strings 2", "Strings 3", "Strings 4",
	    "Brass 1", "Brass 2", "Brass 3", "Brass 4",
	    "Trumpet 1", "Trumpet 2", "Trombone 1", "Trombone 2",
	    "Horn", "Fr Horn", "Engl Horn", "Tuba",
	    "Flute 1", "Flute 2", "Piccolo", "Recorder",
	    "Pan Pipes", "Bottleblow", "Breathpipe", "Whistle",
	    "Sax 1", "Sax 2", "Sax 3", "Clarinet 1",
	    "Clarinet 2", "Oboe", "Bassoon", "Harmonica",
	    "Fantasy", "Harmo Pan", "Chorale", "Glasses",
	    "Soundtrack", "Atmosphere", "Warm Bell", "Space Horn",
	    "Echo Bell", "Ice Rains", "Oboe 2002", "Echo Pan",
	    "Bell Swing", "Reso Synth", "Steam Pad", "Vibe String",
	    "Syn Lead 1", "Syn Lead 2", "Syn Lead 3", "Syn Lead 4",
	    "Syn Bass 1", "Syn Bass 2", "Syn Bass 3", "Syn Bass 4",
	    "Acou Bass 1", "Acou Bass 2", "Elec Bass 1", "Elec Bass 2",
	    "Slap Bass 1", "Slap Bass 2", "Fretless 1", "Fretless 2",
	    "Vibe", "Glock", "Marimba", "Xylophone",
	    "Guitar 1", "Guitar 2", "Elec Gtr 1", "Elec Gtr 2",
	    "Koto", "Shamisen", "Jamisen", "Sho",
	    "Shakuhachi", "Wadaiko Set", "Sitar", "Steel Drum",
	    "Tech Snare", "Elec Tom", "Revrse Cym", "Ethno Hit",
	    "Timpani", "Triangle", "Wind Bell", "Tube Bell",
	    "Orche Hit", "Bird Tweet", "One Note Jam", "Telephone",
	    "Typewriter", "Insect", "Water Bells", "Jungle Tune",
	    0,
	0);
	x = y = 0;
	Controls = window_create(Display, PANEL,
	    WIN_X,			x,
	    WIN_Y,			y,
	    WIN_WIDTH,			WIN_EXTEND_TO_EDGE,
	    WIN_FONT,			Fontp,
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_LABEL_BOLD,		TRUE,
	    PANEL_ITEM_X_GAP,		25,
	0);
	Styl = panel_create_item(Controls, PANEL_CHOICE,
	    PANEL_LABEL_STRING,		"STYLE:",
	    PANEL_DISPLAY_LEVEL,	PANEL_ALL,
	    PANEL_CHOICE_STRINGS,	"", 0,
	    PANEL_FEEDBACK,		PANEL_MARKED,
	    PANEL_MENU_TITLE_STRING, 	"Musical Style",
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	x = (int) panel_get(Styl, PANEL_ITEM_X);
	y = (int) panel_get(Styl, PANEL_ITEM_Y) + 4;
	for (i = 0; S[i].name; i++) {
	    if (i && (i % 5) == 0)
		y += 20;
	    panel_set(Styl,
		PANEL_PAINT,		PANEL_NONE,
		PANEL_CHOICE_STRING,	i, S[i].lname,
		PANEL_MARK_X,		i, 60 + (i % 5) * 100,
		PANEL_MARK_Y,		i, y - 2,
		PANEL_CHOICE_X,		i, 80 + (i % 5) * 100,
		PANEL_CHOICE_Y,		i, y,
	    0);
	}
	panel_set(Styl,
	    PANEL_VALUE,		Sp - S,
	    PANEL_PAINT,		PANEL_NONE,
	    PANEL_MENU_CHOICE_STRINGS,
		S[0].desc, S[1].desc, S[2].desc, S[3].desc, S[4].desc,
		S[5].desc, S[6].desc, S[7].desc, S[8].desc, S[9].desc,
		S[10].desc, S[11].desc, S[12].desc, S[13].desc, S[14].desc,
		S[15].desc, S[16].desc, S[17].desc, S[18].desc, S[19].desc,
		S[20].desc, S[21].desc, S[22].desc, S[23].desc, S[24].desc,
		S[25].desc, S[26].desc, S[27].desc, S[28].desc, S[29].desc,
		S[30].desc, S[31].desc,
	    0,
	0);
	window_fit_width(Controls);
	y += 20;
	Leng = panel_create_item(Controls, PANEL_TEXT,
	    PANEL_LABEL_STRING,		"LENGTH:",
	    PANEL_ITEM_X,		x,
	    PANEL_ITEM_Y,		y,
	    PANEL_VALUE_DISPLAY_LENGTH,	9,
	    PANEL_VALUE_STORED_LENGTH,	9,
	    PANEL_VALUE,		Sp->leng,
	    PANEL_MENU_CHOICE_STRINGS,	"Length of Output",
	    				"in seconds (##.#)",
					"or secs & frames (##:##)",
					0,
	    PANEL_NOTIFY_LEVEL,		PANEL_ALL,
	    PANEL_NOTIFY_PROC,		textproc,
	0);
	Tmpo = panel_create_item(Controls, PANEL_SLIDER,
	    PANEL_LABEL_STRING,		"TEMPO:",
	    PANEL_MIN_VALUE,		20,
	    PANEL_VALUE,		Sp->tempo,
	    PANEL_MAX_VALUE,		240,
	    PANEL_SLIDER_WIDTH,		110,
	    PANEL_SHOW_RANGE,		FALSE,
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	y += 20;
	Stop = panel_create_item(Controls, PANEL_BUTTON,
	    PANEL_ITEM_X,		x,
	    PANEL_ITEM_Y,		y,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 " STOP ", 0, 0),
	    PANEL_NOTIFY_PROC,		butproc,
	0);
	Play = panel_create_item(Controls, PANEL_BUTTON,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 " PLAY ", 0, 0),
	    PANEL_NOTIFY_PROC,		butproc,
	0);
	Save = panel_create_item(Controls, PANEL_BUTTON,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 " SAVE ", 0, 0),
	    PANEL_NOTIFY_PROC,		butproc,
	0);
	Repl = panel_create_item(Controls, PANEL_BUTTON,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "REPLAY", 0, 0),
	    PANEL_NOTIFY_PROC,		butproc,
	0);
	Xprt = panel_create_item(Controls, PANEL_BUTTON,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "XYZZY", 0, 0),
	    PANEL_NOTIFY_PROC,		butproc,
	0);
	Quit = panel_create_item(Controls, PANEL_BUTTON,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 " QUIT ", 0, 0),
	    PANEL_NOTIFY_PROC,		butproc,
	0);
	window_fit_height(Controls);
	window_fit(Display);
	y += 60;
	Xframe = window_create(Display, FRAME,
	    WIN_X,			x,
	    WIN_Y,			y,
	    WIN_FONT,			Fontp,
	    FRAME_EMBOLDEN_LABEL,	TRUE,
	    FRAME_SHOW_LABEL,		TRUE,
	    FRAME_LABEL,		"PANEL OF EXPERTS",
	0);
	Xpanel = window_create(Xframe, PANEL,
	    WIN_FONT,			Fontp,
	    PANEL_LABEL_BOLD,		TRUE,
	0);
	Lead = panel_create_item(Xpanel, PANEL_BUTTON,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "LEAD PARAMS", 0, 0),
	    PANEL_NOTIFY_PROC,		butproc,
	0);
	Bass = panel_create_item(Xpanel, PANEL_BUTTON,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "BASS PARAMS", 0, 0),
	    PANEL_NOTIFY_PROC,		butproc,
	0);
	Chrd = panel_create_item(Xpanel, PANEL_BUTTON,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "CHORD PARAMS", 0, 0),
	    PANEL_NOTIFY_PROC,		butproc,
	0);
	Drum = panel_create_item(Xpanel, PANEL_BUTTON,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "DRUM PARAMS", 0, 0),
	    PANEL_NOTIFY_PROC,		butproc,
	0);
	Revo = panel_create_item(Xpanel, PANEL_BUTTON,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "REVOICE", 0, 0),
	    PANEL_NOTIFY_PROC,		butproc,
	0);
	window_fit_width(Xpanel);
	Key = panel_create_item(Xpanel, PANEL_CYCLE,
	    PANEL_LABEL_STRING,		"KEY:",
	    PANEL_CHOICE_STRINGS,	"C", "Db", "D", "Eb", "E", "F",
					"Gb", "G", "Ab", "A", "Bb", "B",
					0,
	    PANEL_VALUE,		Sp->key,
	    PANEL_MENU_TITLE_STRING, 	"Key of Generated Music",
	    PANEL_MENU_CHOICE_STRINGS,	"C", "Db", "D", "Eb", "E", "F",
					"Gb", "G", "Ab", "A", "Bb", "B",
					0,
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	Cnti = panel_create_item(Xpanel, PANEL_CYCLE,
	    PANEL_LABEL_STRING,		"COUNT-IN:",
	    PANEL_CHOICE_STRINGS,	"NONE", "1 BAR", "2 BARS", 0,
	    PANEL_VALUE,		Sp->cnti,
	    PANEL_MENU_TITLE_STRING, 	"Length of count-in before music",
	    PANEL_MENU_CHOICE_STRINGS,	"No count-in", "One bar of count-in",
					"Two bars of count-in", 0,
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	Opts = panel_create_item(Xpanel, PANEL_TOGGLE,
	    PANEL_LABEL_STRING,		" OPTIMIZE:",
	    PANEL_CHOICE_STRINGS,	"mkcc", "acca", "accl", 0,
	    PANEL_VALUE,		0,
	    PANEL_FEEDBACK,		PANEL_INVERTED,
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_MENU_TITLE_STRING,	"Avoid needless regeneration",
	    PANEL_MENU_CHOICE_STRINGS,
		"Don't regenerate chord chart needlessly",
		"Don't regenerate accompaniment needlessly",
		"Don't regenerate lead line needlessly",
	    0,
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	Lock = panel_create_item(Xpanel, PANEL_TOGGLE,
	    PANEL_LABEL_STRING,		" LOCK:",
	    PANEL_CHOICE_STRINGS,	"mkcc", "acca", "accl", 0,
	    PANEL_VALUE,		0,
	    PANEL_FEEDBACK,		PANEL_INVERTED,
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_MENU_TITLE_STRING,	"Lock current version",
	    PANEL_MENU_CHOICE_STRINGS,
		"Don't regenerate chord chart",
		"Don't regenerate accompaniment",
		"Don't regenerate lead line",
	    0,
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	Dbug = panel_create_item(Xpanel, PANEL_TOGGLE,
	    PANEL_LABEL_STRING,		" DEBUG:",
	    PANEL_CHOICE_STRINGS,	"mkcc", "acca", "accl", "bellscore", 0,
	    PANEL_VALUE,		0,
	    PANEL_FEEDBACK,		PANEL_INVERTED,
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_MENU_TITLE_STRING,	"Debugging output",
	    PANEL_MENU_CHOICE_STRINGS,	"chord chart display (in window)",
					"accompaniment generator >/tmp/acca.2",
					"lead line generator >/tmp/acca.2",
					"bellscore commands & misc (in window)",
	    0,
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	window_fit(Xpanel);
	window_fit(Xframe);
	y += 110;
	Lframe = window_create(Display, FRAME,
	    WIN_X,			x,
	    WIN_Y,			y,
	    WIN_FONT,			Fontp,
	    FRAME_SHOW_LABEL,		TRUE,
	    FRAME_LABEL,		"LEAD INSTRUMENT PARAMETERS",
	0);
	Lpanel = window_create(Lframe, PANEL,
	    WIN_FONT,			Fontp,
	    PANEL_LABEL_BOLD,		TRUE,
	0);
	for (i = 0; i < 3; i++) {
	    Linst[i] = panel_create_item(Lpanel, PANEL_BUTTON,
		PANEL_LABEL_IMAGE,		panel_button_image(Controls,
						     "VOICE", 0, 0),
		PANEL_NOTIFY_PROC,		butproc,
	    0);
	    Linam[i] = panel_create_item(Lpanel, PANEL_MESSAGE,
		PANEL_LABEL_STRING,		"             ",
	    0);
	    Lchan[i] = panel_create_item(Lpanel, PANEL_CHOICE,
		PANEL_VALUE,			Sp->lchan[i],
		PANEL_DISPLAY_LEVEL,		PANEL_CURRENT,
		PANEL_LABEL_STRING,		"CHANNEL:",
		PANEL_CHOICE_STRINGS,	"OFF",
		 "1   ", "2   ", "3   ", "4   ", "5   ", "6   ", "7   ", "8   ",
		 "9   ", "10  ", "11  ", "12  ", "13  ", "14  ", "15  ", "16  ",
		0,
		PANEL_SHOW_MENU,		TRUE,
		PANEL_MENU_TITLE_STRING, 	"MIDI Channel",
		PANEL_MENU_CHOICE_STRINGS,	"No Lead Instrument",
		    " 1 CZ-101", " 2 D-110 Part 1", " 3 D-110 Part 2",
		    " 4 D-110 Part 3", " 5 D-110 Part 4", " 6 D-110 Part 5",
		    " 7 D-110 Part 6", " 8 D-110 Part 7", " 9 D-110 Part 8",
		    "10 D-110 Rhythm", "11 ???", "12 ???", "13 ???",
		    "14 ???", "15 ???", "16 SPM 8:2",
		0,
		PANEL_FEEDBACK,			PANEL_NONE,
		PANEL_NOTIFY_PROC,		cntlproc,
	    0);
	    Lvelo[i] = panel_create_item(Lpanel, PANEL_SLIDER,
		PANEL_LABEL_STRING,		"VOLUME:",
		PANEL_MIN_VALUE,		1,
		PANEL_VALUE,			Sp->lvelo[i],
		PANEL_MAX_VALUE,		127,
		PANEL_SLIDER_WIDTH,		127,
		PANEL_SHOW_RANGE,		FALSE,
		PANEL_NOTIFY_PROC,		cntlproc,
	    0);
	    if (i == 0)
		window_fit_width(Lpanel);
	}
	Lener = panel_create_item(Lpanel, PANEL_SLIDER,
	    PANEL_LABEL_STRING,		"ENERGY:",
	    PANEL_MIN_VALUE,		0,
	    PANEL_VALUE,		Sp->lener,
	    PANEL_MAX_VALUE,		100,
	    PANEL_SLIDER_WIDTH,		102,
	    PANEL_SHOW_RANGE,		FALSE,
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	Lpred = panel_create_item(Lpanel, PANEL_SLIDER,
	    PANEL_LABEL_STRING,		"PREDICTABILITY:",
	    PANEL_MIN_VALUE,		0,
	    PANEL_VALUE,		Sp->lpred,
	    PANEL_MAX_VALUE,		100,
	    PANEL_SLIDER_WIDTH,		102,
	    PANEL_SHOW_RANGE,		FALSE,
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	Lseed = panel_create_item(Lpanel, PANEL_TEXT,
	    PANEL_LABEL_STRING,		"SEED:",
	    PANEL_VALUE_DISPLAY_LENGTH,	16,
	    PANEL_VALUE_STORED_LENGTH,	16,
	    PANEL_VALUE,		Sp->lseed,
	    PANEL_MENU_CHOICE_STRINGS,	"Random Number Seed",
	    				"0 => use time of day",
					0,
	    PANEL_NOTIFY_LEVEL,		PANEL_ALL,
	    PANEL_NOTIFY_PROC,		textproc,
	0);
	window_fit_height(Lpanel);
	window_fit(Lframe);
	y += 150;
	Bframe = window_create(Display, FRAME,
	    WIN_X,			x,
	    WIN_Y,			y,
	    WIN_FONT,			Fontp,
	    FRAME_SHOW_LABEL,		TRUE,
	    FRAME_LABEL,		"BASS INSTRUMENT PARAMETERS",
	0);
	Bpanel = window_create(Bframe, PANEL,
	    WIN_FONT,			Fontp,
	    PANEL_LABEL_BOLD,		TRUE,
	0);
	Binst = panel_create_item(Bpanel, PANEL_BUTTON,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "VOICE", 0, 0),
	    PANEL_NOTIFY_PROC,		butproc,
	0);
	Binam = panel_create_item(Bpanel, PANEL_MESSAGE,
	    PANEL_LABEL_STRING,		"             ",
	0);
	Bchan = panel_create_item(Bpanel, PANEL_CHOICE,
	    PANEL_VALUE,		Sp->bchan,
	    PANEL_DISPLAY_LEVEL,	PANEL_CURRENT,
	    PANEL_LABEL_STRING,		"CHANNEL:",
	    PANEL_CHOICE_STRINGS,	"OFF",
		" 1", " 2", " 3", " 4", " 5", " 6", " 7", " 8",
		" 9", "10", "11", "12", "13", "14", "15", "16",
	    0,
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_MENU_TITLE_STRING, 	"MIDI Channel",
	    PANEL_MENU_CHOICE_STRINGS,	"No Bass Instrument",
		" 1 CZ-101", " 2 D-110 Part 1", " 3 D-110 Part 2",
		" 4 D-110 Part 3", " 5 D-110 Part 4", " 6 D-110 Part 5",
		" 7 D-110 Part 6", " 8 D-110 Part 7", " 9 D-110 Part 8",
		"10 D-110 Rhythm", "11 ???", "12 ???", "13 ???",
		"14 ???", "15 ???", "16 SPM 8:2",
	    0,
	    PANEL_FEEDBACK,		PANEL_NONE,
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	Bvelo = panel_create_item(Bpanel, PANEL_SLIDER,
	    PANEL_LABEL_STRING,		"VOLUME:",
	    PANEL_MIN_VALUE,		1,
	    PANEL_VALUE,		Sp->bvelo,
	    PANEL_MAX_VALUE,		127,
	    PANEL_SLIDER_WIDTH,		127,
	    PANEL_SHOW_RANGE,		FALSE,
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	window_fit(Bpanel);
	window_fit(Bframe);
	y += 70;
	Cframe = window_create(Display, FRAME,
	    WIN_X,			x,
	    WIN_Y,			y,
	    WIN_FONT,			Fontp,
	    FRAME_SHOW_LABEL,		TRUE,
	    FRAME_LABEL,		"CHORDING INSTRUMENT PARAMETERS",
	0);
	Cpanel = window_create(Cframe, PANEL,
	    WIN_FONT,			Fontp,
	    PANEL_LABEL_BOLD,		TRUE,
	0);
	Cinst = panel_create_item(Cpanel, PANEL_BUTTON,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "VOICE", 0, 0),
	    PANEL_NOTIFY_PROC,		butproc,
	0);
	Cinam = panel_create_item(Cpanel, PANEL_MESSAGE,
	    PANEL_LABEL_STRING,		"             ",
	0);
	Cchan = panel_create_item(Cpanel, PANEL_CHOICE,
	    PANEL_VALUE,		Sp->cchan,
	    PANEL_DISPLAY_LEVEL,	PANEL_CURRENT,
	    PANEL_LABEL_STRING,		"CHANNEL:",
	    PANEL_CHOICE_STRINGS,	"OFF",
		" 1", " 2", " 3", " 4", " 5", " 6", " 7", " 8",
		" 9", "10", "11", "12", "13", "14", "15", "16",
	    0,
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_MENU_TITLE_STRING, 	"MIDI Channel",
	    PANEL_MENU_CHOICE_STRINGS,	"No Chording Instrument",
		" 1 CZ-101", " 2 D-110 Part 1", " 3 D-110 Part 2",
		" 4 D-110 Part 3", " 5 D-110 Part 4", " 6 D-110 Part 5",
		" 7 D-110 Part 6", " 8 D-110 Part 7", " 9 D-110 Part 8",
		"10 D-110 Rhythm", "11 ???", "12 ???", "13 ???",
		"14 ???", "15 ???", "16 SPM 8:2",
	    0,
	    PANEL_FEEDBACK,		PANEL_NONE,
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	Cvelo = panel_create_item(Cpanel, PANEL_SLIDER,
	    PANEL_LABEL_STRING,		"VOLUME:",
	    PANEL_MIN_VALUE,		1,
	    PANEL_VALUE,		Sp->cvelo,
	    PANEL_MAX_VALUE,		127,
	    PANEL_SLIDER_WIDTH,		127,
	    PANEL_SHOW_RANGE,		FALSE,
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	window_fit(Cpanel);
	window_fit(Cframe);
	y += 70;
	Dframe = window_create(Display, FRAME,
	    WIN_X,			x,
	    WIN_Y,			y,
	    WIN_FONT,			Fontp,
	    FRAME_SHOW_LABEL,		TRUE,
	    FRAME_LABEL,		"DRUM PARAMETERS",
	0);
	Dpanel = window_create(Dframe, PANEL,
	    WIN_FONT,			Fontp,
	    PANEL_LABEL_BOLD,		TRUE,
	0);
	Dpart = panel_create_item(Dpanel, PANEL_CYCLE,
	    PANEL_LABEL_STRING,		"PART:",
	    PANEL_CHOICE_STRINGS,	"primary", "alt 1", "alt 2", "alt 3", 0,
	    PANEL_VALUE,		Sp->dpart,
	    PANEL_MENU_CHOICE_STRINGS,	"Use \"normal\" drum part",
					"Use first alternate drum part",
					"Use second alternate drum part",
					"Use third alternate drum part",
					0,
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	Dchan = panel_create_item(Dpanel, PANEL_CHOICE,
	    PANEL_VALUE,		Sp->dchan,
	    PANEL_DISPLAY_LEVEL,	PANEL_CURRENT,
	    PANEL_LABEL_STRING,		"CHANNEL:",
	    PANEL_CHOICE_STRINGS,	"OFF",
		" 1", " 2", " 3", " 4", " 5", " 6", " 7", " 8",
		" 9", "10", "11", "12", "13", "14", "15", "16",
	    0,
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_MENU_TITLE_STRING, 	"MIDI Channel",
	    PANEL_MENU_CHOICE_STRINGS,	"No Drums",
		" 1 CZ-101", " 2 D-110 Part 1", " 3 D-110 Part 2",
		" 4 D-110 Part 3", " 5 D-110 Part 4", " 6 D-110 Part 5",
		" 7 D-110 Part 6", " 8 D-110 Part 7", " 9 D-110 Part 8",
		"10 D-110 Rhythm", "11 ???", "12 ???", "13 ???",
		"14 ???", "15 ???", "16 SPM 8:2",
	    0,
	    PANEL_FEEDBACK,		PANEL_NONE,
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	Dvelo = panel_create_item(Dpanel, PANEL_SLIDER,
	    PANEL_LABEL_STRING,		"VOLUME:",
	    PANEL_MIN_VALUE,		1,
	    PANEL_VALUE,		Sp->dvelo,
	    PANEL_MAX_VALUE,		127,
	    PANEL_SLIDER_WIDTH,		127,
	    PANEL_SHOW_RANGE,		FALSE,
	    PANEL_NOTIFY_PROC,		cntlproc,
	0);
	window_fit(Dpanel);
	window_fit(Dframe);
	/* these done at end to fix max sizes first */
	panel_set(Linam[0], PANEL_LABEL_STRING, IMGS(Sp->linst[0]), 0);
	panel_set(Linam[1], PANEL_LABEL_STRING, IMGS(Sp->linst[1]), 0);
	panel_set(Linam[2], PANEL_LABEL_STRING, IMGS(Sp->linst[2]), 0);
	panel_set(Binam, PANEL_LABEL_STRING, IMGS(Sp->binst), 0);
	panel_set(Cinam, PANEL_LABEL_STRING, IMGS(Sp->cinst), 0);
	Debug = (int) PGV(Dbug) & M_BELLSCORE;
}

void
butproc(item, event)
Panel_item item;
Event	*event;
{
	register char *cp;
	register int i;
	char tmpmpu[128];

	/* Controls buttons */
	if (item == Play) {
	    gogo();
	} else if (item == Quit) {
	    cleanup(0);
	    /*NOTREACHED*/
	} else if (item == Repl) {
	    replay(event);
	} else if (item == Save) {
	    sprintf(tmpmpu, "%s.mpu", Tmpfile);
	    if (Nsaved <= 0
	     || strcmp(tmpmpu, Sfile[i = (Nsaved - 1) % MAXSAV]))
		fprintf(stderr, "Nothing to save\n");
	    else {
		cp = namegen(Sp, Sp->tempo, Nsaved);
		sprintf(Sysbuf, "mv -f %s %s", tmpmpu, cp);
		sys(Sysbuf);
		strcopy(Sfile[i], cp);
	    }
	} else if (item == Stop) {
	    if (Playpid && reap(Playpid, WNOHANG)) {
		kill(Playpid, SIGINT);
		Playpid = 0;
	    }
	} else if (item == Xprt) {
	    Xprtsho = (TRUE + FALSE) - Xprtsho;
	    window_set(Xpanel, WIN_SHOW, Xprtsho, 0);
	    window_set(Xframe, WIN_SHOW, Xprtsho, 0);
	/* Xpanel buttons */
	} else if (item == Bass) {
	    Basssho = (TRUE + FALSE) - Basssho;
	    window_set(Bpanel, WIN_SHOW, Basssho, 0);
	    window_set(Bframe, WIN_SHOW, Basssho, 0);
	} else if (item == Chrd) {
	    Chrdsho = (TRUE + FALSE) - Chrdsho;
	    window_set(Cpanel, WIN_SHOW, Chrdsho, 0);
	    window_set(Cframe, WIN_SHOW, Chrdsho, 0);
	} else if (item == Drum) {
	    Drumsho = (TRUE + FALSE) - Drumsho;
	    window_set(Dpanel, WIN_SHOW, Drumsho, 0);
	    window_set(Dframe, WIN_SHOW, Drumsho, 0);
	} else if (item == Lead) {
	    Leadsho = (TRUE + FALSE) - Leadsho;
	    window_set(Lpanel, WIN_SHOW, Leadsho, 0);
	    window_set(Lframe, WIN_SHOW, Leadsho, 0);
	} else if (item == Revo) {
	    revoice(event);
	/* Bpanel, Cpanel, Dpanel, & Lpanel buttons */
	} else if (item == Binst) {
	    i = (int) menu_show(Inst, Xframe, event, 0);
	    if (0 < i && i <= 128)
		setinst(Sp->bchan, Sp->binst = i, Binam);
	} else if (item == Cinst) {
	    i = (int) menu_show(Inst, Xframe, event, 0);
	    if (0 < i && i <= 128)
		setinst(Sp->cchan, Sp->cinst = i, Cinam);
	} else if (item == Linst[0]) {
	    i = (int) menu_show(Inst, Xframe, event, 0);
	    if (0 < i && i <= 128)
		setinst(Sp->lchan[0], Sp->linst[0] = i, Linam[0]);
	} else if (item == Linst[1]) {
	    i = (int) menu_show(Inst, Xframe, event, 0);
	    if (0 < i && i <= 128)
		setinst(Sp->lchan[0], Sp->linst[1] = i, Linam[1]);
	} else if (item == Linst[2]) {
	    i = (int) menu_show(Inst, Xframe, event, 0);
	    if (0 < i && i <= 128)
		setinst(Sp->lchan[0], Sp->linst[2] = i, Linam[2]);
	}
}

setinst(chan, inst, inam)
Panel_item	inam;
{
	if (!chan)
	    return;
	sprintf(Sysbuf, "%s %d=%d | %s", INST, chan, inst, PLAY);
	sys(Sysbuf);
	panel_set(inam, PANEL_LABEL_STRING, IMGS(inst), 0);
	sprintf(Sysbuf, "%s </u/psl/midi/etc/sample %d=1-16 | %s",
	 CHMAP, chan, PLAY);
	sys(Sysbuf);
}

gogo()
{
	char *cp, *bp, buf[256], obuf[128];
	int tempo, frames, clox, unit, i, t;
	double bars;
	extern double atof();

	tempo = Sp->tempo;
	cp = (char *) PGV(Leng);
	for (bp = cp; *bp && *bp != ':'; bp++);
	if (*bp++ == ':') {
	    frames = 30 * atoi(cp) + atoi(bp);
	} else {
	    frames = 30 * atof(cp);
	}
	clox = (2 * tempo * frames) / 30;
DBG(stderr, "tempo=%d leng=%s clox=%d\n", tempo, cp, clox);
	if (!LOCKED(M_MKCC) && (!Mkccok || !OPTIM(M_MKCC))) {
	    unit = Sp->units;
	    i = clox % unit;
	    if (i) {			/* adjust to a multiple of unit */
		clox += ((i + i) >= unit? unit : 0) - i;
		for (;;) {		/* by changing tempo */
		    tempo = (15 * clox + frames - 1) / frames;
		    if (tempo > 240)
			clox -= unit;
		    else if (tempo < 10)
			clox += unit;
		    else
			break;
		}
DBG(stderr, "change tempo to %d; ", tempo);
		PSV(Tmpo, Sp->tempo = tempo);
	    }
	    bars = clox / (float) Sp->barlen;
DBG(stderr,"tempo=%d, %d clox, %d frames, %g bars\n",tempo,clox,frames,bars);
	    sprintf(Sysbuf, "%s -b%g -k%s -s%s >%s.cc",
	     MKCC, bars, key2pc(Sp->key), Sp->name, Tmpfile);
	    sys(Sysbuf);
	    Mkccok = TRUE;
	    Accaok = FALSE;
	    if ((int) PGV(Dbug) & M_MKCC) {
		sprintf(Sysbuf, "cat %s.cc", Tmpfile);
		system(Sysbuf);
	    }
	}
	if (!LOCKED(M_ACCA) && (!Accaok || !OPTIM(M_ACCA))) {
	    sprintf(Sysbuf, "%s -c%d=%d -v%d=%d -c%d=%d -v%d=%d -v%d=%d -t%d",
	     ACCA,
	     Sp->bchan, BCHAN, Sp->bchan, Sp->bvelo,
	     Sp->cchan, CCHAN, Sp->cchan, Sp->cvelo,
	     Sp->dchan, Sp->dvelo, Sp->dchan);
	    cp = strcopy(buf, Sysbuf);
	    for (i = 0; i < MAXNDP; i++) {
		if (Sp->dpchn[i]) {
		    sprintf(Sysbuf, " -c%d=%d",
		     Sp->dpart == i? Sp->dchan : 0, Sp->dpchn[i]);
		    cp = strcopy(cp, Sysbuf);
		}
	    }
	    if ((int) PGV(Dbug) & M_ACCA)
		sprintf(Sysbuf, "%s -d %s.cc >%s.acc 2>/tmp/acca.2",
		 buf, Tmpfile, Tmpfile);
	    else
		sprintf(Sysbuf, "%s %s.cc >%s.acc", buf, Tmpfile, Tmpfile);
	    sys(Sysbuf);
	    Accaok = TRUE;
	}
	if (!LOCKED(M_ACCL) && (!Acclok || !OPTIM(M_ACCL))) {
	    bars = clox / (float) Sp->barlen;
	    sprintf(buf, "%s -b%g, -c%d=%d -c%d=%d -c%d=%d -e%d -k%s -p%d -s%d",
	     ACCL, bars, Sp->lchan[0], Sp->lvelo[0],
	     Sp->lchan[1], Sp->lvelo[1], Sp->lchan[2], Sp->lvelo[2],
	     Sp->lener, key2pc(Sp->key), Sp->lpred, (i = atoi(PGV(Lseed))));
	    Acclok = (i? TRUE : FALSE);
	    if ((int) PGV(Dbug) & M_ACCL)
		sprintf(Sysbuf, "%s -d %s.cc >%s.lead 2>/tmp/accl.2",
		 buf, Tmpfile, Tmpfile);
	    else
		sprintf(Sysbuf, "%s %s.cc >%s.lead", buf, Tmpfile, Tmpfile);
	    sys(Sysbuf);
	}
	sprintf(obuf, "%s.mpu", Tmpfile);
	instdump(Sp, obuf, Sp->cnti * (Sp->barlen / 120));
	if (Sp->lchan[0] || Sp->lchan[1] || Sp->lchan[2])
	    sprintf(Sysbuf, "%s %s.lead %s.acc | %s >>%s",
	     MERGE, Tmpfile, Tmpfile, MCLEAN, obuf);
	else
	    sprintf(Sysbuf, "%s <%s.acc >>%s", MCLEAN, Tmpfile, obuf);
	sys(Sysbuf);
	sprintf(Sysbuf, "-t%d", tempo);
	reap(Playpid, 0);			/* wait for Playpid==0 */
	Playpid = forkexec(-1, -1, PLAY, Sysbuf, obuf, 0);
	if (Nsaved <= 0 || strcmp(buf, Sfile[(Nsaved - 1) % MAXSAV]))
	    strcopy(Sfile[Nsaved++ % MAXSAV], obuf);
}

instdump(sp, file, cntin)
struct	stylstr	*sp;
char	*file;
{
	char obuf[256], buf[64], *cp;

	if (cntin) {
	    sprintf(obuf, "echo '0 %x %x 70 78 %x 0' | axtobb | rpt %d >%s",
	     CH_KEY_ON | Sp->dchan - 1, METKEY, METKEY, cntin, file);
	    sys(obuf);
	} else {
	    sprintf(Sysbuf, "cat /dev/null >%s", file);
	    sys(Sysbuf);
	}
	sprintf(buf, "%s -d >>%s %d=%d", INST, file, Syschan, Ipatch);
	cp = strcopy(obuf, buf);
	if (sp->bchan) {
	    sprintf(buf, " %d=%d", sp->bchan, sp->binst);
	    cp = strcopy(cp, buf);
	}
	if (sp->cchan) {
	    sprintf(buf, " %d=%d", sp->cchan, sp->cinst);
	    cp = strcopy(cp, buf);
	}
	if (sp->lchan[0]) {
	    sprintf(buf, " %d=%d", sp->lchan[0], sp->linst[0]);
	    cp = strcopy(cp, buf);
	}
	if (sp->lchan[1]) {
	    sprintf(buf, " %d=%d", sp->lchan[1], sp->linst[1]);
	    cp = strcopy(cp, buf);
	}
	if (sp->lchan[2]) {
	    sprintf(buf, " %d=%d", sp->lchan[2], sp->linst[2]);
	    cp = strcopy(cp, buf);
	}
	sys(obuf);
}

replay(event)
Event	*event;
{
	register int i, j;
	Menu menu;

	menu = menu_create(0);
	i = Nsaved - MAXSAV;
	if (Nsaved > MAXSAV)
	    for (i = Nsaved; (j = --i % MAXSAV) != (Nsaved % MAXSAV); )
		menu_set(menu, MENU_STRING_ITEM, Sfile[j], j + 1, 0);
	else
	    for (i = Nsaved; --i >= 0; )
		menu_set(menu, MENU_STRING_ITEM, Sfile[i], i + 1, 0);
	if (j = (int) menu_show(menu, Controls, event, 0)) {
	    sprintf(Sysbuf, "-t%d", Sp->tempo);
	    reap(Playpid, 0);			/* wait for Playpid==0 */
	    Playpid = forkexec(-1, -1, PLAY, Sysbuf, Sfile[j - 1], 0);
	}
}

revoice(event)
Event	*event;
{
	register int i, j;
	char tfile[128];
	Menu menu;

	menu = menu_create(0);
	i = Nsaved - MAXSAV;
	if (Nsaved > MAXSAV)
	    for (i = Nsaved; (j = --i % MAXSAV) != (Nsaved % MAXSAV); )
		menu_set(menu, MENU_STRING_ITEM, Sfile[j], j + 1, 0);
	else
	    for (i = Nsaved; --i >= 0; )
		menu_set(menu, MENU_STRING_ITEM, Sfile[i], i + 1, 0);
	if (j = (int) menu_show(menu, Xpanel, event, 0)) {
	    sprintf(tfile, "%s.tmp", Tmpfile);
	    instdump(Sp, tfile, 0);
	    sprintf(Sysbuf, "select -P <%s >>%s && mv %s %s",
	     Sfile[j - 1], tfile, tfile, Sfile[j - 1]);
	    sys(Sysbuf);
	}
}

Panel_setting
textproc(item, event)
Panel_item item;
Event	*event;
{
	if (item == Lseed)
	    Acclok = FALSE;
	else if (item == Leng)
	    Mkccok = Accaok = Acclok = FALSE;
	return(panel_text_notify(item, event));
}

void
cntlproc(item, value, event)
Panel_item item;
Event	*event;
{
	int i;

	if (item == Lchan[0] || item == Lchan[1] || item == Lchan[2]
	 || item == Lvelo[0] || item == Lvelo[1] || item == Lvelo[2]
	 || item == Lseed || item == Lener || item == Lpred)
	    Acclok = FALSE;
	else if (item == Bchan || item == Binst || item == Bvelo)
	    Accaok = FALSE;
	else if (item == Cchan || item == Cinst || item == Cvelo)
	    Accaok = FALSE;
	else if (item == Dchan || item == Dvelo)
	    Accaok = FALSE;
	else if (item == Styl || item == Key || item == Tmpo || item == Leng)
	    Mkccok = Accaok = Acclok = FALSE;
	if (item == Key)	Sp->key = value;
	else if (item == Cnti)	Sp->cnti = value;
	else if (item == Tmpo)	Sp->tempo = value;
	else if (item == Bchan)	Sp->bchan = value;
	else if (item == Bvelo)	Sp->bvelo = value;
	else if (item == Cchan)	Sp->cchan = value;
	else if (item == Cvelo)	Sp->cvelo = value;
	else if (item == Dchan)	Sp->dchan = value;
	else if (item == Dvelo)	Sp->dvelo = value;
	else if (item == Lchan[0])	Sp->lchan[0] = value;
	else if (item == Lchan[1])	Sp->lchan[1] = value;
	else if (item == Lchan[2])	Sp->lchan[2] = value;
	else if (item == Lvelo[0])	Sp->lvelo[0] = value;
	else if (item == Lvelo[1])	Sp->lvelo[1] = value;
	else if (item == Lvelo[2])	Sp->lvelo[2] = value;
	else if (item == Lener)	Sp->lener = value;
	else if (item == Lpred)	Sp->lpred = value;
	else if (item == Dbug)
	    Debug = value & M_BELLSCORE;
	else if (item == Dpart) {
	    if (value >= MAXNDP || !Sp->dpchn[value]) {
		if (Sp->dpart == 0) {
		    for (i = MAXNDP; --i >= 0 && !Sp->dpchn[i]; );
		    if (i >= 0)
			fprintf(stderr, "Only %d drum part%s defined for %s\n",
			 i + 1, i==0? "" : "s", Sp->lname);
		    else
			fprintf(stderr, "%s has no drum part\n", Sp->lname);
		}
		PSV(Dpart, Sp->dpart = 0);
	    } else
		Sp->dpart = value;
	} else if (item == Styl) {
	    strcopy(Sp->leng, PGV(Leng));
	    strcopy(Sp->lseed, PGV(Lseed));
	    if (S[value].binst)
		Sp = &S[value];
	    else {
		fprintf(stderr, "\"%s\" not built yet\n", S[value].lname);
		fprintf(stderr,
		 "All styles in lower case are under construction...\n");
		PSV(Styl, Sp - S);
	    }
	    PSV(Key, Sp->key);
	    PSV(Cnti, Sp->cnti);
	    PSV(Leng, Sp->leng);
	    PSV(Tmpo, Sp->tempo);
	    panel_set(Binam, PANEL_LABEL_STRING, IMGS(Sp->binst), 0);
	    PSV(Bchan, Sp->bchan);
	    PSV(Bvelo, Sp->bvelo);
	    panel_set(Cinam, PANEL_LABEL_STRING, IMGS(Sp->cinst), 0);
	    PSV(Cchan, Sp->cchan);
	    PSV(Cvelo, Sp->cvelo);
	    PSV(Dpart, Sp->dpart);
	    PSV(Dchan, Sp->dchan);
	    PSV(Dvelo, Sp->dvelo);
	    panel_set(Linam[0], PANEL_LABEL_STRING, IMGS(Sp->linst[0]), 0);
	    panel_set(Linam[1], PANEL_LABEL_STRING, IMGS(Sp->linst[1]), 0);
	    panel_set(Linam[2], PANEL_LABEL_STRING, IMGS(Sp->linst[2]), 0);
	    PSV(Lchan[0], Sp->lchan[0]);
	    PSV(Lchan[1], Sp->lchan[1]);
	    PSV(Lchan[2], Sp->lchan[2]);
	    PSV(Lvelo[0], Sp->lvelo[0]);
	    PSV(Lvelo[1], Sp->lvelo[1]);
	    PSV(Lvelo[2], Sp->lvelo[2]);
	    PSV(Lener, Sp->lener);
	    PSV(Lpred, Sp->lpred);
	    PSV(Lseed, Sp->lseed);
	}
}

/*VARARGS3*/
forkexec(in, out, p, a1, a2, a3, a4)
char	*p, *a1, *a2, *a3, *a4;
{
	register int pid;

DBG(stderr, "forkexec(%d, %d, %s", in, out, p);
if (Debug) {	if (a1) { fprintf(stderr, ", %s", a1);
		 if (a2) { fprintf(stderr, ", %s", a2);
		  if (a3) { fprintf(stderr, ", %s", a3);
		   if (a4) { fprintf(stderr, ", %s", a4);
		}}}}
}
DBG(stderr, ")\n");
	switch (pid = fork()) {
	case -1:
	    perror("fork()");
	    break;
	case 0:
	    if (in >= 0) {
		close(0);
		dup2(in, 0);
	    }
	    if (out >= 0) {
		close(1);
		dup2(out, 1);
	    }
	    execlp(p, p, a1, a2, a3, a4, 0);
	    perror(p);
	    exit(1);
	}
	return(pid);
}

/* reap dead children; let wait hang unless flg == WNOHANG */
/* ret 0 if pid is dead, 1 otherwise (note with flg==0, return will be 0) */
reap(pid, flg)
{
	int i;

	do {
	    if ((i = wait3(0, flg, 0)) == 0)
		return(1);
	} while (i != -1 && i != pid);
	return(0);
}

void
winevent(window, event, arg)
Window	window;
Event	*event;
caddr_t	arg;
{
	if (event_id(event) == WIN_RESIZE)
	    ;
	window_default_event_proc(window, event, arg);
}

char	*
namegen(sp, tempo, n)
struct	stylstr	*sp;
{
	char *cp;
	static char buf[64];

	cp = "GLACIAL_\0SLOW_\0   \0        HOT_\0    DEMONIC_\0MANIC_";
	sprintf(buf, "/tmp/%s%s_%d", &cp[(tempo / 43) * 9], sp->lname, n);
	return(buf);
}

FILE	*
nopen(file, mode)			/* noisy open */
char	*file, *mode;
{
	FILE *fp;

	if ((fp = fopen(file, mode)) == (FILE *) NULL)
	    perror(file);
	return(fp);
}

cleanup(exstat)
{
	if (*Tmpfile) {
	    sprintf(Sysbuf, "rm -f %s.* /tmp/acc?.2", Tmpfile);
	    sys(Sysbuf);
	}
	window_set(Display, WIN_SHOW, FALSE, 0);
	log("exit");
	unlink("/tmp/mpu0lock");
	exit(exstat);
}

log(msg)
char	*msg;
{
	long now;
	struct passwd *pwent;
	FILE *ofp;

	if (!(ofp = fopen(LOGFILE, "w")))
	    return;
	now = time(0);
	pwent = getpwuid(getuid());
	fprintf(ofp, "%16.16s %-8s %s\n", ctime(&now), pwent->pw_name, msg);
	fclose(ofp);
}

sys(buf)
char	*buf;
{
DBG(stderr, "%s\n", buf);
	system(buf);
}
