/*	DDMT -- DDM Tool; Sunview interface to ddm, psl 1/89 */

#include	<suntool/sunview.h>
#include	<suntool/panel.h>
#include	<midi.h>
#include	<stdio.h>

#define	MAXCHAN		16
#define	MAXKEY		128
#define	MAXVEL		128
#define	FONTPATH	"/usr/lib/fonts/fixedwidthfonts/screen.r.14"
#define	MAXINST		12
#define	MAXRESO		12
#define	LKDRM		35	/* lowest drum key to generate */
#define	HKDRM		97	/* highest drum key to generate */
#define	LKMEL		31	/* lowest melody key to generate */
#define	HKMEL		84	/* highest melody key to generate */
#define	LKRND		31	/* (min(LKMEL, max(13, LKDRM))) */
#define	HKRND		101	/* (4+max(HKMEL,HKDRM)) */
#define	PGV(I)		panel_get_value(I)
#define	PSV(I,V)	panel_set_value(I,V)
#define	GIV(I,F)	((int)panel_get_value(Inst[I].F))
#define	SIV(I,F,V)	panel_set_value(Inst[I].F,V)
#define	DOWN		0	/* for beat */
#define	UP		1	/* for beat */

/* synth specific stuff */
#define	PRCHAN		(9)		/* pick a rhythm channel */
#define	PMCHAN		(rand()%9)	/* pick a melody channel */
#define	ISRCHAN(C)	((C)==9)	/* is a rhythm channel */
#define	ISMCHAN(C)	((C)!=9)	/* is a melody channel */
#define	SYNTHINIT	"inst 10=1 | play"

char	Tmpfile[128];		/* where .ddm lives while we work */
char	*Playcmd = "play";
int	Barlen	= 480;		/* MIDI clocks per bar, 0=> use TCWMEs */
int	Playpid	= 0;		/* >0 ==> something is playing */
Frame	Dframe;			/* whole display's frame */
Pixfont	*Fontp;			/* the font we use (almost everywhere) */
int	Fontwidth;		/* the width of chars in *Fontp */
Panel	Controls;		/* control panel */
int	Cntlshow = TRUE;	/* whether to show control panel */
int	Durs[MAXRESO]	= {	/* used for both duration & resolution */
	64, 32, 16, 8, 4, 2, 1, 0, };

struct	instr	{
	Panel_item	name;	/* text; what we call it */
	Panel_item	note;	/* text; pitch or offset */
	Panel_item	dens;	/* slider; subdivision probability */
	Panel_item	beat;	/* choice; the old up/down dilemma */
	Panel_item	reso;	/* choice; resolution */
	Panel_item	dura;	/* choice; duration */
	Panel_item	velo;	/* slider; velocity */
	Panel_item	chan;	/* choice; channel */
} Inst[MAXINST];
Panel_item	Scal;		/* toggles; scale */
Panel_item	Lolm;		/* text; low limit */
Panel_item	Hilm;		/* text; high limit */
Panel_item	Seed;		/* text; random number seed */
Panel_item	Quit;		/* button; exit */
Panel_item	Rand;		/* button; random parameters */
Panel_item	Nice;		/* button; improve parameters */
Panel_item	Play;		/* button; play it */
Panel_item	Test;		/* button; generate with -debug */
Panel_item	Read;		/* button; read a file */
Panel_item	Writ;		/* button; write a file */
Panel_item	File;		/* text; file name */
Panel_item	Tmpo;		/* slider; tempo */
Panel_item	Bars;		/* slider; number of bars */
Panel_item	Rpts;		/* slider; repeat limit */

#ifdef LINT
short	*Whole, *Half, *Quarter, *Eighth, *Sixteenth,
	*Thirtysecond, *Sixtyfourth, *Percussion;
#else
/* Format_version=1, Width=16, Height=16, Depth=1, Valid_bits_per_item=16
 */
static	short Whole_bits[] = {
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0700,
	0x0880, 0x1040, 0x1040, 0x1040, 0x0880, 0x0700, 0x0000, 0x0000,
};
static	short Half_bits[] = {
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0700,
	0x0f80, 0x1fc0, 0x1fc0, 0x1fc0, 0x0f80, 0x0700, 0x0000, 0x0000,
};
static	short Quarter_bits[] = {
	0x0040, 0x0040, 0x0040, 0x0040, 0x0040, 0x0040, 0x0040, 0x0740,
	0x0fc0, 0x1fc0, 0x1fc0, 0x1fc0, 0x0f80, 0x0700, 0x0000, 0x0000,
};
static	short Eighth_bits[] = {
	0x0040, 0x0040, 0x0060, 0x0050, 0x0048, 0x0048, 0x0048, 0x0740,
	0x0fc0, 0x1fc0, 0x1fc0, 0x1fc0, 0x0f80, 0x0700, 0x0000, 0x0000,
};
static	short Sixteenth_bits[] = {
	0x0040, 0x0040, 0x0060, 0x0050, 0x0048, 0x0068, 0x0058, 0x0748,
	0x0fc8, 0x1fc8, 0x1fc0, 0x1fc0, 0x0f80, 0x0700, 0x0000, 0x0000,
};
static	short Thirtysecond_bits[] = {
	0x0040, 0x0060, 0x0050, 0x0048, 0x0068, 0x0058, 0x0048, 0x0768,
	0x0fd8, 0x1fc8, 0x1fc8, 0x1fc8, 0x0f80, 0x0700, 0x0000, 0x0000,
};
static	short Sixtyfourth_bits[] = {
	0x0040, 0x0060, 0x0050, 0x0068, 0x0058, 0x0068, 0x0058, 0x0768,
	0x0fd8, 0x1fc8, 0x1fc8, 0x1fc0, 0x0f80, 0x0700, 0x0000, 0x0000,
};
static	short Percussion_bits[] = {
	0x0040, 0x0040, 0x0040, 0x0040, 0x0040, 0x0040, 0x0040, 0x0040,
	0x0cC0, 0x0780, 0x0300, 0x0780, 0x0cC0, 0x0000, 0x0000, 0x0000,
};
mpr_static(Whole, 16, 16, 1, Whole_bits);
mpr_static(Half, 16, 16, 1, Half_bits);
mpr_static(Quarter, 16, 16, 1, Quarter_bits);
mpr_static(Eighth, 16, 16, 1, Eighth_bits);
mpr_static(Sixteenth, 16, 16, 1, Sixteenth_bits);
mpr_static(Thirtysecond, 16, 16, 1, Thirtysecond_bits);
mpr_static(Sixtyfourth, 16, 16, 1, Sixtyfourth_bits);
mpr_static(Percussion, 16, 16, 1, Percussion_bits);
#endif

char	*key2name();
FILE	*nopen();

main(argc, argv)
char	*argv[];
{
	char *ifile;
	extern FILE *sopen();

/****/setbuf(stderr, 0);
	ifile = (char *) 0;
	while (--argc > 0) {
	    if (argv[argc][0] == '-') {
		switch (argv[argc][1]) {
		default:
		    goto syntax;
		}
	    } else if (!ifile) {
		ifile = argv[argc];
	    } else {
syntax:
		fprintf(stderr, "Usage: %s [file]\n", argv[0]);
		fprintf(stderr, "file is an instrument definition file\n");
		exit(2);
	    }
	}
	sprintf(Tmpfile, "%d.ddm", getpid());
	if (!(Fontp = pf_open(FONTPATH)))
	    Fontp = pf_default();
	Fontwidth = Fontp->pf_defaultsize.x;
	if (fork() > 0)				/* drop in background */
	    exit(0);
	miscinits();
	if (!ifile || !readdope(ifile))		/* no file read */
	    ifile = (char *) 0;
	window_main_loop(Dframe);
/****
	unlink(Tmpfile);
****/
	exit(0);
}

miscinits()	/* one-time SunView inits */
{
	Rect tmp;
	void winevent();

	tmp.r_left = 8;
	tmp.r_top = 64;
	tmp.r_width = 1136;
	tmp.r_height = 660;
	Dframe = window_create(NULL, FRAME,
	    FRAME_LABEL,
	 "DDMT  Digital Drum & Melody by Stochastic Binary Subdivision",
	    WIN_FONT,			Fontp,
	    WIN_EVENT_PROC,		winevent,
	    FRAME_OPEN_RECT,		&tmp,
	    0);
	gencntl();
}

gencntl()	/* generate control panel; called also to re-generate */
{
	char buf[128];
	int i;
	void cntlproc(), butproc();
	Panel_setting noteproc();

	if (Controls) {
	    window_destroy(Controls);
	    Controls = 0;
	}
	Controls = window_create(Dframe, PANEL,
	    WIN_X,		0,
	    WIN_Y,		0,
	    WIN_WIDTH,		WIN_EXTEND_TO_EDGE,
	    WIN_FONT,		Fontp,
	    PANEL_SHOW_MENU,	FALSE,
	    PANEL_LABEL_BOLD,	TRUE,
	    PANEL_ITEM_X_GAP,	25,
	    0);
	sprintf(buf, "        %s      %s       %s    %s  %s   %s     %s   %s",
	 "Comment", "Note", "Dens[0:99]", "Beat", "Res", "Dur",
	 "Vel[1:127]", "Chan");
	panel_create_item(Controls, PANEL_MESSAGE,
	    PANEL_LABEL_STRING,	buf,
	    0);
	for (i = 0; i < MAXINST; i++) {
	    sprintf(buf, "%2d", i + 1);
	    Inst[i].name = panel_create_item(Controls, PANEL_TEXT,
		PANEL_ITEM_X,			4,
		PANEL_ITEM_Y,			28 + 24 * i,
		PANEL_NOTIFY_LEVEL,		PANEL_NONE,
		PANEL_LABEL_STRING,		buf,
		PANEL_VALUE_DISPLAY_LENGTH,	16,
		PANEL_VALUE_STORED_LENGTH,	16,
		PANEL_MENU_TITLE_STRING, 	"Comment",
		PANEL_SHOW_MENU,		TRUE,
		0);
	    Inst[i].note = panel_create_item(Controls, PANEL_TEXT,
		PANEL_VALUE_DISPLAY_LENGTH,	8,
		PANEL_VALUE_STORED_LENGTH,	8,
		PANEL_VALUE,			"C3",
		PANEL_MENU_TITLE_STRING, 	"Note name or number",
		PANEL_SHOW_MENU,		TRUE,
		PANEL_NOTIFY_PROC,		noteproc,
		0);
	    Inst[i].dens = panel_create_item(Controls, PANEL_SLIDER,
		PANEL_MIN_VALUE,		0,
		PANEL_VALUE,			0,
		PANEL_MAX_VALUE,		99,
		PANEL_SLIDER_WIDTH,		60,
		PANEL_SHOW_RANGE,		FALSE,
		PANEL_NOTIFY_PROC,		cntlproc,
		0);
	    Inst[i].beat = panel_create_item(Controls, PANEL_CHOICE,
		PANEL_DISPLAY_LEVEL,		PANEL_CURRENT,
		PANEL_CHOICE_STRINGS,		 "down", "up", 0,
		PANEL_FEEDBACK,			PANEL_NONE,
		PANEL_VALUE,			0,
		PANEL_MENU_TITLE_STRING, 	"Downbeat/upbeat",
		PANEL_SHOW_MENU,		TRUE,
		PANEL_NOTIFY_PROC,		cntlproc,
		0);
	    Inst[i].reso = panel_create_item(Controls, PANEL_CHOICE,
		PANEL_DISPLAY_LEVEL,		PANEL_CURRENT,
		PANEL_CHOICE_IMAGES,		&Whole,
						&Half,
						&Quarter,
						&Eighth,
						&Sixteenth,
						&Thirtysecond,
						&Sixtyfourth,
						&Percussion,
						0,
		PANEL_SHOW_MENU,		TRUE,
		PANEL_MENU_TITLE_STRING, 	"Subdivision Limit",
		PANEL_MENU_CHOICE_STRINGS,	"Whole note",
						"Half note",
						"Quarter note",
						"Eighth note",
						"Sixteenth note",
						"Thirtysecond note",
						"Sixtyfourth note",
						"No limit",
						0,
		PANEL_FEEDBACK,			PANEL_NONE,
		PANEL_VALUE,			2,
		PANEL_NOTIFY_PROC,		cntlproc,
		0);
	    Inst[i].dura = panel_create_item(Controls, PANEL_CHOICE,
		PANEL_DISPLAY_LEVEL,		PANEL_CURRENT,
		PANEL_CHOICE_IMAGES,		&Whole,
						&Half,
						&Quarter,
						&Eighth,
						&Sixteenth,
						&Thirtysecond,
						&Sixtyfourth,
						&Percussion,
						0,
		PANEL_SHOW_MENU,		TRUE,
		PANEL_MENU_TITLE_STRING, 	"Note Duration",
		PANEL_MENU_CHOICE_STRINGS,	"Whole note",
						"Half note",
						"Quarter note",
						"Eighth note",
						"Sixteenth",
						"Thirtysecond",
						"Sixtyfourth",
						"No duration",
						0,
		PANEL_FEEDBACK,			PANEL_NONE,
		PANEL_VALUE,			7,
		PANEL_NOTIFY_PROC,		cntlproc,
		0);
	    Inst[i].velo = panel_create_item(Controls, PANEL_SLIDER,
		PANEL_MIN_VALUE,		1,
		PANEL_VALUE,			64,
		PANEL_MAX_VALUE,		MAXVEL - 1,
		PANEL_SLIDER_WIDTH,		60,
		PANEL_SHOW_RANGE,		FALSE,
		PANEL_NOTIFY_PROC,		cntlproc,
		0);
	    Inst[i].chan = panel_create_item(Controls, PANEL_CHOICE,
		PANEL_DISPLAY_LEVEL,		PANEL_CURRENT,
		PANEL_CHOICE_STRINGS,		" 1", " 2", " 3", " 4",
						" 5", " 6", " 7", " 8",
						" 9", "10", "11", "12",
						"13", "14", "15", "16",
						0,
		PANEL_SHOW_MENU,		TRUE,
		PANEL_MENU_TITLE_STRING, 	"MIDI Channel",
		PANEL_MENU_CHOICE_STRINGS,	" 1 CZ-101",
						" 2 D-110 Part 1",
						" 3 D-110 Part 2",
						" 4 D-110 Part 3",
						" 5 D-110 Part 4",
						" 6 D-110 Part 5",
						" 7 D-110 Part 6",
						" 8 D-110 Part 7",
						" 9 D-110 Part 8",
						"10 D-110 Rhythm",
						"11 ???",
						"12 ???",
						"13 ???",
						"14 ???",
						"15 ???",
						"16 SPM 8:2",
						0,
		PANEL_FEEDBACK,			PANEL_NONE,
		PANEL_VALUE,			9,
		PANEL_NOTIFY_PROC,		cntlproc,
		0);
	}
	Scal = panel_create_item(Controls, PANEL_TOGGLE,
	    PANEL_ITEM_X,		4,
	    PANEL_ITEM_Y,		28 + 24 * MAXINST,
	    PANEL_LABEL_STRING,		"SCALE",
	    PANEL_CHOICE_STRINGS,	"C", "C#", "D", "D#", "E", "F",
					"F#", "G", "G#", "A", "A#", "B",
					0,
	    PANEL_VALUE,		0x0AB5,		/* C major scale */
	    PANEL_FEEDBACK,		PANEL_INVERTED,
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_MENU_TITLE_STRING,	"Pitches",
	    PANEL_NOTIFY_PROC,		cntlproc,
	    0);
	Lolm = panel_create_item(Controls, PANEL_TEXT,
	    PANEL_VALUE_DISPLAY_LENGTH,	5,
	    PANEL_VALUE_STORED_LENGTH,	5,
	    PANEL_VALUE,		"C-2",
	    PANEL_LABEL_STRING,		"LO:",
	    PANEL_MENU_TITLE_STRING, 	"Relative Motion Low Limit",
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_NOTIFY_PROC,		noteproc,
	    0);
	Hilm = panel_create_item(Controls, PANEL_TEXT,
	    PANEL_VALUE_DISPLAY_LENGTH,	5,
	    PANEL_VALUE_STORED_LENGTH,	5,
	    PANEL_VALUE,		"G8",
	    PANEL_LABEL_STRING,		"HI:",
	    PANEL_MENU_TITLE_STRING, 	"Relative Motion High Limit",
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_NOTIFY_PROC,		noteproc,
	    0);
	Seed = panel_create_item(Controls, PANEL_TEXT,
	    PANEL_VALUE_DISPLAY_LENGTH,	12,
	    PANEL_VALUE_STORED_LENGTH,	12,
	    PANEL_VALUE,		"0",
	    PANEL_LABEL_STRING,		"SEED:",
	    PANEL_MENU_TITLE_STRING, 	"Random Number Seed",
	    PANEL_SHOW_MENU,		TRUE,
	    0);
	Quit = panel_create_item(Controls, PANEL_BUTTON,
	    PANEL_ITEM_X,		4,
	    PANEL_ITEM_Y,		52 + 24 * MAXINST,
	    PANEL_NOTIFY_LEVEL,		PANEL_NONE,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "QUIT", 0, 0),
	    PANEL_MENU_TITLE_STRING, 	"Exit From DDMT",
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_NOTIFY_PROC,		butproc,
	    0);
	Rand = panel_create_item(Controls, PANEL_BUTTON,
	    PANEL_NOTIFY_LEVEL,		PANEL_NONE,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "RAND", 0, 0),
	    PANEL_MENU_TITLE_STRING, 	"Pick Random Parameters",
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_NOTIFY_PROC,		butproc,
	    0);
	Nice = panel_create_item(Controls, PANEL_BUTTON,
	    PANEL_NOTIFY_LEVEL,		PANEL_NONE,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "NICE", 0, 0),
	    PANEL_MENU_TITLE_STRING, 	"Improve Parameters",
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_NOTIFY_PROC,		butproc,
	    0);
	Play = panel_create_item(Controls, PANEL_BUTTON,
	    PANEL_NOTIFY_LEVEL,		PANEL_NONE,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "PLAY", 0, 0),
	    PANEL_MENU_TITLE_STRING, 	"Generate & play",
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_NOTIFY_PROC,		butproc,
	    0);
	Test = panel_create_item(Controls, PANEL_BUTTON,
	    PANEL_NOTIFY_LEVEL,		PANEL_NONE,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "TEST", 0, 0),
	    PANEL_MENU_TITLE_STRING, 	"Generate & display",
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_NOTIFY_PROC,		butproc,
	    0);
	Read = panel_create_item(Controls, PANEL_BUTTON,
	    PANEL_NOTIFY_LEVEL,		PANEL_NONE,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "READ", 0, 0),
	    PANEL_MENU_TITLE_STRING, 	"Read From File",
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_NOTIFY_PROC,		butproc,
	    0);
	Writ = panel_create_item(Controls, PANEL_BUTTON,
	    PANEL_NOTIFY_LEVEL,		PANEL_NONE,
	    PANEL_LABEL_IMAGE,		panel_button_image(Controls,
					 "WRITE", 0, 0),
	    PANEL_MENU_TITLE_STRING, 	"Write to File",
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_NOTIFY_PROC,		butproc,
	    0);
	File = panel_create_item(Controls, PANEL_TEXT,
	    PANEL_NOTIFY_LEVEL,		PANEL_NONE,
	    PANEL_LABEL_STRING,		"File: ",
	    PANEL_VALUE_DISPLAY_LENGTH,	16,
	    PANEL_VALUE_STORED_LENGTH,	64,
	    PANEL_MENU_TITLE_STRING, 	"File for Read & Write",
	    PANEL_SHOW_MENU,		TRUE,
	    0);
	Tmpo = panel_create_item(Controls, PANEL_SLIDER,
	    PANEL_ITEM_X,		4,
	    PANEL_ITEM_Y,		76 + 24 * MAXINST,
	    PANEL_LABEL_STRING,		"TEMPO",
	    PANEL_SLIDER_WIDTH,		60,
	    PANEL_MIN_VALUE,		40,
	    PANEL_VALUE,		100,
	    PANEL_MAX_VALUE,		250,
	    PANEL_SHOW_RANGE,		TRUE,
	    PANEL_NOTIFY_PROC,		cntlproc,
	    0);
	Bars = panel_create_item(Controls, PANEL_SLIDER,
	    PANEL_LABEL_STRING,		"BARS",
	    PANEL_SLIDER_WIDTH,		60,
	    PANEL_MIN_VALUE,		1,
	    PANEL_VALUE,		1,
	    PANEL_MAX_VALUE,		32,
	    PANEL_SHOW_RANGE,		TRUE,
	    PANEL_NOTIFY_PROC,		cntlproc,
	    0);
	Rpts = panel_create_item(Controls, PANEL_SLIDER,
	    PANEL_LABEL_STRING,		"RPT LIM",
	    PANEL_SLIDER_WIDTH,		60,
	    PANEL_MIN_VALUE,		1,
	    PANEL_VALUE,		2,
	    PANEL_MAX_VALUE,		8,
	    PANEL_SHOW_RANGE,		TRUE,
	    PANEL_NOTIFY_PROC,		cntlproc,
	    0);
	window_fit_width(Controls);
	window_fit_height(Controls);
	window_fit(Dframe);
}

Panel_setting
noteproc(item, event)
Panel_item item;
Event	*event;
{
	setnv(item, getnv(item));
	return((Panel_setting) panel_text_notify(item, event));
}

setnv(item, v)
Panel_item item;
{
	char *cp, buf[8];
	int i;

	for (i = MAXINST; --i >= 0 && item != Inst[i].note; );
	if (i >= 0) {
	    if (v <= 12) {
		if (v > 0)
		    sprintf(cp = buf, "+%d (%02x)", v, v);
		else
		    sprintf(cp = buf, "%d (%02x)", v, v & 0xFF);
	    } else
		sprintf(cp = buf, "%4s(%02x)", key2name(v), v);
	} else
	    cp = key2name(v);
	PSV(item, cp);
}

getnv(item)
Panel_item item;
{
	register char *cp;
	register int v;

	cp = (char *) PGV(item);
	if (*cp == '-')
	    v = myatoi(cp);
	else if (*cp == '+')
	    v = name2key(&cp[1]);
	else
	    v = name2key(cp);
	if (v < -12)
	    v = -12;
	else if (v >= MAXKEY)
	    v = MAXKEY - 1;
	return(v);
}

void
butproc(item, event)
Panel_item item;
Event	*event;
{
	register int i;
	char buf[128];

	if (item == Quit) {
	    unlink(Tmpfile);
	    exit(0);
	} else if (item == Rand) {
	    int b, j, minres;

fprintf(stderr, "You asked for it...\n");
	    minres = 0;
	    for (i = 0; i < MAXINST; i++) {
		for (j = 0; -1 < j && j < LKRND; j =  (rand() % HKRND) - 3);
		setnv(Inst[i].note, j);
		if (j < LKDRM || HKDRM < j)
		    j = PMCHAN;			/* pick a melody channel */
		else if (j < LKMEL || HKMEL < j)
		    j = PRCHAN;			/* pick a rhythm channel */
		else if (rand() & 010)
		    j = PMCHAN;			/* pick a melody channel */
		else
		    j = PRCHAN;			/* pick a rhythm channel */
		PSV(Inst[i].chan, j);
		PSV(Inst[i].dens, 20 + rand() % 59);
		PSV(Inst[i].beat, b = ((rand() / 3) & rand() & 1));
		j = minres + (rand() % 3);
		minres += (j > minres && minres < 3 && b == DOWN)? 1 : 0;
		PSV(Inst[i].reso, j);
		PSV(Inst[i].dura, j + 1);
		PSV(Inst[i].velo, 32 + (rand() % 91));
	    }
	} else if (item == Nice) {
	    nicify();
	} else if (item == Play) {
	    if (writedope(Tmpfile))
		return;
	    system(SYNTHINIT);
	    if (i = atoi(PGV(Seed)))
		sprintf(buf, "ddm -b%d -r%d -s%d %s | play -t%d -w1",
		 PGV(Bars), PGV(Rpts), i,
		 Tmpfile, PGV(Tmpo));
	    else
		sprintf(buf, "ddm -b%d -r%d %s | play -t%d -w1",
		 PGV(Bars), PGV(Rpts),
		 Tmpfile, PGV(Tmpo));
	    if (system(buf))
		perror(buf);
	} else if (item == Test) {
	    if (writedope(Tmpfile))
		return;
	    if (i = atoi(PGV(Seed)))
		sprintf(buf, "ddm -b%d -debug -r%d -s%d %s",
		 PGV(Bars), PGV(Rpts), i, Tmpfile);
	    else
		sprintf(buf, "ddm -b%d -debug -r%d %s",
		 PGV(Bars), PGV(Rpts), Tmpfile);
	    if (system(buf))
		perror(buf);
	} else if (item == Read) {
	    readdope(PGV(File));
	} else if (item == Writ) {
	    if (writedope(PGV(File)))
		return;
	}
}

#define	IMPROVE(ERROR,ARRAY,TEST,BEST)		\
{	int E, Emin, I, Ei;			\
	E = (ERROR);				\
	Emin = E;				\
	(BEST) = -1;				\
	for (I = 0; I < MAXINST; I++) {		\
	    if (Probs[I] && (ARRAY[I] TEST)) {	\
		Ei = Ic[I] - E;			\
		Ei = Ei<0? -Ei : Ei;		\
		if (Ei < Emin) {		\
		    Emin = Ei;			\
		    (BEST) = I;			\
		}				\
	    }					\
	}					\
}

/* several things need access to this stuff created by eval() */
int Ic[MAXINST], Chans[MAXINST], Notes[MAXINST], Probs[MAXINST];
int Rc, Mc;

nicify()
{
	int i, j, e, r, w, b, wv, bv, scale[12];
	int lastop, lasti;

	/* help instruments that can't be heard */
	lastop = lasti = 0;
	e = eval(1);
	for (j = 0; j < 10 && e >= 10; j++) {
	    w = -1;		/* help instruments that can't be heard */
	    for (i = 1; i < MAXINST; i++)	/* find least likely inst */
		if (Probs[i] && (w == -1 || Ic[i] < Ic[w]))
		    w = i;
	    if (w == -1)
		return;				/* none turned on */
	    if (Ic[w] >= 5)
		break;
	    b = -1;				/* inst w can't be heard */
	    for (i = w; --i >= 0; )		/* find the best heard one */
		if (Ic[i] > Ic[b])		/* (ahead of it) */
		    b = i;
	    if (b == -1) {			/* none? */
		if ((wv = GIV(w, dens)) < 80 && (lastop != 'd' || lasti != w))
		    e = nicedens(lasti = w, wv + 5, lastop = 'D');
		else
		    break;
		continue;
	    }
	    if ((bv = GIV(b, dens)) > (wv = GIV(w, dens))) {
		if (wv < 50 && (lastop != 'd' || lasti != w)) {
		    e = nicedens(lasti = w, wv + 5, lastop = 'D');
		    continue;
		} else if (lastop != 'D' || lasti != b) {
		    e = nicedens(lasti = b, bv - 5, lastop = 'd');
		    continue;
		}
	    }
	    if ((bv = GIV(b, reso)) > (wv = GIV(w, reso))) {
		if (wv < (i * 2 + 2) / 5 && (lastop != 'r' || lasti != w)) {
		    e = nicereso(lasti = w, wv + 1, lastop = 'R');
		    continue;
		} else if (lastop != 'R' || lasti != b) {
		    e = nicereso(lasti = b, bv - 1, lastop = 'r');
		    continue;
		}
	    }
	    if ((bv = GIV(b, beat)) == (wv = GIV(w, beat))) {
		if (wv == DOWN && (lastop != 'b' || lasti != w)) {
		    e = nicebeat(lasti = w, UP, lastop = 'B', w);
		    continue;
		} else if (lastop != 'B' || lasti != b) {
		    e = nicebeat(lasti = b, DOWN, lastop = 'b', w);
		    continue;
		}
	    }
	    if ((wv = GIV(w, reso)) == 0 && (lastop != 'r' || lasti != w)) {
		e = nicereso(lasti = w, wv + 1, lastop = 'R');
		continue;
	    }
	    fprintf(stderr, "Stuck helping %d be heard\n", w+1);
	    break;
	}
	for (j = 0; j < 10 && e > 2; j++) {	/* balance rhythm & melody */
	    if (Rc >= 2 * Mc) {			/* too much percussion */
		IMPROVE(Rc-Mc,Chans,==10,b)	/* find closest with chan==10 */
		if (b >= 0) {			/* found one that helps */
/****/fprintf(stderr, "switch %d to melody\n", b+1);
		    SIV(b, chan, PMCHAN);	/* make it melody */
		    e = eval(0);
		}
	    } else if (Mc >= 2 * Rc) {		/* too many notes */
		IMPROVE(Mc-Rc,Chans,!=10,b)	/* find best with chan!=10 */
		if (b >= 0) {			/* found one that helps */
/****/fprintf(stderr, "switch %d to rhythm\n", b+1);
		    SIV(b, chan, PRCHAN);	/* make it rhythm */
		    e = eval(0);
		}
	    }
	}
	r = j = 0;				/* balance relative & jumps */
	for (i = 1; i < MAXINST; i++) {
	    if (Probs[i] && (bv = getnv(Inst[i].note)) <= 12)
		r += Ic[i];
	    else
		j += Ic[i];
	}
	if (2 * r > 3 * j) {			/* too much relative */
	    IMPROVE(r-j,Notes,<=12,b)		/* find closest with note<=12 */
	    if (b >= 0)				/* found one that helps */
/****/{fprintf(stderr, "switch %d to absolute\n", b+1);
		setnv(Inst[b].note, LKRND + (rand() % (HKRND - LKRND)));
/****/}
	} else if (2 * j > 3 * r) {		/* too much absolute */
	    IMPROVE(j-r,Notes,>12,b)		/* find closest with note>12 */
	    if (b >= 0)				/* found one that helps */
/****/{fprintf(stderr, "switch %d to relative\n", b+1);
		setnv(Inst[b].note, (rand() % 7) - 3);
/****/}
	}
	b = (int) PGV(Scal);			/* inspect scale spec */
	r = 0;					/* r = # notes in scale */
	for (i = 12; --i >= 0; ) {
	    if (b & (1 << i)) {
		scale[i] = -100;
		r++;
	    } else
		scale[i] = 0;
	}
	for (i = MAXINST; --i >= 0; )
	    if (Probs[i] && Notes[i] > 12)
		scale[Notes[i] % 12]++;
	for (i = 12; --i >= 0; ) {	/* check for inst note not in scale */
	    if (scale[i] > 0) {
		if (r < 9) {
/****/fprintf(stderr, "add %d to scale (for some abs inst)\n", i);
		    b |= (1 << i);
		    scale[i] -= 100;
		    r++;
		} else {
		    w = 0;
		    for (j = 12; --j > 0; )
			if (scale[i] < scale[w])
			    w = i;
		    if (scale[w] + 100 < scale[i]) {
/****/fprintf(stderr, "add %d & del %d from scale (for some abs inst)\n", i, w);
			b = ((1 << i) | b & ~(1 << w));
			scale[i] -= 100;
			scale[w] += 100;
		    }
		}
	    }
	}
	for (i = 12; r >= 2 && --i >= 0; ) {	/* check unneeded scale notes */
	    if (scale[i] == -100) {
/****/fprintf(stderr, "del %d from scale (not needed)\n", i);
		b &= ~(1 << i);
		--r;
		scale[i] += 100;
	    }
	}
	while (r < 7) {			/* fill out the scale */
	    wv = 0;
	    for (i = 0; i < 12; i++) {
		if (scale[i] < 0) {
		    for (j = i + 1; scale[j % 12] >= 0; j++);
		    if (j - i > wv) {
			wv = j - i;
			w = i;
		    }
		}
	    }
	    if (wv == 12)
		w += 7;
	    else if (wv == 11 || wv == 7 || wv == 6)
		w += 4;
	    else if (wv == 10 || wv == 8)
		w += 3;
	    else if (wv == 9)
		w += 5;
	    else if (wv == 5 || wv == 4 || wv == 3)
		w += 2;
	    else if (wv == 2)	/* whole tone scale! */
		w += 1;
	    w %= 12;
/****/fprintf(stderr, "add %d to scale (interval = %d)\n", w, wv);
	    b |= (1 << w);
	    scale[w] -= 100;
	    r++;
	}
	PSV(Scal, b);
}

nicedens(i, v, op)
{
/****/fprintf(stderr, "%2d: %s density to %d%%\n",
/****/ i+1, op=='D'? "raise" : "lower", v);
	SIV(i, dens, v);
	return(eval(0));
}

nicereso(i, v, op)
{
/****/fprintf(stderr, "%2d: %s resolution to %d\n",
/****/ i+1, op=='R'? "raise" : "lower", v);
	SIV(i, reso, v);
	return(eval(0));
}

nicebeat(i, v, op, j)
{
/****/fprintf(stderr, "%2d: change beat to %s (for %d)\n",
/****/ i+1, v? "UP" : "DOWN", j+1);
	SIV(i, beat, v);
	return(eval(0));
}

eval(verbose)
{
	int i, dens, reso, e, b, r, t, q, n, rc[MAXRESO];

	for (i = MAXRESO; --i >= 0; rc[i] = 100);
	e = 0;
	for (i = 0; i < MAXINST; i++) {
	    Probs[i] = dens = (int) PGV(Inst[i].dens);
	    if (!dens)
		continue;
	    Chans[i] = (int) GIV(i, chan);
	    Notes[i] = getnv(Inst[i].note);
	    Ic[i] = 0;
	    reso = (int) PGV(Inst[i].reso);
	    if (reso >= MAXRESO - 1)
		reso = MAXRESO - 1;
	    b = (int) PGV(Inst[i].beat);
	    for (r = 0, t = 100; r <= reso; r++) {
		q = (t * rc[r + b]) / 100;	/* chance @ this res */
		Ic[i] += q;
		n = r? (1 << (r - 1)) : 1;	/* # locs @ this res */
		rc[r + b] -= q / n;		/* chance left @ this res */
		t = (t * dens) / 100;	/* chance of trying next res */
	    }
	}
	for (i = Rc = Mc = 0; i < MAXINST; i++) {
	    if (!Probs[i])
		continue;
	    if (ISRCHAN(Chans[i]))
		Rc += Ic[i];			/* rhythm entry */
	    else
		Mc += Ic[i];			/* melody entry */
	    if (Ic[i] < 5) {
		e += 10;
		if (verbose)
		    printf("Only %d%% chance to hear inst %d\n", Ic[i], i+1);
	    }
	}
	if (Rc >= 2 * Mc) {		/* too much percussion */
	    e++;
	    if (verbose)
		printf("Rhythm heavy;  drum = %d, pitch = %d\n", Rc, Mc);
	} else if (Mc >= 2 * Rc) {	/* too many notes */
	    e++;
	    if (verbose)
		printf("Note heavy;  pitch = %d, drum = %d\n", Mc, Rc);
	}
/****/fprintf(stderr, "eval(), Rc = %d, Mc = %d, return(%d)\n", Rc, Mc, e);
	return(e);
}

writedope(file)
char	*file;
{
	int i, j, k;
	FILE *ofp;

	if (!file || !*file) {
	    fprintf(stderr, "No filename specified - Aborting\n");
	    return(1);
	}
	if (!(ofp = fopen(file, "w"))) {
	    perror(file);
	    return(1);
	}
	j = (int) PGV(Scal);
	fprintf(ofp, "Scale\t");
	for (i = k = 0; i < 12; i++)
	    if (j & (1 << i))
		fprintf(ofp, "%s%d", k++? "," : "", i);
	fprintf(ofp, "\n");
	fprintf(ofp, "Limit\t%d,%d\n", getnv(Lolm), getnv(Hilm));
	for (i = 0; i < MAXINST; i++) {
	    if ((int) PGV(Inst[i].dens) > 0)
		fprintf(ofp, "%d:%d:%c:%d:%d:%d:%d\t%s\n",
		 getnv(Inst[i].note),
		 PGV(Inst[i].dens),
		 PGV(Inst[i].beat)? 'U' : 'D',
		 64 / Durs[(int) PGV(Inst[i].reso)],
		 Durs[(int) PGV(Inst[i].dura)],
		 PGV(Inst[i].velo),
		 PGV(Inst[i].chan) + 1,
		 PGV(Inst[i].name));
	}
	fclose(ofp);
	return(0);
}

void
cntlproc(item, value, event)
Panel_item item;
Event	*event;
{
}

readdope(file)	/* read a .ddm file and set values (if Controls != 0) */
char	*file;	/* return zero for success */
{
	char *cp;
	int i, j;
	SBSDP sbsdp;
	INST in[MAXINST];
	extern char *sbsdinit();

	if (!file || !*file) {
	    fprintf(stderr, "No filename specified - Aborting\n");
	    return(1);
	}
	sbsdp.inst = in;
	if (cp = sbsdinit(file, &sbsdp, MAXINST)) {
	    fprintf(stderr, "%s", cp);
	    return(1);
	}
	for (i = 12, j = 0; --i >= 0; )
	    if (!sbsdp.scaled || sbsdp.scale[i])
		j |= (1 << i);
	PSV(Scal, j);
	for (i = MAXINST; --i >= 0; ) {
	    PSV(Inst[i].name, "");
	    setnv(Inst[i].note, in[i].inum);
	    PSV(Inst[i].dens, in[i].inum);
	    PSV(Inst[i].beat, in[i].density);
	    PSV(Inst[i].reso, tv2i(64 / in[i].res));
	    PSV(Inst[i].dura, tv2i(in[i].dur));
	    PSV(Inst[i].velo, in[i].vel);
	    PSV(Inst[i].chan, in[i].chan);
	}
	PSV(File, file);
	return(0);
}

tv2i(tv)		/* convert time value (in 64ths) to choice index */
{
	register int i, q;

	if ((q = 14 * tv) == 0)
	    return(8);
	for (i = 8; --i > 0 && q >= 10; q >>= 1);
	return(i);
}

/*VARARGS3*/
forkexec(in, out, p, a1, a2, a3, a4)
char	*p;
{
	register int pid;

	switch (pid = fork()) {
	case -1:
	    perror("fork()");
	    break;
	case 0:
	    close(0);
	    if (in >= 0)
		dup2(0, in);
	    close(1);
	    if (out >= 0)
		dup2(1, out);
	    execlp(p, p, a1, a2, a3, a4);
	    perror(p);
	    exit(1);
	}
	return(pid);
}

void
winevent(window, event, arg)
Window	window;
Event	*event;
caddr_t	arg;
{
	if (event_id(event) == WIN_RESIZE)
	    gencntl();
	window_default_event_proc(window, event, arg);
}

FILE	*
nopen(file, mode)			/* noisy open */
char	*file, *mode;
{
	FILE *fp;

	if ((fp = fopen(file, mode)) == (FILE *) NULL)
	    perror(file);
	return(fp);
}
