/* Implementation of Box class
 *
 * Copyright (C)  1993  The Board of Trustees of  
 * The Leland Stanford Junior University.  All Rights Reserved.
 *
 * Authors: Scott Francis, Paul Kunz, Imran Qureshi, and Libing Wang
 *
 * This file is part of an Objective-C class library for a window system
 *
 * Box.m,v 1.29 1994/06/13 23:03:30 pfkeb Exp
 */

#include "Box.h"

#include <coll/List.h>
#include <stdio.h>
#include "TextFieldCell.h"
#include "graphics.h"
#include "stdmacros.h"
#include <objc2/typedstream2.h>

extern char * BoxInstanceName(void);

@interface Box(WidgetSet)
- _init;
- _setBorderType:(int)borderType;
- _managedBy:parent wid:(void *)widget;
@end

@implementation Box

- initFrame:(const NXRect *)frameRect
{
    View               *aView;
    NXRect              rect;

    offsets.width = DEF_OFFSET;
    offsets.height = DEF_OFFSET;
    
    [super initFrame:frameRect];
    instancename = BoxInstanceName();
    /* inherit class from VIEW.  it should be xmBulletinBoardWidgetClass */

    if (!contentView) {
	borderRect = *frameRect;
	rect.origin.x = DEF_OFFSET;
	rect.origin.y = DEF_OFFSET;
	rect.size.width = borderRect.size.width - DEF_OFFSET*2;
	rect.size.height = borderRect.size.height - DEF_OFFSET*2;
	aView = [[View alloc] initFrame:&rect];
	[self setContentView:aView];
    }
    return self;
}

- contentView
{
    return contentView;
}

// For the case of no border, simply set the shadowThickness to 0.  Otherwise
// leave it at its default value, which is 1 or 2.  Keep in mind in the
// future if the border type changes, the shadow-thickness might also have
// to be reset to the default!

- setBorderType:(int)borderType
{
    [self _setBorderType:borderType];
    return self;
}

- (int)borderType
{
	return border;
}

- cell
{
	return cell;
}

- setOffsets:(NXCoord)horizontal :(NXCoord)vertical
{
	offsets.width = horizontal;
	offsets.height = vertical;
	[borderView _setOffsets:&offsets];
	[self sizeToFit];
	return self;
}

- getOffsets:(NXSize *)theSize
{
	theSize->width = offsets.width;
	theSize->height = offsets.height;
	return self;
}

- sizeTo:(NXCoord)width :(NXCoord)height
{
	NXSize newSize;		// for the contentview
	NXCoord dw, dh;

	dw = frame.size.width - width;
	dh = frame.size.height - height;
	[super sizeTo:width :height];

	// now adjust the borderRect and the box widget...

	borderRect.size.width -= dw;
	borderRect.size.height -= dh;
	[borderView _setSize:&(borderRect.size)];

	// now resize the contentview.

	newSize.width = borderRect.size.width - offsets.width*2;
	newSize.height = borderRect.size.height - offsets.height*2;
	[contentView sizeTo:newSize.width :newSize.height];

	// move the cell to new position according to titlePosition

	if (titlePosition > NX_BELOWTOP) {
	[cell _moveTo:(titleRect.origin.x-(dw/2)) :(titleRect.origin.y-dh)];
	} else if (titlePosition) {
	    [cell _moveTo:(titleRect.origin.x - (dw/2)) :titleRect.origin.y];
	} 
	return self; 
}

- sizeToFit
/* This method tries to do a lot of work...
  	Designed to wrap the box around the contentview and title box.
	If either has become too large, we must resize to be larger.
 */
{
	NXRect cview, tbox;
	
	if ([contentView respondsTo:@selector(sizeToFit)]) {
		[(Box *)contentView sizeToFit];
	}
	// should resize contentview if it can't do itself, but I'm
	// not sure how to proceed for this...
	[contentView getFrame:&cview];
	[cell getFrame:&tbox];
	
	if ((borderRect.size.width >= cview.size.width + 2*offsets.width) ||
	    (borderRect.size.height >= cview.size.height + 2*offsets.height) ||
	    (borderRect.size.width >= tbox.size.width) ||
	    (NX_MAXY(&frame) >= NX_MAXY(&tbox)))
	{
		return self;
	}
	
	borderRect.origin.x = cview.origin.x - offsets.width;
	borderRect.origin.y = cview.origin.y - offsets.height;
	borderRect.size.width = MAX((NX_MAXX(&cview) + offsets.width),
				    (tbox.size.width));
	borderRect.size.height = (NX_MAXY(&cview) + offsets.height);
	frame.origin.x = borderRect.origin.x;
	frame.size.width = borderRect.size.width;

	switch (titlePosition) {
	case (NX_NOTITLE):
	case (NX_BELOWTOP):
	    frame.origin.y = titleRect.origin.y;
	    frame.size.height = borderRect.size.height;
	    break;
	case (NX_ABOVETOP):
	    frame.origin.y = tbox.origin.y;
	    frame.size.height = borderRect.size.height + tbox.size.height;
	    break;
	case (NX_ATTOP):
	    frame.origin.y = tbox.origin.y;
	    frame.size.height = borderRect.size.height + tbox.size.height/2;
	    break;
	case (NX_ABOVEBOTTOM):
	    frame.origin.y = borderRect.origin.y;
	    frame.size.height = borderRect.size.height;
	    break;
	case (NX_ATBOTTOM):
	    frame.origin.y = borderRect.origin.y;
	    frame.size.height = borderRect.size.height + tbox.size.height/2;
	    break;
	case (NX_BELOWBOTTOM):
	    frame.origin.y = borderRect.origin.y;
	    frame.size.height = borderRect.size.height+tbox.size.height;
	    break;
	default:
	    fprintf(stderr, "Error:  erroneous title position\n");
	    break;
	}
/*	frame.size.height = (NX_MAXY(&frame) > NX_MAXY(&tbox))  
			    ? frame.size.height 
			    : NX_MAXY(&tbox) - frame.origin.y;
 */
	[self _setBorder:&borderRect];
	return self;
}

- setTitlePosition:(int)pos
{
	NXRect bbox;
	NXRect temp;
	NXCoord x, y;
	
	[[self cell] getFrame:&bbox];
	x = frame.origin.x + (frame.size.width / 2) - (bbox.size.width / 2);
	// x value is the same for all:  centered horizontally.
	
	switch (pos) {
	case (NX_NOTITLE):
	    break;
	case (NX_ABOVETOP):
		y = frame.origin.y;  // at very top
		[cell moveTo:x :y];
		borderRect.size.height = frame.size.height - bbox.size.height;
		borderRect.origin.y = frame.origin.y + bbox.size.height;
		break;
	case (NX_ATTOP):
		y = frame.origin.y;
		[cell moveTo:x :y];
		borderRect.size.height =frame.size.height-(bbox.size.height/2);
		borderRect.origin.y = frame.origin.y + (bbox.size.height/2);
		break;
	case (NX_BELOWTOP):
		y = frame.origin.y;
		[cell moveTo:x :y];
		borderRect.size.height = frame.size.height;
		borderRect.origin.y = frame.origin.y;
		break;
	case (NX_ABOVEBOTTOM):
		y = frame.size.height - bbox.size.height;
		[cell moveTo:x :y];
		borderRect.origin.y = frame.origin.y;
		borderRect.size.height = frame.size.height;
		break;
	case (NX_ATBOTTOM):
		y = frame.size.height - bbox.size.height;
		[cell moveTo:x :y];
		borderRect.origin.y = frame.origin.y;
		borderRect.size.height = frame.size.height-bbox.size.height/2;
		break;
	case (NX_BELOWBOTTOM):
		y = frame.size.height - bbox.size.height;
		[cell moveTo:x :y];
		borderRect.origin.y = frame.origin.y;
		borderRect.size.height = frame.size.height - bbox.size.height;
		break;
	default:
		return self;
	}
	titlePosition = pos;
	temp.size.width = borderRect.size.width - offsets.width*2;
	temp.size.height = borderRect.size.height - offsets.height*2;
	[contentView sizeTo:temp.size.width :temp.size.height];
	[self _setBorder:&borderRect];
	return self;
}

- (int)titlePosition
{
	return titlePosition;
}

- setTitle:(const char *)aString
{
	if (!aString) return self;
	[cell setTitle:aString];	// sets the title
	
	[self sizeToFit];		// size the box to fit this title
					// and size its subviews, etc
	return self;
}

- (char *)title
/*	Ask the cell for its title.  If no title, then return the default
	title of a box, as stated in the NeXT Reference guide under Box
 */
{
	char *cellTitle;
	
	cellTitle = [cell title];
	return ((cellTitle) ? cellTitle : "Title") ;
}

// -------------------------------------------------------------- //

/* Not Yet Implemented */

/* should respond to this message, but do nothing so far... */
/* These methods could be implemented for setting the font in X-fonts,
   but we need to decide on policy for storing x-fonts in objects, and
   how the font object should interact... in otherwords, write the
   font object before implementing these, maybe.
 */
- font
{
	return nil;
}

- setFont:fontObj
{
    return self;
}

/* obscure method someone could implement-- places box so its content
   view lies on contentFrame, reckoned in box's superview's coord system.
 */
- setFrameFromContentFrame:(const NXRect *)contentFrame
{
	return self;
}

- drawSelf:(const NXRect *)rects :(int)rectCount
/* well, one wonders what this one should do in x, if anything.
   but it needs to exist for compatibility's sake
 */
{
	return self;
}

- (int)heightAdjust
{
    int dh;
    
    switch (titlePosition) {
    case (NX_ABOVETOP):
    case (NX_BELOWBOTTOM):
    	dh = titleRect.size.height;
	break;
    case (NX_ATTOP):
    case (NX_ATBOTTOM):
    	dh = titleRect.size.height/2;
	break;
    default:
    	dh = 0;
	break;
    } 
    return dh;
}

- setContentView:aView
{
    View	*oldContentView = nil;
    int dh;
    
/*	Well, this should really resize the view before adding it to the
	box, in order to stay with NeXT compliance.  For now scrap that
	until done debugging box.
 
 	Okay, here is the resizing!
 */
    dh = [self heightAdjust];
    [aView sizeTo:(borderRect.size.width - offsets.width*2) 
    		 :(borderRect.size.height) - offsets.height*2 - dh];
    if ( contentView ) {
	oldContentView = [super replaceSubview:contentView with:aView];
	contentView = aView;
    } else {
	oldContentView = nil;
        [self addSubview:aView];
    }
    // we need something to redraw the text label of the box here //
    return oldContentView;
}

- replaceSubview:oldView with:newView
{
    return [contentView replaceSubview:oldView with:newView];
}

- addSubview:aView
{
    [contentView addSubview:aView];
    [self sizeToFit];
    return self;
}

- read:(TypedStream *)ts
{    
    [super read:ts];

    objc_read_string(ts, &title);
    
    objc_read_type(ts, "i", &border);
    objc_read_type(ts, "i", &titlePosition);
    objc_read_type(ts, "i", &(offsets.width));
    objc_read_type(ts, "i", &(offsets.height));
    objc_read_rect(ts, &borderRect);
    objc_read_rect(ts, &titleRect);

    contentView = [subviews objectAt:0];
    return self;
}

/* Sometimes we need to change the border.  This is where you do it.
	Notice that since it does motif work it starts with the underscore
	in the method name.  Don't call sizeToFit:: because this should only
	be called internally, when we should know whether or not we want to
	sizeToFit:: or not.  Notice we use the borderRect to set the bounds
	of the widget, though the logical space of the box is as large as its
	frame Rect (and its physical size is as large as title widget + 
	bulletin board widget)

   Note that this changes the _boxwid widget, because we want its border
   to be thus set, not the border of the outer widget, or container widget.
 */
- _setBorder:(NXRect *)aBorder
{
    [borderView _setBorderRect:aBorder];
//    [self sizeToFit];
    return self;
}

// do we need to implement awake here?  Currently just uses View's awake

- awake
{
/* lets instantiate the appropriate title cell widget, but let the title cell
   do the work */
   
    [super awake];
    if (!cell && (titlePosition != NX_NOTITLE)) {
    	cell = [[TextFieldCell alloc] init];
    	[cell _setFrame:&titleRect];
	[cell setStringValue:title];
    }
    return self;
}

- (void *)_widget 
{
    return [borderView _widget];
}

@end

