/* Implementation of Objective-C "collection of delegates" object
   Copyright (C) 1993 Free Software Foundation, Inc.
   
   Written by:  R. Andrew McCallum <mccallum@cs.rochester.edu>
   Dept. of Computer Science, U. of Rochester, Rochester, NY  14627
   
   This file is part of the GNU Objective-C Collection library.
   
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   */ 

#include <coll/DelegateList.h>

@implementation DelegateList

+ initialize
{
  return self;
}

+ alloc
{
  return (id)class_create_instance(self);
}

+ new
{
  return [[self alloc] init];
}

- init
{
  _list = [[Array alloc] init];
  _send_behavior = SEND_TO_ALL;
  return self;
}

/* Archiving must mimic the above designated initializer */

- write: (TypedStream*)aStream
{
  objc_write_type(aStream, @encode(unsigned char), &_send_behavior);
  objc_write_object(aStream, _list);
  return self;
}

- read: (TypedStream*)aStream
{
  objc_write_type(aStream, @encode(unsigned char), &_send_behavior);
  objc_read_object(aStream, &_list);
  return self;
}

  
- free
{
  [_list free];
  return (id)object_dispose(self);
}


// MANIPULATING LIST OF DELEGATES;

- delegateListAddObject: anObject
{
  [_list addObject: anObject];
  return self;
}

- delegateListAddObjectIfAbsent: anObject
{
  [_list addObjectIfAbsent: anObject];
  return self;
}

- delegateListRemoveObject: anObject
{
  return [_list removeObject:anObject];
}

- (BOOL) delegateListIncludesObject: anObject
{
  return [_list includesObject:anObject];
}

- delegateListCollection
{
  return _list;
}

- (unsigned char) delegateListSendBehavior
{
  return _send_behavior;
}

- delegateListSetSendBehavior: (unsigned char)b
{
  _send_behavior = b;
  return self;
}


// FOR PASSING ALL OTHER MESSAGES TO DELEGATES;

- forward: (SEL)aSel :(arglist_t)argFrame
{
  void *ret = 0;
  elt delegate;
  
  switch (_send_behavior) 
    {
    case SEND_TO_ALL:
      FOR_ARRAY(_list, delegate)
	{
	  if ([delegate.id_u respondsTo:aSel]) 
	    ret = [delegate.id_u performv:aSel :argFrame];
	}
      FOR_ARRAY_END;
      break;
      
    case SEND_TO_FIRST_RESPONDER:
      FOR_ARRAY(_list, delegate)
	{
	  if ([delegate.id_u respondsTo:aSel]) 
	    return [delegate.id_u performv:aSel :argFrame];
	}
      FOR_ARRAY_END;
      break;
      
    case SEND_UNTIL_YES:
      FOR_ARRAY(_list, delegate)
	{
	  if ([delegate.id_u respondsTo:aSel]) 
	    if ((ret = [delegate.id_u performv:aSel :argFrame]))
	      return ret;
	}
      FOR_ARRAY_END;
      break;
      
    case SEND_UNTIL_NO:
      FOR_ARRAY(_list, delegate)
	{
	  if ([delegate.id_u respondsTo:aSel]) 
	    if (!(ret = [delegate.id_u performv:aSel :argFrame]))
	      return ret;
	}
      FOR_ARRAY_END;
      break;
    }
  return ret;
}

@end
