/* Implementation of SavePanel class
 *
 * Copyright (C)  1994  The Board of Trustees of  
 * The Leland Stanford Junior University.  All Rights Reserved.
 *
 * Authors: Jeff Kamerer and Libing Wang
 *
 * This file is part of an Objective-C class library for a window system
 *
 */

#include "SavePanel.h"

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>          

#include <dirent.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include "Matrix.h"
#include "Motif.h"
#include "MList.h"
#include "NXBrowser.h"
#include "NXBrowserCell.h"
#include "ScrollView.h"
#include "stdmacros.h"
#include <coll/List.h>
#include "Application.h"
#include "Button.h"
#include "TextField.h"

#define LISTINCREMENT	10

/* the DIR_INDICATOR cannot be the same as the pathseparator--
	it will foul up setPath */
#define DIR_INDICATOR	"-->"
#define DIR_INDICATOR_FIRSTCHAR		'-'

/*
	isLower and sort are used to sort the files the same way that NeXT
	sorts them.  Fairly straightforward.
*/

BOOL isLower(const char *wordOne, const char *wordTwo)
{
	int i = 0;
	char one, two;
	BOOL isLetterOne, isLetterTwo;

	while ((wordOne[i] != '\0') && (wordTwo[i] != '\0')) {
		one = tolower(wordOne[i]);
		isLetterOne = ((one >= 'a') && (one <= 'z'));
		two = tolower(wordTwo[i]);
		isLetterTwo = ((two >= 'a') && (two <= 'z'));
		if ((isLetterOne) && (!isLetterTwo))
			return YES;
		if ((!isLetterOne) && (isLetterTwo))
			return NO;
		if (one < two)
			return YES;
		if (one > two)
			return NO;
		i++;
	}
	if (wordOne[i] == '\0')
		return YES;
	return NO;
}

char** sort(char **list, int length)
{
	int i, j, min;
	char *temp;

	for (i = 0; i < length; i++) {
		min = i;
		for (j = i + 1; j < length; j++) {
			if (isLower(list[j], list[min])) {
				min = j;
			}
		}
		if (min != i) {
			temp = list[i];
			list[i] = list[min];
			list[min] = temp;
		}
	}
	return list;
}

@interface SavePanel(PrivateMethods)
- setUpPanel;
- decodeDirectoryAsSeen;
@end

@implementation SavePanel(PrivateMethods)

/*
	setUpPanel - I moved the sizing of all of the elements out of
	initContent... so that all of the elements could be easily resized
	relative to the panel when the panel is resized.  Of course I never
	implemented this...
*/
- setUpPanel
{
	NXRect contentFrame;
	NXRect myFrame;

	[contentView getFrame:&contentFrame];
	
    myFrame.size.width  = contentFrame.size.width - 20;
    myFrame.size.height = contentFrame.size.height - 150;
    myFrame.origin.x = 10;
    myFrame.origin.y = 75;
    [browser setFrame:&myFrame];

    myFrame.size.width  = 75;
    myFrame.size.height = 35;
    myFrame.origin.x = contentFrame.size.width - 80;
	myFrame.origin.y = contentFrame.size.height - 40;
	[okButton setFrame:&myFrame];

    myFrame.origin.x = myFrame.origin.x - 80;
	[_cancelButton setFrame:&myFrame];

	myFrame.size.width  = 30;
    myFrame.size.height = 10;
    myFrame.origin.x = 10;
	myFrame.origin.y = contentFrame.size.height - 60;
	[_formLabel setFrame:&myFrame];

	myFrame.size.width  = contentFrame.size.width - 52;
	myFrame.size.height = 25;
    myFrame.origin.x = 45;
	myFrame.origin.y = contentFrame.size.height - 68;
	[form setFrame:&myFrame];

	return self;
}

/*
	The _directoryAsSeen is the directory path from the NXBrowser with
	all of the DIR_INDICATORs in it (I made it "-->").  This method removes
	the DIR_INDICATORs and puts the result in the instance variable directory.
*/
- decodeDirectoryAsSeen
{
	int i = 0;
	int indicatorLength = strlen(DIR_INDICATOR);
	char *pathPtr;	
	if ((!_directoryAsSeen) || (strlen(_directoryAsSeen) <= 1)) {
		return self;
	}
	pathPtr = _directoryAsSeen + 1;
	strcpy(directory, "/");

	while (pathPtr[0] != '\0') {
		for (i = 0; ((pathPtr[i] != '/') && (pathPtr[i] != '\0')); i++);
		pathPtr[i - indicatorLength] = '\0';
		strcat(directory, pathPtr);
		if (pathPtr[i] != '\0') {
			strcat(directory, "/");
		}
		pathPtr[i - indicatorLength] = DIR_INDICATOR_FIRSTCHAR;
		pathPtr += i;
		if (pathPtr[0] == '/') {
			pathPtr++;
		}
	}
	return self;
}

@end


@implementation SavePanel

+ newContent:(const NXRect *)contentRect style:(int)aStyle
	backing:(int)bufferingType buttonMask:(int)mask defer:(BOOL)flag
{
	return [[SavePanel alloc] initContent:contentRect style:aStyle
				backing:bufferingType buttonMask:mask defer:flag];
}


- init
{
	NXRect myFrame;

    myFrame.size.width  = 300;
    myFrame.size.height = 400;
    myFrame.origin.x = 0;
    myFrame.origin.y = 0;

	return [self initContent:&myFrame style:0 backing:0 buttonMask:0
		defer:YES];
}

- initContent:(const NXRect *)contentRect style:(int)aStyle
	backing:(int)backingType buttonMask:(int)mask defer:(BOOL)flag
{
	NXRect	myFrame;

	[super initContent:contentRect style:aStyle backing:backingType
		buttonMask:mask defer:flag];
	instancename = "SavePanel";
	
	filename = malloc(MAXPATHLEN * sizeof(char));
	directory = malloc(MAXPATHLEN * sizeof(char));
	_directoryAsSeen = malloc(MAXPATHLEN*strlen(DIR_INDICATOR) * sizeof(char));
	filename[0] = directory[0] = _directoryAsSeen[0] = '\0';
	[super setTitle:""];

    myFrame.size.width  = 300;
    myFrame.size.height = 25;
    myFrame.origin.x = 10;
    myFrame.origin.y = 10;
	_titleText = [[TextField alloc] _initAsLabel];
    [_titleText setFrame:&myFrame];
	[_titleText setStringValue:"Save"];
	[contentView addSubview:_titleText];

    browser = [[NXBrowser alloc] init];
    [browser setMaxVisibleColumns:2];
    [browser setDelegate:self];
    [browser setTarget:self];
	[browser setAction:@selector(highlighted:)];
    [browser setDoubleAction:@selector(clickOK:)];
	[contentView addSubview:browser];

	okButton = [[Button alloc] init];
	[okButton setTitle:"OK"];
	[okButton setTarget:self];
	[okButton setAction:@selector(ok:)];
	[contentView addSubview:okButton];

	_cancelButton = [[Button alloc] init];
	[_cancelButton setTitle:"Cancel"];
	[_cancelButton setTarget:self];
	[_cancelButton setAction:@selector(cancel:)];
	[contentView addSubview:_cancelButton];

	_formLabel = [[TextField alloc] _initAsLabel];
	[_formLabel setStringValue:"File:"];
	[contentView addSubview:_formLabel];

	form = [[TextField alloc] init];
	[form setTarget:self];
	[form setAction:@selector(clickOK:)];
	[contentView addSubview:form];

	[self setUpPanel];

	return self;
}

- (int)runModal
{
	FILE* file;
	int result;

	[self makeKeyAndOrderFront:self];
	/* I had to do the following because a call to setDirectory before
		the NXBrowser widget has been formed crashes.  Could probably
		force the widget to be made earlier. */
	if (!_beenUsed) {
		file = popen("csh -f -c 'echo ~'", "r");
		fscanf(file, "%s", directory);
		pclose(file);
		[self setDirectory:directory];
		_beenUsed = YES;
	}
	result = [NXApp runModalFor:self];
	return result;
}

- (int)runModalForDirectory:(const char *)path file:(const char *)filename
{
	if (path) {
		[self setDirectory:path];
	}
	if (filename) {
		[form setStringValue:filename];
		[form selectText:self];
	}
	_beenUsed = YES;
	return [self runModal];
}

- cancel:sender
{
	[self performClose:self];
	[NXApp stopModal:NX_CANCELTAG];
	return self;
}

- ok:sender
{
	const char *formString = [form stringValue];

	if (formString[0] == '\0') {
		return self;
	}
	strcpy(filename, directory);
	strcat(filename, "/");
	strcat(filename, formString);
	[self performClose:self];
	[NXApp stopModal:NX_OKTAG];
	return self;
}

- (const char *)filename
{
	return filename;
}

- (const char *)directory
{
	return directory;
}

/*
	setDirectory also has to encode the DIR_INDICATORs into
	_directoryAsSeen and set the path of the browser with that.
*/
- setDirectory:(const char *)path
{
	int i = 0;
	char *pathPtr;
	char temp;

	if (!path) {
		return self;
	}
	if (directory != path) {
		strcpy(directory, path);
	}
	if (directory[0] == '/') {
		strcpy(_directoryAsSeen, "/");
		pathPtr = directory + 1;
	} else {
		_directoryAsSeen[0] = '\0';
		pathPtr = directory;
	}

	while (pathPtr[0] != '\0') {
		for (i = 0; ((pathPtr[i] != '/') && (pathPtr[i] != '\0')); i++);
		temp = pathPtr[i];
		pathPtr[i] = '\0';
		strcat(_directoryAsSeen, pathPtr);
		strcat(_directoryAsSeen, DIR_INDICATOR);
		pathPtr[i] = temp;
		if ((pathPtr[i] == '/') && (pathPtr[i + 1] != '\0')) {
			strcat(_directoryAsSeen, "/");
		}
		while (pathPtr[i] == '/') {
			i++;
		}
		pathPtr += i;
	}
	[browser setPath:_directoryAsSeen];
	return self;
}

- setPrompt:(const char *)prompt
{
	return self;
}

- setTitle:(const char *)aTitle
{
	[_titleText setStringValue:aTitle];
	return self;
}

- (const char *)requiredFileType
{
	return requiredType;
}

- setRequiredFileType:(const char *)type
{
	requiredType = NXCopyStringBuffer(type);
	/* do something appropriate */
	return self;
}

- setAccessoryView:aView
{
	/* not urgent for phase 1, we can work around not having it */
	return self;
}

@end

@implementation SavePanel(delegateMethods)

/* the browser passes an empty matrix for this delegate to fill in */
- (int)browser:sender fillMatrix:matrix inColumn:(int)column
{
    NXBrowserCell	*aCell;
    DIR			*dirp;
    struct dirent	*entry;
    struct stat 	statbuf;
    int			i;
    char		file_path[MAXPATHLEN];
    char		*p;
	char**		namesList;
	int			numNames;
	int			namesSpaceAvail;
	char		*tempString;

	if (column == 0) {
		strcpy(directory, "/");
	} else {
		[sender getPath:_directoryAsSeen toColumn:column-1];
		[self decodeDirectoryAsSeen];
	}
	if ( (dirp = opendir(directory) ) == NULL ) {
		return 0;
	}

	namesList = malloc(LISTINCREMENT * sizeof(char*));
	numNames = 0;
	namesSpaceAvail = 10;
    while( (entry = readdir(dirp) ) != NULL ) {
		if (strcmp(entry->d_name, ".") == 0 || 
		    strcmp(entry->d_name, "..") == 0) {
		    continue; /* skip self and parent */
		}
		tempString = malloc(sizeof(char) * (strlen(entry->d_name)
			+ strlen(DIR_INDICATOR) + 1));
		strcpy(tempString, entry->d_name);
		if (numNames == namesSpaceAvail) {
			namesSpaceAvail += LISTINCREMENT;
			namesList = (char**)realloc((void *)namesList,
										namesSpaceAvail * sizeof(char*));
		}
		namesList[numNames++] = tempString;
    }

	sort(namesList, numNames);

	strcpy( file_path, directory );
	if ( column != 0 ) 
	strcat( file_path, "/" );
	p = file_path + strlen(file_path);
	i = 0;
	while(i < numNames) {
		if (strcmp(namesList[i], ".") == 0 || 
		    strcmp(namesList[i], "..") == 0) {
		    continue; /* skip self and parent */
		}
		[matrix addRow];
		aCell = [matrix cellAt:i :0];
		*p = '\0';
		strcat( p, namesList[i] );
		stat( file_path, &statbuf );
		if ( (statbuf.st_mode & S_IFMT) == S_IFDIR ) {
		    strcat(namesList[i], DIR_INDICATOR);
		    [aCell setStringValue:namesList[i]];
		    [aCell setLeaf:NO];
		} else {
		    [aCell setStringValue:namesList[i]];
		    [aCell setLeaf:YES];
		}
		free(namesList[i]);
		[aCell setLoaded:YES];
		i++;
    }
	free(namesList);
    return i;
}

/* single click action from browser */
- highlighted:sender
{
	int column;

	column = [sender selectedColumn];
	if ([[sender selectedCell] isLeaf]) {
		[form setStringValue:[[sender selectedCell] stringValue]];
		[form selectText:self];
		if (column > 0) {
	    	[sender getPath:_directoryAsSeen toColumn:column - 1];
			[self decodeDirectoryAsSeen];
		} else {
			strcpy(directory, "/");
		}
	} else {
	    [sender getPath:_directoryAsSeen toColumn:column];
		[self decodeDirectoryAsSeen];
	}
    return self;
}

/* double click action from browser, return key action from form */
- clickOK:sender
{
	[okButton performClick:sender];
	return self;
}
@end
