/* Implemenation of ScrollView class
 *
 * Copyright (C)  1993  The Board of Trustees of  
 * The Leland Stanford Junior University.  All Rights Reserved.
 *
 * Authors: Paul Kunz, Imran Qureshi, and Libing Wang
 *
 * This file is part of an Objective-C class library for a window system
 *
 * ScrollView.m,v 1.26 1994/06/28 20:39:36 kamerer Exp
 */


#include "ScrollView.h"

#include "Box.h"
#include "ClipView.h"
#include "Scroller.h"
#include <coll/List.h>

#define CLIPVIEW_MARGIN 4

@implementation ScrollView:View


/* Private methods not declared in interface file */

- _init
{
    [super _init];
    instancename = "ScrollView";
    return self;
}

- _setBorder
{
    switch (borderType) {
    case NX_NOBORDER :
        [self _setArg:XmNshadowThickness to:0];
        break;
    case NX_LINE :
        break;
    case NX_BEZEL :
    	[self _setArg:XmNshadowType to:(void *)XmSHADOW_IN];
    	[self _setArg:XmNshadowThickness to:(void *)BEZEL_THICKNESS];
        break;
    case NX_GROOVE :
    	[self _setArg:XmNshadowType to:(void *)XmSHADOW_ETCHED_IN];
        break;
    default :
    	break;
    }
    return self;
}

/* Factory methods */

// + getFrameSize:(NXSize *)fSize forContentSize:(const NXSize *)cSize
//   horizScroller:(BOOL)hFlag vertScroller:(BOOL)vFlag borderType:(int)aType;

+ getContentSize:(NXSize *)cSize forFrameSize:(const NXSize *)fSize
                                horizScroller:(BOOL)hFlag 
                                 vertScroller:(BOOL)vFlag 
                                   borderType:(int)aType
{
    if ( vFlag ) {
	cSize->width = fSize->width - NX_SCROLLERWIDTH - CLIPVIEW_MARGIN;
    } else {
	cSize->width = fSize->width;
    }
    if ( hFlag ) {
	cSize->height = fSize->height - NX_SCROLLERWIDTH - CLIPVIEW_MARGIN;
    } else {
	cSize->height = fSize->height;
    }
    if ( aType == NX_BEZEL ) {
        cSize->width  -= (2*BEZEL_THICKNESS);
        cSize->height -= (2*BEZEL_THICKNESS);
    }
    return self;
}

/* Public methods */

- initFrame:(const NXRect *)frameRect
{
    [super initFrame:frameRect];
    if ( !contentView ) {
	contentView = [[ClipView alloc] initFrame:frameRect];
	[self addSubview:contentView];
    }
    if ( lineAmount == 0 ) {
	[self setLineScroll:10.0];
    }
    return self;
}

// - getDocVisibleRect:(NXRect *)aRect;
// - getContentSize:(NXSize *)contentViewSize;
// - resizeSubviews:(const NXSize *)oldSize;
// - drawSelf:(const NXRect *)rects :(int)rectCount;

- setDocView:aView
{
    return [contentView setDocView:aView];
}

- docView
{
    return [contentView docView];
}

- setDocCursor:anObj
{
    return [contentView setDocCursor:anObj];
}

- (int)borderType
{
    return borderType;
}

- setBorderType:(int)aType
{
    borderType = aType;
//    [contentView _setBorderType:borderType];
    [self _setBorder];
    [self tile];
    return self;
}

// - setBackgroundGray:(float)value;
// - (float)backgroundGray;
// - setBackgroundColor:(NXColor)color;
// - (NXColor)backgroundColor;

- setVertScrollerRequired:(BOOL)flag
{
    vScrollerRequired = flag;
    if ( !flag && vScroller ) {
	[vScroller free];
	vScroller = nil;
    }
    [self tile];
    return self;
}

- setHorizScrollerRequired:(BOOL)flag
{
    hScrollerRequired = flag;
    if ( !flag ) {
	[hScroller free];
	hScroller = nil;
    }
    [self tile];	
    return self;
}

- vertScroller
{
    return vScroller;
}

- horizScroller
{
    return hScroller;
}

// - setVertScroller:anObject;
// - setHorizScroller:anObject;
- setLineScroll:(float)value
{
    lineAmount = value;
    return self;
}

- setPageScroll:(float)value
{
    pageContext = value;
    return self;
}

// - setCopyOnScroll:(BOOL)flag;
// - setDisplayOnScroll:(BOOL)flag;
// - setDynamicScrolling:(BOOL)flag;

- setFrame:(const NXRect *)frameRect
{
  /* not implmented in OpenStep, but used to re-tile */
    [super setFrame:frameRect];
    [self tile];
    return self;
}

- tile
{
    NXRect  rect;

    if ( vScrollerRequired ) {
	rect.origin.x = 0;
	rect.origin.y = 0;
	rect.size.width = NX_SCROLLERWIDTH;
	rect.size.height = frame.size.height;
	if ( vScroller ) {
	    [vScroller setFrame:&rect];
	} else {
	    vScroller = [[Scroller alloc] initFrame:&rect];
	    [self addSubview:vScroller];
	    [vScroller setTarget:self];
	    [vScroller setAction:@selector(_scrollViaScroller:)];
	}
    }
    if ( hScrollerRequired ) {
	if ( vScrollerRequired ) {
	    rect.origin.x = NX_SCROLLERWIDTH;
	    rect.size.width = frame.size.width - NX_SCROLLERWIDTH;
	} else {
	    rect.origin.x = 0;
	    rect.size.width = frame.size.width;
	}
	rect.origin.y = frame.size.height - NX_SCROLLERWIDTH;
	rect.size.height = NX_SCROLLERWIDTH;
	if ( hScroller ) {
	    [hScroller setFrame:&rect];
	} else {
	    hScroller = [[Scroller alloc] initFrame:&rect];
	    [self addSubview:hScroller];
	    [hScroller setTarget:self];
	    [hScroller setAction:@selector(_scrollViaScroller:)];
	}
    }
    if ( vScrollerRequired ) {
        rect.origin.x = NX_SCROLLERWIDTH + CLIPVIEW_MARGIN;
    } else {
	rect.origin.x = 0;
    }
    if ( borderType == NX_BEZEL ) {
        rect.origin.x += BEZEL_THICKNESS;
	rect.origin.y = BEZEL_THICKNESS;
    } else {
        rect.origin.y = 0;
    }
    [ScrollView getContentSize:&rect.size forFrameSize:&frame.size
	         horizScroller:hScrollerRequired
                  vertScroller:vScrollerRequired
	            borderType:borderType];
    [contentView setFrame:&rect];
    [self reflectScroll:contentView];
    return self;
}

- reflectScroll:cView
{
    NXRect 	docrect;
    NXRect 	viewrect;
    float	pos, percent, d;
    
    [contentView getDocRect:&docrect];
    [contentView getDocVisibleRect:&viewrect];
    if ( hScrollerRequired ) {
	d = docrect.size.width;
	percent  = viewrect.size.width / d;
	d -= viewrect.size.width;
	if ( d > 0 ) {
	    pos = - docrect.origin.x / d;
	} else {
	    pos = 0;
	}
	[hScroller setFloatValue:pos :percent];
    }
    if ( vScrollerRequired ) {
	d = docrect.size.height;
	percent  = viewrect.size.height / d;
	d -= viewrect.size.height;
	if ( d > 0 ) {
	    pos = - docrect.origin.y / d;
	} else {
	    pos = 0;
	}
	[vScroller setFloatValue:pos :percent];
    }
    return self;
}

// - write:(NXTypedStream *)stream;

- read:(NXTypedStream *)stream
{
    View	*view;
    NXRect	rect;
    int		i;
	
    [super read:stream];
    objc_read_types(stream, "icc", &borderType, 
                                   &vScrollerRequired, &hScrollerRequired);
    i = [subviews count];
    while ( i-- ) {
	view = [subviews objectAt:i];
	if ( [view isKindOf:[ClipView class]] ) {
	    contentView = (ClipView *)view;
//	    [contentView _setBorderType:borderType];
	}
	if ( [view isKindOf:[Scroller class]] ) {
	    [view getFrame:&rect];
	    if ( rect.size.width > rect.size.height ) {
	        hScroller = (Scroller *)view;
	    } else {
	        vScroller = (Scroller *)view;
	    }
	}
    }
    return self;
}

- awake
{
    [super awake];
    if ( vScrollerRequired ) {
	[vScroller setTarget:self];
	[vScroller setAction:@selector(_scrollViaScroller:)];
    } else {
	[vScroller removeFromSuperview];
	[subviews removeObject:vScroller];
    } 
    if ( hScrollerRequired ) {
	[hScroller setTarget:self];
	[hScroller setAction:@selector(_scrollViaScroller:)];
    } else {
    	[hScroller removeFromSuperview];
	[subviews removeObject:hScroller];
    }
    [self setBorderType:borderType];
    [self tile];
    return self;
}

- _scrollViaScroller:sender
{
    float	h, v;
    int		hit;
    BOOL	flag;
    
    flag = (sender == vScroller);
    hit = [sender hitPart];
    switch (hit) {
    case NX_KNOB :
        if ( vScrollerRequired ) {
	    v = [vScroller floatValue];
	} else {
	    v = 0;
	}
	if ( hScrollerRequired ) {
	    h = [hScroller floatValue];
	} else {
	    h = 0;
	}
	[contentView _setScrollFloatValue:h :v];
        break;
    case NX_INCLINE :
	[contentView _scrollLine:lineAmount vertical:flag];
	break;
    case NX_INCPAGE :
        [contentView _scrollPage:pageContext vertical:flag up:NO];
        break;
    case NX_DECLINE :
        [contentView _scrollLine:-lineAmount vertical:flag];
	break;
    case NX_DECPAGE :
        [contentView _scrollPage:-pageContext vertical:flag up:YES];
        break;
    case NX_NOPART :
    default:
    	break;
    }
    return self;
}

- _managedBy:parent wid:(void*)widget
{
    [self _setBorder];
    [super _managedBy:parent wid:widget];
    return self;
}

+ newFrame:(const NXRect *)frameRect
{
    return [[self alloc] initFrame:frameRect];
}

+ new
{
    return [[self alloc] init];
}


@end
