/* WidgetSet category for implementation of Text class
 *
 * Copyright (C)  1994  The Board of Trustees of  
 * The Leland Stanford Junior University.  All Rights Reserved.
 *
 * Authors: Mike Kienenberger (Alaska) and Jeff Kamerer (SLAC)
 *
 * This file is part of an Objective-C class library for X/Motif
 *
 * xmText.m,v 1.5 1994/08/16 00:18:41 kamerer Exp
 */

#include "Text.h"

#include <stdlib.h>
#include <string.h>

#include "Motif.h"
#include "stdmacros.h"
#include <Xm/Text.h>

char	*TextInstanceName( void )
{
    return "Text";
}

static void theToggleCallback(Widget w, XtPointer client_data,
								XtPointer call_data)
{
	Text *textid = (Text *) client_data;
	XmTextVerifyCallbackStruct *cb = (XmTextVerifyCallbackStruct *)call_data;
   
	switch (cb->reason) {
		case XmCR_VALUE_CHANGED:
			[textid _textDidChange];
			break;
		case XmCR_MODIFYING_TEXT_VALUE:
			[textid _textWillChange:cb];
			break;
   }
   return;
}

@implementation Text(WidgetSet)

/* initialization */

- _textWillChange:(XmTextVerifyCallbackStruct *)cb
{
	char *theString;
	int i;
	
	if (_justKilledWidget) {
		_justKilledWidget = NO;
		return self;
	}
	theString = XmTextGetString(widgetid);
	for (i = cb->startPos; i < cb->endPos; i++) {
		if (theString[i] == '\n') {
			_numTextLines--;
		}
	}
	for (i = 0; i < cb->text->length; i++) {
		if (cb->text->ptr[i] == '\n') {
			_numTextLines++;
		}
	}
    return self;
}

- _textDidChange
{
	short numRows;
	Dimension width;
	char *theString;
	int length;
	
	if (_waitToKillWidget) {
		return self;
	}
	theString = XmTextGetString(widgetid);
	length = strlen(theString);
	XtVaGetValues(widgetid, XmNrows, &numRows, NULL);
	if (numRows < _numTextLines) {
		if (_textString != theString) {
			free(_textString);
			_textString = malloc((length + 1) * sizeof(char));
			strcpy(_textString, theString);
		}
		XtVaGetValues(widgetid, XmNwidth, &width, NULL);
		XtDestroyWidget(widgetid);
		widgetid = NULL;
		_justKilledWidget = YES;
    	[self _addArg:XmNrows:(void *)_numTextLines];
    	[self _addArg:XmNeditMode:(void *)XmMULTI_LINE_EDIT];
    	[self _addArg:XmNwidth:(void *)width];
		[self _managedBy:superview];
		[superview descendantFrameChanged:self];
	}
    return self;
}

- _addCallback
{
	XtAddCallback(widgetid, XmNmodifyVerifyCallback,
				theToggleCallback, self);
    XtAddCallback(widgetid, XmNvalueChangedCallback, 
			    theToggleCallback, self);
    return self;
}

- _init_Text
{	
    classname = xmTextWidgetClass;
	_numTextLines = 1;
	_justKilledWidget = NO;
	_waitToKillWidget = NO;

    [self _addArg:XmNeditMode:(void *)XmMULTI_LINE_EDIT];
    
    return self;
}

- _setText
{    
    if (_textString)
		[self _setArg:XmNvalue to:(void *)_textString];    
    return self;
}

- _managedBy:parent
{
    [super _managedBy:parent];
    if (widgetid) {
		Dimension width, height;
		XtVaGetValues(widgetid, XmNwidth, &width, NULL);
		XtVaGetValues(widgetid, XmNheight, &height, NULL);
		[self sizeTo:(int)width :(int)height];
		[self _setArg:XmNvalue to:(void *)_textString];
	}
    return self;
}

- _setSel:(int)start :(int)end
{
	if (widgetid) {
		XmTextSetSelection(widgetid, start, end, 0);
	}
	return self;
}

- (int)_getSubstring:(char *)buf start:(int)startPos length:(int)numChars
{
	char *wholeString = XmTextGetString(widgetid);
	int length = strlen(wholeString);
	int willMove;

	if (startPos >= length) {
		return (-1);
	}
	willMove = MIN(numChars, length - startPos);
	strncpy(buf, &(wholeString[startPos]), willMove);
	return willMove;
}

- (int)_textLength
{
	if (widgetid) {
		return XmTextGetLastPosition(widgetid);
	}
	return 0;
}

- _replaceSel:(const char *)aString
{
	XmTextPosition start, end;

	if (widgetid) {
		XmTextGetSelectionPosition(widgetid, &start, &end);
		XmTextReplace(widgetid, start, end, aString);
	}
	return self;
}

- _preReadText
{
	_waitToKillWidget = YES;
	return self;
}

- _postReadText
{
	_waitToKillWidget = NO;
	[self _textDidChange];
	return self;
}

@end
