/*
	Deck.h
	Version 0.0		6/21/90		Ernest N. Prabhakar
	Version 0.1		6/29/90		Ernest N. Prabhakar
	Version 1.0		7/12/90		Ernest N. Prabhakar
	
*/

#define DECK_VERSION 1
#define DECK_UNMARKED NX_NOT_IN_LIST

#import <objc/List.h>

@interface Deck : List
 /* 
  * Deck implements a collections of Cards as a subclass of List.
  * It provides the usual operations used in card games.
  * It adds a marker so one Deck can be used for both draw and discard.
  * Instances can also be used as hands in a game.
  * It was designed originally for use with memory cards.
  * Deck expects cards to receive the following messages
  *	- (STR) findSTR:(STR)aString (returns string or nil)
  *		or can use the compiled regex
  *	- (int) compare:aCard (-,0,+ if self <,=,> aCard)
  * And fo the same reasons, expects "activeFace" and "sortFlag" status
  * variables in the Card factory object.
  * For manipulating in greater detail, use the List class methods.
  */
{
@public	
	unsigned	marker;	//separates drawn from discarded
}
 /*
  * METHOD TYPES
  * Creating and freeing a Deck object
  * Archiving
  * Setting and clearing marker
  * Giving random position in deck before marker.
  * Drawing and discarding
  * Shuffling, Cutting, and Dealing into hands (deck instances)
  * Finding and sorting cards
  */

+ new;
 /*
  * TYPE: Creating; Returns a new Deck object
  *
  * Returns a new Deck.
  */


+ initialize;
 /*
  * TYPE: Creating; Set initial parameters
  *
  * Sets up a new Deck.
  */

- copy;
 /*
  * TYPE: Creating.  Returns a copy of the deck pointing to same cards.
  */

- setMarkerAt:(unsigned)index;
- setMarkerAtEnd;
 /*
  * TYPE: Manipulating marker; Sets at index
  *
  * Defaults to setting marker at end if index > numElements.
  */
  
- (unsigned) marker;
 /*
  * TYPE: Manipulating marker; Returns marker location without changing it
  *
  * Returns marker, or DECK_UNMARKED if none was set.
  */
  
- clearMarker;
 /*
  * TYPE: Manipulating marker; clears
  *
  * Sets marker to DECK_UNMARKED
  */

- giveCard;
- giveCardAt:(unsigned)index;
 /*
  * TYPE: Removes from Deck; Returns the object at \fIindex\fR
  *
  * Defaults to first object
  * Removes and Returns the \fBid\fR of the object located at slot \fIindex\fR.
  * if \fIindex\fR is beyond marker or end of Deck, return nil
  */
  
- takeCard:anObject;
 /*
  * TYPE: Adds to Deck; Places anObject at end.
  *
  */

- (unsigned) dealTo:(List *)handList;
- (unsigned) dealTo:(List *)handList each:(int)numCards;
 /*
  * TYPE: Removes from Deck; Takes a list of hands, and distributes cards
  * evenly among them.
  * The default numCards is numElements of Deck/ numHands of handList.
  * Excess cards are left in the Deck.  The marker is set to the back.
  * The number of cards left is returned.
  */

- emptyTo:(Deck *)hand;
 /*
  * TYPE: Removes from Deck; Gives hand all its contents (up to marker)
  *
  * Empties deck, adding contents to hand. 
  */

- cutAt:(unsigned)index;
 /*
  * TYPE: Permute Deck; Cuts at specified index
  *
  * Makes card at index new top card, and returns its id.
  * Cards above it are rolled back
  * If index is invalid (e.g. < 0), returns nil.
  * Sets marker at End.
  */

/* Target-action methods */

- shuffle:sender;
 /*
  * TYPE: Permute Deck
  *
  * Randomly permutes deck.
  * Sets marker at End.
  */

- sort:sender;
- reverse:sender;
 /*
  * TYPE: Permute Deck;  Sorts cards by some criterions
  *
  * Sort rearranges the deck according to the 'rankRelativeTo' method.
  * Each Card-type implements this differently, but must do something.
  *	'reverse' merely inverts the order of the deck.
  * Sets marker at End.
  */

- flipAll:sender;
 /*
  * TYPE: Permute Cards;  Exchanges Faces
  *
  * Make the top face of all the cards the new bottom face.
  *
  */

- findSTR:(const char *)aString;
- findAllSTR:(const char *)aString;
 /*
  * TYPE: Select cards; cards containing string; Returns their index;
  *
  * Find returns the id of the first card which contains string.
  * FindAll returns a Deck containing all such cards (even if one).
  * Does NOT remove them from the Deck.
  * Both return nil if there is no match.
  */

@end
