/* 	This file/program is part of the Graphicsland project at 
 *		The University of Calgary, Canada.
 *
 * 	Author:		Angus Davis.
 *	Purpose:	Provides a standard routines to read and write raster image
 * 	   		files.
 *	Modification history:
 * 			rle_reset() added by Blob so that more than raster file
 *			can be read.   They have to be input sequentially.
 *
 *	May 11, 1988.	Created
 */


#include <stdio.h>
#include <stdlib.h>
#include <strings.h>

#define BUFSIZE	8192

static unsigned char	input_buf[BUFSIZE];
static int	input_eof = 0,
		input_buf_count = 0,
		input_buf_index = BUFSIZE;

static char	output_rle_buf[BUFSIZE];
static int	output_rle_buf_index = 0;

static void add_run(unsigned char r,unsigned char g,unsigned char b,int c,FILE *outfile);
int get_rle_header(FILE *inf, 
		   int *size_x, int *size_y, 
		   unsigned char *erase_red, unsigned char *erase_grn, unsigned char *erase_blu, 
		   unsigned char *erase);
void rle_reset(void);
int get_rle_run(FILE *inf, unsigned char *r, unsigned char *g, unsigned char *b, int *len);

void output_rle_header(int size_x, int size_y, 
		       int erase_red, int erase_grn, int erase_blu, int erase, 
		       FILE *outfile);
void output_rle_run(unsigned char r, unsigned char g, unsigned char b, int len, FILE *outfile);
void output_rle_trailer(FILE *outfile);

/*
 *  Reads a raster image header and copies the image dimensions and
 *    background color into the passed structures.  It returns 0 if
 *    an error occured and returns 1 otherwise.
 */

int get_rle_header (FILE *inf, int *size_x, int *size_y, 
		    unsigned char *erase_red, unsigned char *erase_grn, unsigned char *erase_blu, 
		    unsigned char *erase)
{
	unsigned char	buf[80];

	input_eof = 0;
	input_buf_count = 0;
	input_buf_index = BUFSIZE;

	if (fread (buf, sizeof(buf[0]), 80, inf) != 80)
		return 0;
	if (strncmp ((char *)buf, "gl  RASF", 8))
		return 0;

	*size_x = (buf[ 8]<<24) + (buf[ 9]<<16) + (buf[10]<<8) + buf[11];
	*size_y = (buf[12]<<24) + (buf[13]<<16) + (buf[14]<<8) + buf[15];
	*erase_red = buf[24];
	*erase_grn = buf[25];
	*erase_blu = buf[26];
	*erase     = buf[27];

	return 1;
}

/* -------------------------------------------------------------------------- */
/* it didn't occur to anybody when this library was written that somebody may
   want to input more than raster file.  This routine resets the globals
   but note that you can't input the raster files in parrallel. - blob Jul/90
*/
void rle_reset(void)
{
	input_eof = 0;
	input_buf_count = 0,
	input_buf_index = BUFSIZE;
	bzero(input_buf,BUFSIZE);
	bzero(output_rle_buf,BUFSIZE);
}
/* -------------------------------------------------------------------------- */
/*
 *  Reads a run-length encoded run from the input file, storing the
 *    R, G, and B color values and the length of the run in the passed
 *    structures.  It returns 0 if an error occured and returns 1 otherwise.
 */

int get_rle_run (FILE *inf, unsigned char *r, unsigned char *g, unsigned char *b, int *len)
{
	if (input_eof)
		return 0;
	if (!input_buf_count || input_buf_index==input_buf_count) {
		if (input_buf_index < BUFSIZE) {
			input_eof = 1;
			return 0;
		}
		bzero(input_buf,BUFSIZE);
		input_buf_count = fread (input_buf, sizeof(unsigned char), BUFSIZE, inf);
		input_buf_index = 0;
	}
	if (!input_buf_count) {
		input_eof = 1;
		return 0;
	}
	*r = input_buf[input_buf_index++];
	*g = input_buf[input_buf_index++];
	*b = input_buf[input_buf_index++];
	*len = (int)input_buf[input_buf_index++] + 1;

	return 1;
}


/*
 *  Outputs the raster image header with the given image dimensions and 
 *    background color to an output file.
 */

void output_rle_header (size_x, size_y, erase_red, erase_grn, erase_blu, erase, outfile)
	int	size_x, size_y;
	unsigned char	erase_red, erase_grn, erase_blu, erase;
	FILE	*outfile;
{
	char	buf[80];
	int	i;

	output_rle_buf_index = 0;
        strncpy (buf, "gl  RASF", 8);
        buf[ 8] = (char) (size_x>>24) & 0xff;
        buf[ 9] = (char) (size_x>>16) & 0xff;
        buf[10] = (char) (size_x>>8) & 0xff;
        buf[11] = (char) size_x & 0xff;

        buf[12] = (char) (size_y>>24) & 0xff;
        buf[13] = (char) (size_y>>16) & 0xff;
        buf[14] = (char) (size_y>>8) & 0xff;
        buf[15] = (char) size_y & 0xff;

        buf[16] = '\000';       /* bits_per_pixel */
        buf[17] = '\000';
        buf[18] = '\000';
        buf[19] = (char) 24;

        buf[20] = '\000';       /* LUT present */
        buf[21] = '\000';
        buf[22] = '\000';
        buf[23] = '\000';

        buf[24] = erase_red;
        buf[25] = erase_grn;
        buf[26] = erase_blu;
        buf[27] = (char) erase;
	for (i=28; i<80; i++) buf[i] = 0;
        fwrite (buf, sizeof(char), 80, outfile);
        fflush (outfile);
}


/*
 *  Outputs a set of (r,g,b,length-1) run length encodings where the maximun
 *    possible length of a run is 256.  r,g,b specify the 8-bit color values
 *    and len is an integer which indicates the total length of a run.
 */

static void add_run (unsigned char r,unsigned char g,unsigned char b,int c,FILE *outfile)
{
        output_rle_buf[output_rle_buf_index++] = r;
        output_rle_buf[output_rle_buf_index++] = g;
        output_rle_buf[output_rle_buf_index++] = b;
        output_rle_buf[output_rle_buf_index++] = c - 1;
        if (output_rle_buf_index == BUFSIZE) {
                fwrite (output_rle_buf, sizeof(output_rle_buf[0]), output_rle_buf_index, outfile);
                output_rle_buf_index = 0;
        }
}


void output_rle_run (unsigned char r,unsigned char g,unsigned char b, int len, FILE *outfile)
{
	while (len > 256) {
		add_run (r, g, b, 256, outfile);
		len -= 256;
	}
	if (len > 0)
		add_run (r, g, b, len, outfile);
}

void output_rle_trailer (FILE *outfile)
{
        fwrite (output_rle_buf, sizeof(output_rle_buf[0]), output_rle_buf_index, outfile);
        fflush (outfile);
}

