/* Program:
 *
 *		MOTD - 'Motto of the day' window
 *
 * Author:
 *
 *		Christopher Lane (lane@sumex-aim.stanford.edu)
 *		Symbolic Systems Resources Group
 *		Knowledge Systems Laboratory
 *		Stanford University
 *
 * Contributors:
 *
 *		Izumi Ohzawa (izumi@violet.berkeley.edu)
 *		Group in Neurobiology/School of Optometry
 *		University of California, Berkeley
 *
 *
 *		Steve Hayman (sahayman@cs.indiana.edu)
 *		Computer Science Department Workstation Manager
 *
 * Date:
 *
 *		16 April 1991
 *
 * Copyright:
 *
 *		1989, 1990 & 1991 by The Leland Stanford Junior University.
 *      May be distributed without restriction for non-commercial use.
 *
 */

#import "MOTD.h"

extern void timer(DPSTimedEntry, double, id);

static NXDefaultsVector myDefaults = {
	{ "File", MOTDFILE },
	{ "OnlyOnce", "Yes" },
	{ "Hold", "No" },
	{ "Wait", "25" },
	{ "MailWait", "2" },
	{ NULL }
	};
	
@implementation MOTD

+ initialize
{
	(void) NXRegisterDefaults(APPLICATION, myDefaults);
	return [super initialize];
}

+ new
{
	self = [super new];
	[self setFile:getDefault("File")];
	[self setWait:atoi(getDefault("Wait"))];
	[self setHold:(strncmp(getDefault("Hold"), "Yes", 1) == EQ) ? YES : NO];
	return self;
}

- appDidInit:sender
{
	[[self appIcon] orderOut:self];
	if (loginHook) (void) [self activateSelf:YES];
	if (!do_motd) [self setWait:atoi(getDefault("MailWait"))]; /* short timeout */  
	timedEntry = DPSAddTimedEntry(SECOND, (DPSTimedEntryProc) &timer, self, NX_BASETHRESHOLD);
	return self;
}

- windowWillClose:sender
{
	return [self terminate:sender];
}

- free
{
	FILE *f_utmp;
	struct timeval tp;
	struct utmp s_utmp;
	struct lastlog s_lastlog;
    
	if (timedEntry != NULL) DPSRemoveTimedEntry(timedEntry);
    
	if (!loginHook) return [super free];
     
	bzero(s_utmp.ut_name, sizeof(s_utmp.ut_name));
	bzero(s_utmp.ut_line, sizeof(s_utmp.ut_line));
	bzero(s_utmp.ut_host, sizeof(s_utmp.ut_host));
    
	bzero(s_lastlog.ll_line, sizeof(s_lastlog.ll_line));
	bzero(s_lastlog.ll_host, sizeof(s_lastlog.ll_host));
    
	if (gettimeofday(&tp, NULL) == CERROR) return [super free];
	else s_lastlog.ll_time = s_utmp.ut_time = tp.tv_sec;

	(void) strncpy(s_utmp.ut_line, TTYDEVICE, sizeof(s_utmp.ut_line));
	(void) strncpy(s_lastlog.ll_line, TTYDEVICE, sizeof(s_lastlog.ll_line));
	
	(void) strncpy(s_utmp.ut_name, name, sizeof(s_utmp.ut_name));
	
	if ((f_utmp = fopen(WTMPFILE, "a")) != NULL) {
		(void) fwrite(&s_utmp, sizeof(s_utmp), 1, f_utmp);
		(void) fclose(f_utmp);
		}
	
	if ((f_utmp = fopen(LASTLOGFILE, "a")) != NULL) {
	if (fseek(f_utmp, (long) (uid * sizeof(s_lastlog)), SEEK_SET) != CERROR)
		(void) fwrite(&s_lastlog, sizeof(s_lastlog), 1, f_utmp);
		(void) fclose(f_utmp);
		}

	return [super free];
}

- setMailPanel:anObject
{
	char pathnamebuf[MAXPATHLEN];

	(void) strcpy(pathnamebuf, MAILDIR);
	(void) strcat(pathnamebuf, name);
	[(mailPanel = anObject) setTitle:pathnamebuf];
	return self;
}

- setMailText:anObject
{
	char pathnamebuf[MAXPATHLEN];
	const char *value;

	mailText = anObject;
	if (maillevel != NONE) {
		[mailText setStringValue:(maillevel == NEW) ? NEWMAILMESSAGE : MAILMESSAGE];
		[[mailPanel display] orderFront:nil];
		if ((value = getSystemDefault("SystemAlert")) != NULL && strcmp(value, "Panel") != EQ) {
			(void) strcpy(pathnamebuf, SOUNDDIR);
			(void) strcat(pathnamebuf, (maillevel == NEW) ? NEWMAILSND : MAILSND);
			SNDPlaySoundfile(pathnamebuf, 0);
			}
		}
	return self;
}

- setMyWindow:anObject
{
	myWindow = anObject;
	if (do_motd) [[myWindow setTitle:file] makeKeyAndOrderFront:self];
	return self;
}

- setMyText:anObject
{
	NXStream *stream;
	int fileDesc, flag;
 
	if (!do_motd) return self;	/* Don't do it unless we have a MOTD window */

	if ((fileDesc = open(file, O_RDONLY, 0644)) == CERROR) [self terminate:self];
	
	if ((stream = NXOpenFile(fileDesc, NX_READONLY)) == NULL) [self terminate:self];
	
	myText = [anObject docView];
    
	flag = (NXGetc(stream) == '{');
	NXUngetc(stream);
    
	if (flag) [myText readRichText:stream];
	else [myText readText:stream];

	NXClose(stream);
	(void) close(fileDesc);
	return self;
}

- getHoldButton { return holdButton; }

- setHoldButton:anObject
{
	[(holdButton = anObject) setState:(int) hold];
	return self;
}

- (const char *) getUserName { return name; }

- setUserName:(const char *) pw_name
{
	name = NXCopyStringBuffer(pw_name);
	return self;
}

- (int) getUID { return uid; }
- setUID:(int) pw_uid { uid = pw_uid; return self; }

- getCancelButton { return cancelButton; }
- setCancelButton:anObject { cancelButton = anObject; return self; }

- setDo_Motd:(BOOL) aBoolean { do_motd = aBoolean; return self; }
- (BOOL) getDo_Motd { return do_motd; }

- setMailLevel:(MailLevel) aMailLevel { maillevel = aMailLevel; return self; }
- (MailLevel) getMailLevel { return maillevel; }

- (const char *) getFile { return file; }
- setFile:(const char *) aString { file = aString; return self; }

- (BOOL) getHold { return hold; }
- setHold:(BOOL) aBoolean { hold = aBoolean; return self; }
- toggleHold:anObject { hold = ! hold; return self; }

- (BOOL) getLoginHook { return loginHook; }
- setLoginHook:(BOOL) aBoolean { loginHook = aBoolean; return self; }

- (int) getWait { return wait; }
- setWait:(int) anInteger { wait = anInteger; return self; }

- (void) cancelLogin:anObject
{
	[self free];
	exit(EXIT_FAILURE);
}

@end
