//
// SCSI.m
//
// SCSI generic driver object library (phew)
// Version 1 (Revision 1.0.2)
//
// The SCSI class source, header, and documentation are all 
// Copyright (C) 1990 by Canon Inc.
// Copyright (C) 1991 by Jiro Nakamura.
// All Rights Reserved. 
//
// This class is provided for use by the general public. 
// The class interface header file (SCSI.h), source (SCSI.m) 
// and documentation (Class_SCSI.wn) may be freely copied 
// and distributed as long as this legend is included on all 
// storage media and as part of the software program and 
/// documentation in whole or part. 
//
// See Class Documentation for information. I am not a great comment-person,
// although I am not too proud of that fact....
//
// Original Author:	Jiro Nakamura @ Canon
// Created:		June 12, 1990
//
// RCS Information
// Revision Number->	$Revision: 1.8 $
// Last Revised->	$Date: 91/02/20 16:14:09 $
// ------------------
// History:
//	12-Jul-90	Jiro
//		Cleaned up for public release.
// 	12-Jun-90	Jiro Nakamura @ Canon
//		Created.

static char rcsid[]="$Id: SCSI.m,v 1.8 91/02/20 16:14:09 jiro Exp Locker: jiro $";

#define VERSION		1

#import "SCSI.h"
#import <libc.h>
#import <stdio.h>
#import <fcntl.h>
#import <stdio.h>
#import <objc/typedstream.h>
#import <appkit/nextstd.h>	// For NXLogError
#import <c.h>

// SCSI produces a whole truckload of debugging code. Usually we don't
// want it on.
#ifdef DEBUG
	#undef DEBUG
#endif

@implementation SCSI

+new
{	
	self = [super new];
	target = -1;
	lun = 0;
 	dev_name = "/dev/sg0";
	scsiOpen = FALSE;
	return self;
	return self;
}

- (int) version
{
	return VERSION;
}

- (int) openSCSI
{
	// *No* program should try to open a driver that's already open.
	//  Bad, bad, bad.
	if( scsiOpen == TRUE)
		{
		sprintf(errorString, "SCSI Device already open -- "
			"bug in program...");
		return fd;
		}
		
	if ((fd = open (dev_name, O_RDWR)) < 0) 
		{
		sprintf(errorString,"Could not open SCSI "
			"generic driver <%s>.\n"
			"File descriptor = 0x%X\n",dev_name,fd);
		perror("open");
		scsiOpen = FALSE;
		return -1;
		}
	
	scsiOpen = TRUE;
	return 0;
}

- (int) openSCSIAt: (int) trg lun: (int) ln
{
	if( [self openSCSI] )
		return -1;
	
	return(  [self setTarget: trg lun: ln]);
}


- (int) setTarget: (int) trg lun: (int) ln
{
	sa.sa_target = target = trg;
	sa.sa_lun = lun = ln;
	
	if (ioctl(fd,SGIOCSTL,&sa) < 0) 
		{
		sprintf(errorString,"ioctl(SGIOCSTL): "
			"Error setting target %d lun %d (errno = %d)\n",
			target,lun, errno);
		return -1;
		}
	return 0;
}



- (int) closeSCSI
{	
	if( scsiOpen == FALSE)
		{
		sprintf(errorString, "SCSI device already closed. "
				"Error in program...");
		return -1;
		}
		
	if (   close(fd) < 0) 
		{
		sprintf(errorString, 	"Could not close %s - fd = %XH\n"
					"errno = %d\n", dev_name,fd ,errno);
		perror("close");
		return -1;
		}	
	scsiOpen = FALSE;
	return 0;
}

- (BOOL) scsiOpen
{
	return( scsiOpen);
}

- (struct scsi_req *) statusReq
{
	return &sr;
}

- (char *) errorString
{
	return errorString;
}

- (int) findDevice
{
	return [self findDevice: 0];
}

- (int) findDevice: (int) trg
{
	int tmp;
	
	if( scsiOpen)
		return [self findDeviceSCSI: trg];
	else
		{
		if([self openSCSI] )
			{
			sprintf(errorString,"Can't open SCSI driver");
			return -1;
			}		
		tmp = [self findDeviceSCSI: trg];
		[self closeSCSI];
		return( tmp);
		}
}

- (int) findDeviceSCSI: (int) trg
{
	int tmp;
	
	for( tmp = trg; tmp < 8; tmp ++)
		if( [self isDeviceSCSI: tmp ])
			return( tmp);
	return( -1);
}

- (BOOL) isDevice: (int) trg
{
	BOOL tmp;
	
	if( scsiOpen)
		return [self isDeviceSCSI: trg];
	if([self openSCSI] )
		{
		sprintf(errorString,"Can't open SCSI driver");
		NXLogError(errorString);
		return NO;
		}
	tmp = [self isDeviceSCSI: trg];
	[self closeSCSI];
	return( tmp);
}


- (BOOL) isDeviceSCSI: (int) trg	// Subclasses should implement this
{
	[self subclassResponsibility: _cmd];
	return NO;
}



//  =====================================================
// 	Archiving methods
//  =====================================================

- write: (NXTypedStream *) stream
{
	[super write: stream];
	NXWriteTypes( stream, "ii**", &target, &lun, &dev_name, &errorString);
	return self;
}

- read: (NXTypedStream *) stream
{
	[super read: stream];
	NXReadTypes( stream, "ii**", &target, &lun, &dev_name, &errorString);
	return self;
}


//  =====================================================
// 	Internal Routines
//  =====================================================


- (int) performSCSIRequest
{
	if (ioctl(fd,SGIOCREQ, &sr) < 0) {
		#ifdef DEBUG
			printf("..Error executing ioctl\n");
			printf("errno = %d\n",errno);
		#endif
		return errno;
	}
	
	#ifdef DEBUG
		fprintf(stderr,
			"SCSI Command %x ==> %d byte(s) in %d.%06d sec\n", 
			sr.sr_cdb.cdb_c6.c6_opcode,
			sr.sr_dma_xfr,
			(int) sr.sr_exec_time.tv_sec, 
			(int) sr.sr_exec_time.tv_usec);
	#endif
	
	if(sr.sr_io_status || sr.sr_scsi_status) {
		#ifdef DEBUG
		printf("sr_io_status = 0x%02X\n",sr.sr_io_status);
		if(sr.sr_io_status == SR_IOST_CHKSV) {
			printf("   sense key = 0x%02X   sense code = %02XH\n",
				sr.sr_esense.er_sensekey,
				sr.sr_esense.er_addsensecode);
			}
		printf("SCSI status = 0x%02X\n",sr.sr_scsi_status);
		#endif
		return sr.sr_io_status;
	}
	
	return 0;
} 

- clearCommandBlock: (union cdb *) cd
{
	int i;
	char *p;
	
	p = (char *)cd;
	for(i=0; i<sizeof(union cdb); i++)
		*p++ = 0;
	return self;
}


//  =====================================================
// 	SCSI ROUTINES AND MACROS
//  =====================================================




- (int) inquiry: (struct inquiry_reply *) ibuffer
{
	int tmp;
	
	if( scsiOpen)
		return [self inquirySCSI: ibuffer];
	else
		{
		if([self openSCSI] )
			{
			sprintf(errorString,"Can't open SCSI driver");
			return -1;
			}
		
		tmp = [self inquirySCSI: ibuffer];
		[self closeSCSI];
		return( tmp);
		}
}


- (int) inquirySCSI: (struct inquiry_reply *) ibuffer
{	
	struct cdb_6 *cdbp = &sr.sr_cdb.cdb_c6;

	[self clearCommandBlock: (union cdb *) cdbp];
	cdbp->c6_opcode = C6OP_INQUIRY;
	cdbp->c6_lun    = lun;
	cdbp->c6_len	= 36;
	sr.sr_dma_dir	= SR_DMA_RD;
	sr.sr_addr	= (char *) ibuffer;
	sr.sr_dma_max   = 36;
	sr.sr_ioto	= 10;

	return [self performSCSIRequest];
}

- (int) readCapacity: (struct readCapacity_reply *) rbuffer
{
	int tmp;
	
	if( scsiOpen)
		{
		return [self readCapacitySCSI: rbuffer];
		}
	else
		{
		if([self	openSCSI] )
			{
			sprintf(errorString,"Can't open SCSI driver");
			return -1;
			}
		tmp = [self readCapacitySCSI: rbuffer];
		[self closeSCSI];
		return( tmp);
		}
}

- (int) readCapacitySCSI: (struct readCapacity_reply *) rbuffer
{	
	struct cdb_10 *cdbp = &sr.sr_cdb.cdb_c10;

	[self clearCommandBlock: (union cdb *) cdbp];
	cdbp->c10_opcode = C10OP_READCAPACITY;
	cdbp->c10_lun    = lun;
	cdbp->c10_len	= 0;
	sr.sr_dma_dir	= SR_DMA_RD;
	sr.sr_addr	= (char *) rbuffer;
	sr.sr_dma_max   = 8;
	sr.sr_ioto	= 10;
	
	return [self performSCSIRequest];
}


- (int)	requestSense: (struct esense_reply *)  rbuffer
{
	int tmp;
	if( scsiOpen)
		{
		return [self requestSenseSCSI: rbuffer];
		}
	else
		{
		if([self	openSCSI] )
			{
			sprintf(errorString,"Can't open SCSI driver");
			return -1;
			}
		tmp = [self requestSenseSCSI: rbuffer];
		[self closeSCSI];
		return( tmp);
		}
}


- (int)	requestSenseSCSI:	(struct esense_reply *)  buffer
{
	struct cdb_6 *cdbp = &sr.sr_cdb.cdb_c6;

	[self clearCommandBlock: (union cdb *) cdbp];
	cdbp->c6_opcode = C6OP_REQSENSE;
	cdbp->c6_lun    = lun;
	cdbp->c6_len	= sizeof(*buffer);
	sr.sr_dma_dir	= SR_DMA_RD;
	sr.sr_addr	= (char *) buffer;
	sr.sr_dma_max   = sizeof(*buffer);
	sr.sr_ioto	= 10;

	return [self performSCSIRequest];
}


- (int)	testUnitReady
{
	int tmp;
	if( scsiOpen)
		return [self testUnitReadySCSI];
	else
		{
		if([self openSCSI] )
			{
			sprintf(errorString,"Can't open SCSI driver");
			return -1;
			}
		tmp = [self testUnitReadySCSI];
		[self closeSCSI];
		return( tmp);
		}
}


- (int) testUnitReadySCSI
{	
	struct cdb_6 *cdbp = &sr.sr_cdb.cdb_c6;

	[self clearCommandBlock: (union cdb *) cdbp];
	cdbp->c6_opcode = C6OP_TESTRDY;
	cdbp->c6_lun    = lun;
	cdbp->c6_len	= 0;
	sr.sr_dma_dir	= SR_DMA_RD;
	sr.sr_addr	= (char *) 0;
	sr.sr_dma_max   = 0;
	sr.sr_ioto	= 10;

	return([self performSCSIRequest]);
}

@end
