/* Program:	fsectbyname - Mach-0 section extraction/insertion tool
 *
 * Author:	Christopher Lane
 *		Symbolic Systems Resources Group
 *		Knowledge Systems Laboratory
 *		Stanford University
 *
 * Date:	28 August 1989
 *
 * Copyright:	1989 by The Leland Stanford Junior University.  This program
 *		may be distributed without restriction for non-commercial use.
 */

#include <c.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/loader.h>

extern long ftell(FILE *stream);

enum arguments {PROGRAM, MODE, FILENAME, SEGMENT, SECTION, ARGC};

void sys_error(char *string)
{
    perror(string);
    exit(EXIT_FAILURE);
}

void main(int argc, char *argv[])
{
    FILE *stream;
    long offset;
    unsigned long ncmds, nsects, size = 0L;
    int c, wflg, Wflg;
    
    struct mach_header m_header;
    struct load_command l_command;
    struct segment_command s_command;
    struct section s_section;
    struct stat f_stat;
    
    if (argc != ARGC) {
        fprintf(stderr, "Usage:  fsectbyname r Application Segment Section > File\n");
        fprintf(stderr, "        fsectbyname w Application Segment Section < File\n\n");
        fprintf(stderr, "Application = Mach-O object file (eg. /NextApps/Preferences)\n");
        fprintf(stderr, "Segment     = segment name (eg. __NIB or __TIFF)\n");
        fprintf(stderr, "Section     = section name (eg. mouse or weeks.tiff)\n\n");
        fprintf(stderr, "mode r/w/W  = r: copy named section to stdout\n");
        fprintf(stderr, "            = w: copy stdin to named section (size must be <= original)\n");
        fprintf(stderr, "            = W: same as 'w' without size restriction (can be used in pipe)\n\n");
        fprintf(stderr, "use 'size -m Application' to lookup segment/section names and sizes\n");
        exit(EXIT_FAILURE);
	}
    
    wflg = (Wflg = argv[MODE][0] == 'W') || argv[MODE][0] == 'w';
    
    if ((stream = fopen(argv[FILENAME], wflg ? "r+" : "r")) == NULL) sys_error("fopen");
    if (fread(&m_header, sizeof(m_header), 1, stream) == 0) sys_error("fread");
    if (m_header.magic != MH_MAGIC) {
	fprintf(stderr, "Incorrect Mach magic number %x!\n", m_header.magic);
	exit(EXIT_FAILURE);
	}
	
    for (ncmds = m_header.ncmds; ncmds > 0; ncmds--) {
	offset = ftell(stream);
	if (fread(&l_command, sizeof(l_command), 1, stream) == 0) sys_error("fread");
        if (l_command.cmd == LC_SEGMENT) {	    
	    if (fseek(stream, offset, SEEK_SET) == CERROR) sys_error("fseek");
	    offset = ftell(stream);  
	    if (fread(&s_command, sizeof(s_command), 1, stream) == 0) sys_error("fread");	    
	    if (strcmp(s_command.segname, argv[SEGMENT]) == 0) {        
		for (nsects = s_command.nsects; nsects > 0; nsects--) {
  	    	    if (fread(&s_section, sizeof(s_section), 1, stream) == 0) sys_error("fread");	    
	            if (strcmp(s_section.sectname, argv[SECTION]) == 0) {	        
			offset = ftell(stream);
			if (fseek(stream, s_section.offset, SEEK_SET) == CERROR) sys_error("fseek");
			if (wflg) {
			    if (Wflg) {
			        while ((c = getc(stdin)) != EOF) { putc(c, stream); size++; }
				s_section.size = size;
				}
			    else {
			        if (fstat(stdin->_file, &f_stat) == CERROR) sys_error("fstat");
			        if ((long) f_stat.st_size > s_section.size) {
			            fprintf(stderr, "Stdin size (%d) larger than Segment:Section %s:%s size (%d)!\n", 
				            f_stat.st_size, argv[SEGMENT], argv[SECTION], s_section.size);
			            exit(EXIT_FAILURE);
				    }
				else s_section.size = f_stat.st_size;
		                for (size = 0L; size < f_stat.st_size; size++) putc(getc(stdin), stream);	    	
			        }
			    if (fseek(stream, offset - sizeof(s_section), SEEK_SET) == CERROR) sys_error("fseek");
			    if (fwrite(&s_section, sizeof(s_section), 1, stream) == 0) sys_error("fwrite");		
			    }
			else for (size = 0L; size < s_section.size; size++) putc(getc(stream), stdout);
			exit(EXIT_SUCCESS);
			}  
		    }
	        if (fseek(stream, s_command.cmdsize + offset, SEEK_SET) == CERROR) sys_error("fseek");
	        }
	    else if (fseek(stream, s_command.cmdsize - sizeof(s_command), SEEK_CUR) == CERROR) sys_error("fseek");
	    }
	else if (fseek(stream, l_command.cmdsize - sizeof(l_command), SEEK_CUR) == CERROR) sys_error("fseek");
        }
    fprintf(stderr, "Requested Segment:Section %s:%s not found!\n", argv[SEGMENT], argv[SECTION]);
    
    exit(EXIT_FAILURE);
}
