/*
 * Distributed as part of the Mach Operating System
 */
/*
 * @OSF_FREE_COPYRIGHT@
 * 
 * Copyright (c) 1990, 1991
 * Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 */
/*
 * HISTORY
 * $Log:	wh.c,v $
 * Revision 2.2  92/05/20  20:15:48  mrt
 * 	Added comment about program's origin
 * 	[92/05/20            mrt]
 * 	First checkin
 * 	[92/05/20  18:09:23  mrt]
 * 
 */
/* This is basically the same as CS's wh.c except it does not
 * use quit ( a libcs feature) and it includes all the code for expand
 * and searchp also from libcs. 
 * 
 * Author: Robert Fitzgerald - Carnegie Mellon University
 * Date:   Mar. 1982
 */
/*
 *  wh -- a program for finding instances of files along paths
 *      Composed from pieces of jag's wh.c for path searching,
 *	uses expand(3) to grok wildcards,
 *	normally uses ls(1) to format output, but
 *	handles emacs(1) error messages (-X) locally.
 *
 *      Options known to wh(1):
 *        name    -- search along current path for instances of name
 *        -f name -- search etc., useful if name starts with '-'
 *        -q      -- quit search after finding first instance of file
 *        -p path -- set path to search along
 *        -C      -- set path to CPATH
 *        -E      -- set path to EPATH
 *        -L      -- set path to LPATH
 *        -M      -- set path to MPATH
 *        -P      -- set path to  PATH
 *        -R      -- recursive directory search
 *	  -X      -- list names in emacs(1) error format
 *        --      -- pass remainder of switch to ls(1)
 *	All other switches (arguments starting with '-') are passed
 *	through as formatting options to ls(1) (collisions on -fqC).
 *
 *	Exit codes:
 *	  0 - if at least 1 one file was found,
 *	  1 - if no files were found,
 *	  2 - if some error was encountered
 */

#ifdef notdef
static char sccsid[] = "@(#) wh.c 3.2  3-Mar-86";
#endif

#include <sys/param.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <stdio.h>
#include <pwd.h>
#include <ctype.h>
#include <setjmp.h>
#include <sys/signal.h>
#include <sys/wait.h>
#if __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#ifdef NO_DIRENT
#define dirent direct
#endif

extern unsigned char errno;
extern char *getenv();
extern char *malloc();

char *progname;			/* program name */
char ls[] = "ls";

int    lsargc;			/* number of free slots in ls arglist */
char **lsargv;			/* pointer to front of entire ls arglist */
char **lsargf;			/* pointer to first file name in ls arglist */
char **lsargp;			/* pointer to free slots in ls arglist */
char **lsargs;			/* pointer to saved spot in ls arglist */

char *pathname;
char *path;
char *givenname;
char  namebuffer[256];

int   qflag;			/* quit after first find(s) when set */
int   Rflag;			/* recursive directory search when set */
int   Xflag;			/* emacs(1) format error messages when set */

int   namehits;
int   totalhits;


/*
 *  Management of argument list to ls(1).
 *  LSARGSLACK must be at least 1 to allow room for NULL terminator.
 *  Slots in use are at the beginning (lsargp-lsargv) and end (LSARGSLACK).
 *  List is doubled in size each time it is grown.
 */
#define LSARGSLACK  2
#define LSARGINIT   126
#define lsargused   ((lsargp-lsargv)+LSARGSLACK)
#define lsargnext(cur) (cur<<1)


/*
 *  newpath -- takes the string name of a path (e.g. "PATH"),
 *      looks it up in the environment and remembers pointers
 *      to both pathname and path in global variables.
 *      Make sure storage for pathname and path remains valid,
 *	as only pointers are preserved here.
 */
newpath(fpathname)
    char *fpathname;
  {
    pathname = fpathname;
    path = getenv(fpathname);
    if (!path) {
	fprintf(stderr, "%s:  path %s not found in environment\n",
		progname, pathname);
        exit(2);
    }
  }


/*
 *  lsarggrow -- increase size of arg list to expand(3) and ls(1).
 *	Copies front of old arglist into new arglist.
 *	Updates lsargc, lsargv, lsargf, lsargs, lsargp for new arg list.
 */
lsarggrow(fullname)
    char *fullname;
  {
    register char **oldp;
    register char **newp;

    lsargc = lsargnext(lsargc+lsargused);
    oldp = lsargv;
    newp = (char **)malloc(lsargc*sizeof(char *));
    if (newp <= (char **)0) {
	fprintf(stderr, "%s:  ran out of free space expanding %s\n",
		progname, fullname);
	exit(2);
    }
    lsargf = newp + (lsargf - lsargv);
    lsargs = newp + (lsargs - lsargv);
    lsargv = newp;
    while (oldp < lsargp)
        *newp++ = *oldp++;
    lsargp = newp;
    lsargc -= lsargused;
  }


/*
 *  runls -- prints the entries in lsargv
 *	either here (for -X) or by invoking ls(1)
 *	unwinds lsargv back to lsargf.
 */
runls()
  {
    if (Xflag)
      {
        char **lead;
        for (lead = lsargs; *lead; lead++)
            printf("\"%s\", line 1:\n", *lead);
      }
    else
      {
        int retcode;

        retcode = runvp(ls, lsargv);
        if (-1 == retcode) {
	    fprintf(stderr, "%s:  error %d in executing %s\n", progname, errno, ls);
	    exit(2);
	}
        else if (0 < retcode) {
	    fprintf(stderr, "%s:  %s returned status %d\n", progname, ls, retcode);
	    exit(2);
	}
      }
    lsargc += (lsargp - lsargf);
    lsargp = lsargf;
  }

/*
 *  lookup -- takes a full file name, looks it up and
 *	generates output for anything found.
 *      Records successful lookups by incrementing namehits and totalhits.
 *      Returns 0 on a hit with qflag set, otherwise returns non-zero
 */
int lookup(fullname)
    char *fullname;
  {
    lsargs = lsargp;
    if (*fullname != '\0')
	    strcat(fullname, "/");
    reclookup(fullname);
    if (lsargp == lsargs)
        return(1);

#ifndef LAZYLS
    runls();
#endif
    namehits++;
    totalhits++;
    return(!qflag);
  }

/*
 *  reclookup --
 */
reclookup(fullname)
    char *fullname;
  {
    int found;
    char **lead;

    if (Rflag)
      {
        int pathlen;
        DIR *dirp;

        pathlen = strlen(fullname);
        if (pathlen > 0)
          {
            fullname[pathlen-1] = NULL;
            if (NULL == (dirp = opendir(fullname)))
                fprintf(stderr, "couldn't open \"%s\"\n", fullname);
            strcat(fullname,"/");
          }
        else
          {
            if (NULL == (dirp = opendir(".")))
                fprintf(stderr, "couldn't open \".\"\n");
          }
        if (NULL != dirp)
          {
            struct dirent *dp;

            while (NULL != (dp = readdir(dirp)))
              {
                if ( (0 != dp->d_ino) &&
                     (('.' != dp->d_name[0])) &&
                     (0 != strcmp(dp->d_name, ".")) &&
                     (0 != strcmp(dp->d_name, "..")) )
                  {
                    struct stat statbuf;
                    strcat(fullname, dp->d_name);
                    if (-1 != stat(fullname, &statbuf))
                      {
                        if (S_IFDIR & statbuf.st_mode)
                          {
                            strcat(fullname, "/");
                            reclookup(fullname);
                          }
                      }
                    else fprintf(stderr, "%s:  can't open directory %s\n", progname, fullname);
                    fullname[pathlen] = NULL;
                  }
              }
            closedir(dirp);
          }
      }

    /*
     *  expand wildcards
     *  return non-zero for nothing found
     *  check for expansion errors (expand apparently returns -1 for
     *    too-many-names, not lsargc+1 as is documented)
     *  NULL-terminate parameter list
     */
    strcat(fullname, givenname);
    found = expand(fullname,lsargp,lsargc);
    while ( (found < 0) || (lsargc < found) )
      {
        lsarggrow(fullname);
        found = expand(fullname,lsargp,lsargc);
      }
    *(lsargp+found) = 0;

    /*
     *  scan expanded list, making sure that the files really exist
     *  (since expand doesn't bother to check while expanding wildcards
     *  in directory names in the middle of paths).
     *  compress any bogus entries (before ls(1) gets them and prints
     *  some idiot message about file not existing).
     *  Check again for no acceptable files, returning non-zero if so
     */
    lead = lsargp;
    while (*lsargp = *lead)
      {
        static struct stat buf;
        if (!stat(*lead++, &buf))
          {
            lsargp++;
            lsargc--;
          }
      }
  }


/*
 *  searchpath -- look for instances of filename on path recorded
 *      in global variable path.  Gripe if nothing found.
 *	Global givenname is used to pass filename into lookup.
 *	Global namehits is incremented by lookup when appropriate.
 */
searchpath(filename)
    char *filename;
  {
    namehits = 0;
    givenname = filename;
    searchp(path, "", namebuffer, lookup);
    if (!namehits)
        fprintf(stderr, "%s:  %s not found on %s\n", progname, filename, pathname);
  }


/*
 *  switchblock -- parse one switch block, being given a pointer to
 *	the first character after the '-'.
 */
switchblock(swp)
    char *swp;
  {
    char option;
    char *leadp  = swp;			/* next option to look at */
    char *trailp = swp;			/* next place to put ls(1) option */

    /*
     *  Scan over switches in block
     *  processing those we know (qCLMPRX) and eliminating them,
     *  compacting those we doesn't know about for later.
     */
    while (option = *leadp++)
      {
        switch (option)
          {
          case 'q': qflag++; break;
          case 'R': Rflag++; break;
          case 'X': Xflag++; break;
          case 'C': newpath("CPATH"); break;
#ifdef	CMUCS
          case 'E': newpath("EPATH"); break;
#endif
          case 'L': newpath("LPATH"); break;
          case 'M': newpath("MPATH"); break;
          case 'P': newpath( "PATH"); break;
          default : *trailp++ = option; break;
          }
      }

    /*
     *  If anything remains to be passed to ls(1),
     *  NULL terminate the switch block and back up over the '-'
     *  before appending block to the ls arg list.
     */
    if (trailp != swp)
      {
#ifdef LAZYLS
        if (lsargf != lsargp)
            runls();
#endif
        *trailp++ = 0;
        *lsargp++ = swp-1;
        lsargc--;
        lsargf = lsargp;
      }
  }


/*
 *  Main program -- parse command line
 */
main(argc, argv)
    int    argc;
    char **argv;
  {
    progname = argv[0];
    if (1 == argc) {
	print_usage();	
	exit(2);
    }

    totalhits = 0;
    newpath("PATH");
    lsargc = LSARGINIT;
    lsargp = lsargv = 0;
    lsarggrow(ls);
    *lsargp++ = ls;
    lsargc--;
    lsargf = lsargp;
    while (0 < --argc)
      {
        if ('-' == **++argv)
          {
            switch (*++*argv)
              {

	      case 'v':
		  print_revision();
		  exit(0);
		  break;

              case 'p':
                if (0 >= --argc) {
		    fprintf(stderr, "%s:  path name expected after -p\n", progname);
		    exit(2);
		}
                newpath(*++argv);
              break;

              case 'f':
                if (0 >= --argc) {
		    fprintf(stderr, "%s:  file name expected after -f\n", progname);
		    exit(2);
		}
                searchpath(*++argv);
              break;

              case '-':
#ifdef LAZYLS
                if (lsargf != lsargp)
                    runls();
#endif
                *lsargp++ = *argv;
                lsargc--;
                lsargf = lsargp;
              break;

              default :
                switchblock(*argv);
              break;
              }
          }
         else
            searchpath(*argv);
      }
#ifdef LAZYLS
    if (lsargf != lsargp)
        runls();
#endif
    exit(totalhits == 0);
  }

/*
 *  expand - expand wildcard filename specifications
 *
 *  Usage:
 *	int expand(spec, buffer, bufsize);
 *	char *spec, **buffer;
 *	int bufsize;
 *
 *  Expand takes a file specification, and expands it into filenames
 *  by resolving the characters '*', '?', '[', ']', '{', '}' and '~'
 *  in the same manner as the shell.  You provide "buffer", which is
 *  an array of char *'s, and you tell how big it is in bufsize.
 *  Expand will compute the corresponding filenames, and will fill up
 *  the entries of buffer with pointers to malloc'd strings.
 *
 *  The value returned by expand is the number of filenames found.  If
 *  this value is -1, then malloc failed to allocate a string.  If the
 *  value is bufsize + 1, then too many names were found and you can try
 *  again with a bigger buffer.
 *
 *  This routine was basically created from the csh sh.glob.c file with
 *  the following intended differences:
 *
 *	Filenames are not sorted.
 *	All expanded filenames returned exist.
 */

static	jmp_buf	sjbuf;

static	char	pathbuf[MAXPATHLEN];
static	char	*pathptr, *pathp, *lastpathp;

static	char	*globchars = "{[*?";	/* meta characters */
static	char	*entp;			/* current dir entry pointer */

static	char	**BUFFER;		/* pointer to the buffer */
static	int	BUFSIZE;		/* maximum number in buffer */
static	int	bufcnt;			/* current number in buffer */

static glob(), matchdir(), execbrc(), match(), amatch(), addone(),
	addpath(), gethdir(), dorun();

int expand(spec, buffer, bufsize)
	register char *spec;
	char **buffer;
	int bufsize;
{
	pathp = pathptr = pathbuf;
	*pathp = 0;
	lastpathp = &pathptr[MAXPATHLEN - 2];
	BUFFER = buffer;
	BUFSIZE = bufsize;
	bufcnt = 0;
	if (setjmp(sjbuf) == 0)
	    glob(spec);
	return(bufcnt);
}

static glob(as)
	char *as;
{
	register char *cs;
	register char *spathp, *oldcs;
	struct stat stb;

	spathp = pathp;
	cs = as;
	if (*cs == '~' && pathp == pathptr) {
		if (addpath('~')) goto endit;
		for (cs++; isalnum(*cs) || *cs == '_' || *cs == '-';)
			if (addpath(*cs++)) goto endit;
		if (!*cs || *cs == '/') {
			if (pathp != pathptr + 1) {
				*pathp = 0;
				if (gethdir(pathptr + 1)) goto endit;
				strcpy(pathptr, pathptr + 1);
			} else
				strcpy(pathptr, getenv("HOME"));
			pathp = pathptr;
			while (*pathp) pathp++;
		}
	}
	while (*cs == 0 || index(globchars, *cs) == 0) {
		if (*cs == 0) {
			if (lstat(pathptr, &stb) >= 0) addone(pathptr, "");
			goto endit;
		}
		if (addpath(*cs++)) goto endit;
	}
	oldcs = cs;
	while (cs > as && *cs != '/')
		cs--, pathp--;
	if (*cs == '/')
		cs++, pathp++;
	*pathp = 0;
	if (*oldcs == '{') {
		execbrc(cs, NULL);
		return;
	}
	/* this should not be an lstat */
	if (stat(pathptr, &stb) >= 0 && (stb.st_mode&S_IFMT) == S_IFDIR)
		matchdir(cs);
endit:
	pathp = spathp;
	*pathp = 0;
	return;
}

static matchdir(pattern)
	char *pattern;
{
	register struct dirent *dp;
	DIR *dirp;

	dirp = opendir(pathptr);
	if (dirp == NULL)
		return;
	while ((dp = readdir(dirp)) != NULL) {
		if (dp->d_ino == 0) continue;
		if (match(dp->d_name, pattern))
			addone(pathptr, dp->d_name);
	}
	closedir(dirp);
	return;
}

static execbrc(p, s)
	char *p, *s;
{
	char restbuf[MAXPATHLEN + 1];
	register char *pe, *pm, *pl;
	int brclev = 0;
	char *lm, savec, *spathp;

	for (lm = restbuf; *p != '{'; *lm++ = *p++)
		continue;
	for (pe = ++p; *pe; pe++)
	switch (*pe) {
	case '{':
		brclev++;
		continue;
	case '}':
		if (brclev == 0) goto pend;
		brclev--;
		continue;
	case '[':
		for (pe++; *pe && *pe != ']'; pe++)
			continue;
		if (!*pe) break;
		continue;
	}
pend:
	if (brclev || !*pe) return (0);
	for (pl = pm = p; pm <= pe; pm++)
		switch (*pm & 0177) {
		case '{':
			brclev++;
			continue;
		case '}':
			if (brclev) {
				brclev--;
				continue;
			}
			goto doit;
		case ',':
			if (brclev) continue;
doit:
			savec = *pm;
			*pm = 0;
			strcpy(lm, pl);
			strcat(restbuf, pe + 1);
			*pm = savec;
			if (s == 0) {
				spathp = pathp;
				glob(restbuf);
				pathp = spathp;
				*pathp = 0;
			} else if (amatch(s, restbuf))
				return (1);
			pl = pm + 1;
			continue;

		case '[':
			for (pm++; *pm && *pm != ']'; pm++)
				continue;
			if (!*pm) break;
			continue;
		}
	return (0);
}

static match(s, p)
	char *s, *p;
{
	register int c;
	register char *sentp;

	if (*s == '.' && *p != '.') return(0);
	sentp = entp;
	entp = s;
	c = amatch(s, p);
	entp = sentp;
	return (c);
}

static amatch(s, p)
	register char *s, *p;
{
	register int scc;
	int ok, lc;
	char *spathp;
	struct stat stb;
	int c, cc;

	for (;;) {
		scc = *s++ & 0177;
		switch (c = *p++) {
		case '{':
			return (execbrc(p - 1, s - 1));
		case '[':
			ok = 0;
			lc = 077777;
			while (cc = *p++) {
				if (cc == ']') {
					if (ok) break;
					return (0);
				}
				if (cc == '-') {
					if (lc <= scc && scc <= *p++)
						ok++;
				} else
					if (scc == (lc = cc))
						ok++;
			}
			if (cc == 0) return (0);
			continue;
		case '*':
			if (!*p) return (1);
			if (*p == '/') {
				p++;
				goto slash;
			}
			for (s--; *s; s++)
				if (amatch(s, p))
					return (1);
			return (0);
		case 0:
			return (scc == 0);
		default:
			if (c != scc) return (0);
			continue;
		case '?':
			if (scc == 0) return (0);
			continue;
		case '/':
			if (scc) return (0);
slash:
			s = entp;
			spathp = pathp;
			while (*s)
				if (addpath(*s++)) goto pathovfl;
			if (addpath('/')) goto pathovfl;
			if (stat(pathptr, &stb) >= 0 &&
			    (stb.st_mode&S_IFMT) == S_IFDIR)
				if (*p == 0)
					addone(pathptr, "");
				else
					glob(p);
pathovfl:
			pathp = spathp;
			*pathp = 0;
			return (0);
		}
	}
}

static addone(s1, s2)
	register char *s1, *s2;
{
	register char *ep;

	if (bufcnt >= BUFSIZE) {
		bufcnt = BUFSIZE + 1;
		longjmp(sjbuf, 1);
	}
	ep = malloc(strlen(s1) + strlen(s2) + 1);
	if (ep == 0) {
		bufcnt = -1;
		longjmp(sjbuf, 1);
	}
	BUFFER[bufcnt++] = ep;
	while (*s1) *ep++ = *s1++;
	while (*ep++ = *s2++);
}

static addpath(c)
	char c;
{
	if (pathp >= lastpathp)
		return(1);
	*pathp++ = c;
	*pathp = 0;
	return(0);
}

static gethdir(home)
	char *home;
{
	struct passwd *getpwnam();
	register struct passwd *pp = getpwnam(home);

	if (pp == 0)
		return(1);
	strcpy(home, pp->pw_dir);
	return(0);
}

/*  searchp  --  search through pathlist for file
 *
 *  Usage:  p = searchp (path,file,fullname,func);
 *	char *p, *path, *file, *fullname;
 *	int (*func)();
 *
 *  Searchp will parse "path", a list of pathnames separated
 *  by colons, prepending each pathname to "file".  The resulting
 *  filename will be passed to "func", a function provided by the
 *  user.  This function must return zero if the search is
 *  successful (i.e. ended), and non-zero if the search must
 *  continue.  If the function returns zero (success), then
 *  searching stops, the full filename is placed into "fullname",
 *  and searchp returns 0.  If the pathnames are all unsuccessfully
 *  examined, then searchp returns -1.
 *  If "file" begins with a slash, it is assumed to be an
 *  absolute pathname and the "path" list is not used.  Note
 *  that this rule is used by Bell's cc also; whereas Bell's
 *  sh uses the rule that any filename which CONTAINS a slash
 *  is assumed to be absolute.  The execlp and execvp procedures
 *  also use this latter rule.  In my opinion, this is bogosity.
 */

int searchp (path,file,fullname,func)
char *path,*file,*fullname;
int (*func)();
{
	register char *nextpath,*nextchar,*fname,*lastchar;
	int failure;

	nextpath = ((*file == '/') ? "" : path);
	do {
		fname = fullname;
		nextchar = nextpath;
		while (*nextchar && (*nextchar != ':'))
			*fname++ = *nextchar++;
		if (nextchar != nextpath && *file) *fname++ = '/';
		lastchar = nextchar;
		nextpath = ((*nextchar) ? nextchar + 1 : nextchar);
		nextchar = file;	/* append file */
		while (*nextchar)  *fname++ = *nextchar++;
		*fname = '\0';
		failure = (*func) (fullname);
	} 
	while (failure && (*lastchar));
	return (failure ? -1 : 0);
}

/*
 *  run, runv, runp, runvp --  execute process and wait for it to exit
 *
 *  Usage:
 *	i = run (file, arg1, arg2, ..., argn, 0);
 *	i = runv (file, arglist);
 *	i = runp (file, arg1, arg2, ..., argn, 0);
 *	i = runvp (file, arglist);
 *
 *  Run, runv, runp and runvp have argument lists exactly like the
 *  corresponding routines, execl, execv, execlp, execvp.  The run
 *  routines perform a fork, then:
 *  IN THE NEW PROCESS, an execl[p] or execv[p] is performed with the
 *  specified arguments.  The process returns with a -1 code if the
 *  exec was not successful.
 *  IN THE PARENT PROCESS, the signals SIGQUIT and SIGINT are disabled,
 *  the process waits until the newly forked process exits, the
 *  signals are restored to their original status, and the return
 *  status of the process is analyzed.
 *  All run routines return:  -1 if the exec failed or if the child was
 *  terminated abnormally; otherwise, the exit code of the child is
 *  returned.
 */

#if	VA_ARGV_IS_RECAST
#define va_argv(list) ((char **) list)
#else
#if	!VA_ARGV_IS_ROUTINE
error Please define va_argv() macro(?) for your machine!!
#endif
#endif

#if __STDC__
int run (char *name, ...)
#else
int run (va_alist)
va_dcl
#endif
{
#if !__STDC__
	char *name;
#endif
	va_list ap;
	int val;

#if __STDC__
	va_start(ap, name);
#else
	va_start(ap);
	name = va_arg(ap, char *);
#endif
	val = runv (name, va_argv(ap));
	va_end(ap);
	return(val);
}

int runv (name,argv)
char *name,**argv;
{
	return (dorun (name, argv, 0));
}

#if __STDC__
int runp (char *name, ...)
#else
int runp (va_alist)
va_dcl
#endif
{
#if !__STDC__
	char *name;
#endif
	va_list ap;
	int val;

#if __STDC__
	va_start(ap, name);
#else
	va_start(ap);
	name = va_arg(ap, char *);
#endif
	val = runvp (name, va_argv(ap));
	va_end(ap);
	return (val);
}

int runvp (name,argv)
char *name,**argv;
{
	return (dorun (name, argv, 1));
}

static
int dorun (name,argv,usepath)
char *name,**argv;
int usepath;
{
	int wpid;
	register int pid;
	struct sigvec ignoresig,intsig,quitsig;
	union wait status;
	int execvp(), execv();
	int (*execrtn)() = usepath ? execvp : execv;

	if ((pid = vfork()) == -1)
		return(-1);	/* no more process's, so exit with error */

	if (pid == 0) {			/* child process */
		setgid (getgid());
		setuid (getuid());
		(*execrtn) (name,argv);
		fprintf (stderr,"run: can't exec %s\n",name);
		_exit (0377);
	}

	ignoresig.sv_handler = SIG_IGN;	/* ignore INT and QUIT signals */
	ignoresig.sv_mask = 0;
	ignoresig.sv_onstack = 0;
	sigvec (SIGINT,&ignoresig,&intsig);
	sigvec (SIGQUIT,&ignoresig,&quitsig);
	do {
		wpid = wait3 (&status, WUNTRACED, 0);
		if (WIFSTOPPED (status)) {
		    kill (0,SIGTSTP);
		    wpid = 0;
		}
	} while (wpid != pid && wpid != -1);
	sigvec (SIGINT,&intsig,0);	/* restore signals */
	sigvec (SIGQUIT,&quitsig,0);

	if (WIFSIGNALED (status) || status.w_retcode == 0377)
		return (-1);

	return (status.w_retcode);
}


/* show invocation options */
print_usage()
{
    fprintf(stderr,
#if	CMUCS
	    "usage:  %s { -qCELMPRX | -version | -p path | -f file | file }\n",
#else
	    "usage:  %s { -qCLMPRX | -version | -p path | -f file | file }\n",
#endif
	    progname);
}


/* show the revision of this program */
print_revision()
{
    printf("%s $Revision: 2.2 $ $Date: 92/05/20 20:15:48 $\n", progname);
}
