/* Interface GDB to Mach 3.0 operating systems.
   (Most) Mach 3.0 related routines live in this file.

   Copyright (C) 1992 Free Software Foundation, Inc.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/*
 * Author: Jukka Virtanen <jtv@hut.fi>
 *	   Computing Centre
 *         Helsinki University of Technology
 *         Finland
 *
 * Thanks to my friends who helped with ideas and testing:
 *
 *	Johannes Helander, Antti Louko, Tero Mononen and Tero Kivinen
 *	jvh@cs.hut.fi	   alo@hut.fi   tmo@cs.hut.fi    kivinen@cs.hut.fi
 */

#include <stdio.h>

#include <mach.h>
#include <servers/netname.h>
#include <servers/machid.h>
#include <servers/machid_lib.h>
#include <mach/message.h>
#include <mach/notify.h>
#include <mach_error.h>
#include <mach/exception.h>
#include <mach/vm_attributes.h>

#include <obstack.h>

/* For EMULATOR_BASE and EMULATOR_END */
#include <machine/vmparam.h>

#include "defs.h"
#include "inferior.h"
#include "symtab.h"
#include "value.h"
#include "language.h"
#include "target.h"
#include "wait.h"
#include "gdb-threads.h"

/* Included only for signal names and NSIG
 *
 * note: There are many problems in signal handling with
 *       gdb in Mach 3.0 in general.
 */
#include <signal.h>
#define SIG_UNKNOWN 0	/* Exception that has no matching unix signal */

#define private static

/* 
 * Actions for Mach exceptions.
 *
 * sigmap field maps the exception to corresponding Unix signal.
 *
 * I do not know how to map the exception to unix signal
 * if SIG_UNKNOWN is specified.
 */

struct exception_list {
  char *name;
  boolean_t forward;
  boolean_t print;
  int       sigmap;
} exception_map[] = {
  {"not_mach3_exception",	FALSE, TRUE,  SIG_UNKNOWN},
  {"EXC_BAD_ACCESS",		FALSE, TRUE,  SIGSEGV},
  {"EXC_BAD_INSTRUCTION",	FALSE, TRUE,  SIGILL},
  {"EXC_ARITHMETIC",		FALSE, TRUE,  SIGFPE},
  {"EXC_EMULATION",		FALSE, TRUE,  SIGEMT},	/* ??? */
  {"EXC_SOFTWARE",		FALSE, TRUE,  SIG_UNKNOWN},
  {"EXC_BREAKPOINT",		FALSE, FALSE, SIGTRAP}
};

/* Mach exception table size */
int max_exception = sizeof(exception_map)/sizeof(struct exception_list) - 1;

#define MAX_EXCEPTION max_exception

WAITTYPE wait_status;

/* If you define this, intercepted bsd server calls will be
 * dumped while waiting the inferior to EXEC the correct
 * program
 */
/* #define DUMP_SYSCALL		/* debugging interceptor */


/* debug() printfs if nonzero.
 * If > 1, DUMP_SYSCALL will dump message contents.
 */
int debug_level = 0;

/* "Temporary" debug stuff */
void
debug (fmt, a,b,c)
char *fmt;
int a,b,c;
{
  if (debug_level)
    message (fmt, a, b, c);
}

/* This is in libmach.a */
extern  mach_port_t  name_server_port;

/* Set in catch_exception_raise */
int stop_exception, stop_code, stop_subcode;
int stopped_in_exception;

char *hostname = "";

/* Set when task is attached or created */
boolean_t emulator_present = FALSE;

task_t   inferior_task;
thread_t current_thread;

/* Exception ports for inferior task */
mach_port_t inferior_exception_port     = MACH_PORT_NULL;
mach_port_t inferior_old_exception_port = MACH_PORT_NULL;

/* task exceptions and notifications */
mach_port_t inferior_wait_port_set	= MACH_PORT_NULL;
mach_port_t our_notify_port       	= MACH_PORT_NULL;

/* A port for external messages to gdb.
 * External in the meaning that they do not come
 * from the inferior_task, but rather from external
 * tasks.
 *
 * As a debugging feature:
 * A debugger debugging another debugger can stop the
 * inferior debugger by the following command sequence
 * (without running external programs)
 *
 *    (top-gdb) set stop_inferior_gdb ()
 *    (top-gdb) continue
 */
mach_port_t our_message_port       	= MACH_PORT_NULL;

/* For single stepping */
mach_port_t thread_exception_port 	= MACH_PORT_NULL;
mach_port_t thread_saved_exception_port = MACH_PORT_NULL;
mach_port_t singlestepped_thread_port   = MACH_PORT_NULL;

/* For machid calls */
mach_port_t mid_server = MACH_PORT_NULL;
mach_port_t mid_auth   = MACH_PORT_NULL;

#if 0
#define MACH_TYPE_EXCEPTION_PORT	-1
#endif

/* Chain of ports to remember requested notifications. */

struct port_chain {
  struct port_chain *next;
  mach_port_t	     port;
  int		     type;
  int		     mid;  /* Now only valid with MACH_TYPE_THREAD and */
  			   /*  MACH_TYPE_THREAD */
};
typedef struct port_chain *port_chain_t;

/* Room for chain nodes comes from pchain_obstack */
struct obstack pchain_obstack;
struct obstack *port_chain_obstack = &pchain_obstack;

/* For thread handling */
struct obstack Cproc_obstack;
struct obstack *cproc_obstack = &Cproc_obstack;

/* the list of notified ports */
port_chain_t notify_chain = (port_chain_t) NULL;

port_chain_t
port_chain_insert (list, name, type)
     port_chain_t list;
     mach_port_t name;
     int	 type;
{
  kern_return_t ret;
  port_chain_t new;
  int mid;

  if (! MACH_PORT_VALID (name))
    return list;
  
  if (type == MACH_TYPE_TASK || type == MACH_TYPE_THREAD)
    {
      if (! MACH_PORT_VALID (mid_server))
	{
	  message ("Machid server port invalid, can not map port 0x%x to MID", name);
	  mid = name;
	}
      else
	{
	  ret = machid_mach_register (mid_server, mid_auth, name, type, &mid);
	  
	  if (ret != KERN_SUCCESS)
	    {
	      message ("Can not map name (0x%x) to MID with machid", name);
	      mid = name;
	    }
	}
    }
  else
    mid = 3735928559;	/* 0x?-) */

  new = (port_chain_t) obstack_alloc (port_chain_obstack,
				      sizeof (struct port_chain));
  new->next  = list;
  new->port  = name;
  new->type  = type;
  new->mid   = mid;

  return new;
}

port_chain_t
port_chain_delete (list, elem)
     port_chain_t list;
     mach_port_t elem;
{
  if (list)
    if (list->port == elem)
      list = list->next;
    else
      while (list->next)
	{
	  if (list->next->port == elem)
	    list->next = list->next->next; /* GCd with obstack_free() */
	  else
	    list = list->next;
	}
  return list;
}

void
port_chain_destroy (ostack)
     struct obstack *ostack;
{
  obstack_free (ostack, 0);
  obstack_init (ostack);
}

port_chain_t
port_chain_member (list, elem)
     port_chain_t list;
     mach_port_t elem;
{
  while (list)
    {
      if (list->port == elem)
	return list;
      list = list->next;
    }
  return (port_chain_t) NULL;
}

int
map_port_name_to_mid (name, type)
mach_port_t name;
int         type;
{
  port_chain_t elem;

  if (!MACH_PORT_VALID (name))
    return -1;

  elem = port_chain_member (notify_chain, name);

  if (elem && (elem->type == type))
    return elem->mid;
  
  if (elem)
    return -1;
  
  if (! MACH_PORT_VALID (mid_server))
    {
      message ("Machid server port invalid, can not map port 0x%x to mid",
	       name);
      return -1;
    }
  else
    {
      int mid;
      kern_return_t ret;

      ret = machid_mach_register (mid_server, mid_auth, name, type, &mid);
      
      if (ret != KERN_SUCCESS)
	{
	  message ("Can not map name (0x%x) to mid with machid", name);
	  return -1;
	}
      return mid;
    }
}

private void
setup_single_step (thread, start_step)
     thread_t  thread;
     boolean_t start_step;
{
  kern_return_t ret;

  if (! MACH_PORT_VALID (thread))
    error ("Invalid thread supplied to setup_single_step");
  else
    {
      mach_port_t teport;

      /* Get the current thread exception port */
      ret = thread_get_exception_port (thread, &teport);
      CHK ("Getting thread's exception port", ret);
	  
      if (start_step)
	{
	  if (MACH_PORT_VALID (singlestepped_thread_port))
	    {
	      message ("Singlestepped_thread_port (0x%x) is still valid?",
		       singlestepped_thread_port);
	      singlestepped_thread_port = MACH_PORT_NULL;
	    }
      
	  /* If we are already stepping this thread */
	  if (MACH_PORT_VALID (teport) && teport == thread_exception_port)
	    {
	      ret = mach_port_deallocate (mach_task_self (), teport);
	      CHK ("Could not deallocate thread exception port", ret);
	    }
	  else
	    {
	      ret = thread_set_exception_port (thread, thread_exception_port);
	      CHK ("Setting exception port for thread", ret);
	      
	      /* Insert thread exception port to wait port set */
	      ret = mach_port_move_member (mach_task_self(), 
					   thread_exception_port,
					   inferior_wait_port_set);
	      CHK ("Moving thread exception port to inferior_wait_port_set",
		   ret);

	      thread_saved_exception_port = teport;
	    }
	  
	  thread_trace (thread, TRUE);
	  
	  singlestepped_thread_port   = thread_exception_port;
	}
      else
	{
	  if (! MACH_PORT_VALID (teport))
	    error ("Single stepped thread had an invalid exception port?");

	  if (teport != thread_exception_port)
	    error ("Single stepped thread had an unknown exception port?");
	  
	  ret = mach_port_deallocate (mach_task_self (), teport);
	  CHK ("Couldn't deallocate thread exception port", ret);

	  /* Remove thread exception port from wait port set */
	  ret = mach_port_move_member (mach_task_self(), 
				       thread_exception_port,
				       MACH_PORT_NULL);
	  CHK ("Removing thread exception port from inferior_wait_port_set",
	       ret);
	  
	  /* Restore thread's old exception port */
	  ret = thread_set_exception_port (thread,
					   thread_saved_exception_port);
	  CHK ("Restoring stepped thread's exception port", ret);
	  
	  if (MACH_PORT_VALID (thread_saved_exception_port))
	    (void) mach_port_deallocate (mach_task_self (),
					 thread_saved_exception_port);
	  
	  thread_trace (thread, FALSE);
	  
	  singlestepped_thread_port = MACH_PORT_NULL;
	}
    }
}

private
request_notify (name, variant, type)
     mach_port_t	name;
     mach_msg_id_t	variant;
     int	        type;
{
  kern_return_t ret;
  mach_port_t	previous_port_dummy = MACH_PORT_NULL;
  
  if (! MACH_PORT_VALID (name))
    return;
  
  if (port_chain_member (notify_chain, name))
    return;

  ret = mach_port_request_notification (mach_task_self(),
					name,
					variant,
					1,
					our_notify_port,
					MACH_MSG_TYPE_MAKE_SEND_ONCE,
					&previous_port_dummy);
  CHK ("Serious: request_notify failed", ret);

  (void) mach_port_deallocate (mach_task_self (),
			       previous_port_dummy);

  notify_chain = port_chain_insert (notify_chain, name, type);
}

reverse_msg_bits(msgp, type)
     mach_msg_header_t	*msgp;
     int type;
{
  int		rbits,lbits;
  rbits = MACH_MSGH_BITS_REMOTE(msgp->msgh_bits);
  lbits = type;
  msgp->msgh_bits =
    (msgp->msgh_bits & ~MACH_MSGH_BITS_PORTS_MASK) |
      MACH_MSGH_BITS(lbits,rbits);
}

/* On the third day He said:

   	Let this be global
	and then it was global.

   When creating the inferior fork, the
   child code in inflow.c sets the name of the
   bootstrap_port in its address space to this
   variable.

   The name is transferred to our address space
   with mach3_read_inferior().

   Thou shalt not do this with
   task_get_bootstrap_port() in this task, since
   the name in the inferior task is different than
   the one we get.

   For blessed are the meek, as they shall inherit
   the address space.
 */
mach_port_t original_server_port_name = MACH_PORT_NULL;


/* Called from inferior after FORK but before EXEC */
prepare_inferior_task ()
{
  kern_return_t ret;
  
  /* Get the NAME of the bootstrap port in this task
     so that GDB can read it */
  ret = task_get_bootstrap_port (mach_task_self (),
				 &original_server_port_name);
  if (ret != KERN_SUCCESS)
    abort ();
  ret = mach_port_deallocate (mach_task_self (),
			      original_server_port_name);
  if (ret != KERN_SUCCESS)
    abort ();
  
  /* Suspend this task to let the parent change my ports.
     Resumed by the debugger */
  ret = task_suspend (mach_task_self ());
  if (ret != KERN_SUCCESS)
    abort ();
}

/*
 * Intercept system calls to Unix server.
 * After EXEC_COUNTER calls to exec(), return.
 *
 * Pre-assertion:  Child is suspended. (NOT VERIFIED)
 * Post-condition: Child is suspended after EXEC_COUNTER exec() calls.
 */

void
intercept_exec_calls (exec_counter)
     int exec_counter;
{
  struct syscall_msg_t {
    mach_msg_header_t	header;
    mach_msg_type_t 	type;
    char room[ 2000 ];	/* Enuff space */
  };

  struct syscall_msg_t syscall_in, syscall_out;

  mach_port_t fake_server;
  mach_port_t original_server_send;
  mach_port_t original_exec_reply;
  mach_port_t exec_reply;
  mach_port_t exec_reply_send;
  mach_msg_type_name_t acquired;
  mach_port_t emulator_server_port_name;
  struct task_basic_info info;
  mach_msg_type_number_t info_count;

  kern_return_t ret;

  if (exec_counter <= 0)
    return;		/* We are already set up in the correct program */

  ret = mach_port_allocate(mach_task_self(), 
			   MACH_PORT_RIGHT_RECEIVE,
			   &fake_server);
  CHK("create inferior_fake_server port failed", ret);
  
  /* Wait for inferior_task to suspend itself */
  while(1)
    {
      info_count = sizeof (info);
      ret = task_info (inferior_task,
		       TASK_BASIC_INFO,
		       (task_info_t)&info,
		       &info_count);
      CHK ("Task info", ret);

      if (info.suspend_count)
	break;
      (void) swtch_pri (42); /* Universal Priority Value */
    }

  /* Read the inferior's bootstrap port name */
  if (!mach3_read_inferior (&original_server_port_name,
			    &original_server_port_name,
			    sizeof (original_server_port_name)))
    error ("Can't read inferior task bootstrap port name");

  /* @@ BUG: If more than 1 send right GDB will FAIL!!! */
  /*      Should get refs, and set them back when restoring */
  /* Steal the original bsd server send right from inferior */
  ret = mach_port_extract_right (inferior_task,
				 original_server_port_name,
				 MACH_MSG_TYPE_MOVE_SEND,
				 &original_server_send,
				 &acquired);
  CHK("mach_port_extract_right (bsd server send)",ret);
  
  if (acquired != MACH_MSG_TYPE_PORT_SEND)
    error("Incorrect right extracted, send right to bsd server excpected");

  ret = mach_port_insert_right (inferior_task,
				original_server_port_name,
				fake_server,
				MACH_MSG_TYPE_MAKE_SEND);
  CHK("mach_port_insert_right (fake server send)",ret);

  debug ("inferior task bsd server ports set up \nfs %x, ospn %x, oss %x\n",
	 fake_server,
	 original_server_port_name, original_server_send);

  /* A receive right to the reply generated by unix server exec() request */
  ret = mach_port_allocate(mach_task_self(), 
			   MACH_PORT_RIGHT_RECEIVE,
			   &exec_reply);
  CHK("create intercepted_reply_port port failed", ret);
    
  /* Pass this send right to Unix server so it replies to us after exec() */
  ret = mach_port_extract_right (mach_task_self (),
				 exec_reply,
				 MACH_MSG_TYPE_MAKE_SEND_ONCE,
				 &exec_reply_send,
				 &acquired);
  CHK("mach_port_extract_right (exec_reply)",ret);

  if (acquired != MACH_MSG_TYPE_PORT_SEND_ONCE)
    error("Incorrect right extracted, send once excpected for exec reply");

  ret = mach_port_move_member(mach_task_self(), 
			      fake_server,
			      inferior_wait_port_set);
  CHK ("Moving fake syscall port to inferior_wait_port_set", ret);

  debug ("syscall fake server set up, resuming inferior\n");
  
  ret = task_resume (inferior_task);
  CHK("task_resume (startup)", ret);
	
  /* Read requests from the inferior.
     Pass directly through everything else except exec() calls.
   */
  while(exec_counter > 0)
    {
      ret = mach_msg (&syscall_in.header,	/* header */
		      MACH_RCV_MSG,		/* options */
		      0,			/* send size */
		      sizeof (struct syscall_msg_t), /* receive size */
		      inferior_wait_port_set,        /* receive_name */
		      MACH_MSG_TIMEOUT_NONE,
		      MACH_PORT_NULL);
      CHK("mach_msg (intercepted sycall)", ret);
	    
#ifdef DUMP_SYSCALL
      print_msg (&syscall_in.header);
#endif

      /* ASSERT : msgh_local_port == fake_server */

      if (notify_server (&syscall_in.header, &syscall_out.header))
	error ("received a notify while intercepting syscalls");

      if (syscall_in.header.msgh_id == MIG_EXEC_SYSCALL_ID)
	{
	  debug ("Received EXEC SYSCALL, counter = %d\n", exec_counter);
	  if (exec_counter == 1)
	    {
	      original_exec_reply = syscall_in.header.msgh_remote_port;
	      syscall_in.header.msgh_remote_port = exec_reply_send;
	    }
	  exec_counter--;
	}
	    
      syscall_in.header.msgh_local_port  = syscall_in.header.msgh_remote_port;
      syscall_in.header.msgh_remote_port = original_server_send;

      reverse_msg_bits(&syscall_in.header, MACH_MSG_TYPE_COPY_SEND);

      ret = mach_msg_send (&syscall_in.header);
      CHK ("Forwarded syscall", ret);
    }
	
  ret = mach_port_move_member(mach_task_self(), 
			      fake_server,
			      MACH_PORT_NULL);
  CHK ("Moving fake syscall out of inferior_wait_port_set", ret);

  ret = mach_port_move_member(mach_task_self(), 
			      exec_reply,
			      inferior_wait_port_set);
  CHK ("Moving exec_reply to inferior_wait_port_set", ret);

  ret = mach_msg (&syscall_in.header,	/* header */
		  MACH_RCV_MSG,		/* options */
		  0,			/* send size */
		  sizeof (struct syscall_msg_t),	/* receive size */
		  inferior_wait_port_set,		/* receive_name */
		  MACH_MSG_TIMEOUT_NONE,
		  MACH_PORT_NULL);
  CHK("mach_msg (exec reply)", ret);

  ret = task_suspend (inferior_task);
  CHK ("Suspending inferior after last exec", ret);

  debug ("Received exec reply from bsd server, suspended inferior task\n");

#ifdef DUMP_SYSCALL
      print_msg (&syscall_in.header);
#endif

  /* Message should appear as if it came from the unix server */
  syscall_in.header.msgh_local_port = MACH_PORT_NULL;

  /*  and go to the inferior task original reply port */
  syscall_in.header.msgh_remote_port = original_exec_reply;

  reverse_msg_bits(&syscall_in.header, MACH_MSG_TYPE_MOVE_SEND_ONCE);

  ret = mach_msg_send (&syscall_in.header);
  CHK ("Forwarding exec reply to inferior", ret);

  /* Garbage collect */
  ret = mach_port_deallocate (inferior_task,
			      original_server_port_name);
  CHK ("deallocating fake server send right", ret);

  ret = mach_port_insert_right (inferior_task,
				original_server_port_name,
				original_server_send,
				MACH_MSG_TYPE_MOVE_SEND);
  CHK ("Restoring the original bsd server send right", ret);

  ret = mach_port_destroy (mach_task_self (),
			   fake_server);
  fake_server = MACH_PORT_DEAD;
  CHK("mach_port_destroy (fake_server)", ret);

  ret = mach_port_destroy (mach_task_self (),
			   exec_reply);
  exec_reply = MACH_PORT_DEAD;
  CHK("mach_port_destroy (exec_reply)", ret);

  debug ("Done with exec call interception\n");
}

void
consume_send_rights (thread_list, thread_count)
     thread_array_t thread_list;
     int            thread_count;
{
  int index;
  kern_return_t ret;

  if (!thread_count)
    return;

  for (index = 0; index < thread_count; index++)
    {
      ret = mach_port_deallocate (mach_task_self (),
				  thread_list [ index ]);
      CHK ("Could not consume a right from port named", ret);
    }
}

/* THREAD_ID 0 is special; it selects the first kernel
 * thread from the list. This is used when starting the
 * program with 'run' or when attaching.
 *
 * If SETUP is 0 the context is not changed, and the registers, frame, etc
 * will continue to describe the old thread.
 */
kern_return_t
select_thread (task, thread_id, setup)
     mach_port_t task;
     int thread_id;
     int setup;
{
  thread_array_t thread_list;
  int thread_count;
  kern_return_t ret;
  int index;
  thread_t new_thread = MACH_PORT_NULL;

  if (thread_id < 0)
    error ("Can't (yet) select cprocs without kernel thread");

  ret = task_threads (task, &thread_list, &thread_count);
  if (ret != KERN_SUCCESS)
    {
      message ("Can not select a thread from a dead task");
      kill_inferior (0, 0);
      return KERN_FAILURE;
    }

  if (thread_count == 0)
    {
      /* The task can not do anything anymore, but it still
       * exists as a container for memory and ports.
       */
      registers_changed ();
      message ("Task %d has no threads",
	       map_port_name_to_mid (task, MACH_TYPE_TASK));
      current_thread = MACH_PORT_NULL;
      return KERN_FAILURE;
    }

  if (! thread_id)
    new_thread = thread_list[0];
  else
    {
      for (index = 0; index < thread_count; index++)
	if (thread_id == map_port_name_to_mid (thread_list [index],
					       MACH_TYPE_THREAD))
	  {
	    new_thread = thread_list [index];
	    index = -1;
	    break;
	  }

      if (index != -1)
	error ("No thread with mid %d", thread_id);
    }

  if (current_thread == new_thread)
    return KERN_SUCCESS;

  /* Notify when the selected thread dies */
  request_notify (new_thread, MACH_NOTIFY_DEAD_NAME, MACH_TYPE_THREAD);
  
  ret = vm_deallocate(mach_task_self(),
		      (vm_address_t) thread_list,
		      (thread_count * sizeof(mach_port_t)));
  CHK ("vm_deallocate", ret);
  
  if (! setup)
    current_thread = new_thread;
  else
    {
#if 0
      if (MACH_PORT_VALID (current_thread))
	{
	  /* Store the gdb's view of the thread we are deselecting
	   *
	   * @@ I think gdb updates registers immediately when they are
	   * changed, so don't do this.
	   */
	  ret = thread_abort (current_thread);
	  CHK ("Could not abort system calls when saving state of old thread",
	       ret);
	  child_prepare_to_store ();
	  target_store_registers (-1);
	}
#endif

      registers_changed ();

      current_thread = new_thread;

      ret = thread_abort (current_thread);
      CHK ("Could not abort system calls when selecting a thread", ret);

      stop_pc = read_pc();
      set_current_frame (create_new_frame (read_register (FP_REGNUM),
					   stop_pc));

      select_frame (get_current_frame (), 0);

      stop_frame_address = FRAME_FP (get_current_frame ());
    }

  return KERN_SUCCESS;
}

/*
 * Switch to use thread named NEW_THREAD.
 * Return it's MID
 */
int
switch_to_thread (new_thread)
     thread_t new_thread;
{
  thread_t saved_thread = current_thread;
  int mid;
  
  mid = map_port_name_to_mid (new_thread,
			      MACH_TYPE_THREAD);
  if (mid == -1)
    message ("Can't map thread name 0x%x to mid", new_thread);
  else if (select_thread (inferior_task, mid, 1) != KERN_SUCCESS)
    {
      if (current_thread)
	current_thread = saved_thread;
      error ("Could not select thread %d", mid);
    }
	
  return mid;
}

/* Do this in gdb after doing FORK but before STARTUP_INFERIOR.
 * Note that the registers are not yet valid in the inferior task.
 */
void
mach_create_inferior_hook (pid)
     int pid;
{
  kern_return_t ret;

  inferior_task = task_by_pid (pid);

  if (! MACH_PORT_VALID (inferior_task))
    error ("Can not map Unix pid %d to Mach task", pid);

  /* Clean up previous notifications and create new ones */
  setup_notify_port (1);

  /* When notification appears, the inferior task has died */
  request_notify (inferior_task, MACH_NOTIFY_DEAD_NAME, MACH_TYPE_TASK);

  emulator_present = have_emulator_p (inferior_task);

  /* By default, select the first thread,
   * If task has no threads, gives a warning
   * Does not fetch registers, since they are not yet valid.
   */
  select_thread (inferior_task, 0, 0);

  inferior_exception_port = MACH_PORT_NULL;

  setup_exception_port ();

  debug("Now the debugged task is created\n");
}

setup_exception_port ()
{
  kern_return_t ret;

  ret = mach_port_allocate (mach_task_self(), 
			    MACH_PORT_RIGHT_RECEIVE,
			    &inferior_exception_port);
  CHK("mach_port_allocate",ret);

  /* add send right */
  ret = mach_port_insert_right (mach_task_self (),
				inferior_exception_port,
				inferior_exception_port,
				MACH_MSG_TYPE_MAKE_SEND);
  CHK("mach_port_insert_right",ret);

  ret = mach_port_move_member (mach_task_self(), 
			       inferior_exception_port,
			       inferior_wait_port_set);
  CHK("mach_port_move_member",ret);

  ret = task_get_special_port (inferior_task, 
			       TASK_EXCEPTION_PORT,
			       &inferior_old_exception_port);
  CHK ("task_get_special_port(old exc)",ret);

  ret = task_set_special_port (inferior_task,
			       TASK_EXCEPTION_PORT, 
			       inferior_exception_port);
  CHK("task_set_special_port",ret);

  ret = mach_port_deallocate (mach_task_self (),
			      inferior_exception_port);
  CHK("mack_port_deallocate",ret);

#if 0
  /* When notify appears, the inferior_task's exception
   * port has been destroyed.
   *
   * Not used, since the dead_name_notification already
   * appears when task dies.
   *
   */
  request_notify (inferior_exception_port,
		  MACH_NOTIFY_NO_SENDERS,
		  MACH_TYPE_EXCEPTION_PORT);
#endif
}


/* Wait for the inferior to stop for some reason.
   - Loop on notifications until inferior_task dies.
   - Loop on exceptions until stopped_in_exception comes true.
     (e.g. we receive a single step trace trap)
   - a message arrives to gdb's message port

   There is no other way to exit this loop.

   Returns the inferior_pid for rest of gdb.
   Side effects: Set unix exit value to *w.
 */
int
mach_really_wait (w)
     WAITTYPE *w;
{
  int pid;
  kern_return_t ret;

  struct msg {
    mach_msg_header_t    header;
    mach_msg_type_t foo;
    int             data[8000];
  } in_msg, out_msg;

  stop_exception = stop_code = stop_subcode = -1;

  /* Either notify (death), exception or message can stop the inferior */
  stopped_in_exception = FALSE;

  while (1)
    {
      ret = mach_msg (&in_msg.header,		/* header */
		      MACH_RCV_MSG,		/* options */
		      0,			/* send size */
		      sizeof (struct msg),	/* receive size */
		      inferior_wait_port_set,	/* receive_name */
		      MACH_MSG_TIMEOUT_NONE,
		      MACH_PORT_NULL);
      CHK("mach_msg (receive)", ret);

      /* Check if we received a notify of the childs' death */
      if (notify_server (&in_msg.header, &out_msg.header))
	{
	  /* If inferior_task is null then the inferior has
	     gone away and we want to return to command level.
	     Otherwise it was just an informative message and we
	     need to look to see if there are any more. */
	  if (inferior_task != MACH_PORT_NULL)
	    continue;
	  else
	    {
	      /* Collect Unix exit status for gdb */

	      wait3(w, WNOHANG, 0);

	      /* This mess is here to check that the rest of
	       * gdb knows that the inferior died. It also
	       * tries to hack around the fact that Mach 3.0 (mk69)
	       * unix server (ux28) does not always know what
	       * has happened to it's children when mach-magic
	       * is applied on them.
	       */
	      if ((!WIFEXITED(*w) && WIFSTOPPED(*w))         ||
		  (WIFEXITED(*w)  && WEXITSTATUS(*w) > 0377))
		{
		  WSETEXIT(*w, 0);
		  message ("Using exit value 0 for terminated task");
		}
	      else if (!WIFEXITED(*w))
		{
		  int sig = WTERMSIG(*w);

		  /* Signals cause problems. Warn the user. */
		  if (sig != SIGKILL) /* Bad luck if garbage matches this */
		    message ("The terminating signal stuff may be nonsense");
		  else if (sig > NSIG)
		    {
		      WSETEXIT(*w, 0);
		      message ("Using exit value 0 for terminated task");
		    }
		}
	      return inferior_pid;
	    }
	}

      /* Hmm. Check for exception, as it was not a notification.
	 exc_server() does an upcall to catch_exception_raise()
	 if this rpc is an exception. Further actions are decided
	 there.
       */
      if (! exc_server (&in_msg.header, &out_msg.header))
	{

	  /* Not an exception, check for message.
	   *
	   * Messages don't come from the inferior, or if they
	   * do they better be asynchronous or it will hang.
	   */
	  if (gdb_message_server (&in_msg.header))
	    continue;

	  error ("Unrecognized message received in mach_really_wait");
	}

      /* Send the reply of the exception rpc to the suspended task */
      ret = mach_msg_send (&out_msg.header);
      CHK ("mach_msg_send (exc reply)", ret);
      
      if (stopped_in_exception)
	{
	  /* Get unix state. May be changed in mach3_exception_actions() */
	  wait3(w, WNOHANG, 0);

	  mach3_exception_actions (w, FALSE, "Task");

	  return inferior_pid;
	}
    }
}

/*
 * Gdb message server.
 * Currently implemented is the STOP message, that causes
 * gdb to return to the command level like ^C had been typed from terminal.
 */
int
gdb_message_server (InP)
     mach_msg_header_t *InP;
{
  kern_return_t ret;
  int mid;

  if (InP->msgh_local_port == our_message_port)
    {
      /* A message coming to our_message_port. Suspend the inferior */
      ret = task_suspend (inferior_task);
      if (ret != KERN_SUCCESS)
	{
	  message ("Could not suspend task for message %d:%s",
		   InP->msgh_id,
		   mach_error_string (ret));
	  return 1;
	}

      /* A message coming to our_message_port. Check validity */
      switch (InP->msgh_id) {

      case GDB_MESSAGE_ID_STOP:
	mid = map_port_name_to_mid (current_thread, MACH_TYPE_THREAD);
	if (mid == -1)
	  {
	    message ("Selecting first existing kernel thread");
	    mid = 0;
	  }

	current_thread = MACH_PORT_NULL; /* Force setup */
	select_thread (inferior_task, mid, 1);
	request_quit ();
	QUIT;			/* Return to command level */
	abort ();
	
      default:
	message ("Invalid message id %d received, ignored.",
		 InP->msgh_id);
	return 1;
      }

#if 0
      /* Not reached now */
      ret = task_resume (inferior_task);
      CHK ("Could not resume task for message", ret);

      return 1;
#endif
    }
  return 0;
}



/* this stuff here is an upcall via libmach/excServer.c 
   and mach_really_wait which does the actual upcall.

   The code will pass the exception to the inferior if:

     - The task that signaled is not the inferior task
       (e.g. when debugging another debugger)

     - The user has explicitely requested to pass on the exceptions.
       (e.g to the default unix exception handler, which maps
	exceptions to signals, or the usr has her own signal handler)

     - If the thread that signaled is being single-stepped and it
       has set it's own exception port and the exception is not
       EXC_BREAKPOINT. (Maybe this is not desirable?)
 */

kern_return_t
catch_exception_raise (port, thread, task, exception, code, subcode)
     mach_port_t port;
     thread_t thread;
     task_t task;
     int exception, code, subcode;
{
  kern_return_t ret;
  boolean_t signal_thread;
  int mid = map_port_name_to_mid (thread, MACH_TYPE_THREAD);

  if (! MACH_PORT_VALID (thread))
    {
      /* If the exception was sent and thread dies before we
	 receive it, THREAD will be MACH_PORT_DEAD
       */

      current_thread = thread = MACH_PORT_NULL;
      error ("Received exception from nonexistent thread");
    }

  /* Check if the task died in transit.
   * @@ Isn't the thread also invalid in such case?
   */
  if (! MACH_PORT_VALID (task))
    {
      current_thread = thread = MACH_PORT_NULL;
      error ("Received exception from nonexistent task");
    }

  if (exception < 0 || exception > MAX_EXCEPTION)
    fatal ("catch_exception_raise: unknown exception code %d thread %d",
	   exception,
	   mid);

  if (! MACH_PORT_VALID (inferior_task))
    error ("got an exception, but inferior_task is null or dead");
  
  stop_exception = exception;
  stop_code      = code;
  stop_subcode   = subcode;  
  
  signal_thread = exception != EXC_BREAKPOINT       &&
    		  port == singlestepped_thread_port &&
		  MACH_PORT_VALID (thread_saved_exception_port);

  /* If it was not our inferior or if we want to forward
   * the exception to the inferior's handler, do it here
   */
  if (task != inferior_task ||
      signal_thread         ||
      exception_map [exception].forward)
    {
      mach_port_t eport = inferior_old_exception_port;

      if (signal_thread)
	{
	  /*
	    GDB now forwards the exeption to thread's original handler,
	    since the user propably knows what he is doing.
	    Give a message, though.
	   */

	  mach3_exception_actions ((WAITTYPE *)NULL, TRUE, "Thread");
	  eport = thread_saved_exception_port;
	}

      /* Send the exception to the original handler */
      ret = exception_raise (eport,
			     thread, 
			     task,
			     exception,
			     code,
			     subcode);

      (void) mach_port_deallocate (mach_task_self (), task);
      (void) mach_port_deallocate (mach_task_self (), thread);

      /* Do not stop the inferior */
      return ret;
    }
  
  /* Now gdb handles the exception */
  stopped_in_exception = TRUE;

  ret = task_suspend (task);
  CHK ("Error suspending inferior after exception", ret);

  if (current_thread != thread)
    {
      if (port == singlestepped_thread_port)
	{
	  setup_single_step (current_thread, FALSE);
	  error
	    ("Exception from wrong thread %d while singlestepping thread %d",
	     mid,
	     map_port_name_to_mid (current_thread, MACH_TYPE_THREAD));
	}

      if (select_thread (inferior_task, mid, 0) != KERN_SUCCESS)
	error ("Could not select thread %d causing exception", mid);
      else
	message ("Gdb selected thread %d", mid);
    }

  /* If we receive an exception that is not breakpoint
   * exception, we interrupt the single step and return to
   * debugger. Trace condition is cleared.
   */
  if (MACH_PORT_VALID (singlestepped_thread_port) &&
      stop_exception != EXC_BREAKPOINT)
    {
      message ("Single step interrupted by exception");
      setup_single_step (current_thread, FALSE);
    }
  else if (port == singlestepped_thread_port)
    {
      /* Single step exception occurred, remove trace bit
       * and return to gdb.
       */
      if (! MACH_PORT_VALID (current_thread))
	error ("Single stepped thread is not valid");

      ret = thread_suspend (current_thread);
      CHK ("thread_suspend in catch_exception_raise", ret);
      
      setup_single_step (current_thread, FALSE);
      
      /* Resume threads, but leave the task suspended */
      resume_all_threads (0);
    }

  (void) mach_port_deallocate (mach_task_self (), task);
  (void) mach_port_deallocate (mach_task_self (), thread);

  return KERN_SUCCESS;
}

int
port_valid (port, mask)
  mach_port_t port;
  int         mask;
{
  kern_return_t ret;
  mach_port_type_t type;

  ret = mach_port_type (mach_task_self (),
			port,
			&type);
  if (ret != KERN_SUCCESS || (type & mask) != mask)
    return 0;
  return 1;
}

/* @@ No vm read cache implemented yet */
boolean_t vm_read_cache_valid = FALSE;

/*
 * Read inferior task's LEN bytes from ADDR and copy it to MYADDR
 * in gdb's address space.
 */
int
mach3_read_inferior (addr, myaddr, length)
     CORE_ADDR addr;
     char *myaddr;
     int length;
{
  kern_return_t ret;
  vm_address_t low_address       = (vm_address_t) trunc_page (addr);
  vm_size_t    aligned_length = 
    			(vm_size_t) round_page (addr+length) - low_address;
  pointer_t    copied_memory;
  int	       copy_count;

  /* Get memory from inferior with page aligned addresses */
  ret = vm_read (inferior_task,
		 low_address,
		 aligned_length,
		 &copied_memory,
		 &copy_count);
  if (ret != KERN_SUCCESS)
    {
      /* the problem is that the inferior might be killed for whatever reason
       * before we go to mach_really_wait. This is one place that ought to
       * catch many of those errors.
       * @@ A better fix would be to make all external events to GDB
       * to arrive via a SINGLE port set. (Including input!)
       */

      if (! port_valid (inferior_task, MACH_PORT_TYPE_SEND))
	{
	  kill_inferior (0, 0);
	  error ("Inferior killed (task port invalid)");
	}
      else
	CHK("mach3_read_inferior vm_read failed", ret);
    }

  bcopy ((char *)addr - low_address + copied_memory, myaddr, length);

  ret = vm_deallocate (mach_task_self (),
		       copied_memory,
		       copy_count);
  CHK("mach3_read_inferior vm_deallocate failed", ret);

  return length;
}

#ifdef __STDC__
#define CHK_GOTO_OUT(str,ret) \
  do if (ret != KERN_SUCCESS) { errstr = #str; goto out; } while(0)
#else
#define CHK_GOTO_OUT(str,ret) \
  do if (ret != KERN_SUCCESS) { errstr = str; goto out; } while(0)
#endif

struct vm_region_list {
  struct vm_region_list *next;
  vm_prot_t	protection;
  vm_address_t  start;
  vm_size_t	length;
};

struct obstack  region_obstack;

/*
 * Write inferior task's LEN bytes from ADDR and copy it to MYADDR
 * in gdb's address space.
 */
int
mach3_write_inferior (addr, myaddr, length)
     CORE_ADDR addr;
     char *myaddr;
     int length;
{
  kern_return_t ret;
  vm_address_t low_address       = (vm_address_t) trunc_page (addr);
  vm_size_t    aligned_length = 
    			(vm_size_t) round_page (addr+length) - low_address;
  pointer_t    copied_memory;
  int	       copy_count;
  int	       deallocate = 0;

  char         *errstr = "Bug in mach3_write_inferior";

  struct vm_region_list *region_element;
  struct vm_region_list *region_head = (struct vm_region_list *)NULL;

  /* Get memory from inferior with page aligned addresses */
  ret = vm_read (inferior_task,
		 low_address,
		 aligned_length,
		 &copied_memory,
		 &copy_count);
  CHK_GOTO_OUT ("mach3_write_inferior vm_read failed", ret);

  deallocate++;

  bcopy (myaddr, (char *)addr - low_address + copied_memory, length);

  obstack_init (&region_obstack);

  /* Do writes atomically.
   * First check for holes and unwritable memory.
   */
  {
    vm_size_t    remaining_length  = aligned_length;
    vm_address_t region_address    = low_address;

    struct vm_region_list *scan;

    while(region_address < low_address + aligned_length)
      {
	vm_prot_t protection;
	vm_prot_t max_protection;
	vm_inherit_t inheritance;
	boolean_t shared;
	mach_port_t object_name;
	vm_offset_t offset;
	vm_size_t   region_length = remaining_length;
	vm_address_t old_address  = region_address;
    
	ret = vm_region (inferior_task,
			 &region_address,
			 &region_length,
			 &protection,
			 &max_protection,
			 &inheritance,
			 &shared,
			 &object_name,
			 &offset);
	CHK_GOTO_OUT ("vm_region failed", ret);

	/* Check for holes in memory */
	if (old_address != region_address)
	  {
	    message ("No memory at 0x%x. Nothing written",
		     old_address);
	    ret = KERN_SUCCESS;
	    length = 0;
	    goto out;
	  }

	if (!(max_protection & VM_PROT_WRITE))
	  {
	    message ("Memory at address 0x%x is unwritable. Nothing written",
		     old_address);
	    ret = KERN_SUCCESS;
	    length = 0;
	    goto out;
	  }

	/* Chain the regions for later use */
	region_element = 
	  (struct vm_region_list *)
	    obstack_alloc (&region_obstack, sizeof (struct vm_region_list));
    
	region_element->protection = protection;
	region_element->start      = region_address;
	region_element->length     = region_length;

	/* Chain the regions along with protections */
	region_element->next = region_head;
	region_head          = region_element;
	
	region_address += region_length;
	remaining_length = remaining_length - region_length;
      }

    /* If things fail after this, we give up.
     * Somebody is messing up inferior_task's mappings.
     */
    
    /* Enable writes to the chained vm regions */
    for (scan = region_head; scan; scan = scan->next)
      {
	boolean_t protection_changed = FALSE;
	
	if (!(scan->protection & VM_PROT_WRITE))
	  {
	    ret = vm_protect (inferior_task,
			      scan->start,
			      scan->length,
			      FALSE,
			      scan->protection | VM_PROT_WRITE);
	    CHK_GOTO_OUT ("vm_protect: enable write failed", ret);
	  }
      }

    ret = vm_write (inferior_task,
		    low_address,
		    copied_memory,
		    aligned_length);
    CHK_GOTO_OUT ("vm_write failed", ret);
	
    /* Set up the original region protections, if they were changed */
    for (scan = region_head; scan; scan = scan->next)
      {
	boolean_t protection_changed = FALSE;
	
	if (!(scan->protection & VM_PROT_WRITE))
	  {
	    ret = vm_protect (inferior_task,
			      scan->start,
			      scan->length,
			      FALSE,
			      scan->protection);
	    CHK_GOTO_OUT ("vm_protect: enable write failed", ret);
	  }
      }
  }

 out:
  if (deallocate)
    {
      obstack_free (&region_obstack, 0);
      
      (void) vm_deallocate (mach_task_self (),
			    copied_memory,
			    copy_count);
    }

  if (ret != KERN_SUCCESS)
    {
      message ("%s %s", errstr, mach_error_string (ret));
      return 0;
    }

  return length;
}

int
child_xfer_memory (memaddr, myaddr, len, write, target)
     CORE_ADDR memaddr;
     char *myaddr;
     int len;
     int write;
     struct target_ops *target;	/* IGNORED */
{
  int result;

  if (write)
    result = mach3_write_inferior (memaddr, myaddr, len);
  else
    result = mach3_read_inferior  (memaddr, myaddr, len);

  return result;
}


private char *
translate_state(state)
int	state;
{
  switch (state) {
  case TH_STATE_RUNNING:	return("R");
  case TH_STATE_STOPPED:	return("S");
  case TH_STATE_WAITING:	return("W");
  case TH_STATE_UNINTERRUPTIBLE: return("U");
  case TH_STATE_HALTED:		return("H");
  default:			return("?");
  }
}

private char *
translate_cstate(state)
int	state;
{
  switch (state) {
  case CPROC_RUNNING:	return "R";
  case CPROC_SWITCHING: return "S";
  case CPROC_BLOCKED:   return "B";
  case CPROC_CONDWAIT:  return "C";
  case CPROC_CONDWAIT|CPROC_SWITCHING:
    			return "CS";
  default:		return "?";
  }
}

/* type == MACH_MSG_TYPE_COPY_SEND || type == MACH_MSG_TYPE_MAKE_SEND
 */

mach_port_t           /* no mach_port_name_t found in include files. */
map_inferior_port_name (inferior_name, type)
     mach_port_t inferior_name;
     mach_msg_type_name_t type;
{
  kern_return_t        ret;
  mach_msg_type_name_t acquired;
  mach_port_t          iport;
  
  ret = mach_port_extract_right (inferior_task,
				 inferior_name,
				 type,
				 &iport,
				 &acquired);
  CHK("mach_port_extract_right (map_inferior_port_name)", ret);

  if (acquired != MACH_MSG_TYPE_PORT_SEND)
    error("Incorrect right extracted, (map_inferior_port_name)");

  ret = mach_port_deallocate (mach_task_self (),
			      iport);
  CHK ("Deallocating mapped port (map_inferior_port_name)", ret);

  return iport;
}

/*
 * Naming convention:
 *  Always return user defined name if found.
 *  _K == A kernel thread with no matching CPROC
 *  _C == A cproc with no current kernel thread
 *  _t == A cthread with no user defined name
 *
 * The digits that follow the _names are just sequential numbers
 * incremented for each thread.
 */

private char buf[7];

private char *
get_thread_name (one_cproc, order)
     cproc_t one_cproc;
     int order;
{
  if (one_cproc)
    if (one_cproc->incarnation == NULL)
      {
	/* A cproc, but not currently matching any kernel threads */
	sprintf(buf, "_C%d", order);
      }
    else if (! one_cproc->incarnation->name)
      {
	/* kernel thread, cproc and cthread, but no name */
	sprintf(buf, "_t%d", order);
      }
    else
      return (one_cproc->incarnation->name);
  else
    {
      /* Kernel thread without cproc */
      sprintf(buf, "_K%d", order);
    }

  return buf;
}

int
fetch_thread_info (task, mthreads_out)
     mach_port_t	task;
     gdb_thread_t	*mthreads_out;	/* out */
{
  kern_return_t  ret;
  thread_array_t th_table;
  int		 th_count;
  gdb_thread_t mthreads = NULL;
  int 		 index;

  ret = task_threads (task, &th_table, &th_count);
  if (ret != KERN_SUCCESS)
    {
      message ("Error getting inferior's thread list:%s",
	       mach_error_string(ret));
      kill_inferior (0, 0);
      return -1;
    }
  
  mthreads = (gdb_thread_t)
    		obstack_alloc
		  (cproc_obstack,
		   th_count * sizeof (struct gdb_thread));

  for (index = 0; index < th_count; index++)
    {
      thread_t saved_thread = MACH_PORT_NULL;
      int mid;

      if (th_table[index] != current_thread)
	{
	  saved_thread = current_thread;
	  
	  mid = switch_to_thread (th_table[ index ]);
	}

      mthreads[index].name  = th_table[index];
      mthreads[index].cproc = NULL;	/* map_cprocs_to_kernel_threads() */
      mthreads[index].in_emulator = FALSE;
      
      mthreads[index].sp = read_register (SP_REGNUM);
      mthreads[index].fp = read_register (FP_REGNUM);
      mthreads[index].pc = read_pc ();

      if (MACH_PORT_VALID (saved_thread))
	mid = switch_to_thread (saved_thread);
    }
  
  consume_send_rights (th_table, th_count);
  ret = vm_deallocate (mach_task_self(), (vm_address_t)th_table, 
		       (th_count * sizeof(mach_port_t)));
  if (ret != KERN_SUCCESS)
    {
      message ("Error trying to deallocate thread list : %s",
	       mach_error_string (ret));
    }

  *mthreads_out = mthreads;

  return th_count;
}


/*
 * Current emulator always saves the USP on top of
 * emulator stack below struct emul_stack_top stuff.
 */
CORE_ADDR
fetch_usp_from_emulator_stack (sp)
     CORE_ADDR sp;
{
  CORE_ADDR stack_pointer;

  sp = (sp & ~(EMULATOR_STACK_SIZE-1)) +
    	EMULATOR_STACK_SIZE - sizeof (struct emul_stack_top);
  
  if (mach3_read_inferior (sp,
			   &stack_pointer,
			   sizeof (CORE_ADDR)) != sizeof (CORE_ADDR))
    {
      message ("Can't read user sp from emulator stack address 0x%x", sp);
      return 0;
    }

  return stack_pointer;
}

/* Check if the emulator exists at task's address space.
 */
boolean_t
have_emulator_p (task)
     task_t task;
{
  kern_return_t	ret;
  vm_offset_t	*emulation_vector;
  int		n;
  int		i;
  int		vector_start;
  
  ret = task_get_emulation_vector (task,
				   &vector_start,
				   &emulation_vector,
				   &n);
  CHK("task_get_emulation_vector", ret);
  debug ("%d vectors from %d at 0x%08x\n",
	 n, vector_start, emulation_vector);
  
  for(i = 0; i < n; i++)
    {
      vm_offset_t entry = *(emulation_vector+i);
      
      if (EMULATOR_BASE <= entry && entry <= EMULATOR_END)
	return TRUE;
      else if (entry)
	{
	  static boolean_t informed = FALSE;
	  if (!informed)
	    {
	      message("Emulation vector address 0x08%x outside emulator space",
		      entry);
	      informed = TRUE;
	    }
	}
    }
  return FALSE;
}

/*
 * Map cprocs to kernel threads and vice versa.
 *
 * For reverse mapping the code mis-uses one struct cproc field,
 * see "os-mach3.h" and code here.
 *
 */

void
map_cprocs_to_kernel_threads (cprocs, mthreads, thread_count)
     cproc_t	        cprocs;
     gdb_thread_t       mthreads;
     int thread_count;
{
  int index;
  cproc_t scan;
  boolean_t   all_mapped = TRUE;

  for (scan = cprocs; scan; scan = scan->list)
    {
      /* Default to: no kernel thread for this cproc */
      CPROC_REVERSE_MAP (scan) = -1;

      /* Check if the cproc is found by its stack */
      for (index = 0; index < thread_count; index++)
	{
	  if ((mthreads + index)->sp > scan->stack_base &&
	      (mthreads + index)->sp <= scan->stack_base + scan->stack_size)
	    {
	      (mthreads + index)->cproc = scan;
	      CPROC_REVERSE_MAP (scan) = index;
	      break;
	    }
	}
      all_mapped &= (CPROC_REVERSE_MAP(scan) != -1);
    }

  /* Check for threads that are currently in the emulator.
   * If so, they have a different stack, and the still unmapped
   * cprocs may well get mapped to these threads.
   * 
   * If:
   *  - cproc stack does not match any kernel thread stack pointer
   *  - there is at least one extra kernel thread
   *    that has no cproc mapped above.
   *  - some kernel thread stack pointer points to emulator space
   *  then we find the user stack pointer saved in the emulator
   *  stack, and try to map that to the cprocs.
   *
   * Also set in_emulator for kernel threads.
   */ 

  if (emulator_present)
    {
      for (index = 0; index < thread_count; index++)
	{
	  CORE_ADDR emul_sp;
	  CORE_ADDR usp;

	  gdb_thread_t mthread = (mthreads+index);
	  emul_sp = mthread->sp;

	  if (! mthread->cproc &&
	      EMULATOR_BASE <= emul_sp && emul_sp <= EMULATOR_END)
	    {
	      mthread->in_emulator = emulator_present;
	      
	      if (!all_mapped && cprocs)
		{
		  usp = fetch_usp_from_emulator_stack (emul_sp);
		  
		  /* @@ Could be more accurate */
		  if (! usp)
		    error ("Zero stack pointer read from emulator?");
		  
		  /* Try to match this stack pointer to the cprocs that
		   * don't yet have a kernel thread.
		   */
		  for (scan = cprocs; scan; scan = scan->list)
		    {
		      
		      /* Check is this unmapped CPROC stack contains
		       * the user stack pointer saved in the
		       * emulator.
		       */
		      if (CPROC_REVERSE_MAP (scan) == -1 &&
			  usp > scan->stack_base &&
			  usp <= scan->stack_base + scan->stack_size)
			{
			  mthread->cproc = scan;
			  CPROC_REVERSE_MAP (scan) = index;
			  break;
			}
		    }
		}
	    }
	}
    }
}

/*
 * Format of the thread_list command
 *
 * 	          mid sel   name  emul ks susp  cstate wired   address
 */
#define TL_FORMAT "%d%c\t%-10.10s %1.1s%s%-5.5s %-2.2s %-5.5s "

#define TL_HEADER "\nMID     Name        KState CState   Where\n"

void
print_tl_address (stream, pc)
     FILE *stream;
     CORE_ADDR pc;
{
  if (find_pc_misc_function (pc) < 0)
    fprintf_filtered (stream, local_hex_format(), pc);
  else
    {
      extern int addressprint;
      extern int asm_demangle;

      int store    = addressprint;
      addressprint = 0;
      print_address_symbolic (pc, stream, asm_demangle, "");
    }
}

/* For thread names, but also for gdb_message_port external name */
#define MAX_NAME_LEN 50

#define CPROC_LIST "cproc_list"

private cproc_t
get_cprocs()
{
  extern int lookup_misc_func ();
  
  register struct symbol *sym;
  cproc_t their_cprocs, cproc_head, cproc_copy;
  char *name;
  cthread_t cthread;
  

  their_cprocs = NO_CPROC;

  sym = lookup_symbol(CPROC_LIST, 0, VAR_NAMESPACE, 0, 0);

  /* Get the address of the first cproc in the task */
  if (sym)
    {
      if (!mach3_read_inferior(SYMBOL_VALUE(sym),
				&their_cprocs,
				sizeof(cproc_t)))
	  error("Can't read sym ptr to threads (0x%x).", SYMBOL_VALUE(sym));
    }
  else
    { /* cproc_list is not in a file compiled with debugging
	 symbols, but don't give up yet */
      
      int ind = lookup_misc_func (CPROC_LIST);
      
      if (ind == -1)
	{
	  ind = lookup_misc_func ("cprocs");
	  if (ind != -1)
	    {
	      static int informed = 0;
	      if (!informed) {
		informed++;
		message ("Your program is loaded with an old threads library.");
		message ("GDB does not know the old form of threads");
		message ("so things may not work.");
	      }
	    }
	}
	  
      /* Stripped or no -lthreads loaded, or CPROC_LIST is
       * in wrong segment.
       */
      if (ind == -1 || misc_function_vector[ ind ].type != mf_data)
	return NO_CPROC;
      
      if (!mach3_read_inferior(misc_function_vector [ ind ].address,
			       &their_cprocs,
			       sizeof(cproc_t)))
	error("Can't read misc ptr to threads (0x%x).",
	      misc_function_vector [ ind ]);
    }

  /* Scan the CPROCs in the task.
     CPROCs are chained with LIST field, not NEXT field, which
     chains mutexes, condition variables and queues */
  
  cproc_head = NO_CPROC;

  while (their_cprocs != NO_CPROC)
    {
      cproc_copy = (cproc_t) obstack_alloc(cproc_obstack,
					   sizeof(struct cproc));
      
      if (!mach3_read_inferior(their_cprocs,
				cproc_copy,
				sizeof(struct cproc)))
	error("Can't read next cproc at 0x%x.", their_cprocs);
      
      their_cprocs = cproc_copy->list;
      
      if (cproc_copy->incarnation != NULL)
	{
	  /* This CPROC has an attached CTHREAD. Get it's name */
	  cthread = (cthread_t)obstack_alloc (cproc_obstack,
					      sizeof(struct cthread));
	  
	  if (!mach3_read_inferior(cproc_copy->incarnation,
				    cthread,
				    sizeof(struct cthread)))
	    error("Can't read next thread at 0x%x.",
		  cproc_copy->incarnation);
	  
	  cproc_copy->incarnation = cthread;
	  name = (char *) obstack_alloc (cproc_obstack, MAX_NAME_LEN);
	  
	  if (!mach3_read_inferior(cthread->name, name, MAX_NAME_LEN))
	    error("Can't read next thread's name at 0x%x.", cthread->name);
	  
	  cthread->name = name;
	}
      
      /* insert in front */
      cproc_copy->list = cproc_head;
      cproc_head       = cproc_copy;
    }
  return(cproc_head);
}

#ifndef FETCH_CPROC_STATE
/*
 * Check if your machine does not grok the way this routine
 * fetches the FP,PC and SP of a cproc that is not
 * currntly attached to any kernel thread (e.g. it's cproc.context
 * field points to the place in stack where the context
 * is saved).
 *
 * If it doesn't, define your own routine.
 */
#define FETCH_CPROC_STATE(mth) mach3_cproc_state (mth)

int
mach3_cproc_state (mthread)
     gdb_thread_t mthread;
{
  int context;

  if (! mthread || !mthread->cproc || !mthread->cproc->context)
    return -1;

  context = mthread->cproc->context;

  mthread->sp = context + MACHINE_CPROC_SP_OFFSET;

  if (mach3_read_inferior (context + MACHINE_CPROC_PC_OFFSET,
			   &mthread->pc,
			   sizeof (CORE_ADDR)) != sizeof (CORE_ADDR))
    {
      message ("Can't read cproc pc from inferior");
      return -1;
    }

  if (mach3_read_inferior (context + MACHINE_CPROC_FP_OFFSET,
			   &mthread->fp,
			   sizeof (CORE_ADDR)) != sizeof (CORE_ADDR))
    {
      message ("Can't read cproc fp from inferior");
      return -1;
    }

  return 0;
}
#endif /* FETCH_CPROC_STATE */


void
thread_list_command()
{
  int 		 thread_count;
  thread_basic_info_data_t ths;
  cproc_t cprocs;
  cproc_t scan;
  int     index;
  char *name;
  char selected;
  char *wired;
  int infoCnt;
  kern_return_t ret;
  mach_port_t mid_or_port;
  gdb_thread_t their_threads;
  gdb_thread_t kthread;

  int neworder = 1;

  char *fmt = "There are %d kernel threads in task %d.\n";
  
  int tmid = map_port_name_to_mid (inferior_task, MACH_TYPE_TASK);
  
  MACH_ERROR_NO_INFERIOR;
  
  thread_count = fetch_thread_info (inferior_task,
				    &their_threads);
  if (thread_count == -1)
    return;
  
  if (thread_count == 1)
    fmt = "There is %d kernel thread in task %d.\n";
  
  printf (fmt, thread_count, tmid);
  
  printf(TL_HEADER);
  
  cprocs = get_cprocs();
  
  map_cprocs_to_kernel_threads (cprocs, their_threads, thread_count);
  
  for (scan = cprocs; scan; scan = scan->list, neworder++)
    {
      int mid;
      char buf[10];
      
      selected = ' ';
      
      /* a wired cproc? */
      wired    = scan->wired ? "wired" : "";
      
      if (CPROC_REVERSE_MAP(scan) != -1)
	kthread  = (their_threads + CPROC_REVERSE_MAP(scan));
      else
	kthread  = NULL;

      if (kthread)
	{
	  /* These cprocs have a kernel thread */
	  
	  mid = map_port_name_to_mid (kthread->name, MACH_TYPE_THREAD);
	  
	  infoCnt = THREAD_BASIC_INFO_COUNT;
	  
	  ret = thread_info (kthread->name,
			     THREAD_BASIC_INFO,
			     (thread_info_t)&ths,
			     &infoCnt);
	  
	  if (ret != KERN_SUCCESS)
	    {
	      message ("Unable to get basic info on thread %d : %s",
		       mid,
		       mach_error_string (ret));
	      continue;
	    }

	  if (kthread->name == current_thread)
	    selected = '*';
	  
	  if (ths.suspend_count)
	    sprintf (buf, "%d", ths.suspend_count);
	  else
	    buf[0] = '\000';

#if 0
	  if (ths.flags & TH_FLAGS_SWAPPED)
	    strcat (buf, "S");
#endif

	  if (ths.flags & TH_FLAGS_IDLE)
	    strcat (buf, "I");

	  printf (TL_FORMAT,
		  mid,
		  selected,
		  get_thread_name (scan, neworder),
		  kthread->in_emulator ? "E" : "",
		  translate_state (ths.run_state),
		  buf,
		  translate_cstate (scan->state),
		  wired);
	  print_tl_address (stdout, kthread->pc);
	}
      else
	{
	  /* These cprocs don't have a kernel thread.
	   * find out the calling frame with machine
	   * dependent routine.
	   */

	  struct gdb_thread state;

	  printf (TL_FORMAT,
		  -neworder,	/* Pseudo MID */
		  selected,
		  get_thread_name (scan, neworder),
		  "",
		  "-",	/* kernel state */
		  "",
		  translate_cstate (scan->state),
		  "");

	  state.cproc = scan;

	  if (FETCH_CPROC_STATE (&state) == -1)
	    printf ("???");
	  else
	    print_tl_address (stdout, state.pc);
	}
      printf ("\n");
    }
  
  /* Scan for kernel threads without cprocs */
  for (index = 0; index < thread_count; index++, neworder++)
    {
      if (! their_threads[index].cproc)
	{
	  int mid;
	  
	  char buf[10];

	  mach_port_t name = their_threads[index].name;
	  
	  mid = map_port_name_to_mid (name, MACH_TYPE_THREAD);
	  
	  infoCnt = THREAD_BASIC_INFO_COUNT;
	  
	  ret = thread_info(name,
			    THREAD_BASIC_INFO,
			    (thread_info_t)&ths,
			    &infoCnt);
	    
	  if (ret != KERN_SUCCESS)
	    {
	      message ("Unable to get basic info on thread %d : %s",
		       mid,
		       mach_error_string (ret));
	      continue;
	    }
	  if (name == current_thread)
	    selected = '*';
	  else
	    selected = ' ';

	  if (ths.suspend_count)
	    sprintf (buf, "%d", ths.suspend_count);
	  else
	    buf[0] = '\000';

#if 0
	  if (ths.flags & TH_FLAGS_SWAPPED)
	    strcat (buf, "S");
#endif

	  if (ths.flags & TH_FLAGS_IDLE)
	    strcat (buf, "I");

	  printf (TL_FORMAT,
		  mid,
		  selected,
		  get_thread_name (NULL, neworder),
		  their_threads[index].in_emulator ? "E" : "",
		  translate_state (ths.run_state),
		  buf,
		  "",   /* No cproc state */
		  "");	/* Can't be wired */
	  print_tl_address (stdout, their_threads[index].pc);
	  printf ("\n");
	}
    }
  
  obstack_free (cproc_obstack, 0);
  obstack_init (cproc_obstack);
}

void
thread_select_command(args, from_tty)
     char *args;
     int from_tty;
{
  int mid;
  thread_array_t thread_list;
  int thread_count;
  kern_return_t ret;

  MACH_ERROR_NO_INFERIOR;

  if (!args)
    error_no_arg ("Kernel thread MID to select");

  mid = atoi(args);

  if (mid == 0)
    error ("You must select threads by MID");

  if (select_thread (inferior_task, mid, 1) != KERN_SUCCESS)
    return;

  if (from_tty)
    printf ("Thread %d selected\n", mid);
}

thread_trace (thread, set)
mach_port_t thread;
boolean_t   set;
{
  int			flavor   = TRACE_FLAVOR;
  unsigned int		stateCnt = TRACE_FLAVOR_SIZE;
  kern_return_t		ret;
  thread_state_data_t	state;

  if (! MACH_PORT_VALID (thread))
    {
      message ("thread_trace: invalid thread");
      return;
    }

  ret = thread_get_state(thread, flavor, state, &stateCnt);
  CHK ("thread_trace: error reading thread state", ret);
  
  if (set)
    {
      TRACE_SET (thread, state);
    }
  else
    {
      TRACE_CLEAR (thread, state);
    }

  ret = thread_set_state(thread, flavor, state, stateCnt);
  CHK ("thread_trace: error writing thread state", ret);
}  

#ifdef	FLUSH_INFERIOR_CACHE

/* When over-writing code on some machines the I-Cache must be flushed
   explicitly, because it is not kept coherent by the lazy hardware.
   This definitely includes breakpoints, for instance, or else we
   endup looping in misterious Bpt traps */

flush_inferior_icache(pc, amount)
     CORE_ADDR pc;
{
  vm_machine_attribute_val_t flush = MATTR_VAL_ICACHE_FLUSH;
  kern_return_t   ret;
  
  ret = vm_machine_attribute (inferior_task,
			      pc,
			      amount,
			      MATTR_CACHE,
			      &flush);
  if (ret != KERN_SUCCESS)
    message ("Error flushing inferior's cache : %s",
	     mach_error_string (ret));
}
#endif	FLUSH_INFERIOR_CACHE


private
suspend_all_threads (from_tty)
     int from_tty;
{
  kern_return_t	   ret;
  thread_array_t   thread_list;
  int		   thread_count, index;
  int		   infoCnt;
  thread_basic_info_data_t th_info;

  
  ret = task_threads (inferior_task, &thread_list, &thread_count);
  if (ret != KERN_SUCCESS)
    {
      message ("Could not suspend inferior threads.");
      kill_inferior (0,0);
      return_to_top_level ();
    }
  
  for (index = 0; index < thread_count; index++)
    {
      int mid;

      mid = map_port_name_to_mid (thread_list[ index ],
				  MACH_TYPE_THREAD);
	  
      ret = thread_suspend(thread_list[ index ]);

      if (ret != KERN_SUCCESS)
	message ("Error trying to suspend thread %d : %s",
		 mid, mach_error_string (ret));

      if (from_tty)
	{
	  infoCnt = THREAD_BASIC_INFO_COUNT;
	  ret = thread_info (thread_list[ index ],
			     THREAD_BASIC_INFO,
			     (thread_info_t) &th_info,
			     &infoCnt);
	  CHK ("suspend can't get thread info", ret);
	  
	  message ("Thread %d suspend count is %d",
		   mid, th_info.suspend_count);
	}
    }

  consume_send_rights (thread_list, thread_count);
  ret = vm_deallocate(mach_task_self(),
		      (vm_address_t)thread_list, 
		      (thread_count * sizeof(int)));
  CHK ("Error trying to deallocate thread list", ret);
}

suspend_command (args, from_tty)
     char *args;
     int from_tty;
{
  kern_return_t ret;
  int           mid;
  mach_port_t   saved_thread;
  int           infoCnt;
  thread_basic_info_data_t th_info;
  
  MACH_ERROR_NO_INFERIOR;

  if (!args)
    error_no_arg ("thread mid to suspend or ALL");

  if (!strcasecmp (args, "all")) {
    suspend_all_threads (from_tty);
    return;
  }

  saved_thread = current_thread;

  mid = atoi(args);

  if (mid <= 0)
    error ("You can suspend only kernel threads by MID");

  if (select_thread (inferior_task, mid, 0) != KERN_SUCCESS)
    {
      if (current_thread)
	current_thread = saved_thread;
      error ("Could not select thread %d", mid);
    }

  ret = thread_suspend (current_thread);
  if (ret != KERN_SUCCESS)
    message ("thread_suspend failed : %s",
	     mach_error_string (ret));

  infoCnt = THREAD_BASIC_INFO_COUNT;
  ret = thread_info (current_thread,
		     THREAD_BASIC_INFO,
		     (thread_info_t) &th_info,
		     &infoCnt);
  CHK ("suspend can't get thread info", ret);
  
  message ("Thread %d suspend count is %d", mid, th_info.suspend_count);
  
  current_thread = saved_thread;
}

resume_all_threads (from_tty)
     int from_tty;
{
    kern_return_t  ret;
    thread_array_t thread_list;
    int		   thread_count, index;
    int            mid;
    int		   infoCnt;
    thread_basic_info_data_t th_info;

    ret = task_threads (inferior_task, &thread_list, &thread_count);
    if (ret != KERN_SUCCESS)
      {
	kill_inferior (0,0);
	error("task_threads", mach_error_string( ret));
      }

    for (index = 0; index < thread_count; index++)
      {
	infoCnt = THREAD_BASIC_INFO_COUNT;
	ret = thread_info (thread_list [ index ],
			   THREAD_BASIC_INFO,
			   (thread_info_t) &th_info,
			   &infoCnt);
	CHK ("resume_all can't get thread info", ret);
	
	mid = map_port_name_to_mid (thread_list[ index ],
				    MACH_TYPE_THREAD);
	
	if (! th_info.suspend_count)
	  {
	    if (mid != -1 && from_tty)
	      message ("Thread %d is not suspended", mid);
	    continue;
	  }

	ret = thread_resume (thread_list[ index ]);

	if (ret != KERN_SUCCESS)
	  message ("Error trying to resume thread %d : %s",
		   mid, mach_error_string (ret));
	else if (mid != -1 && from_tty)
	  message ("Thread %d suspend count is %d",
		   mid, --th_info.suspend_count);
      }

    consume_send_rights (thread_list, thread_count);
    ret = vm_deallocate(mach_task_self(),
			(vm_address_t)thread_list, 
			(thread_count * sizeof(int)));
    CHK("Error trying to deallocate thread list", ret);
}

resume_command (args, from_tty)
     char *args;
     int from_tty;
{
  int mid;
  mach_port_t saved_thread;
  kern_return_t ret;
  thread_basic_info_data_t th_info;
  int infoCnt = THREAD_BASIC_INFO_COUNT;
  
  MACH_ERROR_NO_INFERIOR;

  if (!args)
    error_no_arg ("thread mid to resume or ALL to resume all threads");

  if (!strcasecmp (args, "all")) {
    resume_all_threads (from_tty);
    return;
  }

  saved_thread = current_thread;

  mid = atoi(args);

  if (mid <= 0)
    error ("You can resume only kernel threads with MID");

  if (select_thread (inferior_task, mid, 0) != KERN_SUCCESS)
    {
      if (current_thread)
	current_thread = saved_thread;
      return_to_top_level ();
    }

  ret = thread_info (current_thread,
		     THREAD_BASIC_INFO,
		     (thread_info_t) &th_info,
		     &infoCnt);
  CHK ("resume can't get thread info", ret);
  
  if (! th_info.suspend_count)
    {
      message ("Thread %d is not suspended", mid);
      goto out;
    }

  ret = thread_resume (current_thread);
  if (ret != KERN_SUCCESS)
    message ("thread_resume failed : %s",
	     mach_error_string (ret));
  else
    {
      th_info.suspend_count--;
      message ("Thread %d suspend count is %d", mid, th_info.suspend_count);
    }
      
 out:
  current_thread = saved_thread;
}

/* You may either FORWARD the exception to the inferior, or KEEP
 * it and return to GDB command level.
 *
 * exception mid [ forward | keep ]
 */

private void
exception_command (args, from_tty)
     char *args;
     int from_tty;
{
  char *scan = args;
  int exception;
  int len;

  if (!args)
    error_no_arg ("exception number action");

  while (*scan == ' ' || *scan == '\t') scan++;
  
  if ('0' <= *scan && *scan <= '9')
    while ('0' <= *scan && *scan <= '9')
      scan++;
  else
    error ("exception number action");

  exception = atoi (args);
  if (exception <= 0 || exception > MAX_EXCEPTION)
    error ("Allowed exception numbers are in range 1..%d",
	   MAX_EXCEPTION);

  if (*scan != ' ' && *scan != '\t')
    error ("exception number must be followed by a space");
  else
    while (*scan == ' ' || *scan == '\t') scan++;

  args = scan;
  len = 0;
  while (*scan)
    {
      len++;
      scan++;
    }

  if (!len)
    error("exception number action");

  if (!strncasecmp (args, "forward", len))
    exception_map[ exception ].forward = TRUE;
  else if (!strncasecmp (args, "keep", len))
    exception_map[ exception ].forward = FALSE;
  else
    error ("exception action is either \"keep\" or \"forward\"");
}

private void
print_exception_info (exception)
     int exception;
{
  boolean_t forward = exception_map[ exception ].forward;

  printf_filtered ("%s\t(%d): ", exception_map[ exception ].name,
		   exception);
  if (!forward)
    if (exception_map[ exception ].sigmap != SIG_UNKNOWN)
      printf_filtered ("keep and handle as signal %d\n",
		       exception_map[ exception ].sigmap);
    else
      printf_filtered ("keep and handle as unknown signal %d\n",
		       exception_map[ exception ].sigmap);
  else
    printf_filtered ("forward exception to inferior\n");
}

void
exception_info (args, from_tty)
     char *args;
     int from_tty;
{
  int exception;

  if (!args)
    for (exception = 1; exception <= MAX_EXCEPTION; exception++)
      print_exception_info (exception);
  else
    {
      exception = atoi (args);

      if (exception <= 0 || exception > MAX_EXCEPTION)
	error ("Invalid exception number, values from 1 to %d allowed",
	       MAX_EXCEPTION);
      print_exception_info (exception);
    }
}

/* Check for actions for mach exceptions.
 */
mach3_exception_actions (w, force_print_only, who)
     WAITTYPE *w;
     boolean_t force_print_only;
     char *who;
{
  boolean_t force_print = FALSE;

  
  if (force_print_only ||
      exception_map[stop_exception].sigmap == SIG_UNKNOWN)
    force_print = TRUE;
  else
    WSETSTOP (*w, exception_map[stop_exception].sigmap);

  if (exception_map[stop_exception].print || force_print)
    {
      int giveback = grab_terminal ();
      
      printf_filtered ("\n%s received %s exception : ",
		       who,
		       exception_map[stop_exception].name);
      
      wrap_here ("   ");

      switch(stop_exception) {
      case EXC_BAD_ACCESS:
	printf_filtered ("referencing address 0x%x : %s\n",
			 stop_subcode,
			 mach_error_string (stop_code));
	break;
      case EXC_BAD_INSTRUCTION:
	printf_filtered
	  ("illegal or undefined instruction. code %d subcode %d\n",
	   stop_code, stop_subcode);
	break;
      case EXC_ARITHMETIC:
	printf_filtered ("code %d\n", stop_code);
	break;
      case EXC_EMULATION:
	printf_filtered ("code %d subcode %d\n", stop_code, stop_subcode);
	break;
      case EXC_SOFTWARE:
	printf_filtered ("%s specific, code 0x%x\n",
			 stop_code < 0xffff ? "hardware" : "os emulation",
			 stop_code);
	break;
      case EXC_BREAKPOINT:
	printf_filtered ("type %d (machine dependent)\n",
			 stop_code);
	break;
      default:
	fatal ("Unknown exception");
      }
      
      if (giveback)
	terminal_inferior ();
    }
}

setup_notify_port (create_new)
     int create_new;
{
  kern_return_t ret;

  if (MACH_PORT_VALID (our_notify_port))
    {
      ret = mach_port_destroy (mach_task_self (), our_notify_port);
      CHK ("Could not destroy our_notify_port", ret);
    }

  our_notify_port = MACH_PORT_NULL;
  notify_chain    = (port_chain_t) NULL;
  port_chain_destroy (port_chain_obstack);

  if (create_new)
    {
      ret = mach_port_allocate (mach_task_self(),
				MACH_PORT_RIGHT_RECEIVE,
				&our_notify_port);
      if (ret != KERN_SUCCESS)
	fatal("Creating notify port %s", mach_error_string(ret));
      
      ret = mach_port_move_member(mach_task_self(), 
				  our_notify_port,
				  inferior_wait_port_set);
      if (ret != KERN_SUCCESS)
	fatal("initial move member %s",mach_error_string(ret));
    }
}

/*
 * Register our message port to the net name server
 *
 * Currently used only by the external stop-gdb program
 * since ^C does not work if you would like to enter
 * gdb command level while debugging your program.
 *
 * NOTE: If the message port is sometimes used for other
 * purposes also, the NAME must not be a guessable one.
 * Then, there should be a way to change it.
 */

char registered_name[ MAX_NAME_LEN ];

void
message_port_info (args, from_tty)
     char *args;
     int from_tty;
{
  if (registered_name[0])
    printf_filtered ("gdb's message port name: '%s'\n",
		     registered_name);
  else
    printf_filtered ("gdb's message port is not currently registered\n");
}

void
gdb_register_port (name, port)
     char *name;
     mach_port_t port;
{
  kern_return_t ret;
  static int already_signed = 0;
  int len;

  if (! MACH_PORT_VALID (port) || !name || !*name)
    {
      message ("Invalid registration request");
      return;
    }

  if (! already_signed)
    {
      ret = mach_port_insert_right (mach_task_self (),
				    our_message_port,
				    our_message_port,
				    MACH_MSG_TYPE_MAKE_SEND);
      CHK ("Failed to create a signature to our_message_port", ret);
      already_signed = 1;
    }
  else if (already_signed > 1)
    {
      ret = netname_check_out (name_server_port,
			       registered_name,
			       our_message_port);
      CHK ("Failed to check out gdb's message port", ret);
      registered_name[0] = '\000';
      already_signed = 1;
    }

  ret = netname_check_in (name_server_port,	/* Name server port */
			  name,			/* Name of service */
			  our_message_port,	/* Signature */
			  port); 		/* Creates a new send right */
  CHK("Failed to check in the port", ret);
  
  len = 0;
  while(len < MAX_NAME_LEN && *(name+len))
    {
      registered_name[len] = *(name+len);
      len++;
    }
  registered_name[len] = '\000';
  already_signed = 2;
}  

add_mach_specific_commands ()
{
  add_com ("thread-select", class_stack, thread_select_command,
	   "Select and print a thread.");
  add_com_alias ("ts", "thread-select", class_stack, 0);

  add_com ("thread-list", class_stack, thread_list_command,
	   "List all of threads.");
  add_com_alias ("tl", "thread-list", class_stack, 0);

  add_com ("suspend", class_run, suspend_command,
	   "Suspend one or all of the threads in the inferior.");
  add_com ("resume", class_run, resume_command,
	   "Resume one or all of the threads in the inferior.");

  add_info ("exceptions", exception_info,
	    "What debugger does when program gets various exceptions.\n\
Specify an exception number as argument to print info on that\n\
exception only.");

  add_info ("message-port", message_port_info,
	    "Returns the name of gdb's message port in the netnameserver");

  add_com ("exception", class_run, exception_command,
	   "Specify how to handle an exception.\n\
Args are exception number followed by \"forward\" or \"keep\".\n\
`Forward' means forward the exception to the program's normal exception\n\
handler.\n\
`Keep' means reenter debugger if this exception happens, and GDB maps\n\
the exception to some signal (see info exception)\n\
Normally \"keep\" is used to return to GDB on exception.");
}

void
_initialize_mach_os ()
{
  kern_return_t ret;

  ret = mach_port_allocate(mach_task_self(), 
			   MACH_PORT_RIGHT_PORT_SET,
			   &inferior_wait_port_set);
  if (ret != KERN_SUCCESS)
    fatal("initial port set %s",mach_error_string(ret));

  ret = netname_look_up(name_server_port, hostname, "MachID", &mid_server);
  if (ret != KERN_SUCCESS)
    {
      mid_server = MACH_PORT_NULL;

      message ("initialize machid: netname_lookup_up(MachID) : %s",
	       mach_error_string(ret));
      message ("Some features disabled...");
    }

  mid_auth = mach_privileged_host_port();
  if (mid_auth == MACH_PORT_NULL)
    mid_auth = mach_task_self();
    
  obstack_init (port_chain_obstack);

  ret = mach_port_allocate (mach_task_self (), 
			    MACH_PORT_RIGHT_RECEIVE,
			    &thread_exception_port);
  CHK ("Creating thread_exception_port for single stepping", ret);

  ret = mach_port_insert_right (mach_task_self (),
				thread_exception_port,
				thread_exception_port,
				MACH_MSG_TYPE_MAKE_SEND);
  CHK ("Inserting send right to thread_exception_port", ret);

  /* Allocate message port */
  ret = mach_port_allocate (mach_task_self (),
			    MACH_PORT_RIGHT_RECEIVE,
			    &our_message_port);
  if (ret != KERN_SUCCESS)
    message ("Creating message port %s", mach_error_string (ret));
  else
    {
      char buf[ MAX_NAME_LEN ];
      ret = mach_port_move_member(mach_task_self (),
				  our_message_port,
				  inferior_wait_port_set);
      if (ret != KERN_SUCCESS)
	message ("message move member %s", mach_error_string (ret));


      /* @@@@ No way to change name currently */
      /* Foo. This assumes gdb has a unix pid */
      sprintf (buf, "gdb-%d", getpid ());
      gdb_register_port (buf, our_message_port);
    }
  
  /* Heap for thread commands */
  obstack_init (cproc_obstack);

  add_mach_specific_commands ();
}

kern_return_t
do_mach_notify_dead_name (notify, name)
     mach_port_t notify;
     mach_port_t name;
{
  kern_return_t kr = KERN_SUCCESS;

  /* Find the thing that notified */
  port_chain_t element = port_chain_member (notify_chain, name);

  /* Take name of from unreceived dead name notification list */
  notify_chain = port_chain_delete (notify_chain, name);

  if (! element)
    error ("Received a dead name notify from unchained port (0x%x)", name);
  
  switch (element->type) {

  case MACH_TYPE_THREAD:
    if (name == current_thread)
      {
	message ("\nCurrent thread %d died", element->mid);
	current_thread = MACH_PORT_NULL;
      }
    else
      message ("\nThread %d died", element->mid);

    break;

  case MACH_TYPE_TASK:
    if (name != inferior_task)
      message ("Task %d died, but it was not the selected task",
	       element->mid);
    else	       
      {
	message ("Current task %d died", element->mid);
	
	mach_port_destroy (mach_task_self(), name);
	inferior_task = MACH_PORT_NULL;
	
	if (notify_chain)
	  message("There were still unreceived dead_name_notifications???");
	
	/* Destroy the old notifications */
	setup_notify_port (0);

      }
    break;

  default:
    error ("Unregistered dead_name 0x%x notification received. Type is %d, mid is 0x%x",
	   name, element->type, element->mid);
    break;
  }

  return KERN_SUCCESS;
}

kern_return_t
do_mach_notify_msg_accepted (notify, name)
     mach_port_t notify;
     mach_port_t name;
{
  message ("do_mach_notify_msg_accepted : notify %x, name %x",
	   notify, name);
  return KERN_SUCCESS;
}

kern_return_t
do_mach_notify_no_senders (notify, mscount)
     mach_port_t notify;
     mach_port_mscount_t mscount;
{
  message ("do_mach_notify_no_senders : notify %x, mscount %x",
	   notify, mscount);
  return KERN_SUCCESS;
}

kern_return_t
do_mach_notify_port_deleted (notify, name)
     mach_port_t notify;
     mach_port_t name;
{
  message ("do_mach_notify_port_deleted : notify %x, name %x",
	   notify, name);
  return KERN_SUCCESS;
}

kern_return_t
do_mach_notify_port_destroyed (notify, rights)
     mach_port_t notify;
     mach_port_t rights;
{
  message ("do_mach_notify_port_destroyed : notify %x, rights %x",
	   notify, rights);
  return KERN_SUCCESS;
}

kern_return_t
do_mach_notify_send_once (notify)
     mach_port_t notify;
{
#ifdef DUMP_SYSCALL
  /* MANY of these are generated. */
  message ("do_mach_notify_send_once : notify %x",
	   notify);
#endif
  return KERN_SUCCESS;
}

/* NOTE: This is not an RPC call. It is a simpleroutine.
 *
 * This routine is not never called by this program.
 * It may be called by another debugger to cause this
 * debugger to enter command level:
 *
 *            (gdb) set stop_inferior_gdb ()
 *            (gdb) continue
 *
 * External program "stop-gdb" implements this also.
 */
stop_inferior_gdb ()
{
  kern_return_t ret;

  /* Code generated by mig, with minor cleanups :-)
   *
   * simpleroutine stop_inferior_gdb (our_message_port : mach_port_t);
   */

  typedef struct {
    mach_msg_header_t Head;
  } Request;

  Request Mess;

  register Request *InP = &Mess;

  InP->Head.msgh_bits = MACH_MSGH_BITS(MACH_MSG_TYPE_COPY_SEND, 0);

  /* msgh_size passed as argument */
  InP->Head.msgh_remote_port = our_message_port;
  InP->Head.msgh_local_port  = MACH_PORT_NULL;
  InP->Head.msgh_seqno       = 0;
  InP->Head.msgh_id          = GDB_MESSAGE_ID_STOP;

  ret = mach_msg (&InP->Head,
		  MACH_SEND_MSG|MACH_MSG_OPTION_NONE,
		  sizeof(Request),
		  0,
		  MACH_PORT_NULL,
		  MACH_MSG_TIMEOUT_NONE,
		  MACH_PORT_NULL);
}

/* Kills the inferior. It's gone when you call this */
void
kill_inferior_fast ()
{
  WAITTYPE w;

  if (inferior_pid == 0 || inferior_pid == 1)
    return;

  /* kill() it, since the Unix server does not otherwise notice when
   * killed with task_terminate().
   */
  if (inferior_pid > 0)
    kill (inferior_pid, SIGKILL);

  /* It's propably terminate already */
  (void) task_terminate (inferior_task);

  inferior_task  = MACH_PORT_NULL;
  current_thread = MACH_PORT_NULL;

  wait3 (&w, WNOHANG, 0);

  setup_notify_port (0);
}

void
kill_inferior (args, from_tty)
     char *args;
     int from_tty;
{
  kill_inferior_fast ();
  target_mourn_inferior ();
}


/* Mach 3.0 does not need ptrace for anything
 * Make sure nobody uses it on mach.
 */
ptrace (a,b,c,d)
int a,b,c,d;
{
  error ("Loose, Loose! Somebody called ptrace\n");
}

/* Resume execution of the inferior process.
   If STEP is nonzero, single-step it.
   If SIGNAL is nonzero, give it that signal.  */

child_resume (step, signal)
     int step;
     int signal;
{
  kern_return_t	ret;

  vm_read_cache_valid = FALSE;

  if (signal && inferior_pid > 0) /* Do not signal, if attached by MID */
    kill (inferior_pid, signal);

  if (step)
    {
      /* There is no point in single stepping when current_thread
       * is dead.
       */
      if (! MACH_PORT_VALID (current_thread))
	error ("No thread selected; can not single step");

      suspend_all_threads (0);
      setup_single_step (current_thread, TRUE);

      ret = thread_resume (current_thread);
      CHK ("thread_resume", ret);
    }

  ret = task_resume (inferior_task);
  CHK ("Resuming task", ret);


  /* HACK HACK This is needed by the multiserver system HACK HACK */
  while ((ret = task_resume(inferior_task)) == KERN_SUCCESS)
    /* make sure it really runs */;
  /* HACK HACK This is needed by the multiserver system HACK HACK */
}

#ifdef ATTACH_DETACH

/* Start debugging the process with the given task */
void
task_attach (tid)
  task_t tid;
{
  kern_return_t ret;
  inferior_task = tid;

  ret = task_suspend (inferior_task);
  CHK("task_attach: task_suspend", ret);

  setup_notify_port (1);

  request_notify (inferior_task, MACH_NOTIFY_DEAD_NAME, MACH_TYPE_TASK);

  setup_exception_port ();
  
  emulator_present = have_emulator_p (inferior_task);

  select_thread (inferior_task, 0, 1);

  attach_flag = 1;
}

mid_attach (mid)
    int	mid;
{
    kern_return_t ret;

    ret = machid_mach_port (mid_server, mid_auth, mid, &inferior_task);
    CHK("mid_attach: machid_mach_port", ret);

    task_attach (inferior_task);

    return mid;
}

/* 
 * Start debugging the process whose unix process-id is PID.
 * A negative "pid" value is legal and signifies a mach_id not a unix pid.
 *
 * Prevent (possible unwanted) dangerous operations by enabled users
 * like "atta 0" or "atta foo" (equal to the previous :-) and
 * "atta pidself". Anyway, the latter is allowed by specifying a MID.
 */
attach (pid)
     int pid;
{
  kern_return_t ret;

  if (pid == 0)
    error("MID=0, Debugging the master unix server does not compute");

  /* Foo. This assumes gdb has a unix pid */
  if (pid == getpid())
    error ("I will debug myself only by mid. (Gdb would suspend itself!)");

  if (pid < 0)
    {
      mid_attach (-(pid));

      /* inferior_pid will be NEGATIVE! */
      inferior_pid = pid;

      return inferior_pid;
    }

  inferior_task = task_by_pid (pid);
  if (! MACH_PORT_VALID (inferior_task))
    error("Cannot map Unix pid %d to Mach task port", pid);

  task_attach (inferior_task);

  inferior_pid = pid;

  return inferior_pid;
}

void
deallocate_inferior_ports ()
{
  kern_return_t  ret;
  thread_array_t thread_list;
  int		 thread_count, index;

  if (!MACH_PORT_VALID (inferior_task))
    return;

  ret = task_threads (inferior_task, &thread_list, &thread_count);
  if (ret != KERN_SUCCESS)
    {
      message ("deallocate_inferior_ports: task_threads",
	       mach_error_string(ret));
      return;
    }

  /* Get rid of send rights to task threads */
  for (index = 0; index < thread_count; index++)
    {
      int rights;
      ret = mach_port_get_refs (mach_task_self (),
				thread_list[index],
				MACH_PORT_RIGHT_SEND,
				&rights);
      CHK("deallocate_inferior_ports: get refs", ret);

      if (rights > 0)
	{
	  ret = mach_port_mod_refs (mach_task_self (),
				    thread_list[index],
				    MACH_PORT_RIGHT_SEND,
				    -rights);
	  CHK("deallocate_inferior_ports: mod refs", ret);
	}
    }

  ret = mach_port_mod_refs (mach_task_self (),
			    inferior_exception_port,
			    MACH_PORT_RIGHT_RECEIVE,
			    -1);
  CHK ("deallocate_inferior_ports: cannot get rid of exception port", ret);

  ret = mach_port_deallocate (mach_task_self (),
			      inferior_task);
  CHK ("deallocate_task_port: deallocating inferior_task", ret);

  current_thread = MACH_PORT_NULL;
  inferior_task  = MACH_PORT_NULL;
}

/* Stop debugging the process whose number is PID
   and continue it with signal number SIGNAL.
   SIGNAL = 0 means just continue it.  */

void
detach (signal)
     int signal;
{
  kern_return_t ret;

  MACH_ERROR_NO_INFERIOR;

  if (current_thread != MACH_PORT_NULL)
    {
      /* Store the gdb's view of the thread we are deselecting
       * before we detach.
       * @@ I am really not sure if this is ever needeed.
       */
      child_prepare_to_store ();
      target_store_registers (-1);
    }

  ret = task_set_special_port (inferior_task,
			       TASK_EXCEPTION_PORT, 
			       inferior_old_exception_port);
  CHK ("task_set_special_port", ret);

  /* Discard all requested notifications */
  setup_notify_port (0);

  if (remove_breakpoints ())
    message ("Could not remove breakpoints when detaching");
  
  if (signal && inferior_pid > 0)
    kill (inferior_pid, signal);
  
  /* the task might be dead by now */
  (void) task_resume (inferior_task);
  
  deallocate_inferior_ports ();
  
  attach_flag = 0;
}
#endif /* ATTACH_DETACH */

#ifdef DUMP_SYSCALL
#ifdef __STDC__
#define STR(x) #x
#else
#define STR(x) "x"
#endif

char	*bsd1_names[] = {
  "execve",
  "fork",
  "take_signal",
  "sigreturn",
  "getrusage",
  "chdir",
  "chroot",
  "open",
  "creat",
  "mknod",
  "link",
  "symlink",
  "unlink",
  "access",
  "stat",
  "readlink",
  "chmod",
  "chown",
  "utimes",
  "truncate",
  "rename",
  "mkdir",
  "rmdir",
  "xutimes",
  "mount",
  "umount",
  "acct",
  "setquota",
  "write_short",
  "write_long",
  "send_short",
  "send_long",
  "sendto_short",
  "sendto_long",
  "select",
  "task_by_pid",
  "recvfrom_short",
  "recvfrom_long",
  "setgroups",
  "setrlimit",
  "sigvec",
  "sigstack",
  "settimeofday",
  "adjtime",
  "setitimer",
  "sethostname",
  "bind",
  "accept",
  "connect",
  "setsockopt",
  "getsockopt",
  "getsockname",
  "getpeername",
  "init_process",
  "table_set",
  "table_get",
  "pioctl",
  "emulator_error",
  "readwrite",
  "share_wakeup",
  0,
  "maprw_request_it",
  "maprw_release_it",
  "maprw_remap",
  "pid_by_task",
};

int	bsd1_nnames = sizeof(bsd1_names)/sizeof(bsd1_names[0]);

char*
name_str(name,buf)

int	name;
char	*buf;

{
  switch (name) {
  case MACH_MSG_TYPE_BOOLEAN:
    return "boolean";
  case MACH_MSG_TYPE_INTEGER_16:
    return "short";
  case MACH_MSG_TYPE_INTEGER_32:
    return "long";
  case MACH_MSG_TYPE_CHAR:
    return "char";
  case MACH_MSG_TYPE_BYTE:
    return "byte";
  case MACH_MSG_TYPE_REAL:
    return "real";
  case MACH_MSG_TYPE_STRING:
    return "string";
  default:
    sprintf(buf,"%d",name);
    return buf;
  }
}

char *
id_str(id,buf)

int	id;
char	*buf;

{
  char	*p;
  if (id >= 101000 && id < 101000+bsd1_nnames) {
    if (p = bsd1_names[id-101000])
      return p;
  }
  if (id == 102000)
    return "psignal_retry";
  if (id == 100000)
    return "syscall";
  sprintf(buf,"%d",id);
  return buf;
}

print_msg(mp)
mach_msg_header_t	*mp;
{
  char	*fmt_x = "%20s : 0x%08x\n";
  char	*fmt_d = "%20s : %10d\n";
  char	*fmt_s = "%20s : %s\n";
  char	buf[100];

  printf("\n");
#define pr(fmt,h,x) printf(fmt,STR(x),(h).x)
  pr(fmt_x,(*mp),msgh_bits);
  pr(fmt_d,(*mp),msgh_size);
  pr(fmt_x,(*mp),msgh_remote_port);
  pr(fmt_x,(*mp),msgh_local_port);
  pr(fmt_d,(*mp),msgh_kind);
  printf(fmt_s,STR(msgh_id),id_str(mp->msgh_id,buf));
  
  if (debug_level > 1)
  {
    char	*p,*ep,*dp;
    int		plen;
    p = (char*)mp;
    ep = p+mp->msgh_size;
    p += sizeof(*mp);
    for(; p < ep; p += plen) {
      mach_msg_type_t	*tp;
      mach_msg_type_long_t	*tlp;
      int	name,size,number;
      tp = (mach_msg_type_t*)p;
      if (tp->msgt_longform) {
	tlp = (mach_msg_type_long_t*)tp;
	name = tlp->msgtl_name;
	size = tlp->msgtl_size;
	number = tlp->msgtl_number;
	plen = sizeof(*tlp);
      } else {
	name = tp->msgt_name;
	size = tp->msgt_size;
	number = tp->msgt_number;
	plen = sizeof(*tp);
      }
      printf("name=%-16s size=%2d number=%7d inline=%d long=%d deal=%d\n",
	     name_str(name,buf),size,number,tp->msgt_inline,tp->msgt_longform,
	     tp->msgt_deallocate);
      dp = p+plen;
      if (tp->msgt_inline) {
	int	l;
	l = size*number/8;
	l = (l+sizeof(long)-1)&~((sizeof(long))-1);
	plen += l;
	print_data(dp,size,number);
      } else {
	plen += sizeof(int*);
      }
      printf("plen=%d\n",plen);
    }
  }
}

print_data(p,size,number)

char	*p;

{
  int	*ip;
  short	*sp;
  int	i;

  switch (size) {
  case 8:
    for(i = 0; i < number; i++) {
      printf(" %02x",p[i]);
    }
    break;
  case 16:
    sp = (short*)p;
    for(i = 0; i < number; i++) {
      printf(" %04x",sp[i]);
    }
    break;
  case 32:
    ip = (int*)p;
    for(i = 0; i < number; i++) {
      printf(" %08x",ip[i]);
    }
    break;
  }
  printf("\n");
}
#endif  DUMP_SYSCALL
