/* scsi_wini.c
 *
 * The driver supports the following operations (using message format m2):
 *
 *	m_type	  DEVICE   PROC_NR	COUNT	 POSITION  ADRRESS
 * ----------------------------------------------------------------
 * |  DISK_READ | device  | proc nr |  bytes  |	 offset | buf ptr |
 * |------------+---------+---------+---------+---------+---------|
 * | DISK_WRITE | device  | proc nr |  bytes  |	 offset | buf ptr |
 * ----------------------------------------------------------------
 * |SCATTERED_IO| device  | proc nr | requests|         | iov ptr |
 * ----------------------------------------------------------------
 *
 * The file contains one entry point:
 *
 *	 winchester_task:	main entry when system is brought up
 *
 */

#include "kernel.h"
#include <minix/callnr.h>
#include <minix/com.h>
#include <minix/partition.h>
#include "scsi.h"

/* Globals */
PUBLIC int using_bios = FALSE;

/* Defines for host adapter */
#define HA_BASE		0x330

/* Error codes */
#define ERR		  -1	/* general error */

/* Parameters for the disk drive. */
#define MAX_DRIVES         2	/* this driver supports 2 drives (hd0 - hd9)*/
#define DEV_PER_DRIVE   (1 + NR_PARTITIONS)	/* whole drive & each partn */
#define NR_DEVICES      (MAX_DRIVES * DEV_PER_DRIVE)
#define SECTOR_SIZE	 512	/* physical sector size in bytes */
#define SEC_PER_CYL	(64 * 32)

PRIVATE struct mbox_s mboxes[2];
PRIVATE struct ccb_s  ccb;
PRIVATE unsigned char buf[BLOCK_SIZE];
PRIVATE message w_mess;		/* message buffer for in and out */

PRIVATE struct wini {		/* main drive struct, one entry per drive */
  int wn_opcode;		/* DISK_READ or DISK_WRITE */
  int wn_procnr;		/* which proc wanted this operation? */
  int wn_count;			/* byte count */
  vir_bytes wn_address;		/* user virtual address */
  int wn_sid;			/* SCSI id of drive */
  int wn_secsize;		/* Logical block size */
  long wn_first;		/* First block # of this partition */
  long wn_size;			/* Size in blocks of partition */
} wini[NR_DEVICES];


PRIVATE void
ha_clen(byte_array, l)
unsigned char *byte_array;
unsigned long l;
{
    *byte_array++ = (l >> 16) & 0xFF;
    *byte_array++ = (l >> 8) & 0xFF;
    *byte_array++ = l & 0xFF;
}

PRIVATE void
ha_caddr(byte_array, ptr)
unsigned char *byte_array;
phys_bytes ptr;
{
    unsigned long l;

    l = (unsigned long) ptr;
    *byte_array++ = (l >> 16) & 0xFF;
    *byte_array++ = (l >> 8) & 0xFF;
    *byte_array++ = l & 0xFF;
}

PRIVATE unsigned char
ha_reset()
{
    unsigned char stat;

    out_byte(HA_BASE, 0x80);
    while ((stat = in_byte(HA_BASE)) & 0x80)
	;

    return stat;
}

PRIVATE void
ha_intr_reset()
{
    out_byte(HA_BASE, 0x20);
}

PRIVATE void
ha_command(cmd_buffer, cmd_len)
unsigned char *cmd_buffer;
int cmd_len;
{
    int i;

    for (i = 0; i < cmd_len; i++)
    {
	while (in_byte(HA_BASE) & 0x08)
	    ;
	out_byte(HA_BASE+1, cmd_buffer[i]);
    }
}

PRIVATE void
ha_init()
{
    unsigned char mybuf[5];
    phys_bytes paddr;
    
    ha_reset();
    
    mybuf[0] = 1;
    mybuf[1] = 1;
    paddr = numap(WINCHESTER, mboxes, 8);
    ha_caddr(mybuf+2, paddr);
    paddr = numap(WINCHESTER, &ccb, sizeof(ccb));
    ha_caddr(mboxes[0].mb_addr, paddr);
    
    cim_scsi_wini();
    ha_command(mybuf, 5);
    receive(HARDWARE, &w_mess);
    ha_intr_reset();
}

PRIVATE int
scsi_mode_sense(sid, databuf)
unsigned char sid;
phys_bytes databuf;
{
    unsigned char mybuf;
    
    ccb.ccb_buf[0] = 0x1A;		/* Mode sense command */
    ccb.ccb_buf[1] = 0;			/* LUN = 0 */
    ccb.ccb_buf[2] = 3;			/* Current values of DASD Parms */
    ccb.ccb_buf[3] = 0;			/* Reserved */
    ccb.ccb_buf[4] = 12;		/* Allocation length */
    ccb.ccb_buf[5] = 0;			/* Vendor unique, link flag */
    
    ccb.ccb_op_code = 0;		/* SCSI Initiator command */
    ccb.ccb_target = (sid << 5) & 0xE0;	/* SCSI target id */
    ccb.ccb_cmd_len = 6;		/* SCSI command length */
    ccb.ccb_rs_len = 12;		/* Request Sense buffer length */
    ha_clen(ccb.ccb_dlen, (long) 12);
    ha_caddr(ccb.ccb_daddr, databuf);
    ha_caddr(ccb.ccb_laddr, (phys_bytes) 0);
    ccb.ccb_lid = 0;
    
    mboxes[0].mb_status = 1;
    mboxes[1].mb_status = 0;
    mybuf = 2;
    ha_command(&mybuf, 1);
}

PRIVATE int
scsi_xfer(sid, cmd, block, nblocks, databuf, bufsize)
unsigned char sid;
unsigned char cmd;
unsigned long block;
unsigned char nblocks;
phys_bytes    databuf;
unsigned long bufsize;
{
    unsigned char mybuf;
    
    ccb.ccb_buf[0] = cmd;		/* Mode sense command */
    ccb.ccb_buf[1] = (block >> 16) & 0x1F;
    ccb.ccb_buf[2] = (block >> 8) & 0xFF;
    ccb.ccb_buf[3] = block & 0xFF;
    ccb.ccb_buf[4] = nblocks;
    ccb.ccb_buf[5] = 0;			/* Vendor unique, link flag */
    
    ccb.ccb_op_code = 0;		/* SCSI Initiator command */
    ccb.ccb_target = (sid << 5) & 0xE0;	/* SCSI target id */
    ccb.ccb_cmd_len = 6;		/* SCSI command length */
    ccb.ccb_rs_len = 12;		/* Request Sense buffer length */
    ha_clen(ccb.ccb_dlen, bufsize);
    ha_caddr(ccb.ccb_daddr, databuf);
    ha_caddr(ccb.ccb_laddr, (phys_bytes) 0);
    ccb.ccb_lid = 0;
    
    mboxes[0].mb_status = 1;
    mboxes[1].mb_status = 0;
    mybuf = 2;
    ha_command(&mybuf, 1);
    
    return 0;
}

PRIVATE void
scsi_wait()
{
    unsigned char stat;
    
    stat = 0;
    while ((stat & 0x01) == 0)
    {
	cim_scsi_wini();
	receive(HARDWARE, &w_mess);
	stat = in_byte(HA_BASE+2);
	ha_intr_reset();
    }	
}

PRIVATE int
read_parms(drive)
int drive;				/* Drive number to access */
{
    int retries;			/* allow for two retries */
    int wini_idx, rc;
    long l;
    phys_bytes paddr;
    
    paddr = numap(WINCHESTER, buf, 12);
    
    rc = ERR;
    for (retries = 0; retries < 2; retries++)
    {
        scsi_mode_sense(drive, paddr);
        scsi_wait();
        if (ccb.ccb_hastat == 0 && ccb.ccb_tdstat == 0)
        {
            rc = 0;
            break;
        }    
    }
    
    if (!rc)
    {
        wini_idx = drive * DEV_PER_DRIVE;
        
        l = ((long) buf[9] << 16) +  ((long) buf[10] << 8) + (long) buf[11];
        if (l != (long) SECTOR_SIZE)
            rc = ERR;
        else
        {    
	    wini[wini_idx].wn_sid = drive;
	    wini[wini_idx].wn_secsize = (int) l;
	    
	    l = ((long) buf[5] << 16) +  ((long) buf[6] << 8) + (long) buf[7];
	    wini[wini_idx].wn_size = l;
	}    
    }           
    
    return rc;
}    

PRIVATE int
read_prt(drive)
int drive;
{
    register struct part_entry *pe;
    register struct wini *wn;
    int wini_idx, rc, i, secsize;
    phys_bytes paddr;
    
    paddr = numap(WINCHESTER, buf, SECTOR_SIZE);
    scsi_xfer(drive, SCSI_READ, (long) 0, 1, paddr, SECTOR_SIZE);
    scsi_wait();
    if (ccb.ccb_hastat != 0 || ccb.ccb_tdstat != 0)
        return ERR;
        
    wini_idx = drive * DEV_PER_DRIVE;
    secsize = wini[wini_idx].wn_secsize;
    
    wn = &wini[wini_idx+1];
    pe = (struct part_entry *) &buf[PART_TABLE_OFF];
    for (i = 0; i < NR_PARTITIONS; i++, wn++, pe++)
    {
        wn->wn_sid = drive;
        wn->wn_secsize = secsize;
        wn->wn_first = pe->lowsec;
        wn->wn_size = pe->size;
        
        if (pe->sysind == OLD_MINIX_PART && wn->wn_first & 1)
        {
            wn->wn_first++;
            wn->wn_size--;
        }
    }    
    
    return 0;
}

PRIVATE int
w_do_rdwt(m_ptr)
message *m_ptr;
{
    register struct wini *wn;
    phys_bytes paddr;
    int device, nsectors, cmd, rc;
    long sector;
    
    /* Check parms */
    device = m_ptr->DEVICE;
    if (device < 0 || device > NR_DEVICES)
        return EIO;
        
    if ((m_ptr->COUNT != BLOCK_SIZE) || (m_ptr->POSITION % BLOCK_SIZE != 0))
        return EINVAL;
        
    wn = &wini[device];
    nsectors = BLOCK_SIZE / wn->wn_secsize;
    sector = m_ptr->POSITION / wn->wn_secsize;
    
    if (sector + nsectors >= wn->wn_size)
        return 0;
    sector += wn->wn_first;    
    
    paddr = numap(m_ptr->PROC_NR, m_ptr->ADDRESS, BLOCK_SIZE);
    if (m_ptr->m_type == DISK_READ)
        cmd = SCSI_READ;
    else
        cmd = SCSI_WRITE;
        
    rc = scsi_xfer(wn->wn_sid, cmd, sector, nsectors, paddr, BLOCK_SIZE);
    scsi_wait();
    if (ccb.ccb_hastat != 0 || ccb.ccb_tdstat != 0 || rc)
        return EIO;
        
    return BLOCK_SIZE;
}

/*=========================================================================*
 *			winchester_task					   * 
 *=========================================================================*/
PUBLIC void winchester_task()
{
    int caller, proc_nr, rc;
    int i, j, widx;

    /* First initialize the controller */
    ha_init();

    /*
     * Read drive parms from drives.  If that is successful, then read
     * partition tables.
     */
    for (i = 0; i < MAX_DRIVES; i++)
        if (read_parms(i) == 0)
            read_prt(i);
            
    for (i = 0; i < MAX_DRIVES; i++)
    {
        widx = i * DEV_PER_DRIVE;
        printf("DRIVE %d:\n", i);
        for (j = 0; j < NR_PARTITIONS + 1; j++)
            printf("    P%d: START=%ld, SIZE=%ld, D=%d, SS=%d\n", 
                   j, wini[widx+j].wn_first, wini[widx+j].wn_size,
                   wini[widx+j].wn_sid, wini[widx+j].wn_secsize);
    }               

    while (TRUE)
    {
        receive(ANY, &w_mess);
        caller = w_mess.m_source;
        proc_nr = w_mess.PROC_NR;
        
        switch (w_mess.m_type)
        {
            case DISK_READ:
            case DISK_WRITE:
                rc = w_do_rdwt(&w_mess);
                break;
            case SCATTERED_IO:
                rc = do_vrdwt(&w_mess, w_do_rdwt);
                break;    
            default:
                rc = EINVAL;
                break;
        }
        
        w_mess.m_type = TASK_REPLY;
        w_mess.REP_PROC_NR = proc_nr;
        w_mess.REP_STATUS = rc;
        send(caller, &w_mess);
    }    
}
