/* Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * mini-X graphics - public definitions
 */

#ifndef	GRAPHICS_H
#define	GRAPHICS_H

/* Basic typedefs. */
typedef short GR_COORD;		/* coordinate value */
typedef short GR_SIZE;		/* size value */
typedef short GR_COUNT;		/* number of items */
typedef unsigned short GR_COLOR;/* full color value */
typedef unsigned char GR_COLOR8;/* eight bit color value */
typedef unsigned short GR_BITMAP;	/* bitmap unit */
typedef unsigned short GR_MODE;	/* drawing mode */
typedef unsigned char GR_CHAR;	/* text character */
typedef unsigned char GR_CHAR_WIDTH;	/* width of character */
typedef unsigned long GR_ID;	/* resource ids */
typedef GR_ID GR_DRAW_ID;	/* drawable id */
typedef GR_DRAW_ID GR_WINDOW_ID;/* window id */
typedef GR_DRAW_ID GR_PIXMAP_ID;/* pixmap id (not yet used) */
typedef GR_ID GR_GC_ID;		/* graphics context id */
typedef unsigned short GR_FONT;	/* font number */
typedef unsigned short GR_BOOL;	/* boolean value */
typedef unsigned short GR_FUNC;	/* function codes */
typedef int GR_ERROR;		/* error value */
typedef short GR_EVENT_TYPE;	/* event types */
typedef unsigned short GR_BUTTON;	/* button flags */
typedef unsigned short GR_MODIFIER;	/* modifier flags (CTRL, SHIFT, etc) */
typedef unsigned long GR_EVENT_MASK;	/* event masks */
typedef char GR_FUNC_NAME[20];	/* function name */
typedef void (*GR_ERROR_FUNC) ();	/* error function */


/* Limits */
#define	GR_COORD_MIN	((GR_COORD) -32768)	/* minimum coordinate value */
#define	GR_COORD_MAX	((GR_COORD) 32767)	/* maximum coordinate value */
#define	GR_MAX_CLIPRECTS ((GR_COUNT) 200)	/* maximum clip rectangles */
#define	GR_MAX_CURSOR_SIZE ((GR_SIZE) 16)	/* maximum cursor size */


/* Size of bitmaps. */
#define	GR_BITMAP_SIZE(width, height)	((height) * \
  (((width) + sizeof(GR_BITMAP) * 8 - 1) / (sizeof(GR_BITMAP) * 8)))

#define	GR_MAX_BITMAP_SIZE \
  GR_BITMAP_SIZE(GR_MAX_CURSOR_SIZE, GR_MAX_CURSOR_SIZE)


/* The root window id. */
#define	GR_ROOT_WINDOW_ID	((GR_WINDOW_ID) 1)


/* Drawing mode (used with GrSetGCMode). */
#define	GR_MODE_SET	((GR_MODE) 0)	/* draw pixels as given (default) */
#define	GR_MODE_XOR	((GR_MODE) 1)	/* draw pixels using XOR */
#define	GR_MODE_OR	((GR_MODE) 2)	/* draw pixels using OR */
#define	GR_MODE_AND	((GR_MODE) 3)	/* draw pixels using AND */
#define	GR_MAX_MODE	((GR_MODE) 3)	/* maximum legal mode */


/* Booleans. */
#define	GR_FALSE	((GR_BOOL) 0)
#define	GR_TRUE		((GR_BOOL) 1)


/* Definition of a point. */
typedef struct {
  GR_COORD x;			/* x coordinate */
  GR_COORD y;			/* y coordinate */
} GR_POINT;


/* Definition of a line. */
typedef struct {
  GR_COORD x1;			/* x coordinate of first point */
  GR_COORD y1;			/* y coordinate of first point */
  GR_COORD x2;			/* x coordinate of second point */
  GR_COORD y2;			/* y coordinate of second point */
} GR_LINE;


/* Definition of an ellipse. */
typedef struct {
  GR_COORD x;			/* x coordinate of center */
  GR_COORD y;			/* y coordinate of center */
  GR_SIZE rx;			/* radius in x direction */
  GR_SIZE ry;			/* radius in y direction */
} GR_ELLIPSE;


/* Definition of a rectangle. */
typedef struct {
  GR_COORD x;			/* x coordinate of top left corner */
  GR_COORD y;			/* y coordinate of top left corner */
  GR_SIZE width;		/* width of rectangle */
  GR_SIZE height;		/* height of rectangle */
} GR_RECT;


/* Screen properties returned by the GrGetScreenInfo call.
 * This describes the size of the screen and relative dots/cm of the pixels.
 * The actual dots/mm vary depending on the actual monitor in use, but the
 * relative sizes are still useful to compensate for non-square pixels for
 * drawing perfect circles.  (I am avoiding the use of floating point.)
 */
typedef struct {
  GR_SIZE rows;			/* number of rows on screen */
  GR_SIZE cols;			/* number of columns on screen */
  GR_SIZE xdpcm;		/* dots/centimeter in x direction */
  GR_SIZE ydpcm;		/* dots/centimeter in y direction */
  GR_COLOR maxcolor;		/* maximum legal color value */
  GR_COLOR black;		/* the color black */
  GR_COLOR white;		/* the color white */
  GR_BUTTON buttons;		/* buttons which are implemented */
  GR_MODIFIER modifiers;	/* modifiers which are implemented */
  GR_COUNT fonts;		/* number of built-in fonts */
} GR_SCREEN_INFO;


/* Window properties returned by the GrGetWindowInfo call. */
typedef struct {
  GR_WINDOW_ID wid;		/* window id (or 0 if no such window) */
  GR_WINDOW_ID parent;		/* parent window id */
  GR_WINDOW_ID child;		/* first child window id (or 0) */
  GR_WINDOW_ID sibling;		/* next sibling window id (or 0) */
  GR_BOOL inputonly;		/* TRUE if window is input only */
  GR_BOOL mapped;		/* TRUE if window is mapped */
  GR_COUNT unmapcount;		/* reasons why window is unmapped */
  GR_COORD x;			/* absolute x position of window */
  GR_COORD y;			/* absolute y position of window */
  GR_SIZE width;		/* width of window */
  GR_SIZE height;		/* height of window */
  GR_SIZE bordersize;		/* size of border */
  GR_COLOR bordercolor;		/* color of border */
  GR_COLOR background;		/* background color */
  GR_EVENT_MASK eventmask;	/* current event mask for this client */
} GR_WINDOW_INFO;


/* Graphics context properties returned by the GrGetGCInfo call. */
typedef struct {
  GR_GC_ID gcid;		/* GC id (or 0 if no such GC) */
  GR_MODE mode;			/* drawing mode */
  GR_FONT font;			/* font number */
  GR_COLOR foreground;		/* foreground color */
  GR_COLOR background;		/* background color */
  GR_BOOL usebackground;	/* use background in bitmaps */
} GR_GC_INFO;


/* Font information. */
typedef struct {
  GR_FONT font;			/* font number */
  GR_SIZE height;		/* height of font */
  GR_SIZE maxwidth;		/* maximum width of any char */
  GR_SIZE baseline;		/* baseline of font */
  GR_BOOL fixed;		/* TRUE if font is fixed width */
  GR_CHAR_WIDTH widths[256];	/* table of character widths */
} GR_FONT_INFO;


/* Error codes */
#define	GR_ERROR_BAD_WINDOW_ID		((GR_ERROR) 1)
#define	GR_ERROR_BAD_GC_ID		((GR_ERROR) 2)
#define	GR_ERROR_BAD_CURSOR_SIZE	((GR_ERROR) 3)
#define	GR_ERROR_MALLOC_FAILED		((GR_ERROR) 4)
#define	GR_ERROR_BAD_WINDOW_SIZE	((GR_ERROR) 5)
#define	GR_ERROR_KEYBOARD_ERROR		((GR_ERROR) 6)
#define	GR_ERROR_MOUSE_ERROR		((GR_ERROR) 7)
#define	GR_ERROR_INPUT_ONLY_WINDOW	((GR_ERROR) 8)
#define	GR_ERROR_ILLEGAL_ON_ROOT_WINDOW	((GR_ERROR) 9)
#define	GR_ERROR_TOO_MUCH_CLIPPING	((GR_ERROR) 10)
#define	GR_ERROR_SCREEN_ERROR		((GR_ERROR) 11)
#define	GR_ERROR_UNMAPPED_FOCUS_WINDOW	((GR_ERROR) 12)
#define	GR_ERROR_BAD_DRAWING_MODE	((GR_ERROR) 13)


/* Event types.
 * Mouse motion is generated for every motion of the mouse, and is used to
 * track the entire history of the mouse (many events and lots of overhead).
 * Mouse position ignores the history of the motion, and only reports the
 * latest position of the mouse by only queuing the latest such event for
 * any single client (good for rubber-banding).
 */
#define	GR_EVENT_TYPE_ERROR		((GR_EVENT_TYPE) -1)
#define	GR_EVENT_TYPE_NONE		((GR_EVENT_TYPE) 0)
#define	GR_EVENT_TYPE_EXPOSURE		((GR_EVENT_TYPE) 1)
#define	GR_EVENT_TYPE_BUTTON_DOWN	((GR_EVENT_TYPE) 2)
#define	GR_EVENT_TYPE_BUTTON_UP		((GR_EVENT_TYPE) 3)
#define	GR_EVENT_TYPE_MOUSE_ENTER	((GR_EVENT_TYPE) 4)
#define	GR_EVENT_TYPE_MOUSE_EXIT	((GR_EVENT_TYPE) 5)
#define	GR_EVENT_TYPE_MOUSE_MOTION	((GR_EVENT_TYPE) 6)
#define	GR_EVENT_TYPE_MOUSE_POSITION	((GR_EVENT_TYPE) 7)
#define	GR_EVENT_TYPE_KEY_DOWN		((GR_EVENT_TYPE) 8)
#define	GR_EVENT_TYPE_KEY_UP		((GR_EVENT_TYPE) 9)
#define	GR_EVENT_TYPE_FOCUS_IN		((GR_EVENT_TYPE) 10)
#define	GR_EVENT_TYPE_FOCUS_OUT		((GR_EVENT_TYPE) 11)


/* Event masks */
#define	GR_EVENTMASK(n)			(((GR_EVENT_MASK) 1) << (n))

#define	GR_EVENT_MASK_NONE		GR_EVENTMASK(GR_EVENT_TYPE_NONE)
#define	GR_EVENT_MASK_ERROR		GR_EVENTMASK(GR_EVENT_TYPE_ERROR)
#define	GR_EVENT_MASK_EXPOSURE		GR_EVENTMASK(GR_EVENT_TYPE_EXPOSURE)
#define	GR_EVENT_MASK_BUTTON_DOWN	GR_EVENTMASK(GR_EVENT_TYPE_BUTTON_DOWN)
#define	GR_EVENT_MASK_BUTTON_UP		GR_EVENTMASK(GR_EVENT_TYPE_BUTTON_UP)
#define	GR_EVENT_MASK_MOUSE_ENTER	GR_EVENTMASK(GR_EVENT_TYPE_MOUSE_ENTER)
#define	GR_EVENT_MASK_MOUSE_EXIT	GR_EVENTMASK(GR_EVENT_TYPE_MOUSE_EXIT)
#define	GR_EVENT_MASK_MOUSE_MOTION	GR_EVENTMASK(GR_EVENT_TYPE_MOUSE_MOTION)
#define	GR_EVENT_MASK_MOUSE_POSITION	GR_EVENTMASK(GR_EVENT_TYPE_MOUSE_POSITION)
#define	GR_EVENT_MASK_KEY_DOWN		GR_EVENTMASK(GR_EVENT_TYPE_KEY_DOWN)
#define	GR_EVENT_MASK_KEY_UP		GR_EVENTMASK(GR_EVENT_TYPE_KEY_UP)
#define	GR_EVENT_MASK_FOCUS_IN		GR_EVENTMASK(GR_EVENT_TYPE_FOCUS_IN)
#define	GR_EVENT_MASK_FOCUS_OUT		GR_EVENTMASK(GR_EVENT_TYPE_FOCUS_OUT)
#define	GR_EVENT_MASK_ALL		((GR_EVENT_MASK) -1L)


/* Modifiers generated by special keyboard shift-like keys.
 * The state of these keys can be read as up or down, and don't
 * generate any characters by themselves.
 */
#define	GR_MODIFIER_SHIFT	((GR_MODIFIER) 0x01)	/* shift */
#define	GR_MODIFIER_CTRL	((GR_MODIFIER) 0x02)	/* control */
#define	GR_MODIFIER_META	((GR_MODIFIER) 0x03)	/* meta or alt */
#define	GR_MODIFIER_ANY		((GR_MODIFIER) 0x07)	/* any modifier */


/* Button flags. */
#define	GR_BUTTON_1	((GR_BUTTON) 0x01)	/* button 1 */
#define	GR_BUTTON_2	((GR_BUTTON) 0x02)	/* button 2 */
#define	GR_BUTTON_3	((GR_BUTTON) 0x04)	/* button 3 */
#define	GR_BUTTON_ANY	((GR_BUTTON) 0x07)	/* any button */


/* Event for errors detected by the server.
 * These events are not delivered to GrGetNextEvent, but instead call
 * the user supplied error handling function.  Only the first one of
 * these errors at a time is saved for delivery to the client since
 * there is not much to be done about errors anyway except complain
 * and exit.
 */
typedef struct {
  GR_EVENT_TYPE type;		/* event type */
  GR_FUNC_NAME name;		/* function name which failed */
  GR_ERROR code;		/* error code */
  GR_ID id;			/* resource id (maybe useless) */
} GR_EVENT_ERROR;


/* Event for a mouse button pressed down or released. */
typedef struct {
  GR_EVENT_TYPE type;		/* event type */
  GR_WINDOW_ID wid;		/* window id event delivered to */
  GR_WINDOW_ID subwid;		/* sub-window id (pointer was in) */
  GR_COORD rootx;		/* root window x coordinate */
  GR_COORD rooty;		/* root window y coordinate */
  GR_COORD x;			/* window x coordinate of mouse */
  GR_COORD y;			/* window y coordinate of mouse */
  GR_BUTTON buttons;		/* current state of all buttons */
  GR_BUTTON changebuttons;	/* buttons which went down or up */
  GR_MODIFIER modifiers;	/* modifiers (SHIFT, CTRL, etc) */
} GR_EVENT_BUTTON;


/* Event for a keystroke typed for the window with has focus. */
typedef struct {
  GR_EVENT_TYPE type;		/* event type */
  GR_WINDOW_ID wid;		/* window id event delived to */
  GR_WINDOW_ID subwid;		/* sub-window id (pointer was in) */
  GR_COORD rootx;		/* root window x coordinate */
  GR_COORD rooty;		/* root window y coordinate */
  GR_COORD x;			/* window x coordinate of mouse */
  GR_COORD y;			/* window y coordinate of mouse */
  GR_BUTTON buttons;		/* current state of buttons */
  GR_MODIFIER modifiers;	/* modifiers (SHIFT, CTRL, etc) */
  GR_CHAR ch;			/* keystroke or function key */
} GR_EVENT_KEYSTROKE;


/* Event for exposure for a region of a window. */
typedef struct {
  GR_EVENT_TYPE type;		/* event type */
  GR_WINDOW_ID wid;		/* window id */
  GR_COORD x;			/* window x coordinate of exposure */
  GR_COORD y;			/* window y coordinate of exposure */
  GR_SIZE width;		/* width of exposure */
  GR_SIZE height;		/* height of exposure */
} GR_EVENT_EXPOSURE;


/* General events for focus in or focus out for a window, or mouse enter
 * or mouse exit from a window, or window unmapping or mapping, etc.
 */
typedef struct {
  GR_EVENT_TYPE type;		/* event type */
  GR_WINDOW_ID wid;		/* window id */
} GR_EVENT_GENERAL;


/* Events for mouse motion or mouse position. */
typedef struct {
  GR_EVENT_TYPE type;		/* event type */
  GR_WINDOW_ID wid;		/* window id event delivered to */
  GR_WINDOW_ID subwid;		/* sub-window id (pointer was in) */
  GR_COORD rootx;		/* root window x coordinate */
  GR_COORD rooty;		/* root window y coordinate */
  GR_COORD x;			/* window x coordinate of mouse */
  GR_COORD y;			/* window y coordinate of mouse */
  GR_BUTTON buttons;		/* current state of buttons */
  GR_MODIFIER modifiers;	/* modifiers (ALT, SHIFT, etc) */
} GR_EVENT_MOUSE;


/* Union of all possible event structures.
 * This is the structure returned by the GrGetNextEvent and similar routines.
 */
typedef union {
  GR_EVENT_TYPE type;		/* event type */
  GR_EVENT_ERROR error;		/* error event */
  GR_EVENT_GENERAL general;	/* general window events */
  GR_EVENT_BUTTON button;	/* button events */
  GR_EVENT_KEYSTROKE keystroke;	/* keystroke events */
  GR_EVENT_EXPOSURE exposure;	/* exposure events */
  GR_EVENT_MOUSE mouse;		/* mouse motion events */
} GR_EVENT;


/* Public graphics routines. */
extern int GrOpen();
extern void GrClose();
extern void GrGetScreenInfo();
extern GR_ERROR_FUNC GrSetErrorHandler();
extern GR_WINDOW_ID GrNewWindow();
extern GR_WINDOW_ID GrNewInputWindow();
extern void GrDestroyWindow();
extern GR_GC_ID GrNewGC();
extern void GrGetGCInfo();
extern void GrDestroyGC();
extern void GrMapWindow();
extern void GrUnmapWindow();
extern void GrRaiseWindow();
extern void GrLowerWindow();
extern void GrMoveWindow();
extern void GrResizeWindow();
extern void GrGetWindowInfo();
extern void GrGetFontInfo();
extern void GrSetFocus();
extern void GrSetBorderColor();
extern void GrClearWindow();
extern void GrSelectEvents();
extern void GrGetNextEvent();
extern void GrCheckNextEvent();
extern void GrPeekEvent();
extern void GrFlush();
extern void GrLine();
extern void GrPoint();
extern void GrRect();
extern void GrFillRect();
extern void GrPoly();
extern void GrFillPoly();
extern void GrEllipse();
extern void GrFillEllipse();
extern void GrSetGCForeground();
extern void GrSetGCBackground();
extern void GrSetGCUseBackground();
extern void GrSetGCMode();
extern void GrSetGCFont();
extern void GrGetGCTextSize();
extern void GrReadArea8();
extern void GrArea8();
extern void GrBitmap();
extern void GrText();
extern void GrSetCursor();
extern void GrMoveCursor();

#endif

/* END CODE */
