/* Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Generic routines to keep track of the cursor.
 * This implements totally software driven graphics cursors.
 * If your system has hardware cursors, then none of this is
 * needed, and a set of functions for your hardware will suffice.
 */

#include "kernel.h"
#include "graph_dev.h"
#include <minix/graph_msg.h>


/* The following definition is used to save old colors for the cursor.
 * For space reasons, this is defined to be 8 bit colors.  If your colors
 * are more than 8 bits, then redefine this.
 */
#ifndef GR_COLORN
#define	GR_COLORN	GR_COLOR8
#endif


/* Cursor size, position, colors, and bitmap. */
PRIVATE GR_BOOL curenabled;	/* TRUE if cursor is visible */
PRIVATE GR_BOOL curneedfix;	/* TRUE if cursor needs fixing */
PRIVATE GR_COORD curminx;	/* minimum x value of cursor */
PRIVATE GR_COORD curminy;	/* minimum y value of cursor */
PRIVATE GR_COORD curmaxx;	/* maximum x value of cursor */
PRIVATE GR_COORD curmaxy;	/* maximum y value of cursor */
PRIVATE GR_COLOR curfg;		/* foreground color of cursor */
PRIVATE GR_COLOR curbg;		/* background color of cursor */
PRIVATE GR_BITMAP curfgbitmap[GR_MAX_CURSOR_SIZE];	/* foreground bitmap */
PRIVATE GR_BITMAP curbgbitmap[GR_MAX_CURSOR_SIZE];	/* background bitmap */


/* Saved pixels from graphics screen so they can be restored later.
 * The only pixels that are actually saved are the ones where the foreground
 * or background bitmaps are nonzero.
 */
PRIVATE GR_COLORN savedpoints[GR_MAX_CURSOR_SIZE * GR_MAX_CURSOR_SIZE];


FORWARD removecursor();
FORWARD showcursor();


/*===========================================================================*
 *				gen_initcursor				     *
 *===========================================================================*/
PUBLIC void gen_initcursor()
{
/* Initialize the cursor when graphics is initialized.  This just indicates
 * that the cursor is not being show.
 */
  curenabled = GR_FALSE;
  curneedfix = GR_FALSE;
}


/*===========================================================================*
 *				gen_setcursor				     *
 *===========================================================================*/
PUBLIC void gen_setcursor(width, height, foreground, background, fgbitmap, bgbitmap)
GR_SIZE width;			/* width of cursor */
GR_SIZE height;			/* height of cursor */
GR_COLOR foreground;		/* foreground color */
GR_COLOR background;		/* background color */
GR_BITMAP *fgbitmap;		/* foreground bitmap table */
GR_BITMAP *bgbitmap;		/* background bitmap table */
{
/* Set the size, shape, and colors of the cursor.
 * This first removes the old cursor if there was one visible.
 * The new cursor is only drawn if the old one was too.
 */
  GR_BOOL saveenabled;		/* saved enabled flag */
  int bytes;			/* number of bytes to copy */

  if ((width <= 0) || (width > GR_MAX_CURSOR_SIZE)
      || (height <= 0) || (height > GR_MAX_CURSOR_SIZE))
	return;

  saveenabled = removecursor();
  curfg = foreground;
  curbg = background;
  curmaxx = curminx + width - 1;
  curmaxy = curminy + height - 1;
  bytes = BITMAP_WORDS(width) * height * sizeof(GR_BITMAP);
  memcpy(curfgbitmap, fgbitmap, bytes);
  memcpy(curbgbitmap, bgbitmap, bytes);
  if (saveenabled) showcursor();
}


/*===========================================================================*
 *				gen_movecursor				     *
 *===========================================================================*/
PUBLIC void gen_movecursor(x, y)
GR_COORD x;			/* new x position of cursor */
GR_COORD y;			/* new y position of cursor */
{
/* Change the position of the cursor to the specified coordinates.
 * This makes the cursor visible if it was not previously visible.
 * The cursor can be made invisible by moving it off of the screen.
 */
  GR_COORD shiftx;
  GR_COORD shifty;

  shiftx = x - curminx;
  shifty = y - curminy;
  if ((shiftx == 0) && (shifty == 0)) return;
  removecursor();
  curminx += shiftx;
  curmaxx += shiftx;
  curminy += shifty;
  curmaxy += shifty;
  if ((curmaxx >= 0) && (curminx < gr_dev.cols) && (curmaxy >= 0)
      && (curminy < gr_dev.rows))
	showcursor();
}


/*===========================================================================*
 *				removecursor				     *
 *===========================================================================*/
PRIVATE int removecursor()
{
/* Remove the cursor from the screen.
 * This just restores the pixels that were saved before the cursor was put
 * on the screen.  Returns the previous state of the cursor enabled flag.
 */
  GR_COLORN *saveptr;
  GR_BITMAP *fgbitptr;
  GR_BITMAP *bgbitptr;
  GR_COORD x;
  GR_COORD y;
  GR_BITMAP allbits;
  GR_BITMAP curbit;

  if (!curenabled) return GR_FALSE;

  if (gr_mode != GR_MODE_SET) (*gr_dev.setmode) (GR_MODE_SET);

  saveptr = savedpoints;
  fgbitptr = curfgbitmap;
  bgbitptr = curbgbitmap;

  for (y = curminy; y <= curmaxy; y++) {
	curbit = 0;
	for (x = curminx; x <= curmaxx; x++) {
		if (curbit == 0) {
			allbits = *fgbitptr++ | *bgbitptr++;
			curbit = GR_FIRSTBIT;
		}
		if (curbit & allbits) (*gr_dev.drawpoint) (x, y, *saveptr++);
		curbit = GR_NEXTBIT(curbit);
	}
  }
  if (gr_mode != GR_MODE_SET) (*gr_dev.setmode) (gr_mode);

  curenabled = GR_FALSE;

  return GR_TRUE;
}


/*===========================================================================*
 *				showcursor				     *
 *===========================================================================*/
PRIVATE int showcursor()
{
/* Show the cursor on the screen.
 * This must first save the pixel values on the screen that will be
 * overwritten by the cursor, so that they can be restored later.
 * Returns the previous state of the cursor enabled flag.
 */
  GR_COLORN *saveptr;
  GR_BITMAP *fgbitptr;
  GR_BITMAP *bgbitptr;
  GR_COORD x;
  GR_COORD y;
  GR_BITMAP fgbits;
  GR_BITMAP bgbits;
  GR_BITMAP curbit;
  GR_COLOR oldcolor;
  GR_COLOR newcolor;
  GR_BOOL dosave;

  if (curenabled) return GR_TRUE;

  if (gr_mode != GR_MODE_SET) (*gr_dev.setmode) (GR_MODE_SET);

  saveptr = savedpoints;
  fgbitptr = curfgbitmap;
  bgbitptr = curbgbitmap;

  for (y = curminy; y <= curmaxy; y++) {
	curbit = 0;
	for (x = curminx; x <= curmaxx; x++) {
		if (curbit == 0) {
			fgbits = *fgbitptr++;
			bgbits = *bgbitptr++;
			curbit = GR_FIRSTBIT;
		}
		dosave = GR_FALSE;
		if (curbit & fgbits) {
			newcolor = curfg;
			dosave = GR_TRUE;
		} else if (curbit & bgbits) {
			newcolor = curbg;
			dosave = GR_TRUE;
		}
		if (dosave) {
			oldcolor = (*gr_dev.readpoint) (x, y);
			if (oldcolor != newcolor)
				(*gr_dev.drawpoint) (x, y, newcolor);
			*saveptr++ = oldcolor;
		}
		curbit = GR_NEXTBIT(curbit);
	}
  }

  if (gr_mode != GR_MODE_SET) (*gr_dev.setmode) (gr_mode);

  curenabled = GR_TRUE;
  curneedfix = GR_FALSE;

  return GR_FALSE;
}


/*===========================================================================*
 *				gen_checkcursor				     *
 *===========================================================================*/
PUBLIC void gen_checkcursor(x1, y1, x2, y2)
GR_COORD x1;
GR_COORD y1;
GR_COORD x2;
GR_COORD y2;
{
/* Check to see if the cursor is about to be overwritten.
 * If so, then remove the cursor so that the graphics operation
 * works correctly.  If the cursor is removed, then this fact will
 * be remembered and a later call to fixcursor will restore it.
 */
  GR_COORD temp;

  if (!curenabled) return;

  if (x1 > x2) {
	temp = x1;
	x1 = x2;
	x2 = temp;
  }
  if (y1 > y2) {
	temp = y1;
	y1 = y2;
	y2 = temp;
  }
  if ((x1 > curmaxx) || (x2 < curminx) ||
      (y1 > curmaxy) || (y2 < curminy))
	return;

  removecursor();
  curneedfix = GR_TRUE;
  return;
}


/*===========================================================================*
 *				gen_fixcursor				     *
 *===========================================================================*/
PUBLIC void gen_fixcursor()
{
/* Fix back up the cursor if it was removed because of a graphics operation. */
  if (curneedfix) showcursor();
}

/* END CODE */
