/* Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Routines to determine clipping regions.
 */

#include "kernel.h"
#include "graph_dev.h"


/* Clip cache rectangle information.
 * After calling clippoint, this rectangle is guaranteed to contain the
 * specified point (among others), and all points in the rectangle are
 * plottable or not according to the value of clipresult.
 */
GR_COORD clipminx;		/* minimum x value of cache rectangle */
GR_COORD clipminy;		/* minimum y value of cache rectangle */
GR_COORD clipmaxx;		/* maximum x value of cache rectangle */
GR_COORD clipmaxy;		/* maximum y value of cache rectangle */
GR_BOOL clipresult;		/* whether clip rectangle is plottable */


PRIVATE GR_COUNT clipcount;	/* number of clip rectangles */
PRIVATE GR_RECT cliprects[GR_MAX_CLIPRECTS];	/* clip rectangles */


/*===========================================================================*
 *				clipinit				     *
 *===========================================================================*/
PUBLIC void clipinit(count, table)
GR_COUNT count;			/* number of clip rectangles */
GR_RECT *table;			/* rectangle table */
{
/* Specify a new set of clip rectangles.
 * This should only be called after the device has been initialized.
 * If zero clip rectangles are specified, then clipping is done to the
 * full device area.  All clip rectangles are modified if necessary to
 * lie within the device area.
 */
  register GR_RECT *rp;		/* current rectangle */

  /* If there are no clip rectangles, then default to the full device area. */
  if (count <= 0) {
	clipminx = 0;
	clipminy = 0;
	clipmaxx = gr_dev.cols - 1;
	clipmaxy = gr_dev.rows - 1;
	clipcount = 0;
	clipresult = GR_TRUE;
	return;
  }

  /* Copy the clip table to our own static array, modifying each
   * rectangle as necesary to fit within the device area.  If the clip
   * rectangle lies entirely outside of the device area, then skip it.
   */
  rp = cliprects;
  clipcount = 0;
  if (count > GR_MAX_CLIPRECTS) count = GR_MAX_CLIPRECTS;
  while (count-- > 0) {
	*rp = *table++;
	if (rp->x < 0) {
		rp->width += rp->x;
		rp->x = 0;
	}
	if (rp->y < 0) {
		rp->height += rp->y;
		rp->y = 0;
	}
	if ((rp->x >= gr_dev.cols) || (rp->width <= 0) ||
	    (rp->y >= gr_dev.rows) || (rp->height <= 0))
		continue;
	if (rp->x + rp->width > gr_dev.cols) rp->width = gr_dev.cols - rp->x;
	if (rp->y + rp->height > gr_dev.rows)
		rp->height = gr_dev.rows - rp->y;
	rp++;
	clipcount++;
  }

  /* If there were no surviving clip rectangles, then set the clip
   * cache to prevent all drawing.
   */
  if (clipcount == 0) {
	clipminx = GR_COORD_MIN;
	clipminy = GR_COORD_MIN;
	clipmaxx = GR_COORD_MAX;
	clipmaxy = GR_COORD_MAX;
	clipresult = GR_FALSE;
	return;
  }

  /* There was at least one valid clip rectangle. Default the clip
   * cache to be the first clip rectangle.
   */
  clipminx = cliprects[0].x;
  clipminy = cliprects[0].y;
  clipmaxx = clipminx + cliprects[0].width - 1;
  clipmaxy = clipminy + cliprects[0].height - 1;
  clipresult = GR_TRUE;
}


/*===========================================================================*
 *				clippoint				     *
 *===========================================================================*/
PUBLIC GR_BOOL
 clippoint(x, y)
GR_COORD x;
GR_COORD y;
{
/* Check a point against the list of clip rectangles.
 * Returns TRUE if the point is within one or more rectangles and thus
 * can be plotted, or FALSE if the point is not within any rectangle and
 * thus cannot be plotted.  Also remembers the coordinates of a clip cache
 * rectangle containing the specified point such that every point in the
 * rectangle would give the same result.  By examining this clip cache
 * rectangle after a call to this routine, the caller can efficiently
 * check many nearby points without needing any further calls.  If the
 * point lies within the cursor, then the cursor is removed.
 */
  GR_COUNT count;
  GR_RECT *rp;
  GR_COORD temp;

  /* First see whether the point lies within the current clip cache
   * rectangle.  If so, then we already know the result.
   */
  if ((x >= clipminx) && (x <= clipmaxx) &&
      (y >= clipminy) && (y <= clipmaxy)) {
	if (clipresult) (*gr_dev.checkcursor) (x, y, x, y);
	return clipresult;
  }

  /* If the point is outside of the screen area, then it is not
   * plottable, and the clip cache rectangle is the whole half-plane
   * outside of the screen area.
   */
  if (x < 0) {
	clipminx = GR_COORD_MIN;
	clipmaxx = -1;
	clipminy = GR_COORD_MIN;
	clipmaxy = GR_COORD_MAX;
	clipresult = GR_FALSE;
	return GR_FALSE;
  }
  if (y < 0) {
	clipminx = GR_COORD_MIN;
	clipmaxx = GR_COORD_MAX;
	clipminy = GR_COORD_MIN;
	clipmaxy = -1;
	clipresult = GR_FALSE;
	return GR_FALSE;
  }
  if (x >= gr_dev.cols) {
	clipminx = gr_dev.cols;
	clipmaxx = GR_COORD_MAX;
	clipminy = GR_COORD_MIN;
	clipmaxy = GR_COORD_MAX;
	clipresult = GR_FALSE;
	return GR_FALSE;
  }
  if (y >= gr_dev.rows) {
	clipminx = GR_COORD_MIN;
	clipmaxx = GR_COORD_MAX;
	clipminy = gr_dev.rows;
	clipmaxy = GR_COORD_MAX;
	clipresult = GR_FALSE;
	return GR_FALSE;
  }

  /* The point is within the screen area. If there are no clip
   * rectangles, then the point is plottable and the rectangle is the
   * whole screen.
   */
  count = clipcount;
  if (count <= 0) {
	clipminx = 0;
	clipmaxx = gr_dev.cols - 1;
	clipminy = 0;
	clipmaxy = gr_dev.rows - 1;
	clipresult = GR_TRUE;
	(*gr_dev.checkcursor) (x, y, x, y);
	return GR_TRUE;
  }

  /* We need to scan the list of clip rectangles to calculate a new
   * clip cache rectangle containing this point, and the result. First
   * see if the point lies within any of the clip rectangles. If so,
   * then it is plottable and use that clip rectangle as the cache
   * rectangle.  This is not necessarily the best result, but works ok
   * and is fast.
   */
  for (rp = cliprects; count-- > 0; rp++) {
	if ((x >= rp->x) && (y >= rp->y) && (x < rp->x + rp->width)
	    && (y < rp->y + rp->height)) {
		clipminx = rp->x;
		clipminy = rp->y;
		clipmaxx = rp->x + rp->width - 1;
		clipmaxy = rp->y + rp->height - 1;
		clipresult = GR_TRUE;
		(*gr_dev.checkcursor) (x, y, x, y);
		return GR_TRUE;
	}
  }

  /* The point is not plottable. Scan the clip rectangles again to
   * determine a rectangle containing more non-plottable points.
   * Simply pick the largest rectangle whose area doesn't contain any
   * of the same coordinates as appropriate sides of the clip
   * rectangles.  This is not necessarily the best result, but works ok
   * and is fast.
   */
  clipminx = GR_COORD_MIN;
  clipminy = GR_COORD_MIN;
  clipmaxx = GR_COORD_MAX;
  clipmaxy = GR_COORD_MAX;
  count = clipcount;
  for (rp = cliprects; count-- > 0; rp++) {
	if ((x < rp->x) && (rp->x <= clipmaxx)) clipmaxx = rp->x - 1;
	temp = rp->x + rp->width - 1;
	if ((x > temp) && (temp >= clipminx)) clipminx = temp + 1;
	if ((y < rp->y) && (rp->y <= clipmaxy)) clipmaxy = rp->y - 1;
	temp = rp->y + rp->height - 1;
	if ((y > temp) && (temp >= clipminy)) clipminy = temp + 1;
  }
  clipresult = GR_FALSE;
  return GR_FALSE;
}


/*===========================================================================*
 *				cliparea				     *
 *===========================================================================*/
PUBLIC int cliparea(x1, y1, x2, y2)
GR_COORD x1;
GR_COORD y1;
GR_COORD x2;
GR_COORD y2;
{
/* Check the area determined by the specified pair of points against the
 * list of clip rectangles.  The area will either be totally visible,
 * totally visible, or possibly partially visible.  This routine updates
 * the clip cache rectangle, and returns one of the following values:
 *	GR_CLIP_VISIBLE		The whole rectangle is visible
 *	GR_CLIP_INVISIBLE	The whole rectangle is invisible
 *	GR_CLIP_PARTIAL		The rectangle may be partially visible
 * In the case that the area is totally visible, the cursor is removed
 * if it overlaps the clip area.
 */
  if ((x1 < clipminx) || (x1 > clipmaxx) ||
      (y1 < clipminy) || (y1 > clipmaxy))
	clippoint(x1, y1);

  if ((x2 >= clipminx) && (x2 <= clipmaxx) &&
      (y2 >= clipminy) && (y2 <= clipmaxy)) {
	if (!clipresult) return GR_CLIP_INVISIBLE;
	(*gr_dev.checkcursor) (x1, y1, x2, y2);
	return GR_CLIP_VISIBLE;
  }
  return GR_CLIP_PARTIAL;
}

/* END CODE */
