/*
 * Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Private definitions for the graphics server.
 * These definitions are not to be used by the clients.
 * Clients must call the GrXXXX routines instead of these GsXXXX routines.
 */

#ifndef	GRAPH_SERV_H
#define	GRAPH_SERV_H

#include <graphics.h>


/*
 * Drawing types.
 */
typedef	int	GR_DRAW_TYPE;

#define	GR_DRAW_TYPE_NONE	((GR_DRAW_TYPE) 0)	/* none or error */
#define	GR_DRAW_TYPE_WINDOW	((GR_DRAW_TYPE) 1)	/* windows */
#define	GR_DRAW_TYPE_PIXMAP	((GR_DRAW_TYPE) 2)	/* pixmaps */


/*
 * Maximum number of clients that can be simultaneously run by the server.
 * This is limited to 255 because of resource ids, but the number of
 * file descriptor can also limit it.
 */
#define	GR_MAX_CLIENTS	1	/* maximum number of clients */


/*
 * The following is used to determine whether a client has already been
 * given an event.  For each event, a global value is incremented, and
 * this value is saved in the client structure when it is delivered
 * to the client.
 */
typedef	unsigned long GR_EVENT_NUMBER;


/*
 * List of elements for events.
 */
typedef	struct gr_event_list GR_EVENT_LIST;
struct gr_event_list {
	GR_EVENT_LIST	*next;		/* next element in list */
	GR_EVENT	event;		/* event */
};


/*
 * Data structure to keep track of state of clients.
 */
typedef	struct {
	int	infd;			/* handle for input from client */
	int	outfd;			/* handle for output to client */
	int	num;			/* client number */
	GR_ID	allocid;		/* next resource id to allocate */
	GR_EVENT_NUMBER	eventnumber;	/* event number delivered to client */
	GR_EVENT_LIST	*eventhead;	/* head of event chain (or NULL) */
	GR_EVENT_LIST	*eventtail;	/* tail of event chain (or NULL) */
	GR_EVENT_ERROR	errorevent;	/* waiting error event to deliver */
} GR_CLIENT;


/*
 * Structure to remember clients associated with events.
 */
typedef	struct gr_event_client	GR_EVENT_CLIENT;
struct gr_event_client	{
	GR_EVENT_CLIENT	*next;		/* next interested client or NULL */
	GR_EVENT_MASK	eventmask;	/* events client wants to see */
	GR_CLIENT	*client;	/* client who is interested */
};


/*
 * Structure to remember graphics contexts.
 */
typedef	struct gr_gc	GR_GC;
struct gr_gc {
	GR_GC_ID	id;		/* graphics context id */
	GR_MODE		mode;		/* drawing mode */
	GR_FONT		font;		/* current font number */
	GR_COLOR	foreground;	/* foreground color */
	GR_COLOR	background;	/* background color */
	GR_BOOL		usebackground;	/* actually display the background */
	GR_BOOL		changed;	/* graphics context has been changed */
	GR_GC		*next;		/* next graphics context */
	int		usecount;	/* usecount */
};


/*
 * Structure to remember cursor definitions.
 */
typedef	struct {
	int		usecount;	/* use counter */
	GR_SIZE		width;		/* width of cursor */
	GR_SIZE		height;		/* height of cursor */
	GR_COORD	hotx;		/* relative x position of hot spot */
	GR_COORD	hoty;		/* relative y position of hot spot */
	GR_COLOR	foreground;	/* foreground color */
	GR_COLOR	background;	/* background color */
	GR_BITMAP	fgbitmap[GR_MAX_BITMAP_SIZE];	/* foreground bitmap */
	GR_BITMAP	bgbitmap[GR_MAX_BITMAP_SIZE];	/* background bitmap */
} GR_CURSOR;


/*
 * Window structure
 */
typedef struct gr_window GR_WINDOW;
struct gr_window {
	GR_WINDOW_ID	id;		/* window id */
	GR_WINDOW	*parent;	/* parent window */
	GR_WINDOW	*children;	/* first child window */
	GR_WINDOW	*siblings;	/* next sibling window */
	GR_WINDOW	*next;		/* next window in complete list */
	GR_COORD	x;		/* absolute x position */
	GR_COORD	y;		/* absolute y position */
	GR_SIZE		width;		/* width */
	GR_SIZE		height;		/* height */
	GR_SIZE		bordersize;	/* size of border */
	GR_COLOR	bordercolor;	/* color of border */
	GR_COLOR	background;	/* background color */
	GR_EVENT_MASK	nopropmask;	/* events not to be propagated */
	GR_EVENT_CLIENT	*eventclients;	/* clients interested in events */
	GR_CURSOR	*cursor;	/* cursor for this window */
	GR_BOOL		mapped;		/* TRUE if explicitly mapped */
	GR_COUNT	unmapcount;	/* count of reasons not really mapped */
	GR_BOOL		output;		/* TRUE if window can do output */
};


/*
 * Pixmap structure (not yet implemented).
 * Pixmaps of 2 colors use bitmaps, otherwise they use color values.
 */
typedef struct gr_pixmap	GR_PIXMAP;
struct gr_pixmap {
	GR_PIXMAP_ID	id;		/* pixmap id */
	GR_PIXMAP	*next;		/* next pixmap in list */
	GR_SIZE		width;		/* width of pixmap */
	GR_SIZE		height;		/* height of pixmap */
	GR_COLOR	maxcolor;	/* maximum color used in pixmap */
	GR_COLOR	*colors;	/* table of color values */
	GR_COLOR8	*color8;	/* OR: table of 8 bit color values */
	GR_BITMAP	*bitmap;	/* OR: table of bitmap values */
};


/*
 * Macros to obtain the client number from a resource id, and to
 * produce the first resource id to be used for a client number.
 * Client numbers must not be zero.  This allows for 255 clients.
 */
#define	GR_ID_CLIENT(n)	(((unsigned long) (n)) >> 24)
#define	GR_ID_BASE(n)	(((unsigned long) (n)) << 24)


/*
 * Graphics server routines.
 */
extern	int		GdOpenScreen();
extern	void		GdCloseScreen();
extern	void		GdGetScreenInfo();
extern	void		GdGetFontInfo();
extern	void		GdTextMode();
extern	void		GdFlush();
extern	GR_BOOL		GdCheckError();

extern	int		GsInitialize();
extern	void		GsTerminate();
extern	GR_ID		GsOpen();
extern	void		GsClose();
extern	void		GsFlush();
extern	void		GsError();
extern	GR_WINDOW_ID	GsNewWindow();
extern	GR_WINDOW_ID	GsNewInputWindow();
extern	void		GsDestroyWindow();
extern	void		GsGetWindowInfo();
extern	GR_GC_ID	GsNewGC();
extern	GR_GC_ID	GsCopyGC();
extern	void		GsDestroyGC();
extern	void		GsSetGCForeground();
extern	void		GsSetGCBackground();
extern	void		GsSetGCUseBackground();
extern	void		GsSetGCMode();
extern	void		GsSetGCFont();
extern	void		GsGetGCTextSize();
extern	void		GsGetScreenInfo();
extern	void		GsClearWindow();
extern	void		GsSelectEvents();
extern	void		GsPoint();
extern	void		GsPoly();
extern	void		GsFillPoly();
extern	void		GsLine();
extern	void		GsEllipse();
extern	void		GsFillEllipse();
extern	void		GsRect();
extern	void		GsFillRect();
extern	void		GsBitmap();
extern	void		GsArea8();
extern	void		GsReadArea8();
extern	void		GsText();
extern	void		GsSetCursor();
extern	void		GsMoveCursor();
extern	void		GsDrawBorder();
extern	void		GsMapWindow();
extern	void		GsUnmapWindow();
extern	void		GsRaiseWindow();
extern	void		GsLowerWindow();
extern	void		GsMoveWindow();
extern	void		GsResizeWindow();
extern	void		GsSetFocus();
extern	void		GsSetBorderColor();
extern	void		GsGetNextEvent();
extern	void		GsPeekEvent();
extern	void		GsCheckEvent();

extern	int		GsReadMouse();
extern	int		GsReadKeyboard();
extern	void		GsMoveMouse();
extern	int		GsOpenMouse();
extern	int		GsOpenKeyboard();
extern	void		GsCloseMouse();
extern	void		GsCloseKeyboard();
extern	void		GsSetAccelMouse();
extern	void		GsRestrictMouse();

extern	void		GsExposeArea();
extern	void		GsCheckCursor();
extern	void		GsWpSetFocus();
extern	void		GsWpClearWindow();
extern	void		GsWpUnmapWindow();
extern	void		GsWpDestroyWindow();
extern	void		GsSetClipWindow();
extern	GR_COUNT	GsSplitClipRect();
extern	void		GsHandleMouseStatus();
extern	void		GsFreePositionEvent();
extern	void		GsDeliverButtonEvent();
extern	void		GsDeliverMotionEvent();
extern	void		GsDeliverKeyboardEvent();
extern	void		GsDeliverExposureEvent();
extern	void		GsDeliverGeneralEvent();
extern	void		GsCheckMouseWindow();
extern	void		GsCheckFocusWindow();
extern	GR_DRAW_TYPE	GsPrepareDrawing();
extern	GR_BOOL		GsCheckOverlap();
extern	GR_EVENT	*GsAllocEvent();
extern	GR_WINDOW	*GsFindWindow();
extern	GR_GC		*GsFindGC();
extern	GR_WINDOW	*GsPrepareWindow();
extern	GR_WINDOW	*GsFindVisibleWindow();
extern	void		GsDrawBorder();
extern	void		GsMoveMouse();


/*
 * External data definitions.
 */
extern	char *		curfunc;		/* current function name */
extern	GR_WINDOW_ID	cachewindowid;		/* cached window id */
extern	GR_GC_ID	cachegcid;		/* cached graphics context id */
extern	GR_GC		*cachegcp;		/* cached graphics context */
extern	GR_GC		*listgcp;		/* list of all gc */
extern	GR_GC		*curgcp;		/* current graphics context */
extern	GR_WINDOW	*cachewp;		/* cached window pointer */
extern	GR_WINDOW	*listwp;		/* list of all windows */
extern	GR_WINDOW	*rootwp;		/* root window pointer */
extern	GR_WINDOW	*clipwp;		/* window clipping is set for */
extern	GR_WINDOW	*focuswp;		/* focus window for keyboard */
extern	GR_WINDOW	*mousewp;		/* window mouse is currently in */
extern	GR_WINDOW	*grabbuttonwp;		/* window grabbed by button */
extern	GR_CURSOR	*curcursor;		/* currently enabled cursor */
extern	GR_COORD	cursorx;		/* x position of cursor */
extern	GR_COORD	cursory;		/* y position of cursor */
extern	GR_BUTTON	curbuttons;		/* current state of buttons */
extern	GR_CLIENT	*curclient;		/* current client */
extern	GR_EVENT_LIST	*eventfree;		/* list of free events */
extern	GR_EVENT_NUMBER	eventnumber;		/* current event number */
extern	GR_BOOL		inited;			/* TRUE if been initialized */
extern	GR_BOOL		focusfixed;		/* TRUE if focus is fixed */
extern	GR_SCREEN_INFO	sinfo;			/* screen information */
extern	GR_FONT_INFO	curfont;		/* current font information */
extern	GR_CLIENT	clients[GR_MAX_CLIENTS];	/* table of clients */

#endif

/* END CODE */
