/*
 * Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Graphics server top level routines to execute client functions.
 * These functions are the server side of the client GrXXXX functions.
 * Clients should not call these GsXXXX functions, but should only call
 * the GrXXXX functions.  In this way, when the server becomes a separate
 * process, the client code will not need changes.
 *
 * When the server is its own process, these functions will be called by
 * a dispatcher routine which can read messages from multiple clients.
 * The dispatcher should set the curclient variable to the client which
 * is being executed, and the curfunc variable to the current function name.
 * Then the setting of curfunc throughout this file can be removed.
 */
#include <stdio.h>
#include "graph_serv.h"


/*
 * Open a connection from a new client to the server.
 * This returns the first resource id to be allocated by the client.
 * On an error, zero is returned.
 */
GR_ID
GsOpen()
{
	curfunc = "GrOpen";		/* temporary */

	if (GsInitialize())
		return 0;

	curclient = &clients[0];
	curclient->num = 1;
	curclient->infd = -1;		/* not used yet */
	curclient->outfd = -1;
	curclient->allocid = GR_ID_BASE(curclient->num);
	curclient->errorevent.type = GR_EVENT_TYPE_NONE;
	curclient->eventhead = NULL;
	curclient->eventhead = NULL;

	return curclient->allocid;
}


/*
 * Close the connection to the server.
 */
void
GsClose()
{
	curfunc = "GrClose";		/* temporary */

	/* MUST RELEASE ALL RESOURCES ALLOCATED BY THIS CLIENT */

	curclient->num = 0;
	curclient->infd = -1;		/* not used yet */
	curclient->outfd = -1;

	GsTerminate();
}


/*
 * Flush graphics to the device.
 */
void
GsFlush()
{
	curfunc = "GrFlush";		/* temporary */

	GdFlush();
}


/*
 * Return information about the screen for clients to use.
 */
void
GsGetScreenInfo(sip)
	GR_SCREEN_INFO	*sip;		/* pointer to screen info */
{
	curfunc = "GrGetScreenInfo";	/* temporary */

	*sip = sinfo;
}


/*
 * Return the size of a text string for the font in a graphics context.
 * This is the width of the string, the height of the string,
 * and the height above the bottom of the font of the baseline for the font.
 */
void
GsGetGCTextSize(gc, cp, len, retwidth, retheight, retbase)
	GR_GC_ID	gc;		/* graphics context containing font */
	GR_CHAR		*cp;		/* address of text string */
	GR_SIZE		len;		/* length of text string */
	GR_SIZE		*retwidth;	/* returned width of string */
	GR_SIZE		*retheight;	/* returned height of string */
	GR_SIZE		*retbase;	/* returned height of baseline */
{
	GR_GC		*gcp;		/* graphics context */
	GR_FONT_INFO	*fip;		/* current font info */

	curfunc = "GrGCTextSize";	/* temporary */

	gcp = GsFindGC(gc);
	if (gcp == NULL) {
		*retwidth = 0;
		*retheight = 1;
		*retbase = 0;
		return;
	}

	fip = &curfont;
	if (fip->font != gcp->font)
		GdGetFontInfo(gcp->font, fip);

	*retheight = fip->height;
	*retbase = fip->baseline;
	if (fip->fixed) {
		*retwidth = fip->maxwidth * len;
		return;
	}

	*retwidth = 0;
	while (len-- > 0)
		*retwidth += fip->widths[*cp++];
}


/*
 * Return the next waiting event for a client, or wait for one if there
 * is none yet.  The event is copied into the specified structure, and
 * then is moved from the event queue to the free event queue.  If there
 * is an error event waiting, it is delivered before any other events.
 */
void
GsGetNextEvent(ep)
	GR_EVENT	*ep;		/* pointer to element to return */
{
	GR_EVENT_LIST	*elp;		/* current element list */

	curfunc = "GrGetNextEvent";	/* temporary */

	if (curclient->errorevent.type == GR_EVENT_TYPE_ERROR) {
		*((GR_EVENT_ERROR *) ep) = curclient->errorevent;
		curclient->errorevent.type = GR_EVENT_TYPE_NONE;
		return;
	}

	if (curclient->eventhead == NULL)
		GsCheckEvent(GR_TRUE);
	if (curclient->eventhead == NULL) {
		ep->type = GR_EVENT_TYPE_NONE;
		return;
	}
	elp = curclient->eventhead;
	*ep = elp->event;
	curclient->eventhead = elp->next;
	if (curclient->eventhead == NULL)
		curclient->eventtail = NULL;
	elp->next = eventfree;
	eventfree = elp;
}


/*
 * Peek at the event queue for the current client to see if there are any
 * outstanding events.  Returns the event at the head of the queue, or
 * else a null event type.  The event is still left in the queue, however.
 */
void
GsPeekEvent(ep)
	GR_EVENT	*ep;		/* pointer to element to return */
{
	curfunc = "GrPeekEvent";	/* temporary */

	if (curclient->errorevent.type == GR_EVENT_TYPE_ERROR) {
		*((GR_EVENT_ERROR *) ep) = curclient->errorevent;
		return;
	}

	if (curclient->eventhead == NULL)
		GsCheckEvent(GR_FALSE);

	if (curclient->eventhead) {
		*ep = curclient->eventhead->event;
		return;
	}
	ep->type = GR_EVENT_TYPE_NONE;
}


/*
 * Return information about a window id.
 */
void
GsGetWindowInfo(wid, infoptr)
	GR_WINDOW_ID		wid;		/* window to find out about */
	GR_WINDOW_INFO		*infoptr;	/* pointer to returned data */
{
	GR_WINDOW	*wp;		/* window structure */
	GR_EVENT_CLIENT	*evp;		/* event-client structure */

	curfunc = "GrGetWindowInfo";	/* temporary */

	/*
	 * Find the window manually so that an error is not generated.
	 */
	for (wp = listwp; wp && (wp->id != wid); wp = wp->next)
		continue;

	if (wp == NULL) {
		infoptr->wid = 0;
		return;
	}

	infoptr->wid = wid;
	infoptr->parent = wp->parent->id;
	infoptr->child = wp->children->id;
	infoptr->sibling = wp->siblings->id;
	infoptr->mapped = wp->mapped;
	infoptr->unmapcount = wp->unmapcount;
	infoptr->inputonly = !wp->output;
	infoptr->x = wp->x;
	infoptr->y = wp->y;
	infoptr->width = wp->width;
	infoptr->height = wp->height;
	infoptr->bordersize = wp->bordersize;
	infoptr->bordercolor = wp->bordercolor;
	infoptr->background = wp->background;
	infoptr->eventmask = 0;

	for (evp = wp->eventclients; evp; evp = evp->next) {
		if (evp->client == curclient)
			infoptr->eventmask = evp->eventmask;
	}
}


/*
 * Destroy an existing window and all of its children.
 */
void
GsDestroyWindow(wid)
	GR_WINDOW_ID	wid;		/* window to destroy */
{
	GR_WINDOW	*wp;		/* window structure */

	curfunc = "GrDestroyWindow";	/* temporary */

	wp = GsFindWindow(wid);
	if (wp)
		GsWpDestroyWindow(wp);
}


/*
 * Raise a window to the highest level among its siblings.
 */
void
GsRaiseWindow(wid)
	GR_WINDOW_ID	wid;		/* window to be raised */
{
	GR_WINDOW	*wp;		/* window structure */
	GR_WINDOW	*prevwp;	/* previous window pointer */
	GR_BOOL		overlap;	/* TRUE if there was overlap */

	curfunc = "GrRaiseWindow";	/* temporary */

	wp = GsFindWindow(wid);
	if ((wp == NULL) || (wp == rootwp))
		return;

	/*
	 * If this is already the highest window then we are done.
	 */
	prevwp = wp->parent->children;
	if (prevwp == wp)
		return;

	/*
	 * Find the sibling just before this window so we can unlink it.
	 * Also, determine if any sibling ahead of us overlaps the window.
	 * Remember that for exposure events.
	 */
	overlap = GR_FALSE;
	while (prevwp->siblings != wp) {
		overlap |= GsCheckOverlap(prevwp, wp);
		prevwp = prevwp->siblings;
	}
	overlap |= GsCheckOverlap(prevwp, wp);

	/*
	 * Now unlink the window and relink it in at the front of the
	 * sibling chain.
	 */
	prevwp->siblings = wp->siblings;
	wp->siblings = wp->parent->children;
	wp->parent->children = wp;

	/*
	 * Finally redraw the window if necessary.
	 */
	if (overlap) {
		GsDrawBorder(wp);
		GsExposeArea(wp, wp->x, wp->y, wp->width, wp->height);
	}
}


/*
 * Lower a window to the lowest level among its siblings.
 */
void
GsLowerWindow(wid)
	GR_WINDOW_ID	wid;		/* window to be lowered */
{
	GR_WINDOW	*wp;		/* window structure */
	GR_WINDOW	*prevwp;	/* previous window pointer */
	GR_WINDOW	*sibwp;		/* sibling window */
	GR_WINDOW	*expwp;		/* siblings being exposed */

	curfunc = "GrLowerWindow";	/* temporary */

	wp = GsFindWindow(wid);
	if ((wp == NULL) || (wp == rootwp) || (wp->siblings == NULL))
		return;

	/*
	 * Find the sibling just before this window so we can unlink us.
	 */
	prevwp = wp->parent->children;
	if (prevwp != wp) {
		while (prevwp->siblings != wp)
			prevwp = prevwp->siblings;
	}

	/*
	 * Remember the first sibling that is after us, so we can
	 * generate exposure events for the remaining siblings.  Then
	 * walk down the sibling chain looking for the last sibling.
	 */
	expwp = wp->siblings;
	sibwp = wp;
	while (sibwp->siblings)
		sibwp = sibwp->siblings;

	/*
	 * Now unlink the window and relink it in at the end of the
	 * sibling chain.
	 */
	if (prevwp == wp)
		wp->parent->children = wp->siblings;
	else
		prevwp->siblings = wp->siblings;
	sibwp->siblings = wp;

	wp->siblings = NULL;

	/*
	 * Finally redraw the sibling windows which this window covered
	 * if they overlapped our window.
	 */
	while (expwp && (expwp != wp)) {
		if (GsCheckOverlap(wp, expwp)) {
			GsExposeArea(expwp, wp->x - wp->bordersize,
				wp->y - wp->bordersize,
				wp->width + wp->bordersize * 2,
				wp->height + wp->bordersize * 2);
		}
		expwp = expwp->siblings;
	}
}


/*
 * Move the window to the specified position relative to its parent.
 */
void
GsMoveWindow(wid, x, y)
	GR_WINDOW_ID	wid;		/* window to be lowered */
	GR_COORD	x;		/* new relative x position */
	GR_COORD	y;		/* new relative y position */
{
	GR_WINDOW	*wp;		/* window structure */

	curfunc = "GrMoveWindow";	/* temporary */

	wp = GsFindWindow(wid);
	if (wp == NULL)
		return;
	if (wp == rootwp) {
		GsError(GR_ERROR_ILLEGAL_ON_ROOT_WINDOW, wid);
		return;
	}

	x += wp->parent->x;
	y += wp->parent->y;

	if ((wp->x == x) && (wp->y == y))
		return;

	if (wp->unmapcount || !wp->output) {
		wp->x = x;
		wp->y = y;
		return;
	}

	/*
	 * This should be optimized to not require redrawing of the window
	 * when possible.
	 */
	GsWpUnmapWindow(wp);
	wp->x = x;
	wp->y = y;
	GsWpMapWindow(wp);
}


/*
 * Resize the window to be the specified size.
 */
void
GsResizeWindow(wid, width, height)
	GR_WINDOW_ID	wid;		/* window to be lowered */
	GR_SIZE		width;		/* new width of window */
	GR_SIZE		height;		/* new height of window */
{
	GR_WINDOW	*wp;		/* window structure */

	curfunc = "GrResizeWindow";	/* temporary */

	wp = GsFindWindow(wid);
	if (wp == NULL)
		return;
	if (wp == rootwp) {
		GsError(GR_ERROR_ILLEGAL_ON_ROOT_WINDOW, wid);
		return;
	}
	if ((width <= 0) || (height <= 0)) {
		GsError(GR_ERROR_BAD_WINDOW_SIZE, wid);
		return;
	}

	if ((wp->width == width) && (wp->height == height))
		return;

	if (wp->unmapcount || !wp->output) {
		wp->width = width;
		wp->height = height;
		return;
	}

	/*
	 * This should be optimized to not require redrawing of the window
	 * when possible.
	 */
	GsWpUnmapWindow(wp);
	wp->width = width;
	wp->height = height;
	GsWpMapWindow(wp);
}


/*
 * Allocate a new GC with default parameters.
 * The GC is owned by the current client.
 */
GR_GC_ID
GsNewGC()
{
	GR_GC	*gcp;

	curfunc = "GrNewGC";		/* temporary */

	/*
	 * Increment this before checking for errors since the client
	 * always increments his.
	 */
	curclient->allocid++;

	gcp = (GR_GC *) malloc(sizeof(GR_GC));
	if (gcp == NULL) {
		GsError(GR_ERROR_MALLOC_FAILED, 0);
		return 0;
	}

	gcp->id = curclient->allocid;
	gcp->mode = GR_MODE_SET;
	gcp->font = 0;
	gcp->foreground = sinfo.white;
	gcp->background = sinfo.black;
	gcp->usebackground = GR_TRUE;
	gcp->changed = GR_TRUE;
	gcp->next = listgcp;

	listgcp = gcp;

	return gcp->id;
}


/*
 * Destroy an existing graphics context.
 */
void
GsDestroyGC(gc)
	GR_GC_ID	gc;		/* graphics context to destroy */
{
	GR_GC		*gcp;		/* graphics context */
	GR_GC		*prevgcp;	/* previous graphics context */

	curfunc = "GrDestroyGC";	/* temporary */

	gcp = GsFindGC(gc);
	if (gcp == NULL)
		return;

	if (gc == cachegcid) {
		cachegcid = 0;
		cachegcp = NULL;
	}

	if (listgcp == gcp) {
		listgcp = gcp->next;
		free(gcp);
		return;
	}

	prevgcp = listgcp;
	while (prevgcp->next != gcp)
		prevgcp = prevgcp->next;

	prevgcp->next = gcp->next;
	free(gcp);
}


/*
 * Allocate a new GC which is a copy of another one.
 * The GC is owned by the current client.
 */
GR_GC_ID
GsCopyGC(gc)
	GR_GC_ID	gc;		/* GC to be copied */
{
	GR_GC		*oldgcp;	/* old graphics context */
	GR_GC		*gcp;		/* new graphics context */

	curfunc = "GrCopyGC";		/* temporary */

	/*
	 * Increment this before checking for errors since the client
	 * always increments his.
	 */
	curclient->allocid++;

	oldgcp = GsFindGC(gc);
	if (oldgcp == NULL)
		return 0;

	gcp = (GR_GC *) malloc(sizeof(GR_GC));
	if (gcp == NULL) {
		GsError(GR_ERROR_MALLOC_FAILED, 0);
		return 0;
	}

	/*
	 * Copy all the old gcp values into the new one, except allocate
	 * a new id for it and link it into the list of GCs.
	 */
	*gcp = *oldgcp;
	gcp->id = curclient->allocid;
	gcp->changed = GR_TRUE;
	gcp->next = listgcp;
	listgcp = gcp;

	return gcp->id;
}


/*
 * Return information about the specified graphics context.
 */
void
GsGetGCInfo(gcid, gcip)
	GR_GC_ID	gcid;		/* graphics context */
	GR_GC_INFO	*gcip;		/* address of graphics context info */
{
	GR_GC		*gcp;

	/*
	 * Find the GC manually so that an error is not generated.
	 */
	for (gcp = listgcp; gcp && (gcp->id != gcid); gcp = gcp->next)
		continue;

	if (gcp == NULL) {
		gcip->gcid = 0;
		return;
	}

	gcip->gcid = gcid;
	gcip->mode = gcp->mode;
	gcip->font = gcp->font;
	gcip->foreground = gcp->foreground;
	gcip->background = gcp->background;
	gcip->usebackground = gcp->usebackground;
}


/*
 * Return useful information about the specified font.
 */
void
GsGetFontInfo(font, fip)
	GR_FONT		font;		/* font number */
	GR_FONT_INFO	*fip;		/* address of font info */
{
	/*
	 * See if the font is built-in or not.  Someday for non-builtin
	 * fonts, we can return something for them.
	 */
	if (font >= sinfo.fonts) {
		fip->font = 0;
		return;
	}
	GdGetFontInfo(font, fip);
}


/*
 * Select events for a window for this client.
 * The events are a bitmask for the events desired.
 */
void
GsSelectEvents(wid, eventmask)
	GR_WINDOW_ID	wid;		/* window id */
	GR_EVENT_MASK	eventmask;	/* mask of events wanted */
{
	GR_WINDOW	*wp;		/* window structure */
	GR_EVENT_CLIENT	*evp;		/* event-client structure */

	curfunc = "GrSelectEvents";	/* temporary */

	wp = GsFindWindow(wid);
	if (wp == NULL)
		return;

	/*
	 * See if this client is already in the event client list.
	 * If so, then just replace the events he is selecting for.
	 */
	for (evp = wp->eventclients; evp; evp = evp->next) {
		if (evp->client == curclient) {
			evp->eventmask = eventmask;
			return;
		}
	}

	/*
	 * A new client for this window, so allocate a new event client
	 * structure and insert it into the front of the list in the window.
	 */
	evp = (GR_EVENT_CLIENT *) malloc(sizeof(GR_EVENT_CLIENT));
	if (evp == NULL) {
		GsError(GR_ERROR_MALLOC_FAILED, wid);
		return;
	}

	evp->client = curclient;
	evp->eventmask = eventmask;
	evp->next = wp->eventclients;
	wp->eventclients = evp;
}


/*
 * Allocate a new window which is a child of the specified window.
 * The window inherits the cursor of the parent window.
 * The window is owned by the current client.
 */
GR_WINDOW_ID
GsNewWindow(parent, x, y, width, height, bordersize, background, bordercolor)
	GR_WINDOW_ID	parent;		/* parent id */
	GR_COORD	x;		/* x position relative to parent */
	GR_COORD	y;		/* y position relative to parent */
	GR_SIZE		width;		/* width */
	GR_SIZE		height;		/* height */
	GR_SIZE		bordersize;	/* size of border */
	GR_COLOR	background;	/* background color */
	GR_COLOR	bordercolor;	/* border color */
{
	GR_WINDOW	*pwp;		/* parent window */
	GR_WINDOW	*wp;		/* new window */

	curfunc = "GrNewWindow";	/* temporary */

	/*
	 * Increment this before checking for errors since the client
	 * always increments his.
	 */
	curclient->allocid++;

	if ((width <= 0) || (height <= 0) || (bordersize < 0)) {
		GsError(GR_ERROR_BAD_WINDOW_SIZE, 0);
		return 0;
	}

	pwp = GsFindWindow(parent);
	if (pwp == NULL)
		return 0;

	if (!pwp->output) {
		GsError(GR_ERROR_INPUT_ONLY_WINDOW, pwp->id);
		return 0;
	}

	wp = (GR_WINDOW *) malloc(sizeof(GR_WINDOW));
	if (wp == NULL) {
		GsError(GR_ERROR_MALLOC_FAILED, 0);
		return 0;
	}

	wp->id = curclient->allocid;
	wp->parent = pwp;
	wp->children = NULL;
	wp->siblings = pwp->children;
	wp->next = listwp;
	wp->x = pwp->x + x;
	wp->y = pwp->y + y;
	wp->width = width;
	wp->height = height;
	wp->bordersize = bordersize;
	wp->background = background;
	wp->bordercolor = bordercolor;
	wp->nopropmask = 0;
	wp->eventclients = NULL;
	wp->cursor = pwp->cursor;
	wp->mapped = GR_FALSE;
	wp->unmapcount = pwp->unmapcount + 1;
	wp->output = GR_TRUE;

	wp->cursor->usecount++;
	pwp->children = wp;
	listwp = wp;

	return wp->id;
}


/*
 * Allocate a new input-only window which is a child of the specified window.
 * Such a window is invisible, cannot be drawn into, and is only used to
 * return events.  The window inherits the cursor of the parent window.
 * The window is owned by the current client.
 */
GR_WINDOW_ID
GsNewInputWindow(parent, x, y, width, height)
	GR_WINDOW_ID	parent;		/* parent id */
	GR_COORD	x;		/* x position relative to parent */
	GR_COORD	y;		/* y position relative to parent */
	GR_SIZE		width;		/* width */
	GR_SIZE		height;		/* height */
{
	GR_WINDOW	*pwp;		/* parent window */
	GR_WINDOW	*wp;		/* new window */

	curfunc = "GrNewInputWindow";	/* temporary */

	/*
	 * Increment this before checking for errors since the client
	 * always increments his.
	 */
	curclient->allocid++;

	if ((width <= 0) || (height <= 0)) {
		GsError(GR_ERROR_BAD_WINDOW_SIZE, 0);
		return 0;
	}

	pwp = GsFindWindow(parent);
	if (pwp == NULL)
		return 0;

	wp = (GR_WINDOW *) malloc(sizeof(GR_WINDOW));
	if (wp == NULL) {
		GsError(GR_ERROR_MALLOC_FAILED, 0);
		return 0;
	}

	wp->id = curclient->allocid;
	wp->parent = pwp;
	wp->children = NULL;
	wp->siblings = pwp->children;
	wp->next = listwp;
	wp->x = pwp->x + x;
	wp->y = pwp->y + y;
	wp->width = width;
	wp->height = height;
	wp->bordersize = 0;
	wp->background = sinfo.black;
	wp->bordercolor = sinfo.black;
	wp->nopropmask = 0;
	wp->eventclients = NULL;
	wp->cursor = pwp->cursor;
	wp->mapped = GR_FALSE;
	wp->unmapcount = pwp->unmapcount + 1;
	wp->output = GR_FALSE;

	wp->cursor->usecount++;
	pwp->children = wp;
	listwp = wp;

	return wp->id;
}


/*
 * Map the window to make it (and possibly its children) visible on the screen.
 */
void
GsMapWindow(wid)
	GR_WINDOW_ID	wid;		/* window to be mapped */
{
	GR_WINDOW	*wp;		/* window structure */

	curfunc = "GrMapWindow";	/* temporary */

	wp = GsFindWindow(wid);
	if ((wp == NULL) || wp->mapped)
		return;

	wp->mapped = GR_TRUE;

	GsWpMapWindow(wp);
}


/*
 * Unmap the window to make it and its children invisible on the screen.
 */
void
GsUnmapWindow(wid)
	GR_WINDOW_ID	wid;		/* window to be unmapped */
{
	GR_WINDOW	*wp;		/* window structure */

	curfunc = "GrUnmapWindow";	/* temporary */

	wp = GsFindWindow(wid);
	if ((wp == NULL) || !wp->mapped)
		return;

	GsWpUnmapWindow(wp);

	wp->mapped = GR_FALSE;
}


/*
 * Clear the specified window.
 * This sets the window to its background color.
 * If the exposeflag is nonzero, then this also creates an exposure
 * event for the window.
 */
void
GsClearWindow(wid, exposeflag)
	GR_WINDOW_ID	wid;		/* window id */
	GR_BOOL		exposeflag;	/* nonzero to cause an exposure */
{
	GR_WINDOW		*wp;	/* window structure */

	curfunc = "GrClearWindow";	/* temporary */

	wp = GsPrepareWindow(wid);
	if (wp)
		GsWpClearWindow(wp, 0, 0, wp->width, wp->height, exposeflag);
}


/*
 * Set the focus to a particular window.
 * This makes keyboard events only visible to that window or children of it,
 * depending on the pointer location.
 */
void
GsSetFocus(wid)
	GR_WINDOW_ID	wid;		/* window id */
{
	GR_WINDOW	*wp;		/* window structure */

	curfunc = "GrSetFocus";		/* temporary */

	wp = GsFindWindow(wid);
	if (wp == NULL)
		return;

	if (wp->unmapcount) {
		GsError(GR_ERROR_UNMAPPED_FOCUS_WINDOW, wid);
		return;
	}

	focusfixed = (wp != rootwp);
	GsWpSetFocus(wp);
}


/*
 * Set the border of a window to the specified color.
 */
void
GsSetBorderColor(wid, color)
	GR_WINDOW_ID	wid;		/* window id */
	GR_COLOR	color;		/* color for border */
{
	GR_WINDOW	*wp;		/* window structure */

	curfunc = "GrSetBorderColor";	/* temporary */

	wp = GsFindWindow(wid);
	if ((wp == NULL) || (wp->bordercolor == color) ||
		(wp->bordersize == 0))
			return;

	wp->bordercolor = color;
	if (wp->unmapcount == 0)
		GsDrawBorder(wp);
}


/*
 * Specify a cursor for a window.
 * This cursor will only be used within that window, and by default
 * for its new children.  If the cursor is currently within this
 * window, it will be changed to the new one immediately.
 */
void
GsSetCursor(wid, width, height, hotx, hoty, foreground, background,
	fgbitmap, bgbitmap)

	GR_WINDOW_ID	wid;		/* window id to set cursor for */
	GR_SIZE		width;		/* width of cursor */
	GR_SIZE		height;		/* height of cursor */
	GR_COORD	hotx;		/* relative x position of hot spot */
	GR_COORD	hoty;		/* relative y position of hot spot */
	GR_COLOR	foreground;	/* foreground color of cursor */
	GR_COLOR	background;	/* background color of cursor */
	GR_BITMAP	*fgbitmap;	/* foreground bitmap */
	GR_BITMAP	*bgbitmap;	/* background bitmap */
{
	GR_WINDOW	*wp;		/* window structure */
	GR_CURSOR	*cp;		/* cursor structure */
	int		bytes;		/* number of bytes of bitmap */

	curfunc = "GrSetCursor";	/* temporary */

	wp = GsFindWindow(wid);
	if (wp == NULL)
		return;

	/*
	 * Make sure the size of the bitmap is reasonable.
	 */
	if ((width <= 0) || (width > GR_MAX_CURSOR_SIZE) ||
		(height <= 0) || (height > GR_MAX_CURSOR_SIZE))
	{
		GsError(GR_ERROR_BAD_CURSOR_SIZE, 0);
		return;
	}
	bytes = GR_BITMAP_SIZE(width, height) * sizeof(GR_BITMAP);

	/*
	 * See if the window is using a shared cursor definition.
	 * If so, then allocate a new private one, otherwise reuse it.
	 */
	cp = wp->cursor;
	if (cp->usecount-- > 1) {
		cp = (GR_CURSOR *) malloc(sizeof(GR_CURSOR));
		if (cp == NULL) {
			GsError(GR_ERROR_MALLOC_FAILED, 0);
			return;
		}
	}

	cp->usecount = 1;
	cp->width = width;
	cp->height = height;
	cp->hotx = hotx;
	cp->hoty = hoty;
	cp->foreground = foreground;
	cp->background = background;
	memcpy((char *) cp->fgbitmap, (char *) fgbitmap, bytes);
	memcpy((char *) cp->bgbitmap, (char *) bgbitmap, bytes);
	wp->cursor = cp;

	/*
	 * If this was the current cursor, then draw the new one.
	 */
	if (cp == curcursor) {
		GdSetCursor(cp->width, cp->height, cp->foreground,
			cp->background, cp->fgbitmap, cp->bgbitmap);
		GdMoveCursor(cursorx - cp->hotx, cursory - cp->hoty);
	}
}


/*
 * Move the cursor to the specified absolute screen coordinates.
 * The coordinates are that of the defined hot spot of the cursor.
 * The cursor's appearance is changed to that defined for the window
 * in which the cursor is moved to.  In addition, mouse enter, mouse
 * exit, focus in, and focus out events are generated if necessary.
 */
void
GsMoveCursor(x, y)
	GR_COORD	x;		/* new x position of cursor */
	GR_COORD	y;		/* new y position of cursor */
{
	curfunc = "GrMoveCursor";	/* temporary */

	/*
	 * Move the cursor only if necessary, offsetting it to
	 * place the hot spot at the specified coordinates.
	 */
	if ((x != cursorx) || (y != cursory)) {
		GdMoveCursor(x - curcursor->hotx, y - curcursor->hoty);
		cursorx = x;
		cursory = y;
	}

	/*
	 * Now check to see which window the mouse is in, whether or
	 * not the cursor shape should be changed, and whether or not
	 * the input focus window should be changed.
	 */
	GsCheckMouseWindow();
	GsCheckFocusWindow();
	GsCheckCursor();
}


/*
 * Set the foreground color in a graphics context.
 */
void
GsSetGCForeground(gc, foreground)
	GR_GC_ID	gc;		/* graphics context id */
	GR_COLOR	foreground;	/* foreground color */
{
	GR_GC		*gcp;		/* graphics context */

	curfunc = "GrSetGCForeground";	/* temporary */

	gcp = GsFindGC(gc);
	if ((gcp == NULL) || (gcp->foreground == foreground))
		return;
	gcp->foreground = foreground;
	gcp->changed = GR_TRUE;
}


/*
 * Set the background color in a graphics context.
 */
void
GsSetGCBackground(gc, background)
	GR_GC_ID	gc;		/* graphics context id */
	GR_COLOR	background;	/* background color */
{
	GR_GC		*gcp;		/* graphics context */

	curfunc = "GrSetGCBackground";	/* temporary */

	gcp = GsFindGC(gc);
	if ((gcp == NULL) || (gcp->background == background))
		return;
	gcp->background = background;
	gcp->changed = GR_TRUE;
}


/*
 * Set whether or not the background color is drawn in bitmaps and text.
 */
void
GsSetGCUseBackground(gc, flag)
	GR_GC_ID	gc;		/* graphics context id */
	GR_BOOL		flag;		/* TRUE if background is drawn */
{
	GR_GC		*gcp;		/* graphics context */

	curfunc = "GrSetGCUseBackground";	/* temporary */

	flag = (flag != 0);
	gcp = GsFindGC(gc);
	if ((gcp == NULL) || (gcp->usebackground == flag))
		return;
	gcp->usebackground = flag;
	gcp->changed = GR_TRUE;
}


/*
 * Set the drawing mode in a graphics context.
 */
void
GsSetGCMode(gc, mode)
	GR_GC_ID	gc;		/* graphics context id */
	GR_MODE		mode;		/* drawing mode */
{
	GR_GC		*gcp;		/* graphics context */

	curfunc = "GrSetGCMode";	/* temporary */

	gcp = GsFindGC(gc);
	if ((gcp == NULL) || (gcp->mode == mode))
		return;
	if (mode > GR_MAX_MODE) {
		GsError(GR_ERROR_BAD_DRAWING_MODE, gc);
		return;
	}
	gcp->mode = mode;
	gcp->changed = GR_TRUE;
}


/*
 * Set the text font in a graphics context.
 */
void
GsSetGCFont(gc, font)
	GR_GC_ID	gc;		/* graphics context id */
	GR_FONT		font;		/* text font */
{
	GR_GC		*gcp;		/* graphics context */

	curfunc = "GrSetGCFont";	/* temporary */

	gcp = GsFindGC(gc);
	if ((gcp == NULL) || (gcp->font == font))
		return;
	gcp->font = font;
	gcp->changed = GR_TRUE;
}


/*
 * Draw a line in the specified drawable using the specified graphics context.
 */
void
GsLine(id, gc, x1, y1, x2, y2)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x1;
	GR_COORD	y1;
	GR_COORD	x2;
	GR_COORD	y2;
{
	GR_WINDOW	*wp;
	GR_PIXMAP	*pmp;

	curfunc = "GrLine";		/* temporary */

	switch (GsPrepareDrawing(id, gc, &wp, &pmp)) {
		case GR_DRAW_TYPE_WINDOW:
			GdLine(wp->x + x1, wp->y + y1, wp->x + x2, wp->y + y2);
			break;
	}
}


/*
 * Draw the boundary of a rectangle in the specified drawable using the
 * specified graphics context.
 */
void
GsRect(id, gc, x, y, width, height)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		width;
	GR_SIZE		height;
{
	GR_WINDOW	*wp;
	GR_PIXMAP	*pmp;

	curfunc = "GrRect";		/* temporary */

	switch (GsPrepareDrawing(id, gc, &wp, &pmp)) {
		case GR_DRAW_TYPE_WINDOW:
			GdRect(wp->x + x, wp->y + y, width, height);
			break;
	}
}


/*
 * Fill a rectangle in the specified drawable using the specified
 * graphics context.
 */
void
GsFillRect(id, gc, x, y, width, height)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		width;
	GR_SIZE		height;
{
	GR_WINDOW	*wp;
	GR_PIXMAP	*pmp;

	curfunc = "GrFillRect";		/* temporary */

	switch (GsPrepareDrawing(id, gc, &wp, &pmp)) {
		case GR_DRAW_TYPE_WINDOW:
			GdFillRect(wp->x + x, wp->y + y, width, height);
			break;
	}
}


/*
 * Draw the boundary of an ellipse in the specified drawable with
 * the specified graphics context.
 */
void
GsEllipse(id, gc, x, y, rx, ry)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		rx;
	GR_SIZE		ry;
{
	GR_WINDOW	*wp;
	GR_PIXMAP	*pmp;

	curfunc = "GrEllipse";		/* temporary */

	switch (GsPrepareDrawing(id, gc, &wp, &pmp)) {
		case GR_DRAW_TYPE_WINDOW:
			GdEllipse(wp->x + x, wp->y + y, rx, ry);
			break;
	}
}


/*
 * Fill an ellipse in the specified drawable using the specified
 * graphics context.
 */
void
GsFillEllipse(id, gc, x, y, rx, ry)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		rx;
	GR_SIZE		ry;
{
	GR_WINDOW	*wp;
	GR_PIXMAP	*pmp;

	curfunc = "GrFillEllipse";	/* temporary */

	switch (GsPrepareDrawing(id, gc, &wp, &pmp)) {
		case GR_DRAW_TYPE_WINDOW:
			GdFillEllipse(wp->x + x, wp->y + y, rx, ry);
			break;
	}
}


/*
 * Draw a rectangular area in the specified drawable using the specified
 * graphics, as determined by the specified bit map.  This differs from
 * rectangle drawing in that the rectangle is drawn using the foreground
 * color and possibly the background color as determined by the bit map.
 * Each row of bits is aligned to the next bitmap word boundary (so there
 * is padding at the end of the row).  The background bit values are only
 * written if the usebackground flag is set in the GC.
 */
void
GsBitmap(id, gc, x, y, width, height, bitmaptable)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		width;
	GR_SIZE		height;
	GR_BITMAP	*bitmaptable;
{
	GR_WINDOW	*wp;
	GR_PIXMAP	*pmp;

	curfunc = "GrBitmap";		/* temporary */

	switch (GsPrepareDrawing(id, gc, &wp, &pmp)) {
		case GR_DRAW_TYPE_WINDOW:
			GdBitmap(wp->x + x, wp->y + y, width, height,
				bitmaptable);
			break;
	}
}


/*
 * Draw a rectangular area in the specified drawable using the specified
 * graphics context.  This differs from rectangle drawing in that the
 * color values for each pixel in the rectangle are specified.  The color
 * values are restricted to 8 bit values.  The color table is indexed
 * row by row.
 */
void
GsArea8(id, gc, x, y, width, height, colortable)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		width;
	GR_SIZE		height;
	GR_COLOR8	*colortable;
{
	GR_WINDOW	*wp;
	GR_PIXMAP	*pmp;

	curfunc = "GrArea8";		/* temporary */

	switch (GsPrepareDrawing(id, gc, &wp, &pmp)) {
		case GR_DRAW_TYPE_WINDOW:
			GdArea8(wp->x + x, wp->y + y, width, height,
				colortable);
			break;
	}
}


/*
 * Read the color values from the specified rectangular area of the
 * specified drawable into a supplied buffer.  If the drawable is a
 * window which is obscured by other windows, then the returned values
 * will include the values from the covering windows.  Regions outside
 * of the screen boundaries, or unmapped windows will return black.
 */
void
GsReadArea8(id, x, y, width, height, colortable)
	GR_DRAW_ID	id;
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		width;
	GR_SIZE		height;
	GR_COLOR8	*colortable;
{
	GR_WINDOW	*wp;
	long		count;

	wp = GsFindWindow(id);			/* wrong, should check pixmap */
	if ((wp == NULL) || wp->unmapcount || (x >= wp->width) ||
		(y >= wp->height) || (x + width <= 0) || (y + height <= 0))
	{
		count = width * height;
		while (count-- > 0)
			*colortable++ = sinfo.black;
		return;
	}
	GdReadArea8(wp->x, wp->y, width, height, colortable);
}


/*
 * Draw a point in the specified drawable using the specified
 * graphics context.
 */
void
GsPoint(id, gc, x, y)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
{
	GR_WINDOW	*wp;
	GR_PIXMAP	*pmp;

	curfunc = "GrPoint";		/* temporary */

	switch (GsPrepareDrawing(id, gc, &wp, &pmp)) {
		case GR_DRAW_TYPE_WINDOW:
			GdPoint(wp->x + x, wp->y + y);
			break;
	}
}


/*
 * Draw a polygon in the specified drawable using the specified
 * graphics context.  The polygon is only complete if the first
 * point is repeated at the end.
 */
void
GsPoly(id, gc, count, pointtable)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COUNT	count;
	GR_POINT	*pointtable;
{
	GR_WINDOW	*wp;
	GR_PIXMAP	*pmp;
	GR_POINT	*pp;
	GR_COUNT	i;

	curfunc = "GrPoly";		/* temporary */

	switch (GsPrepareDrawing(id, gc, &wp, &pmp)) {
		case GR_DRAW_TYPE_WINDOW:
			break;
		default:
			return;
	}

	/*
	 * Here for drawing to a window.
	 * Relocate all the points relative to the window.
	 */
	pp = pointtable;
	for (i = count; i-- > 0; pp++) {
		pp->x += wp->x;
		pp->y += wp->y;
	}

	GdPoly(count, pointtable);

	/*
	 * The following is temporarily necessary until the server
	 * becomes a separate process.  We don't want to change the
	 * user's arguments!
	 */
	pp = pointtable;
	for (i = count; i-- > 0; pp++) {
		pp->x -= wp->x;
		pp->y -= wp->y;
	}
}


/*
 * Draw a filled polygon in the specified drawable using the specified
 * graphics context.  The last point may be a duplicate of the first
 * point, but this is not required.
 */
void
GsFillPoly(id, gc, count, pointtable)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COUNT	count;
	GR_POINT	*pointtable;
{
	GR_WINDOW	*wp;
	GR_PIXMAP	*pmp;
	GR_POINT	*pp;
	GR_COUNT	i;

	curfunc = "GrFillPoly";		/* temporary */

	switch (GsPrepareDrawing(id, gc, &wp, &pmp)) {
		case GR_DRAW_TYPE_WINDOW:
			break;
		default:
			return;
	}

	/*
	 * Here for drawing to a window.
	 * Relocate all the points relative to the window.
	 */
	pp = pointtable;
	for (i = count; i-- > 0; pp++) {
		pp->x += wp->x;
		pp->y += wp->y;
	}

	GdFillPoly(count, pointtable);

	/*
	 * The following is temporarily necessary until the server
	 * becomes a separate process.  We don't want to change the
	 * user's arguments!
	 */
	pp = pointtable;
	for (i = count; i-- > 0; pp++) {
		pp->x -= wp->x;
		pp->y -= wp->y;
	}
}


/*
 * Draw a text string in the specified drawable using the
 * specified graphics context.
 */
void
GsText(id, gc, x, y, str, count)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
	GR_CHAR		*str;
	GR_COUNT	count;
{
	GR_WINDOW	*wp;
	GR_PIXMAP	*pmp;

	curfunc = "GrText";		/* temporary */

	switch (GsPrepareDrawing(id, gc, &wp, &pmp)) {
		case GR_DRAW_TYPE_WINDOW:
			GdText(wp->x + x, wp->y + y, str, count);
			break;
	}
}

/* END CODE */
