/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  DFiltFIR [options] [FiltFile]

Purpose:
  Design minimax linear phase FIR filters

Description:
  This program designs four types of finite impulse response filters.
   1: Equiripple Linear Phase Filter (multiple passbands and stopbands).  The
      filter is specified in terms of the desired response in bands.  The band
      specifications include desired value, relative weighting and limits on
      the allowed response values in the band.  The resulting filter is a
      minimax approximation (with constraints) to the given specifications.
      The filter coefficients have an even symmetry about the middle of the
      filter.
   2: Equiripple Linear Phase Filter (multiple passbands and stopbands) with
      sin(x)/x compensation.  The filter is specified in terms of the desired
      response in bands.  The band specifications include desired value,
      relative weighting and limits on the allowed response values in the band.
      The resulting filter is a minimax approximation (with constraints) to the
      given specifications.  The filter coefficients have an even symmetry
      about the middle of the filter.
   3: Equiripple Differentiator.  The filter is specified in terms of the
      desired slope in bands.  The band specifications include desired slope,
      relative weighting and limits on the allowed slopes in the band.  The
      resulting filter is a minimax approximation (with constraints) to the
      given specifications.  The filter coefficients have an odd symmetry about
      the middle of the filter.
   4: Equiripple Hilbert Transform Filter. The filter is specified in terms of
      the desired response in bands.  The band specifications include desired
      value, relative weighting and limits on the allowed values in the band.
      The resulting filter is a minimax approximation (with constraints) to the
      given specifications.  The filter coefficients have an odd symmetry about
      the middle of the filter.

  This program implements the filter design procedure outlined by McClellan,
  Parks and Rabiner.  This procedure has been modified to include constraints
  as described by Grenez.
  References:
  J. H. McClellan, T. W. Parks and L. R. Rabiner, "A computer program for
  designing optimum FIR linear phase digital filters", IEEE Trans. Audio and
  Electroacoustics, vol. AU-21, pp. 506-526, December 1973.

  F. Grenez, "Design of linear or minimum-phase FIR filters by constrained
  Chebyshev approximation", Signal Processing, vol. 5, pp. 325-332, July 1983.

Parameters:
  This program reads options from the command line and from standard input.
  At a minimum the command line must specify the number of coefficients.
  -t TYPE, --type=TYPE
      Filter type, default multiple passband/stopband
       "bpf" - multiple passband/stopband
       "receive" - multiple passband/stopband filter used as a
           reconstruction filter (sin(x)/x compensation)
       "differentiator" - differentiator
       "hilbert_transform" - Hilbert transform filter
  -n NCOF, --number_coefficients=NCOF
      Number of filter coefficients.
  -s SFREQ, --srate=SFREQ
      Sampling frequency.  If the sampling frequency is not specified, a
      normalized frequency of one is used.
  -g NGRID, --grid_density=NGRID
      Grid density (number of grid points per extremum of the response),
      default 16.
  -h, --help
      Print a list of options and exit.
  -v, --version
      Print the version number and exit.

  The filter specifications are read from standard input.  The specifications
  are discrete values given in order of increasing frequency within a
  frequency band.  Within each band, piecewise monotonic cubic interpolation is
  used between the tabulated values.  Bands are separated by transition lines.
  These are empty (or white-space) lines.  Each specification line has 5 values
  (the last two are optional).  The specifications are of the form,
    Freq Value Weight [Low_Value High_Value]
    Freq       - Frequency
    Value      - Desired filter response value (or desired slope for
                 differentiators)
    Weight     - Relative weighting for the value (or slope)
    Low_Value  - Lower limit on the filter response (or slope for
                 differentiators)
    High_Value - Upper limit on the filter response (or slope for
                 differentiators)
  The limit values are optional, but within a band, all or none of the
  specification lines must have limit values.  The specification lines must be
  in order of increasing frequency.  A maximum of 200 specification lines can
  be given.  The program expects specification lines to be entered until one of
  the following occurs.
   1: A frequency equal to half of the sampling frequency (normalized
      frequency of 0.5) is entered,
   2: an empty line appears after a transition, or
   3: an end-of-file is encountered.

  Example, 32 tap constrained bandpass filter:
      The stopbands are from 0 to 0.1 and 0.425 to 0.5, and the passband is
      from 0.2 to 0.35.  The relative weights are 10 in the stopbands and 1 in
      the passband.  The response is constrained to be positive (between 0 and
      1) in the first stopband. The filter coefficients will be written to file
      filt.cof.  A Unix shell script for this design would be as follows.
       DFiltFIR -t bpf -n 32 filt.cof << EoF
       0   0 10 0 1
       0.1 0 10 0 1
       <empty line>
       0.2  1 1
       0.35 1 1
       <empty line>
       0.425 0 10
       0.5   0 10
       EoF

Author / version:
  P. Kabal / v1r5a  1995/11/23  Copyright (C) 1996

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: DFiltFIR.c 1.13 1996/05/30 FilterDesign-V1R7a $";

#include <stdlib.h>
#include <stdio.h>
#include <libtsp.h>
#include "DFiltFIR.h"


#ifndef EXIT_SUCCESS
#  define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

int
main (argc, argv)

     int argc;
     const char *argv[];

{
  int Ftype, Ncof;
  const char *Fname;
  float Gridd;
  float Sfreq;
  struct Bspec *S;
  float *h;
  int ier;
  float devs;

/* Read the filter parameters */
  MMoptions (argc, argv, &Ftype, &Ncof, &Sfreq, &Gridd, &Fname);

/* Read the filter specifications */
  S = MMreadSpec (Ftype, Ncof, (double) Sfreq);

/* Design the filter */
  h = (float *) UTmalloc (Ncof * sizeof (float));
  ier = MMdesFIR (Ftype, (double) Gridd, S, h, Ncof, &devs);
  if (ier != 0)
    UThalt ("%s: Error detected", PROGRAM);

/* Print a summary and print or write the coefficients */
  if (Fname == NULL) {
    if (Ftype == BPF)
      VRfPrint (stdout, "Linear Phase FIR Filter", h, Ncof);
    else if (Ftype == REC)
      VRfPrint (stdout, "Linear Phase FIR Filter (sin(x)/x compensation)",
		h, Ncof);
    else if (Ftype == DIF)
      VRfPrint (stdout, "Differentiator", h, Ncof);
    else if (Ftype == HIL)
      VRfPrint (stdout, "Hilbert Transform Filter", h, Ncof);
    else
      UThalt ("DFiltFIR: Invalid filter type");
  }
  MMprintSpec (Ftype, (double) devs, S);
  if (Fname != NULL)
    MMwriteCof (Fname, Ftype, h, Ncof, S);

  UTfree (S);
  UTfree (h);

  return EXIT_SUCCESS;
}
