/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int MMdesRemez (const struct Gspec *G, double x[], double y[], int Etype[],
                  int Next, float *devs)

Purpose:
  Remez exchange algorithm for filter design

Description:
  This routine finds a minimum maximum deviation approximation to a  function,
  subject to constraints of the allowed deviation.  The function is specified
  by the desired function values and weights on a dense grid.  The algorithm
  minimizes the Chebyshev error by determining the best locations of the
  extremal points (points of maximum error).  The returned values are the
  extremal points of the approximation.

Parameters:
  <-  int MMdesRemez
      Error parameter coded as follows.
      0  - No error
      1  - Remez algorithm failed to converge, the deviation is not
           monotonically increasing
      2  - Remez algorithm failed to converge, too many iterations
      3  - Constraints too tight for the given filter order
      For these cases, this routine returns coefficient values.  For all other
      errors, an error message is printed and execution is halted.
   -> const struct Gspec *G
      Structure with the arrays specifying the filter specifications on the
      grid
  <-  double x[]
      Array of abscissa values at the extremal points.  Each x[i] takes on a
      unique value from G->grid[k].
  <-  double y[]
      Array of ordinate values at the extremal points.  The Next extrema are
      constrained in such a way as to allow a Next-1 term polynomial (normally
      one term too few) to pass through the Next points.
  <-  int Etype[]
      Type of extremum,
      -2 - Lower constraint limit
      -1 - Lower ripple
      +1 - Upper ripple
      +2 - Upper constraint limit
   -> int Next
      Number of extrema in the approximation
  <-  float *devs
      Resultant deviation (relative to unity weighting)

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.5 $  $Date: 1995/11/24 01:03:21 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MMdesRemez.c 1.5 1995/11/24 FilterDesign-V1R7a $";

#include <math.h>
#include <libtsp.h>
#include "DFiltFIR.h"

#define MINV(a, b)	(((a) < (b)) ? (a) : (b))
#define MAXV(a, b)	(((a) > (b)) ? (a) : (b))
#define MAXEADJ		4

int
MMdesRemez (G, x, y, Etype, Next, devs)

     const struct Gspec *G;
     double x[];
     double y[];
     int Etype[];
     int Next;
     float *devs;

{
  int iter, ie, ier;
  float dens;
  int i, k;
  int Nchange, Ngrid;
  int *Ext;
  struct Pcev *P;
  double dev, devl;

  Ngrid = G->Ngrid;
  devl = -1.0;

/* Allocate storage */
  Ext = (int *) UTmalloc (Next * sizeof (int));
  P = (struct Pcev *) UTmalloc (sizeof (struct Pcev));
  P->ad = (double *) UTmalloc (Next * sizeof (double));
  P->x = x;
  P->y = y;
  P->N = Next;

/* Initial guess for the extremal frequencies (equally spaced  along the
   frequency axis) */
  if (Next > Ngrid)
    UThalt ("MMdesRemez: Too few grid points");
  dens = (float) (Ngrid - 1) / MAXV (Next - 1, 1);
  for (i = 0; i < Next; ++i)
    Ext[i] = floor (i * dens + 0.5);

/* Set up the alternating limit array */
/* The initial setting might be inverted, but is later corrected */
  Etype[0] = LDEV;
  for (i = 1; i < Next; ++i) {
    if (Etype[i-1] == LDEV)
      Etype[i] = UDEV;
    else
      Etype[i] = LDEV;
  }

/* ******************** */
  for (iter = 0; iter < MAXITER; ++iter) {

/* Store the abscissa values for the Lagrange interpolation */
    for (i = 0; i < Next; ++i)
      P->x[i] = G->grid[Ext[i]];

/* Find the Lagrange interpolation coefficients to interpolate Next values */
    MMintCof (P->x, P->ad, Next);

/* Calculate the deviation: */
/* Given a set of Next distinct points, an Next-1 coefficient polynomial
   can give an error curve which exhibits equal amplitude alternations (in a
   weighted sense) about the desired value.  In such a formulation, the free
   parameters are the Next-1 coefficients and the deviation giving a total of
   Next linear equations.
*/

/*
   ** Constraints only **
   Etype[i] represents the type of deviation as determined from the previous
   iteration (and set to UDEV or LDEV prior to the first iteration).  The
   deviation is calculated using MMmmDev.  Using this deviation value, new
   ordinate values are set.  These new ordinate values may change an ordinary
   extremum to a constrained extremum.  This loop recalculates the deviation
   if Etype[i] changes.
*/
    for (ie = 0; ie < MAXEADJ; ++ie) {
      dev = MMmmDev (P->ad, Next, G, Ext, Etype, &ier);
      if (ier != 0)
	break;

/* Invert the signs of the deviations if necessary (first iteration only) */
      if (dev < 0.0) {
	if (iter != 0)
	  UThalt ("MMdesRemez: Change in sign of deviation");
	for (i = 0; i < Next; ++i)
	  Etype[i] = -Etype[i];
	dev = -dev;
      }

/*
   Calculate the ordinates for the Lagrange interpolation.  The polynomial that
   passes through the Next alternations with deviation as calculated above has
   only Next-1 terms.  However, if we keep the ordinate values for the Next
   term interpolation at the Next alternation values, the Lagrange coefficients
   for the Next term case can be used to interpolate the Next-1 term
   polynomial.
*/
      Nchange = 0;
      for (i = 0; i < Next; ++i) {
	k = Ext[i];
	if (Etype[i] == LDEV) {
	  y[i] = G->des[k] - dev / G->wt[k];
	  if (y[i] < G->liml[k]) {
	    Etype[i] = LLIM;
	    ++Nchange;
	  }
	}
	else if (Etype[i] == UDEV) {
	  y[i] = G->des[k] + dev / G->wt[k];
	  if (y[i] > G->limu[k]) {
	    Etype[i] = ULIM;
	    ++Nchange;
	  }
	}
	else if (Etype[i] == LLIM)
	  y[i] = G->liml[k];
	else if (Etype[i] == ULIM)
	  y[i] = G->limu[k];
	else
	  UThalt ("MMdesRemez: Invalid limit type");

      }
      if (Nchange == 0)
	break;
    }
    if (ie > MAXEADJ)
      UThalt ("MMdesRemez: Extremum adjustment fails, constraints too tight");

/* Check the convergence */
    if (dev < devl) {
      ier = 1;
      UTwarn ("MMdesRemez - Fails to converge (Deviation non-increasing)");
      break;
    }
    devl = dev;

/* Remez exchange algorithm */
    Nchange = MMexchExt (P, G, dev, Ext, Etype);
    if (Nchange <= 0)
      break;
  }

/* ******************** */
/* Failed to converge */
  if (iter >= MAXITER) {
    ier = 2;
    UTwarn ("MMdesRemez - Fails to converge (too many iterations)");
  }

  UTfree (Ext);
  UTfree (P->ad);
  UTfree (P);

  *devs = (float) dev;
  return ier;
}
