/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void MMwriteCof (const char Fname[], int Ftype, const float h[], int Ncof,
                   const struct Bspec *S)

Purpose:
  Write an FIR filter coefficient file

Description:
  This routine writes a coefficient file for an FIR filter.  The data written
  to the file includes a header containing the design specifications.  This
  header is followed by the filter coefficients.  The file name is also printed
  on standard output.

Parameters:
   -> const char Fname[]
      File name for the coefficient file
   -> int Ftype
      Filter type
   -> const float h[]
      Array of coefficients (Ncof values)
   -> int Ncof
      Number of filter coefficients
   -> struct Bspec *S
      Band specifications

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.5 $  $Date: 1995/11/24 01:40:23 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MMwriteCof.c 1.5 1995/11/24 FilterDesign-V1R7a $";

#include <stdio.h>
#include <libtsp.h>
#include "DFiltFIR.h"

void
MMwriteCof (Fname, Ftype, h, Ncof, S)

     const char Fname[];
     int Ftype;
     const float h[];
     int Ncof;
     const struct Bspec *S;

{
  FILE *fp;
  char Fullname[FILENAME_MAX+1];
  int i, j, k;

/* Open the coefficient file, print the name to stdout */
  FLbackup (Fname);
  fp = fopen (Fname, "w");
  if (fp == NULL)
    UTerror ("%s: Unable to open coefficient file", PROGRAM);
  FLfullName (Fname, Fullname);
  printf ("Coefficient file: %s\n", Fullname);

/* Preamble */
  if (Ftype == BPF)
    fprintf (fp, "!FIR - Linear Phase FIR Filter\n");
  else if (Ftype == REC)
    fprintf (fp, "!FIR - Linear Phase FIR Filter (sin(x)/x compensation)\n");
  else if (Ftype == DIF)
    fprintf (fp, "!FIR - Differentiator\n");
  else if (Ftype == HIL)
    fprintf (fp, "!FIR - Hilber Transform Filter\n");
  else
    UThalt ("MMwriteCof: Invalid filter type");

/* Design specifications */
  fprintf (fp, "!         Frequency       Value     Weight  ");
  fprintf (fp, "           Limits\n");
  i = 0;
  for (k = 1; k <= S->Nbands; ++k) {
    for (j = 0; j < S->Nval[k-1]; ++j, ++i) {
      if (j == 0)
	fprintf (fp, "!Band %2d: ", k);
      else
	fprintf (fp, "!         ");
      fprintf (fp, "%9.5f %11.4g %10.3g", S->f[i], S->val[i], S->w[i]);
      if (S->liml[i] != NOLIMIT_L)
	fprintf (fp, "%12.4g ", S->liml[i]);
      else
	fprintf (fp, "       ----  ");
      if (S->limu[i] != NOLIMIT_U)
	fprintf (fp, "%12.4g\n", S->limu[i]);
      else
	fprintf (fp, "       ----\n");
    }
  }

/* Coefficient values */
  for (i = 0; i < Ncof; ++i) {
    fprintf (fp, "%15.7g", h[i]);
    if ((i + 1) % 5 == 0 || i + 1 == Ncof)
      fprintf (fp, "\n");
  }

  fclose (fp);
}
