/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  float *INintMSE (const float h[], int Ncof, double Delay, int Ir,
                   const struct PSD *psd, const struct PSD *dis)

Purpose:
  Calculate the mean-square interpolation error

Description:
  This function calculates the mean-square interpolation error for an
  interpolating filter given the power spectrum of the input signal.  The power
  spectrum of the input signal is specified in terms of a continuous component
  and sinusoidal components.  The continuous component is given as a tabulated
  function.  Piecewise monotonic cubic interpolation is used between the given
  values.  The sinusoidal components are specified by frequency and power.  The
  continuous power spectrum is given as pairs of values: a normalized frequency
  and the corresponding power spectral density value.  The frequencies must be
  non-negative and in increasing order. The continuous power spectrum is
  assumed to be zero outside the given range of frequencies.  Sinusoidal
  components of the power spectrum are specified as a normalized frequency
  (non-negative) and the corresponding power.

  Negative mean-square error values are set to zero and a warning message
  is printed.  Such a case can happen if the correlation values are numerically
  non-positive definite.

Parameters:
  <-  float *INintMSE
      Array of normalized mean-square interpolation errors.  This is a pointer
      to an array with Ir elements.  Element i is the normalized mean-square
      interpolation error for the i'th subfilter.
   -> const float h[]
      Filter coefficients.  The middle of the filter is the reference time
      point.
   -> int Ncof
      Number of filter coeffients
   -> double Delay
      Filter delay (samples)
   -> int Ir
      Interpolation ratio
   -> const struct PSD *psd
      Structure describing the continuous power spectral density.  This
      structure contains an array of normalized frequencies and an array of
      the corresponding power spectral values.
   -> const struct PSD *dis
      Structure describing the discrete (sinusoidal) components of the power
      spectrum.  This structure contains an array of normalized frequencies and
      an array of the corresponding powers of the sinusoids.

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.8 $  $Date: 1996/06/17 21:29:17 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: INintMSE.c 1.8 1996/06/17 FilterDesign-V1R7a $";

#include <math.h>
#include <libtsp.h>
#include "DFiltInt.h"

#define ICEILV(n, m)	(((n) + ((m) - 1)) / (m))	/* int n,m >= 0 */

float *
INintMSE (h, Ncof, Delay, Ir, psd, dis)

     const float h[];
     int Ncof;
     double Delay;
     int Ir;
     const struct PSD *psd;
     const struct PSD *dis;

{
  double *rxx, *p, *hs;
  static float *mse = NULL;
  double del;
  int Nm, Nmmax;
  int i, m;
  
/*
   We can associate a delay value (measured in samples at the higher sampling
   rate) relative to the overall filter delay for each coefficient of the
   interpolating filter,
     coefficient   relative delay
       h[n]        n-Delay
   The interpolating filter is divided into Ir subfilters.  Consider the m'th
   coefficient of the i'th subfilter, 0 <= i < Ir, m = 0, 1, ... .  This
   numbering assigns i = 0 to the subfilter starting with the first coefficient
   of the overall filter.
     coefficient        relative delay
     hs[i](m)     <-->  m*Ir+i-Delay

   The mean-square error can be expressed in terms of R, hs[i] and p[i],
   where the matrix R is Toeplitz with elements R(m,k) = rxx(m-k)
         the vector hs[i] has elements hs[i](m),
         the vector p[i] has elements p[i](m) = rxx((m*Ir+i-Delay)/Ir), and
         the function rxx(t) gives the correlation at lag t*Ir.
*/

/* Allocate memory for the arrays */
  Nmmax = ICEILV (Ncof, Ir);
  rxx = (double *) UTmalloc (Nmmax * sizeof (double));
  p = (double *) UTmalloc (Nmmax * sizeof (double));
  hs = (double *) UTmalloc (Nmmax * sizeof (double));

/* Allocate memory for the mean-square error array */
  UTfree (mse);
  mse = (float *) UTmalloc (Ir * sizeof (float));

/* Fill in the autocorrelation vector */
  for (m = 0; m < Nmmax; ++m)
    rxx[m] = INevCorr ((double) m, psd, dis);

/* Loop over the subfilter indices */
  for (i = 0; i < Ir; ++i) {

    Nm = MSiCeil (Ncof-i, Ir);
    if (Nm > Nmmax)
      UThalt ("INintMSE: Consistency check failed");
 
    /* Fill in the right-hand side vector, p[i](.) */
    /* Copy the subfilter coefficients into the subfilter array */
    for (m = 0; m < Nm; ++m) {
      del = (-Delay + m*Ir + i) / Ir;
      p[m] = INevCorr (del, psd, dis);
      hs[m] = h[m*Ir+i];
    }

/*
   The resulting mean-square error for the i'th sub-filter can be expressed as
     E = r(0) - 2 hs'[i] p[i] + hs'[i] R hs[i] ,
   where the matrix R has elements R(m,k) = rxx(m-k). This expression is
   general in that the vector hs[i] need not be the "optimal" value.
*/
    mse[i] = INfiltMSE (rxx, hs, p, Nm) / rxx[0];
    if (mse[i] < 0.0) {
      UTwarn ("INintMSE: Non-positive definite correlation values");
      mse[i] = 0.0;
    }
  }

/* Deallocate the temporary arrays */
  UTfree (rxx);
  UTfree (p);
  UTfree (hs);

/* Return the mean-square error array */
  return mse;
}
