/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  double INtimeResp (double t, const float f[], const float fresp[], int N)

Purpose:
  Evaluate a time response given frequency response values

Description:
  This function returns the value of a time response at a given time.  The time
  response function is calculated as the Fourier transform of a symmetrical
  real-valued frequency response.  The corresponding time response is real and
  symmetrical about zero.

  The frequency response is specified in tabular form.  A piece-wise monotonic
  cubic interpolation is used between the given values.  The frequency response
  is specified for non-negative frequencies.

Parameters:
  <-  double INtimeResp
      Returned time response value
   -> double t
      Time value at which the time response is to be evaluated
   -> const float f[]
      Array of frequency values.  These values must be in increasing order and
      have non-negative values.
   -> const float fresp[]
      Array of frequency response values.  Value fresp[i] is the frequency
      response at frequency f[i].
   -> int N
      Number of frequency response values specified.  N must be at least two.

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.9 $  $Date: 1996/02/15 13:05:15 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: INtimeResp.c 1.9 1996/02/15 FilterDesign-V1R7a $";

#include <math.h>
#include <libtsp.h>
#include <libtsp/nucleus.h>
#include "DFiltInt.h"

static double
IN_cosIntMC p_((double t, double fl, double fh, double vl, double vh,
		double dl, double dh));

#ifndef PI		/* Sometimes in math.h */
#  define PI		3.14159265358979323846
#endif
#define PIx2		6.28318530717958647693

double
INtimeResp (t, f, fresp, N)

     double t;
     const float f[];
     const float fresp[];
     int N;

{
  int i;
  double dh, fh, dl, fl, vh, vl;
  double sum;

/* Error checks */
  if (N <= 1)
    UThalt ("INtimeResp: Too few frequency values");
  if (f[0] < 0.0)
    UThalt ("INtimeResp: Negative frequency");
  if (fresp[0] < 0.0)
    UThalt ("INtimeResp: Negative power spectrum value");

/* Loop over intervals */
  /* Values at the initial point */
  fl = f[0];
  vl = fresp[0];
  if (fl == 0.0 || vl == 0.0)
    dl = 0.0;
  else
    dl = MSslopeMC (0, f, fresp, N);

  sum = 0.0;
  for (i = 1; i < N; ++i) {

    fh = f[i];
    if (fh <= fl)
      UThalt ("INtimeResp: Frequency values not increasing");
    vh = fresp[i];
    if (vh < 0.0)
      UThalt ("INtimeResp: Negative power spectrum value");

    /* Determine the slope at the end of the interval */
    if (i == N - 1 && vh == 0.0)
      dh = 0.0;
    else
      dh = MSslopeMC (i, f, fresp, N);

    /* Evaluate the integral for one cubic segment */
    sum = sum + IN_cosIntMC (t, fl, fh, vl, vh, dl, dh);

    fl = fh;
    vl = vh;
    dl = dh;
  }

  return 2.0 * sum;
}

/*
   The integral used to calculate the Fourier transform is

      f2
     Int P(f) cos(a f) df,  where a=2 Pi t.
      f1

   If a=0, we directly evaluate of the integral without the cosine term.
*/
static double
IN_cosIntMC (t, fl, fh, vl, vh, dl, dh)
     double t;
     double fl;
     double fh;
     double vl;
     double vh;
     double dl;
     double dh;

{
  double a;
  double df, dx, sx;
  double Cosdx, Sindx, Cossx, Sinsx;
  double tdvk, tsvk, tddk, tsdk;
  double val;

  df = fh - fl;
  if (df == 0.0) {
    if (vl != vh || dl != dh)
      UThalt ("INtimeResp: Inconsistent polynomial specification");
    else
      val = 0.0;
  }
  else if (t == 0.0) {

/* Direct evaluation of the integral for t = 0 */
    val = df * (df * (dl-dh) / 12.0 + 0.5 * (vl+vh));

  }
  else {

/*
   Integral with cosine term; express the integral as

      f2                           x2
     Int P(f) cos(a f) df = (1/a) Int P(x/a) cos(x) dx ,  where x=a f.
      f1                           x1

  The cubic polynomial is defined in terms of its values and derivatives at
  the end points of the integral.  The expression below was determined using
  a symbolic math program using the following steps.
  (1) Find the integral in terms of the direct form coefficients;
  (2) Solve for the direct form coefficients in terms of the values and
      derivatives at the end points;
  (3) Substitute for the direct form coefficients, and simplify the result
      algebraically.
  The final form combines terms such that each individual expression goes to
  zero as the interval size decreases (the txxk terms below each go to zero as
  dx goes to zero).
*/
    a = PIx2 * t;
    dx = PI * df * t;
    sx = PI * (fl + fh) * t;
    Cossx = cos (sx);
    Sinsx = sin (sx);
    Cosdx = cos (dx);
    Sindx = sin (dx);

    tdvk = (3.0 * (dx * Cosdx - Sindx) / (dx*dx) + dx * Cosdx) / dx;
    tsvk = Sindx;
    tddk = (dx * Cosdx - Sindx) / dx;
    tsdk = (3.0 * (Sindx - dx * Cosdx) / (dx*dx) - Sindx);
    val = (Sinsx * ((vh-vl) * tdvk + (dl+dh) * tsdk / a) +
	   Cossx * ((vl+vh) * tsvk + (dh-dl) * tddk / a) ) /a;
  }

  return val;
}
