
#ifndef _MUSIC_CLASSES_
#define _MUSIC_CLASSES_

#include <stdio.h>

#include "Notes.h"
#include "Lists.h"
#include "Tags.h"


extern Dimension    GetZeroWidth    (MusicObject);
extern MTime        GetZeroLength   (MusicObject);
extern MusicObject  GetLongestChord (MusicObject); /* Chord not yet defined */
extern NoteVoice   *GetHighestNote  (MusicObject);
extern NoteVoice   *GetLowestNote   (MusicObject);
extern void         WriteNothing    (MusicObject, FILE *, int);
extern void         MidiWriteNothing(MusicObject, List, long, ClefTag);
extern Dimension    DrawNothing     (MusicObject, Drawable, Position,
				     Position, Pitch, Dimension, LineEquation);


typedef struct _GenericPart {
  ClassTag      object_class;
} GenericPart;

typedef struct _Generic {
  GenericPart   generic;
} Generic;

extern Generic *NewGeneric(Generic *);


typedef struct _Methods {
  Dimension     (* get_min_width) (MusicObject);
  MusicObject   (* get_shortest)  (MusicObject);
  NoteVoice   * (* get_highest)   (MusicObject);
  NoteVoice   * (* get_lowest)    (MusicObject);
  MTime         (* get_length)    (MusicObject);
  MusicObject   (* clone)         (MusicObject);
  void          (* write)         (MusicObject, FILE *, int);
  void          (* write_midi)    (MusicObject, List, long, ClefTag);
  /* item, EventList, delta, clef for offset */
  Dimension     (* draw)          (MusicObject, Drawable, Position, Position,
				   Pitch, Dimension, LineEquation);
  /* item, pixmap, x & y coordinates, clef offset, width, grouping line eqn */
  void          (* destroy)       (MusicObject);
} Methods;

typedef struct _ItemPart {
  Methods      *methods;
  MarkList     *marks;
  short         x;
} ItemPart;

typedef struct _Item {
  GenericPart   generic;
  ItemPart      item;
} Item;

extern Methods    itemMethods;
extern Boolean    ReadItem(MusicObject *, FILE *);
extern Item      *NewItem(Item *);
extern void       DestroyItem(MusicObject);
extern MusicObject CloneItem(MusicObject);

typedef struct _ItemListElement {
  ListElement   typeless;
  Item         *item;
} ItemListElement, *ItemList;

extern ItemList NewItemList(Item *); /* one-item list */
extern void     DestroyItemList(ItemList);


typedef struct _IndicatorPart {
  /* empty */
  char          dummy_field;
} IndicatorPart;

typedef struct _Indicator {
  GenericPart   generic;
  ItemPart      item;
  IndicatorPart indicator; 
} Indicator;

extern Methods    indicatorMethods;
extern Indicator *NewIndicator(Indicator *);        /* width (for Item) */
extern void       DestroyIndicator(MusicObject);
extern MusicObject CloneIndicator(MusicObject);


typedef struct _ClefPart {
  ClefTag       clef;
  ClefVisual    visual;
} ClefPart;

typedef struct _Clef {
  GenericPart   generic;
  ItemPart      item;
  IndicatorPart indicator;
  ClefPart      clef;
} Clef;  

extern Methods    clefMethods;
extern Clef      *NewClef(Clef *, ClefTag);	/* clef */
extern Dimension  GetClefWidth(MusicObject);
extern void       WriteClef(MusicObject, FILE *, int);
extern Boolean    ReadClef(MusicObject, FILE *);
extern Dimension  DrawClef(MusicObject, Drawable, Position,
			   Position, Pitch, Dimension, LineEquation);
extern void       DestroyClef(MusicObject);
extern Pitch      ClefPitchOffset(ClefTag);
extern MusicObject CloneClef(MusicObject);

extern Clef       defaultClef;


typedef struct _KeyPart {
  KeyTag        key;
  KeyVisual     visual;
} KeyPart;

typedef struct _Key {
  GenericPart   generic;
  ItemPart      item;
  IndicatorPart indicator;
  KeyPart       key;
} Key;

extern Methods    keyMethods;
extern Key       *NewKey(Key *, KeyTag);	/* key */
extern Dimension  GetKeyWidth(MusicObject);
extern void       WriteKey(MusicObject, FILE *, int);
extern Boolean    ReadKey(MusicObject, FILE *);
extern Dimension  DrawKey(MusicObject, Drawable, Position,
			  Position, Pitch, Dimension, LineEquation);
extern void       DestroyKey(MusicObject);
extern MusicObject CloneKey(MusicObject);


typedef struct _TimeSigPart {
  int           numerator;
  int           denominator;
  MTime         bar_length;
} TimeSigPart;

typedef struct _TimeSignature {
  GenericPart   generic;
  ItemPart      item;
  IndicatorPart indicator;
  TimeSigPart   timesig;
} TimeSignature;

extern Methods        timeSigMethods;
extern TimeSignature *NewTimeSignature(TimeSignature *, int, int);
			                   	/* num, denom */
extern Dimension      GetTimeSignatureWidth(MusicObject);
extern void           WriteTimeSignature(MusicObject, FILE *, int);
extern Boolean        ReadTimeSignature(MusicObject, FILE *);
extern Dimension      DrawTimeSignature(MusicObject, Drawable, Position,
					Position, Pitch, Dimension,
					LineEquation);
extern void           DestroyTimeSignature(MusicObject);
extern MusicObject CloneTimeSignature(MusicObject);

extern TimeSignature  defaultTimeSignature;



typedef struct _TextPart {
  String        text;
  TextPosnTag   position;
} TextPart;

typedef struct _Text {
  GenericPart   generic;
  ItemPart      item;
  IndicatorPart indicator;
  TextPart      text;
} Text;

extern Methods    textMethods;
extern Text      *NewText(Text *, String, TextPosnTag); /* text, position */
extern void       WriteText(MusicObject, FILE *, int);
extern void       MidiWriteText(MusicObject, List, long, ClefTag);
extern Boolean    ReadText(MusicObject, FILE *);
extern Dimension  DrawText(MusicObject, Drawable, Position,
			   Position, Pitch, Dimension, LineEquation);
extern void       DestroyText(MusicObject);
extern MusicObject CloneText(MusicObject);


typedef struct _PhrasePart {
  Boolean       tied_forward;
  Boolean       tied_backward;
} PhrasePart;

typedef struct _Phrase {
  GenericPart   generic;
  ItemPart      item;
  PhrasePart    phrase;
} Phrase;

extern Methods    phraseMethods;
extern Phrase    *NewPhrase(Phrase *);
extern void       DestroyPhrase(MusicObject);
extern MusicObject ClonePhrase(MusicObject);


typedef struct _RestPart {
  MTime         length;
  RestVisual    visual;
} RestPart;

typedef struct _Rest {
  GenericPart   generic;
  ItemPart      item;
  PhrasePart    phrase;
  RestPart      rest;
} Rest;

extern Methods      restMethods;
extern Rest        *NewRest(Rest *, NoteTag, Boolean); 	/* type, dotted */
extern Dimension    GetRestWidth(MusicObject);
extern MTime        GetRestLength(MusicObject);
extern MusicObject  GetRestShortest(MusicObject);
extern void         WriteRest(MusicObject, FILE *, int);
extern void         MidiWriteRest(MusicObject, List, long, ClefTag);
extern Boolean      ReadRest(MusicObject, FILE *);
extern Dimension    DrawRest(MusicObject, Drawable, Position,
			     Position, Pitch, Dimension, LineEquation);
extern void         DestroyRest(MusicObject);
extern MusicObject CloneRest(MusicObject);


typedef struct _GroupPart {
  GroupTag      type;
  ItemList      start;
  ItemList      end;
} GroupPart;

typedef struct _Group {
  GenericPart   generic;
  ItemPart      item;
  PhrasePart    phrase;
  GroupPart     group;
} Group;

extern Methods      groupMethods;
extern Group       *NewGroup(Group *, GroupTag, ItemList, ItemList);
extern Dimension    GetGroupWidth(MusicObject);     /* ^-- type, start, end */
extern MTime        GetGroupLength(MusicObject);
extern MusicObject  GetGroupShortest(MusicObject);
extern NoteVoice   *GetGroupLowest(MusicObject);
extern NoteVoice   *GetGroupHighest(MusicObject);
extern void         WriteGroup(MusicObject, FILE *, int);
extern Boolean      ReadGroup(MusicObject, FILE *);
extern Dimension    DrawGroup(MusicObject, Drawable, Position,
			      Position, Pitch, Dimension, LineEquation);
extern void         DestroyGroup(MusicObject);
extern MusicObject CloneGroup(MusicObject);


typedef struct _BarPart {

  BarTag         start_bar;
  BarTag         end_bar;
  unsigned long  number;

  /* The following are instantiated by the format method ... */

  TimeSignature *time;	       /* bar methods should never try to free these */
  Clef          *clef;
  Key           *key;
  Boolean        still_as_drawn;

  /* This is manipulated sometimes by the Bar and sometimes by the Stave: */

  short          width;		                       /* from last drawing */

} BarPart;

typedef struct _Bar {
  GenericPart   generic;
  ItemPart      item;
  PhrasePart    phrase;
  GroupPart     group;
  BarPart       bar;
} Bar;

extern Methods    barMethods;
extern Bar       *NewBar(Bar *, unsigned long, BarTag, BarTag);
				/* number, start_bar, end_bar */
extern void       DestroyBar(MusicObject);

extern Dimension GetBarWidth(MusicObject);
extern Dimension DrawBar(MusicObject, Drawable, Position,
			 Position, Pitch, Dimension, LineEquation);
extern MusicObject CloneBar(MusicObject);



typedef struct _ChordPart {
  MTime         length;
  ChordMods     modifiers;
  NoteVisual    visual;
  NoteVoice    *voices;
  short         voice_count;
  Dimension     note_modifier_width;
} ChordPart;

typedef struct _Chord {
  GenericPart   generic;
  ItemPart      item;
  PhrasePart    phrase;
  ChordPart     chord;
} Chord;

extern Methods      chordMethods;

extern Chord       *NewChord(Chord *, NoteVoice *, int,
			     ChordMods, NoteTag, Boolean);

				/* note voices, voice count, mods,
				   note length tag, dotted */

extern Dimension    GetChordWidth(MusicObject);
extern MTime        GetChordLength(MusicObject);
extern MusicObject  GetChordShortest(MusicObject);
extern NoteVoice   *GetChordLowest(MusicObject);
extern NoteVoice   *GetChordHighest(MusicObject);
extern void         WriteChord(MusicObject, FILE *, int);
extern void         MidiWriteChord(MusicObject, List, long, ClefTag);
extern Boolean      ReadChord(MusicObject, FILE *);
extern Dimension    DrawChord(MusicObject, Drawable, Position,
			      Position, Pitch, Dimension, LineEquation);
extern void         DestroyChord(MusicObject);
extern MusicObject CloneChord(MusicObject);

extern Chord longestChord;
extern Chord restChord;



typedef struct _MetronomePart {
  Chord         beat;
  MTime         beat_length;
  unsigned int  setting;
} MetronomePart;

typedef struct _Metronome {
  GenericPart   generic;
  ItemPart      item;
  IndicatorPart indicator;
  MetronomePart metronome;
} Metronome;

extern Methods    metronomeMethods;
extern Metronome *NewMetronome(Metronome *, NoteTag, Boolean, unsigned int);
				                /* beat, dotted, setting */
extern void       WriteMetronome(MusicObject, FILE *, int);
extern void       MidiWriteMetronome(MusicObject, List, long, ClefTag);
extern Boolean    ReadMetronome(MusicObject, FILE *);
extern Dimension  DrawMetronome(MusicObject, Drawable, Position,
				Position, Pitch, Dimension, LineEquation);
extern void       DestroyMetronome(MusicObject);
extern MusicObject CloneMetronome(MusicObject);


#endif /* _MUSIC_CLASSES_ */

