
/* Music Editor             */
/* Constructor/Destructor   */
/* Functions for Classes &c */
/* Chris Cannam, Jan 1994   */

/* Destructors should only  */
/* be called for objects    */
/* constructed by passing   */
/* null space into the new  */
/* function and letting it  */
/* allocate space itself    */

#include <Lists.h>

#include "General.h"
#include "Tags.h"
#include "Notes.h"
#include "Classes.h"
#include "Stave.h"
#include "Visuals.h"

#define SPACE(x,y)  { if (!(x)) (x) = (y *)malloc((Cardinal)sizeof(y)); \
                      if (!(x)) Error("Couldn't perform malloc"); }
#define CLASS(x,y)  { (x)->generic.object_class = (y); }


/* The paradigm for constructors is as follows.  The first argument */
/* is always a pointer to the type to be constructed.  If this is   */
/* NULL, space will be allocated using malloc, a new instance put   */
/* in it and a pointer to it returned.  If it is non-NULL, it will  */
/* be initialised with the new instance.  In either case, a pointer */
/* to the newly initialised space is returned.                      */

/* The exception is lists, which are opaque pointers anyway; their  */
/* constructors always use malloc, and don't take the initial arg.  */

/* Strings and lists passed as arguments to constructors will be    */
/* shared (no new copy will be made); destructors will likewise not */
/* attempt to destroy them.                                         */


/* SIMPLE AND COMPOUND STRUCTURES */

/* NewMTime is in MTime.c */


NoteVoice *NewNoteVoice(NoteVoice *p, Pitch pitch, NoteMods modifiers)
{
  Begin("NewNoteVoice");

  SPACE(p, NoteVoice);

  p->pitch             = pitch;
  p->modifiers         = modifiers;
  p->display_mods      = modifiers;

  Return(p);
}


void DestroyNoteVoice(NoteVoice *p)
{
  Begin("DestroyNoteVoice");

  free((void *)p);

  End;
}



/* CLASSES */

Generic *NewGeneric(Generic *p)
{
  Begin("NewGeneric");

  SPACE(p, Generic);
  CLASS(p, GenericClass);

  Return(p);
}


void DestroyGeneric(MusicObject p)
{
  Begin("DestroyGeneric");

  free(p);

  End;
}


Item *NewItem(Item *p)
{
  Begin("NewItem");

  SPACE(p, Item);

  (void)NewGeneric((Generic *)p);

  p->item.methods   = &itemMethods;
  p->item.x         = 0;

  CLASS(p, ItemClass);
  Return(p);
}


MusicObject CloneItem(MusicObject p)
{
  Begin("CloneItem");
  Return((MusicObject)NewItem(NULL));
}


void DestroyItem(MusicObject p)
{
  Begin("DestroyItem");

  DestroyGeneric(p);

  End;
}


Indicator *NewIndicator(Indicator *p)
{
  Begin("NewIndicator");

  SPACE(p, Indicator);

  (void)NewItem((Item *)p);
  p->item.methods = &indicatorMethods;

  CLASS(p, IndicatorClass);
  Return(p);
}


MusicObject CloneIndicator(MusicObject p)
{
  Begin("CloneIndicator");
  Return((MusicObject)NewIndicator(NULL));
}


void DestroyIndicator(MusicObject p)
{
  Begin("DestroyIndicator");

  DestroyItem(p);

  End;
}


Metronome *NewMetronome(Metronome *p, NoteTag beat,
			Boolean dotted, unsigned int setting)
{
  Begin("NewMetronome");

  SPACE(p, Metronome);

  (void)NewIndicator((Indicator *)p);
  (void)NewChord(&(p->metronome.beat),
		 &metronomeNoteVoice, 1, ModNone, beat, dotted);

  p->metronome.beat_length = TagToMTime(beat, dotted);

  /* Don't need to change derived get_length, because */
  /* all Indicators have length 0 anyway...           */

  p->item.methods          = &metronomeMethods;
  p->metronome.setting     = setting;

  CLASS(p, MetronomeClass);
  Return(p);
}


MusicObject CloneMetronome(MusicObject p)
{
  Begin("CloneMetronome");
  Return((MusicObject)NewMetronome
	 (NULL, ((Metronome *)p)->metronome.beat.chord.visual->type,
	  ((Metronome *)p)->metronome.beat.chord.visual->dotted,
	  ((Metronome *)p)->metronome.setting));
}


void DestroyMetronome(MusicObject p)
{
  Begin("DestroyMetronome");

  DestroyIndicator(p);

  End;
}


Clef *NewClef(Clef *p, ClefTag tag)
{
  Begin("NewClef");
  
  SPACE(p, Clef);

  (void)NewIndicator((Indicator *)p);

  p->item.methods       = &clefMethods;
  p->clef.clef          = tag;
  p->clef.visual        = &(clefVisuals[tag]);

  CLASS(p, ClefClass);
  Return(p);
}


MusicObject CloneClef(MusicObject p)
{
  Begin("CloneClef");
  Return(NewClef(NULL, ((Clef *)p)->clef.clef));
}


void DestroyClef(MusicObject p)
{
  Begin("DestroyClef");

  DestroyIndicator(p);

  End;
}


Key *NewKey(Key *p, KeyTag tag)
{
  Begin("NewKey");

  SPACE(p, Key);

  (void)NewIndicator((Indicator *)p);

  p->item.methods       = &keyMethods;
  p->key.key            = tag;
  p->key.visual         = &(keyVisuals[tag]);

  CLASS(p, KeyClass);
  Return(p);
}


MusicObject CloneKey(MusicObject p)
{
  Begin("CloneKey");
  Return(NewKey(NULL, ((Key *)p)->key.key));
}


void DestroyKey(MusicObject p)
{
  Begin("DestroyKey");

  DestroyIndicator(p);

  End;
}


/* trying to use a sig whose denom is not a power of two may be troublesome */

TimeSignature *NewTimeSignature(TimeSignature *p,
				int numerator, int denominator)
{
  int count;

  Begin("NewTimeSignature");

  SPACE(p, TimeSignature);

  (void)NewIndicator((Indicator *)p);

  p->item.methods        = &timeSigMethods;
  p->timesig.numerator   = numerator;
  p->timesig.denominator = denominator;

  for (count = 0; denominator > 1; denominator = denominator/2, ++ count);

  (void)NewMTime(&(p->timesig.bar_length), Semibreve - count, numerator);

  CLASS(p, TimeSignatureClass);
  Return(p);
}


MusicObject CloneTimeSignature(MusicObject p)
{
  Begin("CloneTimeSignature");
  Return(NewTimeSignature(NULL, ((TimeSignature *)p)->timesig.numerator,
			  ((TimeSignature *)p)->timesig.denominator));
}


void DestroyTimeSignature(MusicObject p)
{
  Begin("DestroyTimeSignature");

  DestroyIndicator(p);

  End;
}


Text *NewText(Text *p, String text, TextPosnTag tag)
{
  Begin("NewText");

  SPACE(p, Text);

  (void)NewIndicator((Indicator *)p);

  p->item.methods       = &textMethods;
  p->text.text          = XtNewString(text);
  p->text.position      = tag;

  CLASS(p, TextClass);
  Return(p);
}


MusicObject CloneText(MusicObject p)
{
  Begin("CloneText");
  Return(NewText(NULL, XtNewString(((Text *)p)->text.text),
		 ((Text *)p)->text.position));
}


void DestroyText(MusicObject p)
{
  Begin("DestroyText");

  free(((Text *)p)->text.text);
  DestroyIndicator(p);

  End;
}


Phrase *NewPhrase(Phrase *p)
{
  Begin("NewPhrase");

  SPACE(p, Phrase);

  (void)NewItem((Item *)p);
  p->item.methods = &phraseMethods;

  p->phrase.tied_forward  = False;
  p->phrase.tied_backward = False;

  CLASS(p, PhraseClass);
  Return(p);
}


MusicObject ClonePhrase(MusicObject p)
{
  Begin("ClonePhrase");
  Return(NewPhrase(NULL));
}


void DestroyPhrase(MusicObject p)
{
  Begin("DestroyPhrase");

  DestroyItem(p);

  End;
}


Rest *NewRest(Rest *p, NoteTag tag, Boolean dotted)
{
  Begin("NewRest");

  SPACE(p, Rest);

  (void)NewPhrase((Phrase *)p);

  p->rest.length = TagToMTime(tag, dotted);

  if (dotted) p->rest.visual = &(restVisuals[tag].dotted);
  else        p->rest.visual = &(restVisuals[tag].undotted);

  p->item.methods = &restMethods;

  CLASS(p, RestClass);
  Return(p);
}


MusicObject CloneRest(MusicObject p)
{
  Begin("CloneRest");
  Return(NewRest(NULL, ((Rest *)p)->rest.visual->type,
		 ((Rest *)p)->rest.visual->dotted));
}


void DestroyRest(MusicObject p)
{
  Begin("DestroyRest");

  DestroyPhrase(p);

  End;
}


/* Chord constructor and destructor assume you've got space  */
/* for the NoteVoice array using malloc or something, and    */
/* that it's allowed to free it up again when the destructor */
/* gets called.                                              */

Chord *NewChord(Chord *p, NoteVoice *voices, int voiceCount,
		ChordMods modifiers, NoteTag tag, Boolean dotted)
{
/*  int       i; 
  NoteMods  cmods;
  int       mods; */

  Begin("NewChord");

  SPACE(p, Chord);

  (void)NewPhrase((Phrase *)p);

  p->item.methods              = &chordMethods;
  p->chord.voices              = voices;
  p->chord.voice_count         = voiceCount;
  p->chord.modifiers           = modifiers;
  p->chord.note_modifier_width = 0;
  p->chord.length              = TagToMTime(tag, dotted);
  p->chord.visual =
    dotted ? &(noteVisuals[tag].dotted) : &(noteVisuals[tag].undotted);
/*
  for (i = 0; i < voiceCount; ++i) {

    for (mods = 0, cmods = voices[i].modifiers; cmods; cmods >>= 1)
      if (cmods & 1) ++mods;

    if (mods * NoteModWidth > p->chord.note_modifier_width)
      p->chord.note_modifier_width = mods * NoteModWidth;
  }

  p->chord.note_modifier_width = p->chord.note_modifier_width * 2 / 3;
*/
  CLASS(p, ChordClass);
  Return(p);
}


MusicObject CloneChord(MusicObject p)
{
  int        i;
  NoteVoice *v;
  Chord     *c = (Chord *)p;

  Begin("CloneChord");

  v = (NoteVoice *)XtMalloc(sizeof(NoteVoice) * c->chord.voice_count);
  for (i = 0; i < c->chord.voice_count; ++i) v[i] = c->chord.voices[i];

  Return(NewChord(NULL, v, i, c->chord.modifiers,
		  c->chord.visual->type, c->chord.visual->dotted));
}


void DestroyChord(MusicObject p)
{
  Begin("DestroyChord");

  free((void *)(((Chord *)p)->chord.voices));
  DestroyPhrase(p);

  End;
}


/* Notice that the list given to a group may be a sublist of another   */
/* list; no operations should use First, Last &c. on the group's list. */
/* The group's destructor does not deallocate the itemlist.            */

Group *NewGroup(Group *p, GroupTag type, ItemList start, ItemList end)
{
  Begin("NewGroup");

  SPACE(p, Group);

  (void)NewPhrase((Phrase *)p);

  p->item.methods = &groupMethods;
  p->group.start  = start;
  p->group.end    = end;
  p->group.type   = type;

  CLASS(p, GroupClass);
  Return(p);
}


/* This doesn't do the right thing with its list, because that's quite */
/* difficult to do without knowing the context.  Take care.            */

MusicObject CloneGroup(MusicObject p)
{
  Begin("CloneGroup");

  Return(NewGroup(NULL, ((Group *)p)->group.type,
		  ((Group *)p)->group.start, ((Group *)p)->group.end));
}


void DestroyGroup(MusicObject p)
{
/*  Group   *g = (Group *)p;
  ItemList list; */

  Begin("DestroyGroup");

  DestroyPhrase(p);

  End;
}


ItemList NewItemList(Item *item)
{
  ItemList list;
  
  Begin("NewItemList");

  list = (ItemList)NewList(sizeof(ItemListElement));
  list->item = item;

  Return(list);
}



void DestroyItemList(ItemList list)
{
  ItemList first = (ItemList)First(list);

  Begin("DestroyItemList");

  list = first;

  while (list) {

    list->item->item.methods->destroy((MusicObject)(list->item));
    list = (ItemList)Next(list);
  }

  DestroyList(first);

  End;
}



Bar *NewBar(Bar *p, unsigned long number, BarTag start, BarTag end)
{
  Begin("NewBar");

  SPACE(p, Bar);

  (void)NewGroup((Group *)p, GroupNoDecoration, NULL, NULL);

  p->item.methods       = &barMethods;
  p->bar.start_bar      = start;
  p->bar.end_bar        = end;
  p->bar.number         = number;
  p->bar.still_as_drawn = False;

  CLASS(p, BarClass);
  Return(p);
}


/* This is basically wrong.  Please don't use it. */

MusicObject CloneBar(MusicObject p)
{
  Begin("CloneBar");
  Return(NewBar(NULL, ((Bar *)p)->bar.number, ((Bar *)p)->bar.start_bar,
		((Bar *)p)->bar.end_bar));
}


void DestroyBar(MusicObject p)
{
  Begin("DestroyBar");

  DestroyGroup(p);

  End;
}

