
/* Musical Notation Editor for X, Chris Cannam 1994 */

/* Drawing methods */

/* Feb 95: occurrences of "this" changed to "curr" to  */
/* satisfy C++ compilers                               */

#include "General.h"
#include "Notes.h"
#include "Classes.h"
#include "Visuals.h"
#include "GC.h"
#include "ItemList.h"
#include "Spline.h"

typedef struct _BeamPoint {
  Position  x;
  Position  y;
  Dimension width;
  NoteTag   tag;
  Chord    *chord;
} BeamPoint;

#define BeamPointMax 100

/* Making these global means that I */
/* can't have beamed groups within  */
/* beamed groups. I am unrepentant. */

static int       beampointcount = 0;
static BeamPoint beampoints[BeamPointMax+1];


/* tie[0] holds info for back-tie, tie[1] for forward */

static struct {
  Boolean  present;
  Position x;
  Boolean  above;
} tie[] = { { False, 0, False, }, { False, 0, False, }, };


#define TIE_DIST  (3*NoteHeight)

#define EQN(e,x) ((int)((e)->m*(x))+(e)->c)

/*
   Note on Drawing Coordinates.

   The X coordinate specified is the coordinate of the start of
   the note associated with that object.  Functions are free to
   place their objects to the left or right of this coordinate
   as appropriate.

   The Y coordinate is that of the top line of the stave.  The
   function is always free to draw up to StaveUpperGap pixels
   above, and StaveHeight + StaveLowerGap pixels below, that
   coordinate.

   If a width is passed which is less than the minimum width
   of the object to be drawn, the minimum width will be assumed
   instead.

   All drawing functions return the drawn width of the object.
   If the passed width was wider than the minimum object width,
   the returned width will be equal to the passed width;
   otherwise the minimum width will be returned.
*/



Dimension DrawNothing(MusicObject obj, Drawable drawable, Position x,
		      Position y, Pitch p, Dimension width, LineEquation eqn)
{
  Begin("DrawNothing");
  Return(width);
}



Dimension DrawNoteMod(NoteMods mod, Drawable drawable,
		      Position x, Position y, Pitch pitch)
{
  int n;
  Begin("DrawNoteMod");

  for (n = 0; !(noteModVisuals[n].type & mod); ++n);
  if  (n > 2) Return(0);

  CopyArea(noteModVisuals[n].pixmap, drawable, 0, 0,
	   NoteModWidth - 1, NoteModHeight, x,
	   y + STAVE_Y_COORD(pitch) - (NoteModHeight-NoteHeight)/2 -
	   ((mod & ModFlat) ? 2 : 0));

  Return(NoteModWidth);
}



/* Draws the body of a note, given its visual; this is perhaps misnamed, */
/* as a NoteVoice is only a record of the pitch and modifiers associated */
/* with a sound, and as such it doesn't really have a visual presence.   */

/* The x coord passed to this function should _already_ have had compen- */
/* sation added to it for the width of the NoteMods; this function won't */
/* compensate, it'll just draw the modifiers to the left of the x coord. */

Dimension DrawNoteVoice(NoteVoice *voice, Drawable drawable, Position x,
			Position y, Pitch offset, Dimension width,
			NoteVisual visual)
{
  Position   ny;
  Pitch      p;
  NoteMods   mods;
  int        modno = 0;
  int        modco = 0;
  Dimension  rwidth;
  Pitch      pitch = voice->pitch + offset;

  Begin("DrawNoteVoice");

  ny = y + STAVE_Y_COORD(pitch);          /* work out height of body */

  CopyArea(visual->body, drawable,                      /* draw body */
	   0, 0, NoteWidth, NoteHeight, x, ny);

  if (visual->dotted)
    CopyArea(noteDotMap, drawable, 0, 0, DotWidth, NoteHeight,
	     x + NoteWidth, ny);        /* draw dot to right of note */

  for (mods = voice->display_mods; mods; mods >>= 1, ++ modco)
    if (mods & 1)
      DrawNoteMod(1<<modco, drawable,         /* modify if necessary */
		  x - NoteModWidth * ++modno, y, pitch);

  if (pitch < -1) {                              /* need leger lines */

    p = -pitch;
    if (p/2 != (p+1)/2) p -= 1;
    p = -p;

    for (ny = y + STAVE_Y_COORD(p) + NoteHeight/2 - 1; p < -1;
	 p += 2, ny -= NoteHeight + 1)
      XDrawLine(display, drawable, drawingGC, x-1, ny, x+NoteWidth-1, ny);

  } else if (pitch > 9) {

    p = pitch;
    if (p/2 != (p+1)/2) p -= 1;

    for (ny = y + STAVE_Y_COORD(p) + NoteHeight/2; p > 9;
	 p -= 2, ny += NoteHeight + 1)
      XDrawLine(display, drawable, drawingGC, x-1, ny, x+NoteWidth-1, ny);
  } 

  rwidth = NoteWidth + (visual->dotted ? DotWidth : 0);

  if (width > 0) Return(width);
  else Return(rwidth);
}



Dimension DrawChord(MusicObject obj, Drawable drawable, Position x,
		    Position y, Pitch offset, Dimension width,
		    LineEquation eqn)
{
  Position   nx, ay, by;
  Pitch      lowest  = highestNoteVoice.pitch;
  Pitch      highest =  lowestNoteVoice.pitch;
  Boolean    up = True;
  Dimension  rwidth = NoteWidth;
  Chord     *chord  = (Chord *)obj;
  NoteVoice *voices;
  int        voiceno, voiceCount, i;
  ChordMods  mods;
  int        modno = 0;
  int        modco = 0;
  int        extent;
  NoteTag    type;

  Begin("DrawChord");

  if (!chord) Return(width);

  x      += chord->chord.note_modifier_width;
  rwidth += chord->chord.note_modifier_width;
  type    = chord->chord.visual->type;
  chord->item.x = x;

  if (chord->chord.visual->dotted) rwidth += DotWidth;
 
  /* First draw the notes */

  for (voiceno = 0, voices = chord->chord.voices,
       voiceCount = chord->chord.voice_count; voiceno < voiceCount;
       ++voiceno) {

    /* if there's another note next door to this one, offset this */
    /* one if it's in a space and leave it if it's on a line      */

    if (voiceCount > 1 &&
	(voices[voiceno].pitch + 100)/2 == (voices[voiceno].pitch + 99)/2) {

      for (i = 0; i < voiceCount; ++i)
	if (voices[i].pitch + offset == voices[voiceno].pitch + offset - 1 ||
	    voices[i].pitch + offset == voices[voiceno].pitch + offset + 1) {

	  DrawNoteVoice(&voices[voiceno], drawable, x + NoteWidth - 2, y,
			offset, width, chord->chord.visual);
	  break;
	}

      if (i >= voiceCount)
	DrawNoteVoice(&voices[voiceno], drawable,
		      x, y, offset, width, chord->chord.visual);

    } else DrawNoteVoice(&voices[voiceno], drawable,
			 x, y, offset, width, chord->chord.visual);

    if (voices[voiceno].pitch + offset > highest)
	 highest = voices[voiceno].pitch + offset;

    if (voices[voiceno].pitch + offset <  lowest)
	 lowest  = voices[voiceno].pitch + offset;
  }

  /* Draw the stalk, if there is one */

  if (chord->chord.visual->stalked) {

    if (eqn && eqn->eqn_present)
                        up = EQN(eqn, x) < (y + STAVE_Y_COORD(lowest));
    else
      if  (highest > 4)
	if (lowest > 4) up = False;
	else            up = ((highest - 4) < (5 - lowest));
      else              up = True;

    if (eqn && eqn->reverse) { up = !up; eqn = NULL; }
    
    nx = x + NoteWidth - 2;
    ay = y + STAVE_Y_COORD(lowest);
    by = y + STAVE_Y_COORD(highest);

    extent = eqn ? 0 : (type < Quaver ? 4*(Quaver-type) : 0);

    if (up) {

      XDrawLine
	(display, drawable, drawingGC, nx, ay + NoteHeight/2, (int)nx,
	 (int)((eqn == NULL) ?
	       by - NoteHeight*5/2 - extent : EQN(eqn,nx)));

    } else {
      XDrawLine
	(display, drawable, drawingGC, x, by + NoteHeight/2, (int)x,
	 (int)((eqn == NULL) ?
	       ay + NoteHeight*7/2 + extent : EQN(eqn,x)));
    }

    /* Add note tails if there's no line equation for a beam */

    if (eqn == NULL && type < Crotchet && chord->chord.visual->stalked)
      if (up) {
	CopyArea(tailDownMap[type],
		 drawable, 0, 0, TailWidth, TailHeight + 6*(Quaver-type),
		 nx + 1, by - NoteHeight*5/2 - extent + 1);
      } else {
	CopyArea(tailUpMap[type],
		 drawable, 0, 0, TailWidth, TailHeight + 6*(Quaver-type),
		 x + 1, ay + extent + NoteHeight*7/2 -
		 TailHeight - 6*(Quaver-type));
      }
  }
  
  /* Decorate with modifiers if necessary */

  if (up) {
    ay = y + NoteHeight + STAVE_Y_COORD(lowest);
    if (lowest  > 1) lowest  = 1;
    by = y + NoteHeight + STAVE_Y_COORD(lowest);
  } else {
    ay = y - NoteHeight + STAVE_Y_COORD(highest);
    if (highest < 7) highest = 7;
    by = y - NoteHeight + STAVE_Y_COORD(highest);
  }

  for (modno = 0, mods = chord->chord.modifiers; mods; mods >>= 1, ++ modco)
    if (mods & 1L)
      switch(modco) {

      case ModPausePower:
	CopyArea
	  (chordModVisuals[modco].pixmap, drawable, 0, 0, NoteWidth,
	   ChordModHeight, (int)x,
	   ay < y-NoteHeight ?
	   (int)ay - ChordModHeight*modno++ :
	   (int) y - ChordModHeight*modno++ - NoteHeight);
	break;

      case ModDotPower: case ModLegatoPower:
	CopyArea
	  (chordModVisuals[modco].pixmap, drawable, 0, 0, NoteWidth,
	   ChordModHeight, (int)x,
	   up ? (int)ay + ChordModHeight*modno++ :
	        (int)ay - ChordModHeight*modno++ );
	break;

      default:
	CopyArea
	  (chordModVisuals[modco].pixmap, drawable, 0, 0, NoteWidth,
	   ChordModHeight, (int)x,
	   up ? (int)by + ChordModHeight*modno++ + 3:
	        (int)by - ChordModHeight*modno++ - 3);
	break;
      }

  /* Finally, check out the possibility of a tie */

  if (chord->phrase.tied_backward && !tie[0].present) {
    tie[0].present = True;
    tie[0].x       = x + NoteWidth/3;
    tie[0].above   = !(lowest < 4);
  }

  if (chord->phrase.tied_forward) {
    tie[1].present = True;
    tie[1].above   = !(lowest < 4);
    tie[1].x       = x + (2*NoteWidth)/3;
  }
  
  if (width > 0) Return(width);
  else Return(rwidth);
}


Dimension DrawMetronome(MusicObject obj, Drawable drawable, Position x,
			Position y, Pitch offset, Dimension width,
			LineEquation eqn)
{
  Metronome *mnome = (Metronome *)obj;
  Dimension  off;
  char       string[20];

  Begin("DrawMetronome");

  if (!mnome) Return(width);

  mnome->item.x = x;

  off = DrawChord((MusicObject)&(mnome->metronome.beat),
		  drawable, x, 10, 0, 0, NULL);

  sprintf((char *)string, "= %d", mnome->metronome.setting);

  XDrawString(display, drawable, italicTextGC,
	      x + (int)off + 11, NoteHeight*7/2 + 8, string, strlen(string));

  Return(width);
}


Dimension DrawClef(MusicObject obj, Drawable drawable, Position x,
		   Position y, Pitch offset, Dimension width, LineEquation eqn)
{
  Begin("DrawClef");

  if (!obj) Return(width);

  ((Clef *)obj)->item.x = x;

  CopyArea(((Clef *)obj)->clef.visual->pixmap, drawable, 0, 0,
	   ClefWidth, StaveHeight + 2*NoteHeight, x, y - NoteHeight);

  if (width > 0) Return(width);
  else Return(ClefWidth + 3);
}


Dimension DrawKey(MusicObject obj, Drawable drawable, Position x,
		  Position y, Pitch offset, Dimension width, LineEquation eqn)
{
  int          n;
  Dimension    off;
  Pitch        pitch;
  Boolean      sharps;
  Key         *key = (Key *)obj;

  Begin("DrawKey");

  if (!obj) Return(width);

  key->item.x = x;
  sharps      = key->key.visual->sharps;
  pitch       = sharps ? 8 : 4;

  for (n = off = 0;
       n < key->key.visual->number; n++, off += NoteModWidth - 2) {

    DrawNoteMod(sharps ? ModSharp : ModFlat,
		drawable, x + off, y, pitch + offset);

    if (sharps) { pitch -= 3; if (pitch < 3) pitch += 7; }
    else        { pitch += 3; if (pitch > 7) pitch -= 7; }
  }

  if (width > 0) Return(width);
  else Return(off + 3);
}



Dimension DrawTimeSignature(MusicObject obj, Drawable drawable,
			    Position x, Position y, Pitch offset,
			    Dimension width, LineEquation eqn)
{
  char           a[5], b[5];
  int            dir, asc, desc;
  XID            fid;
  XCharStruct    info;
  int            wid;
  XGCValues      values;
  TimeSignature *sig = (TimeSignature *)obj;

  Begin("DrawTimeSignature");

  if (!obj) Return(width);
  sig->item.x = x;

  sprintf((char *)a, "%2d", sig->timesig.numerator);
  sprintf((char *)b, "%2d", sig->timesig.denominator);

  XDrawString(display, drawable, timeSigGC,
	      (int)x, (int)y + STAVE_Y_COORD(4) + NoteHeight/2 - 2,
	      a, strlen(a));
  XDrawString(display, drawable, timeSigGC,
	      (int)x, (int)y + STAVE_Y_COORD(0) + NoteHeight/2 - 2,
	      b, strlen(b));

  if (width > 0) Return(width);
  else Return(sig->item.methods->get_min_width(obj));
}


Dimension DrawText(MusicObject obj, Drawable drawable, Position x,
		   Position y, Pitch offset, Dimension width, LineEquation eqn)
{
  Text       *text = (Text *)obj;
  int         ty;
  int         dirn;
  int         dir, asc, desc;
  XCharStruct info;
  XGCValues   values;
  GC          gc;

  Begin("DrawText");

  if (!obj) Return(width);
  text->item.x = x;

  switch (text->text.position) {

  case TextAboveStave:
    ty   = y - 10;
    dirn = -1;
    gc   = littleTextGC;
    break;

  case TextAboveStaveLarge:
    ty   = y - StaveUpperGap + 42;
    dirn = -1;
    gc   = bigTextGC;
    break;

  case TextAboveBarLine:
    ty   = y + 2;
    dirn = -1;
    gc   = littleTextGC;
    break;

  case TextBelowStave:
    ty   = y + STAVE_Y_COORD(-5) + NoteHeight*7/2;
    dirn = 0;
    gc   = littleTextGC;
    break;

  case TextBelowStaveItalic:
    ty   = y + STAVE_Y_COORD(-5) + NoteHeight*7/2;
    dirn = 0;
    gc   = italicTextGC;
    break;
  }

  if (dirn != 0) {

    if (XGetGCValues(display, timeSigGC, GCFont, &values) == 0)
      Error("Could not get text graphics-context values");

    XQueryTextExtents(display, values.font, text->text.text,
		      1, &dir, &asc, &desc, &info);
    ty += dirn * asc;
  }

  XDrawString(display, drawable, gc, x, ty,
	      text->text.text, strlen(text->text.text));

  Return(width);
}


Dimension DrawRest(MusicObject obj, Drawable drawable, Position x,
		   Position y, Pitch offset, Dimension width, LineEquation eqn)
{
  Rest *rest = (Rest *)obj;

  Begin("DrawRest");
  
  if (!obj) Return(width);
  rest->item.x = x;

  CopyArea(rest->rest.visual->pixmap, drawable, 0, 0,
	   rest->rest.visual->width, StaveHeight, x, y);

  if (rest->rest.visual->dotted)
    CopyArea(noteDotMap, drawable, 0, 0, DotWidth, NoteHeight,
	     x + rest->rest.visual->width, y + STAVE_Y_COORD(5));

  if (rest->phrase.tied_backward && !tie[0].present) {
    tie[0].present = True;
    tie[0].above   = True;
    tie[0].x       = x + rest->rest.visual->width/3;
  }

  if (rest->phrase.tied_forward) {
    tie[1].present = True;
    tie[1].above   = True;
    tie[1].x       = x + (2*rest->rest.visual->width)/3;
  }
  
  if (width > 0) Return(width);
  else Return(rest->rest.visual->width);
}


void DrawBeam(Drawable drawable, LineEquation eqn, Boolean down)
{
  NoteTag longest = ShortestNote;
  NoteTag curr, next, prev;
  int     x;
  int     dir;
  int     i;

  Begin("DrawBeam");

  dir = down ? -1 : 1;

  for (i = 0; i < beampointcount; ++i)
    if (beampoints[i].tag > longest) longest = beampoints[i].tag;

  for (x = 0; x < Crotchet - longest; ++ x)
    XDrawLine(display, drawable, beamGC,
	      beampoints[0].x, beampoints[0].y + dir*x*5,
	      beampoints[beampointcount-1].x,
	      beampoints[beampointcount-1].y + dir*x*5);

  for (i = 0; i < beampointcount; ++i)
    if (beampoints[i].tag < longest) {

      curr = beampoints[i].tag;

      if (i > 0)                prev = beampoints[i-1].tag;
      if (i < beampointcount-1) next = beampoints[i+1].tag;

      if (i < beampointcount-1 && curr <= next) {

	if (i == 0 || curr < prev)
	  for (x = curr; x < next; ++x)
	    XDrawLine(display, drawable, beamGC, beampoints[i].x,
		      beampoints[i].y + dir*5*(Quaver-x),
		      beampoints[i].x + beampoints[i].width*2/3,
		      EQN(eqn, beampoints[i].x + beampoints[i].width*2/3) +
		      dir*5*(Quaver-x));

	for (x = next; x < longest; ++x)
	  XDrawLine(display, drawable, beamGC, beampoints[i].x,
		    beampoints[i].y + dir*5*(Quaver-x),
		    beampoints[i+1].x, beampoints[i+1].y + dir*5*(Quaver-x));

      } else if (i < beampointcount-1) {

	for (x = curr; x < longest; ++x)
	  XDrawLine(display, drawable, beamGC, beampoints[i].x,
		    beampoints[i].y + dir*5*(Quaver-x),
		    beampoints[i+1].x, beampoints[i+1].y + dir*5*(Quaver-x));

      } else if (i > 0 && curr < prev) {
	
	XDrawLine(display, drawable, beamGC, beampoints[i].x,
		  beampoints[i].y + dir*5*(Quaver - curr),
		  beampoints[i].x - beampoints[i].width*2/3,
		  EQN(eqn, beampoints[i].x - beampoints[i].width*2/3) +
		  dir*5*(Quaver - curr));
      }
    }

  End;
}


void DrawTuplingLine(Drawable drawable, LineEquation eqn, Boolean down)
{
  int         i;
  MTime       curr;
  MTime       length;
  MTime       shortest;
  int         number;
  char        a[5];
  int         dir, asc, desc;
  XID         fid;
  Position    mid;
  XGCValues   values;
  XCharStruct info;

  Begin("DrawTuplingLine");

  for (i = 0; i < beampointcount; ++i) {
    
    curr = beampoints[i].chord->item.methods->get_length
      ((MusicObject)beampoints[i].chord);

    length = AddMTime(length, curr);
    if (MTimeLesser(curr, shortest)) shortest = curr;
  }
    
  number = MTimeToNumber(length)/MTimeToNumber(shortest);
  sprintf(a, "%d", number);

  if (XGetGCValues(display, tinyTextGC, GCFont, &values) == 0)
    Error("Could not get tupling line graphics-context values");

  fid = values.font;
  XQueryTextExtents(display, fid, a, strlen(a), &dir, &asc, &desc, &info);

  beampoints[beampointcount-1].x += 5;

  mid = (beampoints[0].x + beampoints[beampointcount-1].x)/2;

  XDrawLine(display, drawable, drawingGC, 
	    beampoints[0].x, beampoints[0].y + (down? -1:1)*5,
	    beampoints[0].x, beampoints[0].y);

  XDrawLine(display, drawable, drawingGC, beampoints[0].x, beampoints[0].y,
	    mid - info.width/2 - 3, EQN(eqn, mid - info.width/2 - 3));

  XDrawLine(display, drawable, drawingGC, mid + info.width/2 + 3,
	    EQN(eqn, mid + info.width/2 + 3),
	    beampoints[beampointcount-1].x, beampoints[beampointcount-1].y);

  XDrawLine(display, drawable, drawingGC, 
	    beampoints[beampointcount-1].x,
	    beampoints[beampointcount-1].y + (down? -1:1)*5,
	    beampoints[beampointcount-1].x,
	    beampoints[beampointcount-1].y);

  XDrawString(display, drawable, tinyTextGC, mid - info.width/2,
	      EQN(eqn, mid - info.width/2) + desc, a, strlen(a));

  End;
}


Dimension DrawUndecoratedGroup(MusicObject obj, Drawable drawable, Position x,
			       Position y, Pitch offset, Dimension width,
			       LineEquation eqn)
{
  Group       * group = (Group *)obj;
  ItemList      list;
  MTime         length;
  MTime         shortestLength;
  MTime         totalLength;
  static MTime  crotchetTime = zeroTime;
  NoteTag       shortestTag;
  unsigned int  shortnum;
  Dimension     activeWidth = width;
  Dimension     shortestWidth;
  Dimension     itemWidth;
  Chord       * shortest;
  Item        * item;
  int           px, cx;

  Begin("DrawUndecoratedGroup");

  if (!obj) Return(width);
  if (MTimeEqual(crotchetTime, zeroTime))
    (void)NewMTime(&crotchetTime, Crotchet, 1);

  shortestLength = ((Chord *)(GetLongestChord(NULL)))->chord.length;
  totalLength = zeroTime;

  for (ItemList_ITERATE_GROUP(list, group)) {

    /* Get the data */

    item     = list->item;
    shortest = (Chord *)(item->item.methods->get_shortest((MusicObject)item));
    length   = shortest->chord.length;

    /* Do calculations involving length */

    if (MTimeLesser(length, shortestLength)) {
      shortestLength = length;
      shortestTag    = shortest->chord.visual->type;
    }

    length = item->item.methods->get_length((MusicObject)item);
    totalLength = AddMTime(length, totalLength);

    if (MTimeEqual(length, zeroTime))
      activeWidth -=
	item->item.methods->get_min_width((MusicObject)item);
  }

  /* Compute with the acquired length data. */

  length = totalLength;

  if (MTimeEqual(length, zeroTime)) shortestWidth = NoteWidth + 2;
  else shortestWidth =
    (Dimension)((long)activeWidth*(shortnum = MTimeToNumber(shortestLength)) /
		MTimeToNumber(length));

  /* Now draw the notes. */

  cx = x;

  for (ItemList_ITERATE_GROUP(list, group)) {

    px = cx;
    item = list->item;
    
    length = item->item.methods->get_length((MusicObject)item);

    if (MTimeEqual(length, zeroTime)) {

      if (item->generic.object_class == TextClass &&
	  ((Text *)item)->text.position == TextAboveBarLine) {

	cx += item->item.methods->draw
	  ((MusicObject)item, drawable, x - 15, y, offset, 0, eqn);

      } else {

	if (item->generic.object_class == ClefClass)
	  offset = ClefPitchOffset(((Clef *)item)->clef.clef);

	cx += item->item.methods->draw
	  ((MusicObject)item, drawable, cx, y, offset, 0, eqn);
      }
    } else {

      itemWidth = shortestWidth * MTimeToNumber(length) / shortnum;

      if (item->generic.object_class == ChordClass &&
	  !MTimeLesser(length, crotchetTime)) {

	if (itemWidth > NoteWidth +
	    ((Chord *)item)->chord.note_modifier_width + 4) {

	  cx        += (itemWidth > 20 ? 4 : (itemWidth / 5));
	  itemWidth -= cx - px;
	  px         = cx;
	}
      }

      if (itemWidth == 0) itemWidth = 1;

      (void)item->item.methods->draw
	((MusicObject)item, drawable, cx, y, offset, itemWidth, eqn);

      cx += itemWidth;
    }
  }

  if (width > 0) Return(width);
  else Return((Dimension)(cx - x));
}


/* We have the following problems in drawing a beamed group:     */
/*                                                               */
/*   == choosing whether to arrange the beam above or below;     */
/*   == finding a suitable gradient for the beam;                */
/*   == choosing the height of the beam;                         */
/*   == drawing the beam, with the correct number of "tails"     */
/*      for each note in the group.                              */

Dimension DrawGroup(MusicObject obj, Drawable drawable, Position x,
		    Position y, Pitch offset, Dimension width,
		    LineEquation eqn)
{
  Group       * group = (Group *)obj;
  ItemList      list;
  MTime         length;
  MTime         shortestLength;
  MTime         lastLength;
  static MTime  crotchetTime = zeroTime;
  NoteTag       shortestTag;
  unsigned int  shortnum;
  Dimension     activeWidth = width;
  Dimension     shortestWidth;
  Dimension     itemWidth;
  int           tailedNoteCount = 0;
  Chord       * shortest;
  Item        * item;
  Pitch         thispitch[2];	/* Of these, the first element */
  Pitch         prevpitch[2];	/* corresponds to the top line */
  Pitch         firstpitch[2];	/* of notes, & the second to   */
  Pitch         lastpitch[2];	/* the bottom line.            */
  Pitch         minpitch;
  Pitch         maxpitch;
  int           beamAbove = 0;
  int           beamBelow = 0;
  int           down;
  int           direction[2];
  static double grads[] = { (double)(0.1), (double)(0.17), (double)(0.3) };
  double        grad;
  int           nearestY;
  int           nearestX;
  int           px, cx;
  int           i;

  Begin("DrawGroup");

  if (!obj) Return(width);
  if (MTimeEqual(crotchetTime, zeroTime))
    (void)NewMTime(&crotchetTime, Crotchet, 1);

  group->item.x = x;

  if (group->group.type != GroupDeGrace &&
      group->group.start == group->group.end)
    Return(group->group.start->item->item.methods->draw
	   ((MusicObject)(group->group.start->item),
	    drawable, x, y, offset, width, eqn));

  if (group->group.type == GroupNoDecoration)
    Return(DrawUndecoratedGroup(obj, drawable, x, y, offset, width, eqn));

  /* 1. Zip through the group's objects.  Determine whether */
  /*    the note-row is non-increasing, non-decreasing or   */
  /*    neither, in the highest and lowest row of notes.    */
  /*    Determine whether there are more of the outside     */
  /*    notes of chords above or below the middle line, to  */
  /*    decide where to put the beam; also find shortest    */
  /*    note overall, and active width of the group (ie.    */
  /*    total width, minus widths of all zero-length elts). */

  shortestLength = ((Chord *)(GetLongestChord(NULL)))->chord.length;
  direction[0]   = direction[1] = -2;
  prevpitch[0]   = prevpitch[1] = -100;

  minpitch =  16;
  maxpitch = -16;

  for (ItemList_ITERATE_GROUP(list, group)) {

    /* Get the data */

    item     = list->item;
    shortest = (Chord *)(item->item.methods->get_shortest((MusicObject)item));
    length   = shortest->chord.length;

    if (tailedNoteCount < 2 && shortest->chord.visual->type < Crotchet &&
	item->generic.object_class != RestClass) ++tailedNoteCount;

    /* Do calculations involving length */

    if (MTimeLesser(length, shortestLength)) {
      shortestLength = length;
      shortestTag    = shortest->chord.visual->type;
    }

    length = list->item->item.methods->get_length((MusicObject)(list->item));

    if (MTimeEqual(length, zeroTime))
      activeWidth -=
	list->item->item.methods->get_min_width((MusicObject)(list->item));

    lastLength = length;

    /* Now consider the pitch */

    if (item->generic.object_class == ChordClass) {

      thispitch[0] =
	item->item.methods->get_highest((MusicObject)item)->pitch + offset;
      thispitch[1] =
	item->item.methods->get_lowest ((MusicObject)item)->pitch + offset;

      if (thispitch[1] < minpitch) minpitch = thispitch[1];
      if (thispitch[0] > maxpitch) maxpitch = thispitch[0];

      for (i = 0; i < 2; ++i) {

	if (prevpitch[i] == -100) prevpitch[i] = firstpitch[i] = thispitch[i];
	lastpitch[i] = thispitch[i];

	if (thispitch[i] < 5) beamAbove += (5-thispitch[i]);
	else                  beamBelow += (thispitch[i]-4);

	if      (thispitch[i] < prevpitch[i])
	  if (direction[i] ==  1 || direction[i] == 0) direction[i] =  0;
	  else                                         direction[i] = -1;
	else if (thispitch[i] > prevpitch[i])
	  if (direction[i] == -1 || direction[i] == 0) direction[i] =  0;
	  else                                         direction[i] =  1;

	prevpitch[i] = thispitch[i];
      }
    }
  }

  /* 2. Compute with the acquired length data. */

  if (group->group.type == GroupTupled) {

    group->group.type = GroupNoDecoration;
    length = group->item.methods->get_length(obj);
    group->group.type = GroupTupled;

  } else if (group->group.type == GroupDeGrace) {

    group->group.type = GroupNoDecoration;
    length = group->item.methods->get_length(obj);
    group->group.type = GroupDeGrace;

  } else length = group->item.methods->get_length(obj);

  if (MTimeEqual(length, zeroTime)) shortestWidth = NoteWidth + 2;
  else shortestWidth =
    (Dimension)((long)activeWidth*(shortnum = MTimeToNumber(shortestLength)) /
		MTimeToNumber(length));

  /* 3. Compute with the acquired pitch data -- get above or    */
  /*    below, and gradient; have a guess at a suitable offset. */

  if (group->group.type == GroupBeamed ||
      group->group.type == GroupTupled) {

    down = beamAbove > beamBelow ? 0 : 1;
    down = group->group.type == GroupTupled ? 1-down : down;

    if (direction[down] == 0 || direction[down] == -2) {
      if      (firstpitch[down] -  lastpitch[down] > 2) grad =  grads[0];
      else if ( lastpitch[down] - firstpitch[down] > 2) grad = -grads[0];
      else grad = (double)0;

    } else {			                 /* some overall direction */
      if      (firstpitch[down] -  lastpitch[down] > 4) grad =  grads[2];
      else if ( lastpitch[down] - firstpitch[down] > 4) grad = -grads[2];
      else if (firstpitch[down] -  lastpitch[down] > 3) grad =  grads[1];
      else if ( lastpitch[down] - firstpitch[down] > 3) grad = -grads[1];
      else grad = direction[down] > 0 ? -grads[0] : grads[0];
    }

    /* Offset should be such that the point on the line nearest   */
    /* to the note group should be at least NoteHeight away       */
    /* from the note nearest the beam.                            */

    if (down) {

      nearestY = y + STAVE_Y_COORD(minpitch) + NoteHeight*2;
      if (grad < (grads[1] - 0.01) && grad > (-grads[1] + 0.01))
	nearestY += NoteHeight;
      if (grad < (grads[0] - 0.01) && grad > (-grads[0] + 0.01))
	nearestY += NoteHeight/2;

      /* I think this computation of nearestX is at fault; I think it */
      /* should obtain the x coord of the lowest note head (similarly */
      /* in the other conditional branch, with the highest note head) */

      if (direction[down] > 0) nearestX =
	x + width -
	  (shortestWidth * MTimeToNumber(lastLength)/shortnum) - NoteWidth/2;
      else nearestX = x;

      if (shortestTag < Quaver) nearestY += 3*(Quaver-shortestTag);

    } else {

      nearestY = y + STAVE_Y_COORD(maxpitch) - NoteHeight;
      if (grad < (grads[1] - 0.01) && grad > (-grads[1] + 0.01))
	nearestY -= NoteHeight;
      if (grad < (grads[0] - 0.01) && grad > (-grads[0] + 0.01))
	nearestY -= NoteHeight/2;

      if (direction[down] > 0) nearestX = x;
      else nearestX =
	x + width - (shortestWidth * MTimeToNumber(lastLength) / shortnum);

      if (shortestTag < Quaver) nearestY -= 3*(Quaver-shortestTag);
    }

    /* want "nearestY = m * nearestX + c"; we have m already */

    eqn = (LineEquation)XtMalloc(sizeof(LineEquationRec));
    eqn->m = grad;
    eqn->c = nearestY - (int)(grad * nearestX);
    eqn->eqn_present = True;
    eqn->reverse = False;
  }

  /* 3. Now draw the notes, and the beams. */

  beampointcount = 0;
  cx = x;

  for (ItemList_ITERATE_GROUP(list, group)) {
    
    px = cx;
    
    length = list->item->item.methods->get_length((MusicObject)(list->item));

    if (MTimeEqual(length, zeroTime)) {

      if (item->generic.object_class == ClefClass)
	offset = ClefPitchOffset(((Clef *)item)->clef.clef);

      cx += list->item->item.methods->draw
	((MusicObject)(list->item), drawable, cx, y, offset, 0, eqn);

    } else {

      itemWidth = shortestWidth * MTimeToNumber(length) / shortnum;

      if (list->item->generic.object_class == ChordClass) {
	
	if (!(MTimeLesser(length, crotchetTime))) {

	  if (itemWidth > NoteWidth +
	      ((Chord *)list->item)->chord.note_modifier_width + 4) {

	    cx        += (itemWidth > 20 ? 4 : (itemWidth / 5));
	    itemWidth -= cx - px;
	    px         = cx;
	  }

	  if (eqn && group->group.type == GroupBeamed &&
	      list != group->group.start && list != group->group.end)
	    eqn->reverse = True;

	} else {

	  if (group->group.type != GroupBeamed) itemWidth += 3;
	}
      }

      if (itemWidth == 0) itemWidth = 1;

      (void)list->item->item.methods->draw
	((MusicObject)(list->item), drawable, cx, y, offset, itemWidth,
	 group->group.type == GroupBeamed && tailedNoteCount > 1 ? eqn : NULL);

      cx += itemWidth;

      if (eqn && (group->group.type == GroupTupled ||
		  list->item->generic.object_class == ChordClass)) {

	if (group->group.type == GroupBeamed &&
	    ((Chord *)list->item)->chord.visual->type < Crotchet) {

	  if (down) beampoints[beampointcount].x = 
	    px + ((Chord *)(list->item))->chord.note_modifier_width;
	  else      beampoints[beampointcount].x =
	    px+NoteWidth-2+((Chord *)(list->item))->chord.note_modifier_width;

	  beampoints[beampointcount].y = EQN(eqn,beampoints[beampointcount].x);
	  beampoints[beampointcount].chord = (Chord *)(list->item);
	  beampoints[beampointcount].tag =
	    ((Chord *)(list->item))->chord.visual->type;
	  beampoints[beampointcount].width =
	    (Dimension)(shortestWidth * MTimeToNumber(length) / shortnum);

	  if (beampointcount < BeamPointMax) beampointcount ++;
	}
      }

      if (eqn) eqn->reverse = False;
    }
  }

  if (beampointcount > 0) {

    if (group->group.type == GroupBeamed)
      DrawBeam(drawable, eqn, down);
    else if (group->group.type == GroupTupled)
      DrawTuplingLine(drawable, eqn, down);
  }

  if (group->group.type == GroupBeamed ||
      group->group.type == GroupTupled) XtFree((void *)eqn);

  if (width > 0) Return(width);
  else Return((Dimension)(cx - x));
}



Dimension DrawBar(MusicObject obj, Drawable drawable, Position x,
		  Position y, Pitch offset, Dimension width, LineEquation eqn)
{
  Position  sy;
  Bar     * bar = (Bar *)obj;
  Dimension actwid;
  Position  off;
  BarTag    start, end;
  XPoint    tieControl;
  XPoint    tieStart;
  XPoint    tieEnd;
  int       i, j;

  Begin("DrawBar");

  if (!bar) Return(width);

  actwid = width;
  off    = x;
  start  = bar->bar.start_bar;
  end    = bar->bar.end_bar;
  bar->item.x = x;

  if (tie[1].present) tie[0].above = tie[1].above;
  tie[0].present = tie[1].present = False;

  /* If required, draw the opening bar line.  We only need this if it's  */
  /* a double or opening-repeat bar (and then it goes after the clef and */
  /* key sig), or if this is the first bar in the line (in which case    */
  /* see above); otherwise, the ending line on the previous bar will do. */

  switch(start) {

  case SingleBar:
    break;

  case RepeatBar:

    XDrawLine(display, drawable, drawingGC, off+1, y, off+1, y+StaveHeight-1);
    XDrawLine(display, drawable, drawingGC, off+2, y, off+2, y+StaveHeight-1);
    XDrawLine(display, drawable, drawingGC, off+5, y, off+5, y+StaveHeight-1);

    CopyArea(noteDotMap, drawable, 0, 0,
	     DotWidth, NoteHeight, off + 6, y + STAVE_Y_COORD(3));
    CopyArea(noteDotMap, drawable, 0, 0,
	     DotWidth, NoteHeight, off + 6, y + STAVE_Y_COORD(5));

    off += 6 + DotWidth;
    break;

  case DoubleBar:
    
    XDrawLine(display, drawable, drawingGC, off+1, y, off+1, y+StaveHeight-1);
    XDrawLine(display, drawable, drawingGC, off+2, y, off+2, y+StaveHeight-1);
    XDrawLine(display, drawable, drawingGC, off+5, y, off+5, y+StaveHeight-1);

    off += 6;
    break;
  }

  if (!(bar->group.start &&
	bar->group.start->item->generic.object_class == ClefClass)) off += 5;
  actwid -= off-x;

  /* Draw in the closing bar line */
    
  switch(end) {
      
  case SingleBar:

    XDrawLine(display, drawable, drawingGC,
	      x + width-1, y, x + width-1, y + StaveHeight - 1);
    actwid -= 3;
    break;

  case RepeatBar:

    CopyArea(noteDotMap, drawable, 0, 0, DotWidth, NoteHeight,
	     x + width - 5 - DotWidth, y + STAVE_Y_COORD(3));
    CopyArea(noteDotMap, drawable, 0, 0, DotWidth, NoteHeight,
	     x + width - 5 - DotWidth, y + STAVE_Y_COORD(5));

    actwid -= DotWidth;

    /* fall through */

  case DoubleBar:
    
    XDrawLine(display, drawable, drawingGC,
	      x + width-1, y, x + width-1, y + StaveHeight - 1);
    XDrawLine(display, drawable, drawingGC,
	      x + width-2, y, x + width-2, y + StaveHeight - 1);
    XDrawLine(display, drawable, drawingGC,
	      x + width-5, y, x + width-5, y + StaveHeight - 1);

    actwid -= 7;
  }

  /* Now fill in the contents of the bar */

  DrawUndecoratedGroup
    (obj, drawable, off + 2, y, offset, actwid>3 ? actwid-3 : 2, NULL);

  bar->bar.still_as_drawn = True;

  /* Draw the stave lines */
    
  for (sy = 0 ; sy < StaveHeight; sy += NoteHeight + 1)
    XDrawLine(display, drawable, drawingGC, x, y + sy, x+width-1, y + sy);

  /* And finally, draw the ties if they're called for. */

  for (i = 0; i < 2; ++i) {

    if ((i ? bar->phrase.tied_forward : bar->phrase.tied_backward) &&
	tie[i].present) {

      tieStart.x = i ? tie[i].x + 10 : tie[i].x - 10;
      tieEnd.x   = tieControl.x = tie[i].x;

      tieStart.y = tieControl.y =
	tie[i].above ?  y - TIE_DIST - 4 :
	  y + StaveHeight + TIE_DIST + 4;

      tieEnd.y = tieStart.y + (tie[i].above ? 4 : -4);
      
      XDrawLine  (display, drawable, drawingGC,
		  i ? x + width : x, tieStart.y, tieStart.x, tieStart.y);

      DrawSpline (display, drawable, drawingGC,
		  &tieControl, tieStart, tieEnd, 1, True);
    }
  }

  Return(width);
}

