
/*
   Musical Notation Editor for X, Chris Cannam 1994

   Menu handling
*/


#include "General.h"
#include "Palette.h"
#include "IO.h"
#include "Widgets.h"
#include "Visuals.h"
#include "Menu.h"

#include "text_above.xbm"
#include "text_above_large.xbm"
#include "text_below.xbm"
#include "text_below_italic.xbm"

#include <Yawn.h>
#include <YSmeBSB.h>
#include <ILClient.h>

#define EditMenuUndo Unimplemented
#define EditMenuChangeNote Unimplemented
#define StaveMenuFillToEnd Unimplemented
#define StaveMenuConnectAbove Unimplemented
#define StaveMenuConnectBelow Unimplemented
#define StaveMenuDisconnect Unimplemented
#define GroupMenuGrace Unimplemented
#define GroupMenuTuplet Unimplemented
#define GroupMenuSlur Unimplemented
#define GroupMenuCrescendo Unimplemented
#define GroupMenuDecrescendo Unimplemented
#define GroupMenuRemove Unimplemented
#define ToolsMenuPreferences Unimplemented
#define BarMenuAddFixed Unimplemented
#define BarMenuRemoveFixed Unimplemented


static YMenuElement editMenu[] = {
  { "Undo",           FileLoadedMode | FileNotLoadedMode,  /* ie., never */
                                               EditMenuUndo,          NULL, },
  YMenuDivider,
  { "Delete",         NoAreaSweptMode,         EditMenuDelete,        NULL, },
  YMenuDivider,
  { "Cut",            NoAreaSweptMode,         EditMenuCut,           NULL, },
  { "Copy",           NoAreaSweptMode,         EditMenuCopy,          NULL, },
  { "Paste",          CursorNotPlacedMode,     EditMenuPaste,         NULL, },
  { "Show Clipboard", NullMode,                EditMenuShowClipboard, NULL, },
  YMenuDivider,
  { "Change Chord",   NoAreaSweptMode | MultipleItemsSweptMode,
                                               EditMenuChangeNote,    NULL, },
};

static YMenuId editMenuId = NULL;


static YMenuElement staveMenu[] = {
  { "Add Staff",      FileNotLoadedMode,       StaveMenuNewStave,     NULL, },
  { "Delete Staff",   CursorNotPlacedMode,     StaveMenuDeleteStave,  NULL, },
  { "Empty Staff",    CursorNotPlacedMode,     StaveMenuEmptyStave,   NULL, },
  { "Rename Staff",   CursorNotPlacedMode,     StaveMenuRenameStave,  NULL, },
  YMenuDivider,
  { "Connect Above",  CursorNotPlacedMode,     StaveMenuConnectAbove, NULL, },
  { "Connect Below",  CursorNotPlacedMode,     StaveMenuConnectBelow, NULL, },
  { "Disconnect",     CursorNotPlacedMode,     StaveMenuDisconnect,   NULL, },
  YMenuDivider,
  { "Fill To End",    CursorNotPlacedMode,     StaveMenuFillToEnd,    NULL, },
};

static YMenuId staveMenuId = NULL;


static YMenuElement markMenu[] = {
  { "Accent",         NoAreaSweptMode,         MarkMenuMark,          NULL, },
  { "Staccato",       NoAreaSweptMode,         MarkMenuMark,          NULL, },
  { "Legato",         NoAreaSweptMode,         MarkMenuMark,          NULL, },
  { "Sforzando",      NoAreaSweptMode,         MarkMenuMark,          NULL, },
  { "Rinforzando",    NoAreaSweptMode,         MarkMenuMark,          NULL, },
  { "Trill",          NoAreaSweptMode,         MarkMenuMark,          NULL, },
  { "Turn",           NoAreaSweptMode,         MarkMenuMark,          NULL, },
  { "Pause",          NoAreaSweptMode,         MarkMenuMark,          NULL, },
  YMenuDivider,
  { "Unmark",         NoAreaSweptMode,         MarkMenuUnmark,        NULL, },
};

static YMenuId markMenuId = NULL;


static YMenuElement groupMenu[] = {
  { "Chord",          NoAreaSweptMode,         GroupMenuChord,        NULL, },
  { "Unchord",        NoAreaSweptMode,         GroupMenuUnchord,      NULL, },
  YMenuDivider,
  { "Slur",           NoAreaSweptMode,         GroupMenuSlur,         NULL, },
  { "Crescendo",      NoAreaSweptMode,         GroupMenuCrescendo,    NULL, },
  { "Decrescendo",    NoAreaSweptMode,         GroupMenuDecrescendo,  NULL, },
  { "Remove Indication", NoAreaSweptMode,      GroupMenuRemove,       NULL, },
  YMenuDivider,
  { "Beam",           NoAreaSweptMode,         GroupMenuBeam,         NULL, },
  { "Grace",          NoAreaSweptMode,         GroupMenuGrace,        NULL, },
  { "Tuplet",         NoAreaSweptMode,         GroupMenuTuplet,       NULL, },
  { "Break Group",    NoAreaSweptMode,         GroupMenuBreakGroup,   NULL, },
  YMenuDivider,
  { "Auto-Beam",      FileNotLoadedMode,       GroupMenuAutoBeam,     NULL, },
  { "Transpose",      FileNotLoadedMode,       GroupMenuTranspose,    NULL, },
  /* JPff insertion */
  { "Retrograde",     FileNotLoadedMode,       GroupMenuRetrograde,   NULL, },
  { "Invert",         FileNotLoadedMode,       GroupMenuInvert,       NULL, },
};

static YMenuId groupMenuId = NULL;


static YMenuElement textMenu[] = {
  { "Text1",          CursorNotPlacedMode,     TextMenuTextAbove,     NULL, },
  { "Text2",          CursorNotPlacedMode,     TextMenuTextAboveBig,  NULL, },
  { "Text3",          CursorNotPlacedMode,     TextMenuTextBelow,     NULL, },
  { "Text4",          CursorNotPlacedMode,     TextMenuTextBelowI,    NULL, },
  YMenuDivider,
  { "Clear Group",    NoAreaSweptMode,         TextMenuUnlabelGroup,  NULL, },
  { "Clear Bar",      CursorNotPlacedMode,     TextMenuUnlabelBar,    NULL, },
  { "Clear Staff",    CursorNotPlacedMode,     TextMenuUnlabelStaff,  NULL, },
};

static YMenuId textMenuId = NULL;


static YMenuElement toolsMenu[] = {
  { "Import MIDI",    NullMode,                ToolsMenuImportMidi,   NULL, },
  { "Export MIDI",    FileNotLoadedMode,       ToolsMenuWriteMidi,    NULL, },
  { "Play MIDI",      FileNotLoadedMode | SequencerRunningMode,
                                               ToolsMenuPlayMidi,     NULL, },
  { "Sequence",       FileNotLoadedMode | SequencerRunningMode,
                                               ToolsMenuSequence,     NULL, },
  YMenuDivider,
  { "Export MusicTeX",FileNotLoadedMode | Midi2TeXUnavailableMode,
                                               ToolsMenuExportTeX,    NULL, },
  YMenuDivider,
  { "Preferences",    NullMode,                ToolsMenuPreferences,  NULL, },
};

static YMenuId toolsMenuId = NULL;


static YMenuElement barMenu[] = {
  { "Plain",          CursorNotPlacedMode,     BarMenuPlain,          NULL, },
  { "Double Bar",     CursorNotPlacedMode,     BarMenuDoubleBar,      NULL, },
  { "Repeat at Start", CursorNotPlacedMode,    BarMenuRepeatAtStart,  NULL, },
  { "Repeat at End",  CursorNotPlacedMode,     BarMenuRepeatAtEnd,    NULL, },
  YMenuDivider,
  { "Label",          CursorNotPlacedMode,     BarMenuLabel,          NULL, },
  { "Remove Label",   CursorNotPlacedMode,     BarMenuUnlabel,        NULL, },
  YMenuDivider,
  { "Metronome",      CursorNotPlacedMode,     BarMenuMetronome,      NULL, },
  { "Remove Metronome", CursorNotPlacedMode,   BarMenuRemoveMet,      NULL, },
  YMenuDivider,
  { "Time Signature", CursorNotPlacedMode,     BarMenuTimeSignature,  NULL, },
  { "Remove Time Signature", CursorNotPlacedMode, BarMenuRemoveTime,  NULL, },
  YMenuDivider,
  { "Add Fixed Barline", CursorNotPlacedMode,  BarMenuAddFixed,       NULL, },
  { "Remove Fixed Barline", CursorNotPlacedMode, BarMenuRemoveFixed,  NULL, },
};

static YMenuId barMenuId = NULL;


static YMenuElement paletteMenu[] = {
  { "Notes",          NullMode,                PaletteMenuNotes,      NULL, },
  { "Rests",          NullMode,                PaletteMenuRests,      NULL, },
  { "Clefs",          NullMode,                PaletteMenuClefs,      NULL, },
  { "Keys",           NullMode,                PaletteMenuKeys,       NULL, },
};

static YMenuId paletteMenuId = NULL;



void IssueMenuComplaint(String s)
{
  Begin("IssueMenuComplaint");

  XBell(display, 70);
  (void)YQuery(topLevel, s, 1, 0, 0, "OK", NULL);

  End;
}


void Unimplemented(Widget w, XtPointer a, XtPointer b)
{
  Begin("Unimplemented");
  (void)YQuery(topLevel, "Sorry, that function is not yet implemented.",
		1, 0, 0, "Continue", NULL);
  End;
}


void InstallEditMenu(Widget edit)
{
  Begin("InstallEditMenu");

  editMenuId = YCreateMenu(edit, "Edit Menu", XtNumber(editMenu), editMenu);

  End;
}


void InstallStaveMenu(Widget stavew)
{
  Begin("InstallStaveMenu");

  staveMenuId =
    YCreateMenu(stavew, "Stave Menu", XtNumber(staveMenu), staveMenu);

  End;
}


void InstallGroupMenu(Widget group)
{
  Begin("InstallGroupMenu");

  groupMenuId =
    YCreateMenu(group, "Group Menu", XtNumber(groupMenu), groupMenu);

  End;
}


void InstallMarkMenu(Widget mark)
{
  int    i, j;
  Pixmap map;

  Begin("InstallMarkMenu");

  markMenuId =
    YCreateMenu(mark, "Mark Menu", XtNumber(markMenu), markMenu);

  for (i = 0; i < XtNumber(markMenu) - 2; ++i) {

    YSetValue(markMenu[i].widget, XtNleftMargin, NoteWidth + 12);

    for (j = 0; j < chordModVisualCount; ++j)
      if (!strncasecmp(chordModVisuals[j].name, markMenu[i].label,
		       strlen(markMenu[i].label))) break;

    if (j < chordModVisualCount) {

      map = XCreateBitmapFromData
	(display, RootWindowOfScreen(XtScreen(topLevel)),
	 chordModVisuals[j].bitmap, NoteWidth, ChordModHeight);

      YSetValue(markMenu[i].widget, XtNleftBitmap, map);
    }
  }

  End;
}
  


void InstallTextMenu(Widget text)
{
  int    i;
  Pixmap map;

  Begin("InstallTextMenu");

  textMenuId =
    YCreateMenu(text, "Text Menu", XtNumber(textMenu), textMenu);

  for (i = 0;
       i < XtNumber(textMenu) &&
       (textMenu[i].label ? strncmp(textMenu[i].label,"Text",4) : True); ++i);

  if (i < XtNumber(textMenu)) {

    map = XCreateBitmapFromData
      (display, RootWindowOfScreen(XtScreen(text)), text_above_bits,
       text_above_width, text_above_height);
    YSetValue(textMenu[i].widget, XtNleftMargin, text_above_width + 9);
    YSetValue(textMenu[i].widget, XtNleftBitmap, map); i++;

    map = XCreateBitmapFromData
      (display, RootWindowOfScreen(XtScreen(text)), text_above_large_bits,
       text_above_large_width, text_above_large_height);
    YSetValue(textMenu[i].widget, XtNleftMargin, text_above_large_width + 9);
    YSetValue(textMenu[i].widget, XtNleftBitmap, map); i++;

    map = XCreateBitmapFromData
      (display, RootWindowOfScreen(XtScreen(text)), text_below_bits,
       text_below_width, text_below_height);
    YSetValue(textMenu[i].widget, XtNleftMargin, text_below_width + 9);
    YSetValue(textMenu[i].widget, XtNleftBitmap, map); i++;

    map = XCreateBitmapFromData
      (display, RootWindowOfScreen(XtScreen(text)), text_below_italic_bits,
       text_below_italic_width, text_below_italic_height);
    YSetValue(textMenu[i].widget, XtNleftMargin, text_below_italic_width + 9);
    YSetValue(textMenu[i].widget, XtNleftBitmap, map);
  }

  End;
}


void InstallBarMenu(Widget bar)
{
  Begin("InstallBarMenu");

  barMenuId =
    YCreateMenu(bar, "Bar Menu", XtNumber(barMenu), barMenu);

  End;
}


void InstallToolsMenu(Widget tools)
{
  Begin("InstallToolsMenu");

  toolsMenuId =
    YCreateMenu(tools, "Tools Menu", XtNumber(toolsMenu), toolsMenu);

  End;
}


void InstallPaletteMenu(Widget palette)
{
  Begin("InstallPaletteMenu");

  paletteMenuId =
    YCreateMenu(palette, "Palette Menu", XtNumber(paletteMenu), paletteMenu);

  End;
}


void EnterMenuMode(MenuMode mode)
{
  Begin("EnterMenuMode");
  
  YEnterMenuMode(    fileMenuId, mode );
  YEnterMenuMode(    editMenuId, mode );
  YEnterMenuMode(   groupMenuId, mode );
  YEnterMenuMode(   staveMenuId, mode );
  YEnterMenuMode(     barMenuId, mode );
  YEnterMenuMode(    markMenuId, mode );
  YEnterMenuMode(    textMenuId, mode );
  YEnterMenuMode(   toolsMenuId, mode );
  YEnterMenuMode( paletteMenuId, mode );

  End;
}


void LeaveMenuMode(MenuMode mode)
{
  Begin("LeaveMenuMode");
  
  YLeaveMenuMode(    fileMenuId, mode );
  YLeaveMenuMode(    editMenuId, mode );
  YLeaveMenuMode(   groupMenuId, mode );
  YLeaveMenuMode(   staveMenuId, mode );
  YLeaveMenuMode(     barMenuId, mode );
  YLeaveMenuMode(    markMenuId, mode );
  YLeaveMenuMode(    textMenuId, mode );
  YLeaveMenuMode(   toolsMenuId, mode );
  YLeaveMenuMode( paletteMenuId, mode );

  End;
}


Boolean QueryMenuMode(MenuMode mode)
{
  Begin("QueryMenuMode");
  Return(YQueryMenuMode(fileMenuId, mode));
}


void MenuCleanUp(void)
{
  Begin("MenuCleanUp");

  FileMenuCleanUp();

  if (    editMenuId) YDestroyMenu(    editMenuId);
  if (   groupMenuId) YDestroyMenu(   groupMenuId);
  if (     barMenuId) YDestroyMenu(     barMenuId);
  if (   staveMenuId) YDestroyMenu(   staveMenuId);
  if (    textMenuId) YDestroyMenu(    textMenuId);
  if (    markMenuId) YDestroyMenu(    markMenuId);
  if (   toolsMenuId) YDestroyMenu(   toolsMenuId);
  if ( paletteMenuId) YDestroyMenu( paletteMenuId);

  End;
}

