
/* Musical Notation Editor for X, Chris Cannam 1994 */
/* Functions to handle actions from Words menu options */


#include "General.h"
#include "Widgets.h"
#include "Menu.h"
#include "Stave.h"
#include "StaveEdit.h"
#include "StavePrivate.h"
#include "ItemList.h"
#include "GC.h"
#include "Yawn.h"

#include <X11/Xaw/Box.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Repeater.h>

static Widget    metNoteDisplay;
static Dimension metNoteMapWidth;


static void TsNumUp(Widget w, XtPointer client, XtPointer call)
{
  int           i;
  String        s1;
  static String s = "12345";
  Widget        ww = (Widget)client;
  
  Begin("TsNumUp");
  YGetValue(ww, XtNlabel, &s1);

  i = atoi(s1);
  if (i < 99) ++i;
  sprintf(s, "%3d ", i);

  YSetValue(ww, XtNlabel, s);
  End;
}
 

static void TsNumDown(Widget w, XtPointer client, XtPointer call)
{
  int           i;
  String        s1;
  static String s = "12345";
  Widget        ww = (Widget)client;
  
  Begin("TsNumDown");
  YGetValue(ww, XtNlabel, &s1);

  i = atoi(s1);
  if (i > 1) --i;
  sprintf(s, "%3d ", i);

  YSetValue(ww, XtNlabel, s);
  End;
}

 
static void TsDenUp(Widget w, XtPointer client, XtPointer call)
{
  int           i;
  String        s1;
  static String s = "12345";
  Widget        ww = (Widget)client;
  
  Begin("TsDenUp");
  YGetValue(ww, XtNlabel, &s1);

  i = atoi(s1);
  if (i < 64) i *= 2;
  sprintf(s, "%3d ", i);

  YSetValue(ww, XtNlabel, s);
  End;
}

 
static void TsDenDown(Widget w, XtPointer client, XtPointer call)
{
  int           i;
  String        s1;
  static String s = "12345";
  Widget        ww = (Widget)client;
  
  Begin("TsDenDown");
  YGetValue(ww, XtNlabel, &s1);

  i = atoi(s1);
  if (i > 1) i /= 2;
  sprintf(s, "%3d ", i);

  YSetValue(ww, XtNlabel, s);
  End;
}


static void TsCallback(Widget w, XtPointer client, XtPointer call)
{
  Begin("TsCallback");

  *((Widget *)client) = w;

  End;
}
 


void BarMenuTimeSignature(Widget w, XtPointer a, XtPointer b)
{
  MajorStaveRec *mstave  = (MajorStaveRec *)stave;
  int            staveNo = mstave->sweep.stave;
  int            firstStave;
  int            lastStave;
  int            barNo;
  ItemList       list;
  ItemList       newList;
  StaveEltList   barList;
  int            i;
  String         temp;
  String         numl;
  String         denl;
  static int     num = 4;
  static int     den = 4;

  Widget         tsShell;
  Widget         tsPane;
  Widget         tsBox[3];
  Widget         tsNumUp;
  Widget         tsNumDown;
  Widget         tsNumDisplay;
  Widget         tsDenUp;
  Widget         tsDenDown;
  Widget         tsDenDisplay;
  Widget         tsApply;
  Widget         tsCancel;

  XPoint         op;
  XtAppContext   context;
  Widget         value = NULL;

  Begin("BarMenuTimeSignature");

  if (!stave || !mstave->sweep.swept ||
      mstave->sweep.from.left != mstave->sweep.to.left) {
    XBell(display, 70);
    End;
  }

  for (barNo = 0, barList = mstave->bar_list;
       barList && barNo < mstave->sweep.first_bar;
       barList = (StaveEltList)Next(barList), ++barNo);

  tsShell = XtCreatePopupShell("Time Signature", transientShellWidgetClass,
			       topLevel, NULL, 0);

  tsPane = YCreateWidget("Time Signature Pane", panedWidgetClass, tsShell);
  
  for (i = 0; i < 3; ++i)
    tsBox[i] = YCreateShadedWidget
      ("Time Signature Box", formWidgetClass, tsPane,
       i == 2 ? MediumShade : LightShade);

  tsNumDown = YCreateArrowButton("Numerator Down", tsBox[0], YArrowLeft);
  tsNumDisplay = YCreateLabel("   0", tsBox[0]);
  tsNumUp = YCreateArrowButton("Numerator Up", tsBox[0], YArrowRight);

  YSetValue(XtParent(tsNumDisplay), XtNfromHoriz, XtParent(tsNumDown));
  YSetValue(XtParent(tsNumUp),      XtNfromHoriz, XtParent(tsNumDisplay));

  tsDenDown = YCreateArrowButton("Denominator Down", tsBox[1], YArrowLeft);
  tsDenDisplay = YCreateLabel("   1", tsBox[1]);
  tsDenUp = YCreateArrowButton("Denominator Up", tsBox[1], YArrowRight);

  YSetValue(XtParent(tsDenDisplay), XtNfromHoriz, XtParent(tsDenDown));
  YSetValue(XtParent(tsDenUp),      XtNfromHoriz, XtParent(tsDenDisplay));

  tsApply  = YCreateCommand("OK",     tsBox[2]);
  tsCancel = YCreateCommand("Cancel", tsBox[2]);

  YSetValue(XtParent(tsCancel), XtNfromHoriz, XtParent(tsApply));
  YSetValue(XtParent(tsCancel), XtNfromHoriz, XtParent(tsApply));
  YSetValue(XtParent(tsCancel), XtNfromHoriz, XtParent(tsApply));

  XtAddCallback(tsNumUp,     XtNcallback, TsNumUp,    (XtPointer)tsNumDisplay);
  XtAddCallback(tsNumDown,   XtNcallback, TsNumDown,  (XtPointer)tsNumDisplay);
  XtAddCallback(tsDenUp,     XtNcallback, TsDenUp,    (XtPointer)tsDenDisplay);
  XtAddCallback(tsDenDown,   XtNcallback, TsDenDown,  (XtPointer)tsDenDisplay);
  XtAddCallback(tsApply,     XtNcallback, TsCallback, (XtPointer)&value);
  XtAddCallback(tsCancel,    XtNcallback, TsCallback, (XtPointer)&value);

  op = YPlacePopupAndWarp(tsShell, XtGrabExclusive, tsCancel, tsCancel);
  YAssertDialogueActions(tsShell, tsCancel, tsCancel, NULL);

  numl = (String)XtMalloc(5);
  sprintf(numl, "%3d ", num);
  YSetValue(tsNumDisplay, XtNlabel, numl);

  denl = (String)XtMalloc(5);
  sprintf(denl, "%3d ", den);
  YSetValue(tsDenDisplay, XtNlabel, denl);

  context = XtWidgetToApplicationContext(tsShell);
  while (!value || XtAppPending(context)) XtAppProcessEvent(context, XtIMAll);

  YGetValue(tsNumDisplay, XtNlabel, &temp);
  num = atoi(temp);

  YGetValue(tsDenDisplay, XtNlabel, &temp);
  den = atoi(temp);

  XtFree(numl);
  XtFree(denl);

  if (value != tsApply) {
    
    YPopPointerPosition();
    YPopdown(tsShell);
    XtDestroyWidget(tsShell);
    End;
  }

  firstStave = lastStave = staveNo;

  if (mstave->staves > 1 &&
      YQuery(topLevel, NULL, 2, 1, -1, "Change This Staff Only",
	     "Change Whole Stave", "Bar - Time Signature") == 1) {

    firstStave = 0;
    lastStave = mstave->staves - 1;
  }

  YPopPointerPosition();
  YPopdown(tsShell);
  YRetractDialogueActions(tsShell);
  XtDestroyWidget(tsShell);

  for (staveNo = firstStave; staveNo <= lastStave; ++staveNo) {

    if (!barList || !barList->bars[staveNo] ||
	!barList->bars[staveNo]->group.start) continue;

    newList = NewItemList((Item *)NewTimeSignature(NULL, num, den));
    list = barList->bars[staveNo]->group.start;

    while (list && list->item->generic.object_class == ClefClass ||
	   list->item->generic.object_class == KeyClass)
      list = (ItemList)Next(list);

    if (list) Insert(newList, list);
    else Nconc(barList->bars[staveNo]->group.start, newList);

    if (list && list->item->generic.object_class == TimeSignatureClass)
      Remove(list);

    if (list == barList->bars[staveNo]->group.start) {

      if (barList->bars[staveNo]->group.start == mstave->music[staveNo])
	mstave->music[staveNo] = barList->bars[staveNo]->group.start = newList;
      else barList->bars[staveNo]->group.start = newList;
    }

    mstave->formats[staveNo].next  = 0;
    mstave->formats[staveNo].time  = NULL;
    mstave->formats[staveNo].key   = NULL;
    mstave->formats[staveNo].clef  = NULL;
    mstave->formats[staveNo].items = mstave->music[staveNo];
  }

  FileMenuMarkChanged(stave, True);
  StaveRefreshAsDisplayed(stave);
  End;
}


static void MetNoteChange(Chord *c, Boolean up)
{
  int     i;
  Pixmap  map;
  NoteTag tag;
  Boolean dotted;
  static  XExposeEvent event =
    { Expose, 0L, True, 0, 0,
	0, 0, NoteWidth*5, StaveHeight + NoteHeight, 0 };

  Begin("MetNoteChange");

  tag    = c->chord.visual->type;
  dotted = c->chord.visual->dotted;

  if (up)
    if (dotted) {
      if (tag < LongestNote) { ++tag; dotted = False; }
    } else dotted = True;
  else
    if (dotted) dotted = False;
    else if (tag > ShortestNote) { --tag; dotted = True; }

  (void)NewChord(c, c->chord.voices, c->chord.voice_count,
		 c->chord.modifiers, tag, dotted);

  YGetValue(metNoteDisplay, XtNbitmap, &map);

  XFillRectangle(display, map, clearGC, 0, 0,
		 metNoteMapWidth, StaveHeight + NoteHeight);

  c->item.methods->draw((MusicObject)c, map,
			metNoteMapWidth/2 - NoteWidth, 0, 0, 0, NULL);

  event.display = display;
  event.window  = XtWindow(metNoteDisplay);

  XSendEvent(display, XtWindow(metNoteDisplay),
	     False, ExposureMask, (XEvent *)&event);

  End;
}


static void MetNoteUp(Widget w, XtPointer client, XtPointer call)
{
  Begin("MetNoteUp");

  MetNoteChange((Chord *)client, True);

  End;
}


static void MetNoteDown(Widget w, XtPointer client, XtPointer call)
{
  Begin("MetNoteDown");

  MetNoteChange((Chord *)client, False);

  End;
}

 
static void MetBeatUp(Widget w, XtPointer client, XtPointer call)
{
  int           i;
  String        s1;
  static String s = "123456";
  Widget        ww = (Widget)client;
  
  Begin("MetBeatUp");
  YGetValue(ww, XtNlabel, &s1);

  i = atoi(s1);
  if (i < 999) ++i;
  sprintf(s, "%3d  ", i);

  YSetValue(ww, XtNlabel, s);
  End;
}

 
static void MetBeatDown(Widget w, XtPointer client, XtPointer call)
{
  int           i;
  String        s1;
  static String s = "123456";
  Widget        ww = (Widget)client;
  
  Begin("MetBeatDown");
  YGetValue(ww, XtNlabel, &s1);

  i = atoi(s1);
  if (i > 1) --i;
  sprintf(s, "%3d  ", i);

  YSetValue(ww, XtNlabel, s);
  End;
}


static void MetCallback(Widget w, XtPointer client, XtPointer call)
{
  Begin("MetCallback");

  *((Widget *)client) = w;

  End;
}
 


void BarMenuMetronome(Widget w, XtPointer a, XtPointer b)
{
  MajorStaveRec  * mstave  = (MajorStaveRec *)stave;
  int              barNo;
  ItemList         list;
  ItemList         newList;
  StaveEltList     barList;
  static NoteVoice voice;
  static Chord     chord;
  static Boolean   haveChord = False;
  String           temp;
  String           beatl;
  static int       beat = 95;
  int              i;
  Pixmap           noteMap;

  Widget           metShell;
  Widget           metPane;
  Widget           metBox[3];
  Widget           metNoteUp;
  Widget           metNoteDown;
  Widget           metBeatUp;
  Widget           metBeatDisplay;
  Widget           metBeatDown;
  Widget           metApply;
  Widget           metCancel;

  XPoint           op;
  XtAppContext     context;
  Widget           value = NULL;

  Begin("BarMenuTimeSignature");

  if (!stave || !mstave->sweep.swept ||
      mstave->sweep.from.left != mstave->sweep.to.left) {
    XBell(display, 70);
    End;
  }

  for (barNo = 0, barList = mstave->bar_list;
       barList && barNo < mstave->sweep.first_bar;
       barList = (StaveEltList)Next(barList), ++barNo);

  if (!haveChord) {

    (void)NewNoteVoice(&voice, 0, ModNone);
    (void)NewChord(&chord, &voice, 1, ModNone, Crotchet, False);
    haveChord = True;
  }

  metShell = XtCreatePopupShell("Metronome", transientShellWidgetClass,
				topLevel, NULL, 0);

  metPane = YCreateWidget("Metronome Pane", panedWidgetClass, metShell);
  
  for (i = 0; i < 3; ++i)
    metBox[i] = YCreateShadedWidget
      ("Metronome Box", formWidgetClass, metPane,
       i == 2 ? MediumShade : LightShade);

  metNoteDown = YCreateArrowButton("Note Down", metBox[0], YArrowLeft);
  metNoteDisplay = YCreateLabel(" 00 ", metBox[0]);
  metNoteUp   = YCreateArrowButton("Note Up",   metBox[0], YArrowRight);

  YSetValue(XtParent(metNoteDisplay), XtNfromHoriz, XtParent(metNoteDown));
  YSetValue(XtParent(metNoteUp),      XtNfromHoriz, XtParent(metNoteDisplay));

  metBeatDown = YCreateArrowButton("Beat Down", metBox[1], YArrowLeft);
  metBeatDisplay = YCreateLabel(" 00 ", metBox[1]);
  metBeatUp   = YCreateArrowButton("Beat Up",   metBox[1], YArrowRight);

  YSetValue(XtParent(metBeatDisplay), XtNfromHoriz, XtParent(metBeatDown));
  YSetValue(XtParent(metBeatUp),      XtNfromHoriz, XtParent(metBeatDisplay));

  metApply  = YCreateCommand("OK",     metBox[2]);
  metCancel = YCreateCommand("Cancel", metBox[2]);

  YSetValue(XtParent(metCancel), XtNfromHoriz, XtParent(metApply));
  YSetValue(XtParent(metCancel), XtNfromHoriz, XtParent(metApply));
  YSetValue(XtParent(metCancel), XtNfromHoriz, XtParent(metApply));

  XtAddCallback(metBeatUp,    XtNcallback, MetBeatUp,
		(XtPointer)metBeatDisplay);
  XtAddCallback(metBeatDown,  XtNcallback, MetBeatDown,
		(XtPointer)metBeatDisplay);

  XtAddCallback(metNoteUp,    XtNcallback, MetNoteUp,   (XtPointer)&chord);
  XtAddCallback(metNoteDown,  XtNcallback, MetNoteDown, (XtPointer)&chord);
  XtAddCallback(metApply,     XtNcallback, MetCallback, (XtPointer)&value);
  XtAddCallback(metCancel,    XtNcallback, MetCallback, (XtPointer)&value);

  YSetValue(metNoteDisplay, XtNheight, StaveHeight + NoteHeight + 10);
  XtRealizeWidget(metShell);
  YGetValue(metBeatDisplay, XtNwidth, &metNoteMapWidth);
  XtUnrealizeWidget(metShell);

  noteMap = XCreatePixmap(display, RootWindowOfScreen(XtScreen(topLevel)),
			  metNoteMapWidth, StaveHeight + NoteHeight,
			  DefaultDepthOfScreen(XtScreen(topLevel)));

  XFillRectangle(display, noteMap, clearGC, 0, 0,
		 metNoteMapWidth, StaveHeight + NoteHeight);

  chord.item.methods->draw((MusicObject)&chord, noteMap,
			   metNoteMapWidth/2 - NoteWidth, 0, 0, 0, NULL);

  YSetValue(metNoteDisplay, XtNbitmap,                       noteMap);
  YSetValue(metNoteDisplay, XtNwidth,                metNoteMapWidth);

  op = YPlacePopupAndWarp(metShell, XtGrabExclusive, metCancel, metCancel);
  YAssertDialogueActions(metShell, metCancel, metCancel, NULL);

  beatl = (String)XtMalloc(5);
  sprintf(beatl, "%3d  ", beat);
  YSetValue(metBeatDisplay, XtNlabel, beatl);

  context = XtWidgetToApplicationContext(metShell);
  while (!value || XtAppPending(context)) XtAppProcessEvent(context, XtIMAll);

  YPopPointerPosition();
  YPopdown(metShell);
  YRetractDialogueActions(metShell);
  XtDestroyWidget(metShell);
  XFreePixmap(display, noteMap);

  YGetValue(metBeatDisplay, XtNlabel, &temp);
  beat = atoi(temp);
  XtFree(beatl);

  if (value != metApply) End;

  if (!barList || !barList->bars[0] ||
      !barList->bars[0]->group.start) {
    IssueMenuComplaint
      ("There is no bar in the top staff to place the metronome mark above.");
    End;
  }

  newList = NewItemList
    ((Item *)NewMetronome
     (NULL, chord.chord.visual->type, chord.chord.visual->dotted, beat));
  list = barList->bars[0]->group.start;

  while (list && list->item->generic.object_class == TimeSignatureClass)
    list = (ItemList)Next(list);

  if (list) Insert(newList, list);
  else Nconc(barList->bars[0]->group.start, newList);

  while (list && list != (ItemList)Next(barList->bars[0]->group.end))
    if (list && list->item->generic.object_class == MetronomeClass)
      list = (ItemList)Remove(list);
    else list = (ItemList)Next(list);

  if (barList->bars[0]->group.start == mstave->music[0])
    mstave->music[0] = barList->bars[0]->group.start = newList;
  else barList->bars[0]->group.start = newList;

  mstave->formats[0].next  = 0;
  mstave->formats[0].time  = NULL;
  mstave->formats[0].key   = NULL;
  mstave->formats[0].clef  = NULL;
  mstave->formats[0].items = mstave->music[0];

  FileMenuMarkChanged(stave, True);
  StaveRefreshAsDisplayed(stave);
  End;
}


void BarMenuRemoveTime(Widget w, XtPointer a, XtPointer b)
{
  MajorStaveRec *mstave  = (MajorStaveRec *)stave;
  int            staveNo = mstave->sweep.stave;
  int            firstStave;
  int            lastStave;
  int            barNo;
  ItemList       list;
  StaveEltList   barList;
  Boolean        foundSig = False;
  int            i;

  Begin("BarMenuRemoveTime");

  if (!stave || !mstave->sweep.swept ||
      mstave->sweep.from.left != mstave->sweep.to.left) {
    XBell(display, 70);
    End;
  }

  for (barNo = 0, barList = mstave->bar_list;
       barList && barNo < mstave->sweep.first_bar;
       barList = (StaveEltList)Next(barList), ++barNo);

  firstStave = lastStave = staveNo;

  if (mstave->staves > 1 &&
      YQuery(topLevel, NULL, 2, 1, -1, "Remove From This Staff Only",
	     "Remove From Whole Stave", "Bar - Remove Time Signature") == 1) {

    firstStave = 0;
    lastStave = mstave->staves - 1;
  }

  for (staveNo = firstStave; staveNo <= lastStave; ++staveNo) {

    if (!barList || !barList->bars[staveNo] ||
	!barList->bars[staveNo]->group.start) continue;

    list = barList->bars[staveNo]->group.start;

    while (list && list->item->generic.object_class == ClefClass ||
	   list->item->generic.object_class == KeyClass)
      list = (ItemList)Next(list);

    if (list && list->item->generic.object_class == TimeSignatureClass) {
      list = (ItemList)Remove(list);
      foundSig = True;
    }

    if (list == (ItemList)Next(barList->bars[staveNo]->group.start))
      if (barList->bars[staveNo]->group.start == mstave->music[staveNo])
	mstave->music[staveNo] = barList->bars[staveNo]->group.start = list;
      else barList->bars[staveNo]->group.start = list;

    mstave->formats[staveNo].next  = 0;
    mstave->formats[staveNo].time  = NULL;
    mstave->formats[staveNo].key   = NULL;
    mstave->formats[staveNo].clef  = NULL;
    mstave->formats[staveNo].items = mstave->music[staveNo];
  }

  if (!foundSig) IssueMenuComplaint
    ("There doesn't seem to be a time signature there to remove.");

  FileMenuMarkChanged(stave, True);
  StaveRefreshAsDisplayed(stave);
  End;
}


void BarMenuRemoveMet(Widget w, XtPointer a, XtPointer b)
{
  MajorStaveRec *mstave  = (MajorStaveRec *)stave;
  int            barNo;
  ItemList       list;
  StaveEltList   barList;
  Boolean        foundMet = False;
  int            i;

  Begin("BarMenuRemoveMet");

  if (!stave || !mstave->sweep.swept ||
      mstave->sweep.from.left != mstave->sweep.to.left) {
    XBell(display, 70);
    End;
  }

  for (barNo = 0, barList = mstave->bar_list;
       barList && barNo < mstave->sweep.first_bar;
       barList = (StaveEltList)Next(barList), ++barNo);

  if (!barList || !barList->bars[0] ||
      !barList->bars[0]->group.start) {
    IssueMenuComplaint
      ("There is no bar in the top staff in which a metronome mark could be.");
    End;
  }

  list = barList->bars[0]->group.start;
  
  while (list && list->item->generic.object_class == ClefClass ||
	 list->item->generic.object_class == KeyClass)
    list = (ItemList)Next(list);
  
  while (list && list != (ItemList)Next(barList->bars[0]->group.end))
    if (list && list->item->generic.object_class == MetronomeClass) {
      list = (ItemList)Remove(list);
      foundMet = True;
    } else list = (ItemList)Next(list);

  if (list == (ItemList)Next(barList->bars[0]->group.start))
    if (barList->bars[0]->group.start == mstave->music[0])
      mstave->music[0] = barList->bars[0]->group.start = list;
    else barList->bars[0]->group.start = list;

  mstave->formats[0].next  = 0;
  mstave->formats[0].time  = NULL;
  mstave->formats[0].key   = NULL;
  mstave->formats[0].clef  = NULL;
  mstave->formats[0].items = mstave->music[0];

  if (!foundMet) IssueMenuComplaint
    ("There doesn't seem to be a metronome mark there to remove.");

  FileMenuMarkChanged(stave, True);
  StaveRefreshAsDisplayed(stave);
  End;
}


static void ChangeBarStyle(Boolean start, BarTag tag)
{
  MajorStaveRec *mstave  = (MajorStaveRec *)stave;
  StaveEltList   barList;
  int            staveNo;
  int            barNo;

  Begin("ChangeBarStyle");

  if (!stave || !mstave->sweep.swept ||
      mstave->sweep.from.left != mstave->sweep.to.left) {
    XBell(display, 70);
    End;
  }

  for (barNo = 0, barList = mstave->bar_list;
       barList && barNo < mstave->sweep.first_bar;
       barList = (StaveEltList)Next(barList), ++barNo);

  if (!barList || !barList->bars[mstave->sweep.stave]) {
    XBell(display, 70);
    End;
  }

  if (tag != SingleBar)
    for (staveNo = 0; staveNo < mstave->staves; ++staveNo)
      if (!barList->bars[staveNo]) {
	IssueMenuComplaint("There must be a bar here in every staff.");
	End;
      }

  for (staveNo = 0; staveNo < mstave->staves; ++staveNo) {

    if (start) barList->starts[staveNo] = tag;
    else       barList->  ends[staveNo] = tag;
    StaveResetFormatting(stave, staveNo);
  }

  FileMenuMarkChanged(stave, True);
  StaveRefreshAsDisplayed(stave);
  End;
}


void BarMenuPlain(Widget w, XtPointer a, XtPointer b)
{
  Begin("BarMenuPlain");

  ChangeBarStyle(True,  SingleBar);
  ChangeBarStyle(False, SingleBar);

  End;
}


void BarMenuDoubleBar(Widget w, XtPointer a, XtPointer b)
{
  Begin("BarMenuDoubleBar");

  ChangeBarStyle(False, DoubleBar);

  End;
}


void BarMenuRepeatAtStart(Widget w, XtPointer a, XtPointer b)
{
  Begin("BarMenuRepeatAtStart");

  ChangeBarStyle(True, RepeatBar);

  End;
}


void BarMenuRepeatAtEnd(Widget w, XtPointer a, XtPointer b)
{
  Begin("BarMenuRepeatAtEnd");

  ChangeBarStyle(False, RepeatBar);

  End;
}

