
/* Musical Notation Editor for X, Chris Cannam 1994 */


#include "General.h"
#include "IO.h"
#include "Tags.h"
#include "MidiOut.h"
#include "MidiIn.h"
#include "Menu.h"
#include "StaveEdit.h"

#include <Yawn.h>
#include <ILClient.h>

String filename = NULL;


void SequenceAcknowledgeCallback(IL_ReturnCode rtn)
{
  Begin("SequenceCompleteCallback");

  if (filename) (void)unlink(filename);	/* can't do anything about failure */

  StaveBusy(False);

  switch(rtn) {

  case IL_SERVICE_OK:

    LeaveMenuMode(SequencerRunningMode);
    EnterMenuMode(SequencerNotRunningMode);
    break;

  case IL_SERVICE_BUSY:
    
    (void)YQuery(topLevel, "The sequencer is busy.  Try again later.",
		 1, 0, 0, "Okay", NULL);

    LeaveMenuMode(SequencerRunningMode);
    EnterMenuMode(SequencerNotRunningMode);
    break;

  case IL_SERVICE_FAILED:

    (void)YQuery(topLevel, "Sorry, the sequencer doesn't seem to be working.",
		 1, 0, 0, "Okay", NULL);
    break;

  case IL_NO_SUCH_SERVICE:

    (void)YQuery(topLevel, "I can't invoke the sequencer.",
		 1, 0, 0, "Okay", NULL);
    break;
  }

  End;
}


void ToolsMenuSequence(Widget w, XtPointer a, XtPointer b)
{
  String name;

  Begin("ToolsMenuSequence");

  if (!(filename = tmpnam(NULL))) {
    IssueMenuComplaint("Sorry, I couldn't get a temporary file name.");
    End;
  }

  name = (String)XtMalloc(strlen(GetStaveName(stave)) + 25);
  sprintf(name, "%s (temporary)", GetStaveName(stave));

  if (MidiWriteStave(stave, filename, name) == Failed) {
    XtFree(name);
    End;
  }

  XtFree(name);

  StaveBusy(True);
  IL_RequestService("Mseq", SequenceAcknowledgeCallback,
		    filename, strlen(filename) + 1);

  LeaveMenuMode(SequencerNotRunningMode);
  EnterMenuMode(SequencerRunningMode);
  End;
}


void ToolsMenuPlayMidi(Widget w, XtPointer a, XtPointer b)
{
  char *text = "Hello World!  How are you?";

  Begin("ToolsMenuPlayMidi");

  StaveBusy(True);

  IL_RequestService("Play", SequenceAcknowledgeCallback,
		    text, strlen(text) + 1);

  LeaveMenuMode(SequencerNotRunningMode);
  EnterMenuMode(SequencerRunningMode);
  End;
}


void ToolsMenuWriteMidi(Widget w, XtPointer a, XtPointer b)
{
  String fname;

  Begin("ToolsMenuWriteMidi");

  if ((fname = YFileGetWriteFilename(XtParent(XtParent(w)),
				     "Editor Tools - Export MIDI"))
      == NULL) End;
  (void)MidiWriteStave(stave, fname, NULL);
}


void ToolsMenuImportMidi(Widget w, XtPointer a, XtPointer b)
{
  String     fname;
  String     name;
  MajorStave sp;

  Begin("ToolsMenuImportMidi");

  if ((fname = YFileGetReadFilename(XtParent(XtParent(w)),
				    "Editor Tools - Import MIDI"))
      == NULL) End;
  if ((sp = MidiReadStave(fname, &name)) == NULL) { StaveBusy(False); End; }

  AddStaveToFileMenu(sp, name, NULL);
  FileMenuMarkChanged(sp, False);
  XtFree(name);

  staveMoved = True;
  LeaveMenuMode(FileNotLoadedMode);
  EnterMenuMode(FileLoadedMode);

  End;
}



void ToolsMenuEditILCallback(String chunk)
{
  String       name;
  MajorStave   sp;
  XtAppContext appContext;

  Begin("ToolsMenuEditILCallback");

  appContext = XtWidgetToApplicationContext(topLevel);
  while (XtAppPending(appContext)) {
    XtAppProcessEvent(appContext, XtIMAll);
    XSync(display, False);
  }

  if ((sp = MidiReadStave(chunk, &name)) == NULL) {

    IL_AcknowledgeRequest("Edit", IL_SERVICE_FAILED);
    End;
  }

  XRaiseWindow(display, XtWindow(topLevel));
  XRaiseWindow(display, XtWindow(paletteShell));

  AddStaveToFileMenu(sp, name, NULL);
  FileMenuMarkChanged(sp, False);
  XtFree(name);

  staveMoved = True;
  LeaveMenuMode(FileNotLoadedMode);
  EnterMenuMode(FileLoadedMode);

  IL_AcknowledgeRequest("Edit", IL_SERVICE_OK);

  End;
}


void ToolsMenuExportTeX(Widget w, XtPointer a, XtPointer b)
{
  int    sysrtn;
  String name;
  char   midifile[50];
  String outputfile;
  String texfile;
  String command;

  Begin("ToolsMenuExportTeX");

  if (!appData.midi2texName) {

    XBell(display, 70);
    LeaveMenuMode(Midi2TeXUnavailableMode);
    End;
  }

  if (YQuery(XtParent(XtParent(w)), "\n\
 As an interim measure until this function is properly available, \n\
 it works by converting the music to MIDI and using ``midi2tex'', \n\
 by Hans Kuykens and Ad Verbruggen, to produce MusicTeX. \n\n\
 This will inevitably lose some information, and the results are \n\
 known to be often shaky. \n\n\
 Do you want to go on anyway? \n\n", 2, 0, 1,
	     "Yes", "No", "Editor Tools - Export MusicTeX") == 1) End;

  (void)tmpnam(midifile);
  (void)strcat(midifile, ".mid");

  name = (String)XtMalloc(strlen(GetStaveName(stave)) + 25);
  sprintf(name, "%s (temporary)", GetStaveName(stave));

  if (MidiWriteStave(stave, midifile, name) == Failed) {
    XtFree(name);
    End;
  }

  if ((texfile = YFileGetWriteFilename(XtParent(XtParent(w)),
				       "Editor Tools - Export MusicTeX"))
      == NULL)
    End;

  outputfile =
    tempnam(appData.musicDirectory ? appData.musicDirectory : ".", "error");

  command =
    (String)XtMalloc(strlen(appData.midi2texName) +
		     strlen(midifile) + strlen(texfile) + strlen(outputfile)
		     + 15);

  sprintf(command, "%s %s %s > %s",
	  appData.midi2texName, midifile, texfile, outputfile);

  StaveBusy(True);

  sysrtn = system(command);

  StaveBusy(False);
  XtFree(command);

  if (sysrtn == 0) {		/* succeeded */

    (void)unlink(midifile);
    (void)unlink(outputfile);
    XtFree(outputfile);
    End;

  } else {			/* failed */

    if (sysrtn < 0) {

      YQuery(XtParent(XtParent(w)),
	     "Sorry, I couldn't fork the midi2tex process.",
	     1, 0, 0, "Continue", NULL);

      (void)unlink(midifile);
      XtFree(outputfile);
      End;

    } else {

      String message = (String)XtMalloc(strlen(outputfile) + 100);

      sprintf(message, "Sorry, the process failed.  Error report in ``%s''.",
	      outputfile);

      YQuery(XtParent(XtParent(w)), message, 1, 0, 0, "Continue", NULL);
      (void)unlink(midifile);
      XtFree(outputfile);
      XtFree(message);
      End;
    }
  }
}

