
/* Musical Notation Editor for X, Chris Cannam 1994    */
/* Some Midi output fns.  Thanks to Andy for midi code */

#include "General.h"
#include "Tags.h"
#include "Classes.h"
#include "Notes.h"
#include "Stave.h"
#include "StaveEdit.h"
#include "StavePrivate.h"
#include "Format.h"

#include <Yawn.h>
#include <MidiFile.h>
#include <MidiTrack.h>

static int midiBarStart = 0;	/* JPff */
extern int ampBarEmphasis;	/* JPff */


static void MidiWriteTrackZero(MajorStave sp, void *fp, String name)
{
  int            i;
  MajorStaveRec *mstave = (MajorStaveRec *)sp;
  MIDIFileHandle file = (MIDIFileHandle) fp;
  ItemList       items;
  EventList      events;
  MIDIEvent      endEvent;
  long           delta = 0;
  EventList      iTime;
  EventList      iTempo;

  Begin("MidiWriteTrackZero");

  events = Midi_EventCreateList
    (Midi_EventCreateTextEvt(MIDI_TRACK_NAME, 0, name), False);

  /* midi2tex barfs on this (because midi2tex barfs on everything):

  Nconc(events, Midi_EventCreateList
	(Midi_EventCreateTextEvt
	 (MIDI_TEXT_MARKER, 0,
	  "Created by the Rosegarden editor"), False));
  */

  Nconc(events, iTempo =
	Midi_EventCreateList(Midi_EventCreateTempoEvt(0, 120), False));
  Nconc(events, iTime  =
	Midi_EventCreateList(Midi_EventCreateTimeSigEvt(0, 4, 4), False));

  items = (ItemList)First(mstave->music[0]);

  while (items) {

    if (items->item->generic.object_class == GroupClass) {

      items = (ItemList)Next(items);
      continue;
    }

    switch (items->item->generic.object_class) {

    case TimeSignatureClass:

      Nconc(events, Midi_EventCreateList
	    (Midi_EventCreateTimeSigEvt
	     (delta,
	      (byte)((TimeSignature *)items->item)->timesig.numerator,
	      (byte)((TimeSignature *)items->item)->timesig.denominator),
	     False));

      if (iTime && delta == 0) { Remove(iTime); iTime = NULL; }
      break;

    case KeyClass:

      Nconc(events, Midi_EventCreateList
	    (Midi_EventCreateKeySigEvt
	     (delta, (byte)
	      (((Key *)items->item)->key.visual->sharps ?
	       ((Key *)items->item)->key.visual->number :
	       - ((Key *)items->item)->key.visual->number), (byte)0), False));
      break;

    case MetronomeClass:

      Nconc(events, Midi_EventCreateList
	    (Midi_EventCreateTempoEvt
	     (delta, (long)
	      (((Metronome *)items->item)->metronome.setting *
	       MTimeToNumber(((Metronome *)items->item)->metronome.beat_length)
	       / TagToNumber(Crotchet, False))), False));

      if (iTempo && delta == 0) { Remove(iTempo); iTempo = NULL; }
      break;

    default: break;
    }

    delta += (10L * MTimeToNumber(items->item->item.methods->get_length
				  ((MusicObject)items->item)))
      / TagToNumber(Hemidemisemiquaver, False);
  
    items = (ItemList)Next(items);
  }

  endEvent = (MIDIEvent)XtMalloc(sizeof(MIDIEventStruct));
  endEvent->DeltaTime = delta;
  endEvent->EventCode = MIDI_FILE_META_EVENT;
  endEvent->EventData.MetaEvent.MetaEventCode = MIDI_END_OF_TRACK;
  endEvent->EventData.MetaEvent.NBytes = 0;

  Nconc(events, Midi_EventCreateList(endEvent, False));
  Midi_FileWriteTrack(file, events);
  Midi_TrackDelete(events);

  End;
}


static void MidiWriteTrack(String name, ItemList list, void *fp)
{
  MIDIFileHandle file = (MIDIFileHandle)fp;
  ClefTag        clef = TrebleClef;
  long           delta = 0;
  EventList      events;
  EventList      events2p;
  MIDIEvent      endEvent;
  ItemList       i;

  Begin("MidiWriteItemList");

  events = Midi_EventCreateList
    (Midi_EventCreateTextEvt(MIDI_TRACK_NAME, 0, name), False);

  /* Find the original clef */

  for (i = list; i; i = (ItemList)Next(i))
    if (i->item->generic.object_class == ClefClass) {
      clef = ((Clef *)i->item)->clef.clef;
      break;
    }

  /* Get the events */

  for (i = list; i; i = (ItemList)Next(i)) {

    if (i->item->generic.object_class == ClefClass)
      clef = ((Clef *)i->item)->clef.clef;

    if (i->item->generic.object_class == GroupClass) continue;

    i->item->item.methods->write_midi
      ((MusicObject)i->item, (List)events, delta, clef);

    delta += (10L * MTimeToNumber(i->item->item.methods->get_length
				  ((MusicObject)i->item)))
      / TagToNumber(Hemidemisemiquaver, False);
  }

  endEvent = (MIDIEvent)XtMalloc(sizeof(MIDIEventStruct));
  endEvent->DeltaTime = delta;
  endEvent->EventCode = MIDI_FILE_META_EVENT;
  endEvent->EventData.MetaEvent.MetaEventCode = MIDI_END_OF_TRACK;
  endEvent->EventData.MetaEvent.NBytes = 0;
  Nconc(events, Midi_EventCreateList(endEvent, False));

  /* And write them out */

  events2p = Midi_TrackConvertToTwoPointRepresentation(events);
  Midi_FileWriteTrack(file, events2p);
  Midi_TrackDelete(events2p);
  Midi_TrackDelete(events);
}


Result MidiWriteStave(MajorStave sp, String fname, String title)
{
  MajorStaveRec  *mstave = (MajorStaveRec *)sp;
  MIDIHeaderChunk header;    
  MIDIFileHandle  file;
  String          msg;
  int             i;
  
  Begin("MidiWriteStave");

  header.Format = MIDI_SIMULTANEOUS_TRACK_FILE;
  header.NumTracks = mstave->staves + 1;
  header.Timing.Division = 160;

  StaveBusyStartCount(mstave->staves*2 + 1);
  StaveBusyMakeCount(0);

  if ((file = Midi_FileOpen(fname, &header, MIDI_WRITE)) == NULL) {
    
    msg = (String)XtMalloc(300);

    if (!title) sprintf(msg, "Sorry, I can't open `%s' for writing.", fname);
    else        sprintf(msg, "Sorry, I can't open a temporary file.");

    XBell(display, 70);
    (void)YQuery(topLevel, msg, 1, 0, 0, "Cancel", NULL);

    XtFree((void *)msg);
    Return(Failed);
  }

  if (title) {

    MidiWriteTrackZero(sp, (void *)file, title);

  } else {

    for (msg = fname + strlen(fname) - 1; msg >= fname; --msg)
      if (*msg == '/') break;

    if (msg <= fname) MidiWriteTrackZero(sp, (void *)file, fname);
    else              MidiWriteTrackZero(sp, (void *)file, ++msg);
  }

  for (i = 0; i < mstave->staves; ++i) {

    StaveBusyMakeCount(i*2 + 1);
    UnformatItemList(&mstave->music[i], NULL);

    StaveBusyMakeCount(i*2 + 2);
    MidiWriteTrack(mstave->names[i], mstave->music[i], (void *)file);

    mstave->formats[i].next  = 0;
    mstave->formats[i].time  = NULL;
    mstave->formats[i].key   = NULL;
    mstave->formats[i].clef  = NULL;
    mstave->formats[i].items = mstave->music[i];
  }

  Midi_FileClose(file);

  staveMoved = True;
  StaveRefresh(stave, -1);
  StaveBusyFinishCount();

  Return(Succeeded);
}



void MidiWriteNothing(MusicObject obj, List events, long delta, ClefTag clef)
{
  Begin("MidiWriteNothing");
  End;
}


/* So we can emphasise first note (JPff change, not finished) */
void MidiWriteRest(MusicObject obj, List events, long delta, ClefTag clef)
{
  Begin("MidiWriteRest");
  midiBarStart = 0;
  End;
}


/* So we can emphasise first note (JPff change, not finished) */
void MidiWriteBar(MusicObject obj, List events, long delta, ClefTag clef)
{
  Begin("MidiWriteBar");
  midiBarStart = ampBarEmphasis;
  End;
}


void MidiWriteText(MusicObject obj, List events, long delta, ClefTag clef)
{
  Text *text = (Text *)obj;

  Begin("MidiWriteText");

  Nconc(events, Midi_EventCreateList
	(Midi_EventCreateTextEvt(MIDI_TEXT_EVENT, delta, text->text.text),
	 False));

  End;
}


int VoiceToMidiPitch(NoteVoice voice, ClefTag clef)
{
  int rtn;
  int octave = 5;

  Begin("VoiceToMidiPitch");

  while (voice.pitch < 0) { octave -= 1; voice.pitch += 7; }
  while (voice.pitch > 7) { octave += 1; voice.pitch -= 7; }

  if (voice.pitch > 4) ++octave;

  switch(voice.pitch) {

  case 0: rtn =  4; break;	/* bottom line, treble clef: E */
  case 1: rtn =  5; break;	/* F */
  case 2: rtn =  7; break;	/* G */
  case 3: rtn =  9; break;	/* A, in next octave */
  case 4: rtn = 11; break;	/* B, likewise*/
  case 5: rtn =  0; break;	/* C, moved up an octave (see above) */
  case 6: rtn =  2; break;	/* D, likewise */
  case 7: rtn =  4; break;	/* E, likewise */
  }

  if (voice.modifiers & ModSharp) ++ rtn;
  if (voice.modifiers & ModFlat)  -- rtn;

  switch(clef) {

  case TrebleClef: break;
  case  TenorClef: octave -= 1; break;
  case   AltoClef: octave -= 1; break;
  case   BassClef: octave -= 2; break;
  }

  rtn += 12 * octave;

  Return(rtn);
}


void MidiWriteChord(MusicObject obj, List events, long delta, ClefTag clef)
{
  int    i;
  Chord *chord = (Chord *)obj;

  Begin("MidiWriteChord");

  /* arg 4 of Midi_EventCreateNote changed from    */
  /* plain (byte)64 by JPff for bar-start emphasis */

  for (i = 0; i < chord->chord.voice_count; ++i)
    Nconc(events, Midi_EventCreateList
	  (Midi_EventCreateNote
	   (delta, (byte)0,
	    (byte)VoiceToMidiPitch(chord->chord.voices[i], clef),
	    (byte)((chord->chord.modifiers & ModAccent ? 100:64)+midiBarStart),
	    (10 * MTimeToNumber(chord->item.methods->get_length
				((MusicObject)chord)))
	    / TagToNumber(Hemidemisemiquaver, False)), False));
  
  midiBarStart = 0;		/* Emphasis on first note of bar */
  End;
}

