
/* Musical Notation Editor for X, Chris Cannam 1994 -- Palette code */

/* Feb 95: occurrences of "this" changed to "curr" to  */
/* satisfy C++ compilers                               */

#include "General.h"
#include "Tags.h"
#include "Visuals.h"
#include "Palette.h"
#include "GC.h"
#include "Notes.h"
#include "Classes.h"
#include "Yawn.h"
#include "Stave.h"
#include "StaveEdit.h"
#include "StaveCursor.h"

#include <X11/Xaw/Form.h>
#include <X11/Xaw/Toggle.h>


#define PaletteModCount 4


static Widget  paletteShellLocal;
static Widget  paletteForm;	/* Contains palette items (not mods)    */
static Widget  paletteSpace;	/* Grey bit at bottom of palette window */
static Widget *paletteItems;	/* List of toggle widgets, one per item */
static Widget  paletteEmpty;	/* Label used when no palette is shown  */
				/* (needed to avoid `zero width and/or  */
                                /* height' errors from the paletteForm) */

static int     numItems;	/* Maximum number of palette items      */
static int     nothing = 0;


/*
   A palette is a list of visuals.  There are several possible palettes,
   and the current `palette mode' describes which is visible.  Possibly
   the visual is not the best record for the content of a palette as
   we're principally interested just in drawing them and reporting which
   they are; there's no intrinsic function for drawing a given visual.
   In many cases (rests, clefs, keys) we need to create an object
   derived from Item and use its draw method; Marks have no such object.
   This is something of a pity, making the code here rather less
   than elegant.
*/

typedef struct _PaletteRec {
  PaletteTag type;
  XtPointer  visual;
  int       *number;
  int        current;
} PaletteRec;

/*
   We associate the modifiers with the palette modes in which they are
   available, instead of the palette modes with the modifiers available
   in them, because this makes it easier if we want to add more mods.
   The records for the number of palette modes have already to some
   extent been hardcoded already, so we don't lose much by assuming
   that the mods have knowledge of them.  However, just to make things
   difficult we choose an intensely unintuitive representation for the
   availablePalettes field.

   Trying to associate Visuals proper with all the possible sorts of
   modifiers in a mixable fashion would be too difficult; instead, just
   name them and give pixmaps, and discriminate by ad-hoc means later.
*/

typedef struct _PaletteModRec {
  String        name;
  Dimension     width;
  Dimension     height;
  Pixmap       *pixmap;
  unsigned long availablePalettes;
} PaletteModRec;

static PaletteRec palettes[] = {
  { PaletteNothing, (XtPointer)NULL,            &nothing,            -1, },
  { PaletteNotes,   (XtPointer)noteVisuals,     &noteVisualCount,     4, },
  { PaletteRests,   (XtPointer)restVisuals,     &restVisualCount,     4, },
  { PaletteKeys,    (XtPointer)keyVisuals,      &keyVisualCount,      4, },
  { PaletteClefs,   (XtPointer)clefVisuals,     &clefVisualCount,     0, },
};

static PaletteModRec paletteMods[] = {
  {
    "Sharp",   NoteModWidth, NoteModHeight,
    &(noteModVisuals[0].pixmap), 1L << PaletteNotes,
  }, {
    "Flat",    NoteModWidth, NoteModHeight,
    &(noteModVisuals[1].pixmap), 1L << PaletteNotes,
  }, {
    "Natural", NoteModWidth, NoteModHeight,
    &(noteModVisuals[2].pixmap), 1L << PaletteNotes,
  }, {
    "Dot",     DotWidth,     NoteHeight,
    &(noteDotMap), (1L << PaletteNotes) | (1L << PaletteRests),
  },
};

static PaletteRec *currentPalette = NULL;
static Widget      paletteModWidgets[PaletteModCount];



Boolean PaletteModDottedQuery(void)
{
  Boolean rtn = False;
  
  Begin("PaletteModDottedQuery");

  if (paletteModWidgets[3])
    YGetValue(paletteModWidgets[3], XtNstate, &rtn);

  Return(rtn);
}


NoteMods PaletteGetNoteMods(void)
{
  int      i;
  Boolean  curr;
  NoteMods rtn = ModNone;

  Begin("PaletteGetNoteMods");

  for (i = 0; i < 3; ++i) {

    curr = False;

    if (paletteModWidgets[i])
      YGetValue(paletteModWidgets[i], XtNstate, &curr);

    if (curr) rtn |= 1 << i;
  }

  Return(rtn);
}


void PaletteItemCallback(Widget w, XtPointer cl, XtPointer ca)
{
  int i;
  Begin("PaletteItemCallback");

  if (!currentPalette) End;

  if (currentPalette->current == (int)cl) {

    currentPalette->current = -1;
    StaveEditEnterEditMode();

  } else {

    StaveCursorRemove(stave);
    currentPalette->current = (int)cl;

    StaveEditEnterInsertMode();
    StaveEditAssertInsertVisual(currentPalette->type,
				currentPalette->visual,
				currentPalette->current);
  }

  End;
}


void InstallNamedPalette(int i)
{
  int j;
  Begin("InstallNamedPalette");

  for (j = 0; j < *(palettes[i].number); ++j) {

    switch(palettes[i].type) {

    case PaletteKeys:
      YSetValue(paletteItems[j], XtNlabel,
		 ((KeyVisualRec *)(palettes[i].visual))[j].name);
      break;

    default:
      break;
    }
  }

  End;
}


void InstallNamedVisualPalette(int i)
{
  int         j, y;
  Pixmap      bitmap;
  MusicObject obj;
  char        name[100];

  Begin("InstallNamedVisualPalette");

  name[0] = ' ';

  for (j = 0; j < *(palettes[i].number); ++j) {

    switch(palettes[i].type) {

    case PaletteClefs:
 /*
      bitmap = XCreateBitmapFromData(display,
				     RootWindowOfScreen(XtScreen(topLevel)),
				     ((ClefVisualRec *)
				      (palettes[i].visual))[j].bitmap,
				     ClefWidth, StaveHeight + 2*NoteHeight);
 */
      bitmap = XCreatePixmap(display,
			     RootWindowOfScreen(XtScreen(topLevel)),
			     ClefWidth + 65, StaveHeight + 2*NoteHeight,
			     DefaultDepthOfScreen(XtScreen(topLevel)));

      XFillRectangle(display, bitmap, clearGC, 0, 0,
		     ClefWidth + 65, StaveHeight + 2*NoteHeight);

      obj = (MusicObject)NewClef(NULL, ((ClefVisualRec *)
					(palettes[i].visual))[j].type);

      (void)(((Item *)obj)->item.methods->draw
	     (obj, bitmap, 55, NoteHeight, 0, 0, NULL));

      for (y = 0; y < StaveHeight; y += NoteHeight + 1)
	XDrawLine(display, bitmap, drawingGC,
		  0, y + NoteHeight, ClefWidth + 64, y + NoteHeight);

      sprintf(&(name[1]), ((ClefVisualRec *)(palettes[i].visual))[j].name);
      break;

    default:
      bitmap = 0;
      break;
    }

    YSetValue(paletteItems[j], XtNleftBitmap, bitmap);
    YSetValue(paletteItems[j], XtNlabel, name);
  }

  End;
}


void InstallVisualPalette(int i)
{
  int         j;
  Pixmap      bitmap;
  Dimension   width;
  MusicObject obj;
  NoteVoice   voice;

  Begin("InstallVisualPalette");

  for (j = 0; j < *(palettes[i].number); ++j) {

    switch(palettes[i].type) {

    case PaletteNotes:
      (void)NewNoteVoice(&voice, 1, ModNone);
      width = NoteWidth;
      obj   = NewChord
	(NULL, &voice, 1, ModNone,
	 ((NoteVisualCompound *)(palettes[i].visual))[j].undotted.type, False);
      break;

    case PaletteRests:
      width  = ((RestVisualCompound *)(palettes[i].visual))[0].undotted.width;
				/* make them all the same width */
      obj    =
	(MusicObject)NewRest
	  (NULL, ((RestVisualCompound *)
		  (palettes[i].visual))[j].undotted.type, False);
      break;

    default:
      obj = NULL;
      break;
    }

    if (obj) {

      bitmap = XCreatePixmap(display,
			     RootWindowOfScreen(XtScreen(topLevel)),
			     width + 20, StaveHeight + NoteHeight,
			     DefaultDepthOfScreen(XtScreen(topLevel)));

      XFillRectangle(display, bitmap, clearGC, 0, 0,
		     width + 20, StaveHeight + NoteHeight);

      (void)(((Item *)obj)->item.methods->draw
	     (obj, bitmap, 10, NoteHeight, 0, 0, NULL));

      YSetValue(paletteItems[j], XtNbitmap, bitmap);
      YSetValue(paletteItems[j], XtNlabel, NULL);

      XtFree(obj);
    }
  }

  End;
}



void SetPaletteTitle(void)
{
  static Widget top = NULL;
  static String title;
  PaletteTag    type = currentPalette->type;

  Begin("SetPaletteTitle");

  if (top == NULL)
    for (top = paletteForm; XtParent(top); top = XtParent(top));

  title = (type == PaletteNothing ? "Palette" :
	   type == PaletteNotes   ? "Notes"   :
	   type == PaletteRests   ? "Rests"   :
	   type == PaletteKeys    ? "Keys"    :
	   type == PaletteClefs   ? "Clefs"   : "Palette");

  YSetValue(top, XtNtitle, title);
  YSetValue(top, XtNiconName, title);

  End;
}



void InstallPalette(PaletteTag type)
{
  int               i, j;
  Pixmap            bitmap;
  static PaletteTag prev = (PaletteTag) -1;
  static int        shapeStyle = -1;
  static Arg        shapeArg = { XtNshapeStyle, 1 };

  Begin("InstallPalette");

  if (type == prev) return;
  prev = type;

  for (i = 0; i < XtNumber(palettes); ++i) if (palettes[i].type == type) break;
  if (i >= XtNumber(palettes)) return;

  if (shapeStyle < 0)
    YGetValue(paletteItems[0], XtNshapeStyle, &shapeStyle);

  if (XtIsRealized(paletteSpace) && XtIsManaged(paletteSpace))
    XtUnmapWidget(paletteSpace);

  XtUnmanageChild(paletteForm);
  XtUnmanageChild(paletteSpace);
  XtUnmanageChildren(paletteItems, numItems);

  for (j = 0; j < numItems; ++j) {

    YGetValue(paletteItems[j], XtNbitmap, &bitmap);
    
    if (bitmap) {
      XFreePixmap(display, bitmap);
      YSetValue(paletteItems[j], XtNbitmap, NULL);
    }

    YGetValue(paletteItems[j], XtNleftBitmap, &bitmap);
    
    if (bitmap) {
      XFreePixmap(display, bitmap);
      YSetValue(paletteItems[j], XtNleftBitmap, NULL);
    }
  }

  switch(type) {

  case PaletteNotes: case PaletteRests:
    InstallVisualPalette(i); break;

  case PaletteClefs:
    InstallNamedVisualPalette(i); break;

  case PaletteKeys:
    InstallNamedPalette(i); break;

  default: break;
  }

  currentPalette = &(palettes[i]);

  if (palettes[i].current >= 0) {

    for (j = 0; j < numItems; ++j)
      YSetValue(paletteItems[j], XtNstate, j == palettes[i].current);

    StaveCursorRemove(stave);

    StaveEditEnterInsertMode();
    StaveEditAssertInsertVisual(currentPalette->type,
				currentPalette->visual,
				currentPalette->current);

  } else StaveEditEnterEditMode();

  if (*(palettes[i].number) > 0) {

    for (j = 0; j < numItems; ++j)
      XtSetMappedWhenManaged(paletteItems[j], False);

    if (shapeStyle > 1) {
      shapeArg.value = 1;
      for (j = 0; j < numItems; ++j)
	XtSetValues(paletteItems[j], &shapeArg, 1);
    }

    XtUnmanageChild(paletteEmpty);
    XtManageChildren(paletteItems, *(palettes[i].number));
    XtManageChild(paletteForm);

    XtManageChild(paletteSpace);

    if (shapeStyle > 1) {
      shapeArg.value = (XtArgVal)shapeStyle;
      for (j = 0; j < numItems; ++j)
	XtSetValues(paletteItems[j], &shapeArg, 1);
    }

    for (j = 0; j < numItems; ++j)
      XtSetMappedWhenManaged(paletteItems[j], True);

  } else {

    XtManageChild(paletteEmpty);
    XtManageChild(paletteForm);
    XtManageChild(paletteSpace);
  }

  for (j = 0; j < PaletteModCount; ++j)
    XtSetSensitive(paletteModWidgets[j],
		   paletteMods[j].availablePalettes & (1L << i));

  SetPaletteTitle();
  End;
}


void InstallPalettes(Widget form, Widget space)
{
  int i;
  int sz;

  Begin("InstallPalettes");

  paletteForm  = form;
  paletteSpace = space;
  paletteEmpty = YCreateWidget("Palette Empty", labelWidgetClass, form);

  for (paletteShellLocal = paletteForm;
       XtParent(paletteShellLocal);
       paletteShellLocal = XtParent(paletteShellLocal));

  YSetValue(paletteEmpty, XtNbitmap, roseMap);

  for (i = sz = 0; i < XtNumber(palettes); ++i)
    if (*(palettes[i].number) > sz) sz = *(palettes[i].number);

  paletteItems = (Widget *)XtCalloc(sz, sizeof(Widget));
  numItems = sz;

  for (i = 0; i < sz; ++i) {

    paletteItems[i] = YCreateWidget
      ("Palette Item", toggleWidgetClass, paletteForm);

    YSetValue(paletteItems[i], XtNfromVert,   i>0 ? paletteItems[i-1] : NULL);
    YSetValue(paletteItems[i], XtNradioGroup, i>0 ? paletteItems[0]   : NULL);
    YSetValue(paletteItems[i], XtNradioData,                               i);

    XtAddCallback(paletteItems[i], XtNcallback,
		  PaletteItemCallback, (XtPointer)i);
  }

  StaveEditEnterEditMode();
  InstallPalette(PaletteNothing);
  End;
}


void InstallPaletteMods(Widget parent)
{
  int       i;
  Dimension wd = 0;
  Dimension ht = 0;
  Pixmap    bitmap;

  Begin("InstallPaletteMods");

  for (i = 0; i < PaletteModCount; ++i) {

    if (paletteMods[i].width  > wd) wd = paletteMods[i].width;
    if (paletteMods[i].height > ht) ht = paletteMods[i].height;
  }

  for (i = 0; i < PaletteModCount; ++i) {

    bitmap = XCreatePixmap(display, RootWindowOfScreen(XtScreen(topLevel)),
			   wd, ht, DefaultDepthOfScreen(XtScreen(topLevel)));

    XFillRectangle(display, bitmap, clearGC, 0, 0, wd, ht);

    CopyArea(*(paletteMods[i].pixmap), bitmap, 0, 0,
	     paletteMods[i].width, paletteMods[i].height,
	     (wd - paletteMods[i].width)/2, (ht - paletteMods[i].height)/2);

    paletteModWidgets[i] =
      YCreateWidget("Palette Mod Item", toggleWidgetClass, parent);

    YSetValue(paletteModWidgets[i], XtNbitmap, bitmap);
    YSetValue(paletteModWidgets[i], XtNlabel,    NULL);
    YSetValue(paletteModWidgets[i], XtNfromHoriz, 
	      i == 0 || i == PaletteModCount/2 ? NULL :
	      paletteModWidgets[i-1]);
    YSetValue(paletteModWidgets[i], XtNfromVert,
	      i < PaletteModCount/2 ? NULL :
	      paletteModWidgets[i-PaletteModCount/2]);
  }

  End;
}


void PaletteMenuNotes(Widget w, XtPointer a, XtPointer b)
{
  Begin("PaletteMenuNotes");
  InstallPalette(PaletteNotes);
  YSetValue(paletteShellLocal, XtNtitle, "Notes");
  End;
}


void PaletteMenuRests(Widget w, XtPointer a, XtPointer b)
{
  Begin("PaletteMenuRests");
  InstallPalette(PaletteRests);
  YSetValue(paletteShellLocal, XtNtitle, "Rests");
  End;
}


void PaletteMenuClefs(Widget w, XtPointer a, XtPointer b)
{
  Begin("PaletteMenuClefs");
  InstallPalette(PaletteClefs);
  YSetValue(paletteShellLocal, XtNtitle, "Clefs");
  End;
}


void PaletteMenuKeys(Widget w, XtPointer a, XtPointer b)
{
  Begin("PaletteMenuKeys");
  InstallPalette(PaletteKeys);
  YSetValue(paletteShellLocal, XtNtitle, "Keys");
  End;
}

