/* CMATH.C Modified by Robin Whittle 8 September 1995			*/

					/* 31 bit algorithm for random
					 * number generation.
					 *
					 * xunirand ugen added to make the
					 * set complete.
					 *
					 * All comments formatted like this
					 * are by RW.
					 *
					 * Corrections made to code which
					 * auto incremented the pointer
					 * to the output, when this is not
					 * needed at i or k rate.
					 * See the various ikxxx functions.
					 *
					 * Also, the original lines in 
					 * opcodlst in entry.c for kpow and 
					 * apow caused the inorm parameter 
					 * to be mandatory, rather than 
					 * optional as the doco states.
					 *
					 * Change the input parameter flags
					 * to "kkp" and "akp" respectively 
					 * to make inorm optional with a 
					 * default of 1.
					 */


/*	Math functions for Csound coded by Paris Smaragdis 1994         */
/*	Berklee College of Music Csound development team                */

#include "cs.h"
#include "cmath.h"
#include <math.h>

					/* The original code refered to 
					 * RAND_MAX, but now we use
					 * GRAND_MAX for the maximum
					 * value available from the
					 * raw pseudo random number generator.
					 *
					 * The following macro is used
					 * by later code as the sole source
					 * of random numbers.
					 */

#define	GRAND_MAX  0x7FFFFFFFL 

#define unirand() (float)((double)grand()/(double)GRAND_MAX)

					/* Function prototype for grand()
					 * so it can be used properly in 
					 * the unirand() macro.
					 */
long	grand(void);



void ipow(POW *p)		/*      Power for i-rate */
{
    *p->sr = pow( *p->in, *p->pow);	/*      sophisticated code!     */

					/* Paris, you genius! 
					 */
}

void apow(POW *p)		/*      Power routine for a-rate  */
{
    register long n = ksmps;
    register float *in = p->in, *out = p->sr;

    do {
	*out++ = pow( *in++, *p->pow) / *p->norm;
    } while( --n);
}

void kpow(POW *p)		/*      Power routine for k-rate        */
{
    *p->sr = pow( *p->in, *p->pow) / *p->norm;
}


					/* See below for the new
					 * seedrand function.
					 *
					 * This is driven by the 
					 * ugen seed, which Paris
					 * wrote but was not documented
					 * in the doco I saw - RW.
					 */
/*
 * void seedrand(PRAND *p)
 * {
 *     printf("Seeding with %.3f\n", *p->out);
 *     srand( ( unsigned int)*p->out);
 * }
 */

					/* Uniform random functions.
					 * Paris provided ten exotic
					 * flavors of random generator
					 * and all these now run from a 
					 * global 32 bit source.
					 *
					 * All that is missing is a uniform
					 * generator, similar to rand.
					 *
					 * The uniform ugen and functions are
					 * copies of the linear ones.
					 *
					 * Add appropriate entries in 
					 * opcolst at the end of entry.c, 
					 * as well as the function prototypes
					 * for the functions at the start.
					 *
					 * New ungens:
					 *
					 * ir	iunirand irange
					 *
					 * kr	kunirand krange
					 *
					 * ar	aunirand arange
					 *
					 *
					 * New function prototypes:
					 * 
					 * void auniform(void*)
					 * void ikuniform(void*)
					 */

					/* Uniform random functions      
					 *
					 * Equal probability of number
					 * being between 0 and 1.
					 */
void auniform(PRAND *p)		
{
register long n = ksmps;
register float *out = p->out;
register float arg1 = *p->arg1;

	do *out++ = unirand() * arg1;
	while( --n);
}

void ikuniform(PRAND *p)
{
register float *out = p->out;

	*out = unirand() * *p->arg1;
}


					/*-----------------------------------*/


void alinear(PRAND *p)		/* 	Linear random functions      */
					
					/* Range 0 to 1.
					 *
					 * Linear means that the 
					 * probablity of the number 
					 * turning up diminishes from 
					 * a maximum near 0, to a minimum
					 * near 1.
					 */
{
    register long n = ksmps;
    register float *out = p->out;

    do {
	*out++ = linrand( *p->arg1);
    } while( --n);
}

void iklinear(PRAND *p)
{
    register float *out = p->out;
    *out = linrand( *p->arg1);
}




void atrian(PRAND *p)		/*      Triangle random functions       */

					/* Range -1 to +1.
					 *
					 * Probability of number being
					 * near -1 or +1 is low.
					 * Probability of it being near 0
					 * is high.
					 */	
{
    register long n = ksmps;
    register float *out = p->out;

    do {
	*out++ = trirand( *p->arg1);
    } while( --n);
}

void iktrian(PRAND *p)
{
    register float *out = p->out;
    *out = trirand( *p->arg1);
}


void aexp(PRAND *p)		/*      Exponential random functions    */
{
    register long n = ksmps;
    register float *out = p->out;

    do {
	*out++ = exprand( *p->arg1);
    } while( --n);
}

void ikexp(PRAND *p)
{
    register float *out = p->out;
    *out = exprand( *p->arg1);
}


void abiexp(PRAND *p)		/*      Bilateral exponential random functions  */
{
    register long n = ksmps;
    register float *out = p->out;

    do {
	*out++ = biexprand( *p->arg1);
    } while( --n);
}

void ikbiexp(PRAND *p)
{
    register float *out = p->out;
    *out = biexprand( *p->arg1);
}


void agaus(PRAND *p)		/*      Gaussian random functions       */
{
    register long n = ksmps;
    register float *out = p->out;

    do {
	*out++ = gaussrand( *p->arg1);
    } while( --n);
}

void ikgaus(PRAND *p)
{
    register float *out = p->out;
    *out = gaussrand( *p->arg1);
}


void acauchy(PRAND *p)		/*      Cauchy random functions */
{
    register long n = ksmps;
    register float *out = p->out;

    do {
	*out++ = cauchrand( *p->arg1);
    } while( --n);
}

void ikcauchy(PRAND *p)
{
    register float *out = p->out;
    *out = cauchrand( *p->arg1);
}


void apcauchy(PRAND *p)		/*      Positive Cauchy random functions        */
{
    register long n = ksmps;
    register float *out = p->out;

    do {
	*out++ = pcauchrand( *p->arg1);
    } while( --n);
}

void ikpcauchy(PRAND *p)
{
    register float *out = p->out;
    *out = pcauchrand( *p->arg1);
}


void abeta(PRAND *p)		/*      Beta random functions   */
{
    register long n = ksmps;
    register float *out = p->out;

    do {
	*out++ = betarand( *p->arg1, *p->arg2, *p->arg3);
    } while( --n);
}

void ikbeta(PRAND *p)
{
    register float *out = p->out;
    *out = betarand( *p->arg1, *p->arg2, *p->arg3);
}


void aweib(PRAND *p)		/*      Weibull randon functions        */
{
    register long n = ksmps;
    register float *out = p->out;

    do {
	*out++ = weibrand( *p->arg1, *p->arg2);
    } while( --n);
}

void ikweib(PRAND *p)
{
    register float *out = p->out;
    *out = weibrand( *p->arg1, *p->arg2);
}


void apoiss(PRAND *p)		/*      Poisson random funcions */
{
    register long n = ksmps;
    register float *out = p->out;

    do {
	*out++ = poissrand( *p->arg1);
    } while( --n);
}

void ikpoiss(PRAND *p)
{
    register float *out = p->out;
    *out = poissrand( *p->arg1);
}


/* * * * * * RANDOM NUMBER GENERATORS * * * * * */


					/* See start of file for the new
					 * definition of unirand().
					 */

/* #define unirand()	(float)((double)rand()/(double)RAND_MAX)
 */

					/*************************************/
					
					/* grand()
					 *
					 * A 31 bit global random number 
					 * generator.
					 *
					 * This is intended to give a  
					 * long cycle - longer than the
					 * 32767 cycles available 
					 * from ANSI rand() - which show
					 * up as cyclic fluctuations in noise
					 * when we use them at audio rates.
					 *
					 * We need a global variable for
					 * the state - a 32 bit long integer.
					 *
					 * Initialize it with an auspicious 
					 * number derived from 32 tosses of
					 * a coin.
					 *
					 * Also define the maximum value for
					 * the random number generator - see
					 * start of file.
					 *
					 * The generator produces a 31 bit
					 * positive integer. 
					 */

long	grandstat = 0x16BA2118L;

					/* Random number generator.
					 */
long	grand(void)
{
	grandstat = (grandstat * 1103515245 + 12345) & GRAND_MAX;

	return	grandstat;
}		


void seedrand(PRAND *p)
{
					/* New random seed setting function.
					 *
					 * Paris wrote it to use the same
					 * PRAND data structure as the other
					 * functions. It has only one input
					 * parameter, and no output parameters
					 * so the input is in the first 
					 * variable in PRAND.
					 *
					 * Take this to be a floating point
					 * number with three decimal points.
					 * Display this and multiply it by
					 * 1000 to make these parts of the 
					 * number map into the final 32 bit
					 * seed.
					 * 
					 * This is quite arbitrary, but it 
					 * enables people to use numbers 
					 * between 0.000 and 4294967.295
					 * to get unique starting points.
					 * 
					 * Hash the seed with another 
					 * auspicious number before using
					 * it. This number comes from the 
					 * xor of 32 tosses of two coins -
					 * and rounded to an odd number so
					 * bits 0 to 30 of the original
					 * value affect their respective bits 
					 * of the final 31 bit seed.
					 *
					 * Unique starting points can be
					 * had from any number with three
					 * decimal places of precision from
					 * 0.0 to 2147483.647.
 					 */

	printf("Seeding with %.3f\n", *p->out);

	grandstat = ( (unsigned int)*p->out * 1000) * 0xA7DD150D + 1234567;

	grandstat = grandstat &	GRAND_MAX;
}

					/* See the start of the file for
					 * the #define which creates a 
					 * macro for unirand(), but which 
					 * actually calls grand() and divides
					 * the result by GRAND_MAX.
					 */


					/* End of modifications. 
					 */
					 
					/*************************************/

					/* This was in Paris' code, but
					 * not used by anything.
					 * Leave it here, but for speed, 
					 * my uniform random functions 
					 * operate directly with the 
					 * unirand() macro defined above.
					 */ 
float unifrand(void)
{
    return unirand();
}

float linrand(float range)	/*      linear distribution routine     */
{
    register float r1, r2;

    r1 = unirand();
    r2 = unirand();

	printf("r1 =%f, r2 =%f\n", r1, r2);

    if( r1 > r2)
	r1 = r2;

    return( r1 * range);
}

float trirand(float range)	/*      triangle distribution routine   */
{
    register float r1, r2;

    r1 = unirand();
    r2 = unirand();

    return( (( r1 + r2) - 1) * range);
}
		
float exprand(float l)		/*      exponential distribution routine        */
{
    register float r1;

    if( l < 0) return( 0);

    do {
	r1 = unirand();
    } while( r1 == 0);

    return( -log( r1) * l/10);
}

float biexprand(float l)	/* bilateral exponential distribution routine      */
{
    register float r1;

    if( l < 0) return ( 0);

    do {
	r1 = 2 * unirand();
    } while( r1 == 0 || r1 == 2); 

    if( r1 > 1)     {
	r1 = 2.0 - r1;
	return( -log(r1) * l/10);
    }
    return( log(r1) * l/10);
}

float gaussrand(float s)	/*      gaussian distribution routine   */
{
    register float r1 = 0;
    register int n = 12;
    s /= 3.83;

    do {
	r1 += unirand();
    } while( --n);

    return( s * (r1 - 6));
}

float cauchrand(float a)	/*      cauchy distribution routine     */
{
    register float r1;
    a /= 318.3;

    do {
	r1 = unirand();
    } while( r1 == .5);

    r1 *= pi;
    return( a * tan( r1));
}

float pcauchrand(float a)	/*      positive cauchy distribution routine    */
{
    register float r1;
    a /= 318.3;

    do {
	r1 = unirand();
    } while( r1 == 1);

    r1 *= pi / 2;
    return( a * tan( r1));
}

float betarand(float range, float a, float b) /*      beta distribution routine       */
{
    register float r1, r2, r3;

    if( a < 0 || b < 0 ) return( 0); 

    do {
	do {
	    r1 = unirand();
	} while( r1 == 0);
	
	do {
	    r2 = unirand();
	} while( r2 == 0);
	
	r3 = r1 = pow(r1, 1.0 / a);
	r2 = pow(r2, 1.0 / b);
    } while( (r1 + r2) > 1.0);

    return( (r3 / (r1 +r2)) * range);
}

float weibrand(float s, float t) /*      weibull distribution routine    */
{
    register float r1, r2;

    if( t < 0 ) return( 0);

    do {
	r1 = unirand();
    } while( r1 == 0 || r1 == 1);

    r2 = 1.0 / ( 1.0 - r1);

    return( s * pow( log( r2), ( 1.0 / t)));
}

float poissrand(float l)	/*      poisson distribution routine    */
{
    register float r1, r2, r3;

    if( l < 0 ) return( 0);

    r1 = unirand();
    r2 = exp( -l);
    r3 = 0;

    while( r1 >= r2){
	r3++;
	r1 *= unirand();
    }

    return( r3);
}
