#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include "waveform.h"
#include "format.h"


// ***********************************************
// waveform function definitions
// ***********************************************

waveform::waveform(char *filename)
{
// check for existing wave data
if (noise.data)
  {
  // delete existing waveform
  delete noise.data;
  }

ReadWave(filename);

SelectRange.valid = 0;
}

waveform::waveform(void)
{
if (noise.data)
  delete noise.data;

noise.data = NULL;
noise.size = 0;

SelectRange.valid = 0;
}

waveform::waveform(wavedata sound)
{
// check - is there currently a waveform?
if (noise.data)
  {
  // if so, de-allocate the waveform
  delete noise.data;
  noise.data = NULL;
  }

// set noise to new waveform
noise.data = sound.data;
noise.size = sound.size;
SelectRange.valid = 0;
}

void waveform::ReadWave(char *filename)
// Read the given wave-form from disk
{
FILE *wave;
wavedata *wtemp;
struct stat fileinfo;

if (filename)
  {
  if (!(wave = fopen(filename, "r")))
    {
    fprintf(stderr, "Could not open wave-form\n");
    exit(-1);
    }
  }
else 
  {
  fprintf(stderr, "usage: wavedit <wave-file>\n");
  exit(-1);
  }

if (stat(filename, &fileinfo))
  {
  fprintf(stderr, "Error reading file %s\n", filename);
  fclose(wave);
  exit(-1);
  }

// read the waveform into memory
noise.data = new unsigned char [fileinfo.st_size];
if ((noise.size = fread(noise.data, 1, fileinfo.st_size, wave))
    < fileinfo.st_size)
  {
  fprintf(stderr, "Error reading file %s into memory\n", filename);
  fclose(wave);
  exit(-1);
  }
}


void waveform::WriteWave(char *filename)
{
int outfile; // file descriptor

if ((outfile = open(filename, O_WRONLY | O_CREAT)) == -1)
  {
  fprintf(stderr, "Error opening file for output\n");
  exit(-1);
  }

if (write(outfile, noise.data, noise.size) == -1)
  {
  fprintf(stderr, "Error writing data to file\n");
  close(outfile);
  exit(-1);
  }

close(outfile);
}


void waveform::Select(long start, long end)
{
if ((start >= end) || (start < 0) || (end < 0) || (end > noise.size))
  SelectRange.valid = 0;
else
  {
  SelectRange.startindex = start;
  SelectRange.endindex = end;
  SelectRange.valid = 1;
  }
}


// Clipboard management functions
wavedata waveform::Cut(void)
{
// divide the wave into pieces
wavedata front, middle;
unsigned char *temp;
int index = SelectRange.startindex;

if (SelectRange.valid)
  {
  // Copy the selection into middle
  middle.data = new unsigned char[SelectRange.endindex-SelectRange.startindex];
  middle.size = SelectRange.endindex - SelectRange.startindex;

  // copy the sound data
  temp = middle.data;
  while (index < SelectRange.endindex)
    *(temp++) = noise.data[index++];

  front.data = new unsigned char[noise.size - middle.size];
  front.size = noise.size - middle.size;

  // copy the first half of the waveform
  temp = front.data;
  index = 0;
  while (index < SelectRange.startindex)
    *(temp++) = noise.data[index++];

  // copy the end of the waveform
  index = SelectRange.endindex;
  while (index < noise.size)
    *(temp++) = noise.data[index++];

  // replace the original waveform with the remaining segment
  delete noise.data;
  noise.data = front.data;
  noise.size = front.size;

  // invalidate the selection
  SelectRange.valid = 0;
  }
else
  {
  middle.data = NULL;
  middle.size = 0;
  }

return middle;
}

wavedata waveform::Copy(void)
{
unsigned char *temp;
unsigned long index = SelectRange.startindex;
wavedata sound;

if (SelectRange.valid)
  {
  // set info for the new block of memory
  sound.data = new unsigned char[SelectRange.endindex-SelectRange.startindex];
  sound.size = SelectRange.endindex - SelectRange.startindex;

  // copy the sound data
  temp = sound.data;
  while (index < SelectRange.endindex)
    *(temp++) = noise.data[index++];
  }
else
  {
  sound.data = NULL;
  sound.size = 0;
  }

return sound;
}

void waveform::Paste(wavedata sound, unsigned long insert)
{
// the composite waveform
wavedata comp;
unsigned char *temp;
int index = 0;

comp.data = new unsigned char[noise.size + sound.size];
comp.size = noise.size + sound.size;

// copy the first segment of the waveform;
temp = comp.data;
while (index < insert)
  *(temp++) = noise.data[index++];

// copy the insert waveform into the new waveform
index = 0;
while (index < sound.size)
  *(temp++) = sound.data[index++];

// copy the remaining portion of the original waveform
index = insert;
while (index < noise.size)
  *(temp++) = noise.data[index++];

// housekeeping
delete noise.data;
noise.data = comp.data;
noise.size = comp.size;

Select(insert, insert+sound.size);
}


// These should be shorter, but aren't
unsigned long waveform::select_start(void)
{
if (SelectRange.valid)
  return SelectRange.startindex;
else
  return 0;
}

unsigned long waveform::select_end(void)
{
if (SelectRange.valid)
  return SelectRange.endindex;
else
  return 0;
}


// ******************************************************
// ms_wav function definitions
// ******************************************************

ms_wav::ms_wav(char *filename)
    : waveform(filename),
      frequency(11025),
      headers((WaveHeader *) noise.data),
      is_wav(0)
{
wavedata temp;

// check to see if valid wave file
if ((headers->main_chunk != RIFF_ID) || (headers->chunk_type != WAVE_ID)
    || (headers->sub_chunk != FMT))
  // If not a wave, do nothing
  return;

if ((headers->bit_p_spl != 8) || (headers->modus != WAVE_MONO))
  {
  fprintf(stderr, "wavedit can only edit 8-bit mono samples...sorry\n");
  exit(0);
  }

// it's a .wav file, as far as we care
is_wav = 1;

// set the playback frequency and strip the headers
frequency = headers->byte_p_sec;
Select(0,sizeof(WaveHeader));
temp = Cut();
headers = (WaveHeader *) temp.data;
}

ms_wav::ms_wav(void)
    : frequency(11025),
      headers(NULL),
      is_wav(0)
{
}

ms_wav::ms_wav(wavedata sound, WaveHeader *info)
    : waveform(sound),
      is_wav(1)
{
if (headers)
  delete headers;

headers = info;
}


void ms_wav::WriteWave(char *filename)
{
int outfile = open(filename, O_WRONLY | O_CREAT); // file descriptor

if (outfile == -1)
  {
  fprintf(stderr, "Error opening file for output\n");
  exit(-1);
  }

// is it a .wav file, or not?
if (is_wav)
  {
  // set the appropriate headers that I read from someplace else
  headers->length = noise.size;
  headers->data_length = noise.size;
  if (write(outfile, headers, sizeof(WaveHeader)) == -1)
    {
    fprintf(stderr, "Error writing headers for .wav file\n");
    close(outfile);
    exit(-1);
    }
  }

if (write(outfile, noise.data, noise.size) == -1)
  {
  fprintf(stderr, "Error writing file data for .wav file\n");
  close(outfile);
  exit(-1);
  }

close(outfile);
}
