#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "lxt.h"

extern Panelitem_dispatch panelitem_proctab[];

void
panel_draw(p)
/*
   Internal function.
   Plots the panel's contents onto backing store.
*/
Panel *p;
{
	Panel_item *pi;

	XFillRectangle(p->xp_dpy, p->xp_ppm, p->xp_cgc, 0, 0, p->xp_mpw, p->xp_mph);
	for (pi= p->xp_items; pi != (Panel_item *) NULL; pi= pi->xpi_next)
		(panelitem_proctab[pi->xpi_type].xpi_drw_proc)(p, pi);
}

void
panel_display(p)
/*
   Internal function.
   Plots a Panel on the display.
*/
Panel *p;
{
	XCopyArea(p->xp_dpy, p->xp_ppm, p->xp_pwin, p->xp_gc, p->xp_vx, p->xp_vy, p->xp_apw, p->xp_aph, 0, 0);
	XFlush(p->xp_dpy);
	return;
}

int
panel_redisplay(p)
/*
   User-callable.
   Forces the redisplay of a panel.
*/
Panel *p;
{
	XWindowAttributes xwa;
	void panel_unmapsubwins();
	void panel_draw(), panel_display();
	void panelvscroll_draw(), panelhscroll_draw();

	if (p == (Panel *) NULL) {
		(void) fprintf(stderr, "panel_redisplay: null panel\n");
		return(LX_ERROR);
	}
	if (p->xp_magic != LX_PANEL) {
		(void) fprintf(stderr, "panel_redisplay: object is not a panel\n");
		return(LX_ERROR);
	}

	XGetWindowAttributes(p->xp_dpy, p->xp_win, &xwa);
	switch (xwa.map_state) {
	case IsUnviewable:
	case IsViewable:
		break;
	case IsUnmapped:
		(void) fprintf(stderr, "panel_redisplay: panel not mapped\n");
		return(LX_ERROR);
		break;
	default:
		break;
	}

	panel_unmapsubwins(p);
	(void) panel_resize(p);
	(void) panel_config(p);
	p->xp_flags|= LXP_CONFIGDONE;
	XMapWindow(p->xp_dpy, p->xp_pwin);
	panel_draw(p);
	panel_display(p);

	if (p->xp_flags & LXP_VSCROLLVIS) {
		XMapWindow(p->xp_dpy, p->xp_vswin);
		panelvscroll_draw(p);
	}
	if (p->xp_flags & LXP_HSCROLLVIS) {
		XMapWindow(p->xp_dpy, p->xp_hswin);
		panelhscroll_draw(p);
	}
	return(LX_SUCCESS);
}

void
panelvscroll_draw(p)
/*
   Internal function.
   Plots the panel's vertical scrollbar on the display.
*/
Panel *p;
{
	int w, h, bl;
	void panelvscroll_drawbubble();

	w= p->xp_vscroll->xs_barwidth+1;
	if (p->xp_py == p->xp_ibw)
		h= p->xp_aph;
	else
		h= p->xp_aph+1;

	XClearWindow(p->xp_dpy, p->xp_vswin);
	XFillRectangle(p->xp_dpy, p->xp_vspm, p->xp_cgc, 0, 0, w, h);
	if (p->xp_aph < p->xp_h-(2*p->xp_ibw))
		XDrawLine(p->xp_dpy, p->xp_vspm, p->xp_gc, 0, 0, w-1, 0);
	XDrawLine(p->xp_dpy, p->xp_vspm, p->xp_gc, w-1, 0, w-1, h-1);
	if (p->xp_vscroll->xs_flags & LXS_BUTTONVIS) {
		bl= p->xp_vscroll->xs_buttonlen;
		if (p->xp_aph < p->xp_h-(2*p->xp_ibw))
			XDrawLine(p->xp_dpy, p->xp_vspm, p->xp_gc, 0, bl+1, w-1, bl+1);
		else
			XDrawLine(p->xp_dpy, p->xp_vspm, p->xp_gc, 0, bl, w-1, bl);
		XDrawLine(p->xp_dpy, p->xp_vspm, p->xp_gc, 0, h-1-bl, w-1, h-1-bl);
	}
	panelvscroll_drawbubble(p);
	XCopyArea(p->xp_dpy, p->xp_vspm, p->xp_vswin, p->xp_gc, 0, 0, w, h, 0, 0);
}

void
panelvscroll_drawbubble(p)
/*
   Internal function.
   Plots the panel's vertical scrollbar bubble on the display.
*/
Panel *p;
{
	Scrollbar *s;
	float f;

	s= p->xp_vscroll;
	f= ((float) p->xp_vy)/((float) p->xp_mph);
	s->xs_bubblemin= ((int) (f*s->xs_barlen))+s->xs_barstart;
	if (s->xs_bubblemin >= s->xs_barstart+s->xs_barlen)
		s->xs_bubblemin= s->xs_barstart+s->xs_barlen-1;
	f= ((float) p->xp_aph)/((float) p->xp_mph);
	if (p->xp_vy == p->xp_mph-p->xp_aph)
		s->xs_bubblelen= s->xs_barlen-(s->xs_bubblemin-s->xs_barstart);
	else
		s->xs_bubblelen= (int) (f*s->xs_barlen);
	if (s->xs_bubblemin+s->xs_bubblelen >= s->xs_barstart+s->xs_barlen)
		s->xs_bubblelen= s->xs_barstart+s->xs_barlen-s->xs_bubblemin;
	if (2*s->xs_bubblemargin > s->xs_barwidth)
		return;
	XFillRectangle(p->xp_dpy, p->xp_vspm, p->xp_sgc, s->xs_bubblemargin, s->xs_bubblemin, s->xs_barwidth-(2*s->xs_bubblemargin), s->xs_bubblelen);
}

void
panelhscroll_draw(p)
/*
   Internal function.
   Plots the panel's horizontal scrollbar on the display.
*/
Panel *p;
{
	int w, h, bl;
	void panelhscroll_drawbubble();

	if (p->xp_px == p->xp_ibw)
		w= p->xp_apw;
	else
		w= p->xp_apw+1;
	h= p->xp_hscroll->xs_barwidth+1;

	XClearWindow(p->xp_dpy, p->xp_hswin);
	XFillRectangle(p->xp_dpy, p->xp_hspm, p->xp_cgc, 0, 0, w, h);
	if (p->xp_apw < p->xp_w-(2*p->xp_ibw))
		XDrawLine(p->xp_dpy, p->xp_hspm, p->xp_gc, 0, 0, 0, h-1);
	XDrawLine(p->xp_dpy, p->xp_hspm, p->xp_gc, 0, h-1, w-1, h-1);
	if (p->xp_hscroll->xs_flags & LXS_BUTTONVIS) {
		bl= p->xp_hscroll->xs_buttonlen;
		if (p->xp_apw < p->xp_w-(2*p->xp_ibw))
			XDrawLine(p->xp_dpy, p->xp_hspm, p->xp_gc, bl+1, 0, bl+1, h-1);
		else
			XDrawLine(p->xp_dpy, p->xp_hspm, p->xp_gc, bl, 0, bl, h-1);
		XDrawLine(p->xp_dpy, p->xp_hspm, p->xp_gc, w-1-bl, 0, w-1-bl, h-1);
	}
	panelhscroll_drawbubble(p);
	XCopyArea(p->xp_dpy, p->xp_hspm, p->xp_hswin, p->xp_gc, 0, 0, w, h, 0, 0);
}

void
panelhscroll_drawbubble(p)
/*
   Internal function.
   Plots the panel's horizontal scrollbar bubble on the display.
*/
Panel *p;
{
	Scrollbar *s;
	float f;

	s= p->xp_hscroll;
	f= ((float) p->xp_vx)/((float) p->xp_mpw);
	s->xs_bubblemin= ((int) (f*s->xs_barlen))+s->xs_barstart;
	if (s->xs_bubblemin >= s->xs_barstart+s->xs_barlen)
		s->xs_bubblemin= s->xs_barstart+s->xs_barlen-1;
	f= ((float) p->xp_apw)/((float) p->xp_mpw);
	if (p->xp_vx == p->xp_mpw-p->xp_apw)
		s->xs_bubblelen= s->xs_barlen-(s->xs_bubblemin-s->xs_barstart);
	else
		s->xs_bubblelen= (int) (f*s->xs_barlen);
	if (s->xs_bubblemin+s->xs_bubblelen >= s->xs_barstart+s->xs_barlen)
		s->xs_bubblelen= s->xs_barstart+s->xs_barlen-s->xs_bubblemin;
	if (2*s->xs_bubblemargin > s->xs_barwidth)
		return;
	XFillRectangle(p->xp_dpy, p->xp_hspm, p->xp_sgc, s->xs_bubblemin, s->xs_bubblemargin, s->xs_bubblelen, s->xs_barwidth-(2*s->xs_bubblemargin));
}

void
panelitem_draw(p, pi)
/*
   Internal function.
   Plots a Panel_item onto the backing store.
*/
Panel *p;
Panel_item *pi;
{
	Panel_item *xpi;

	if (p == (Panel *) NULL) {
		(void) fprintf(stderr, "panelitem_draw: null panel\n");
		return;
	}
	if (pi == (Panel_item *) NULL) {
		(void) fprintf(stderr, "panelitem_draw: null panel item\n");
		return;
	}
	for (xpi= p->xp_items; xpi != (Panel_item *) NULL; xpi= xpi->xpi_next) {
		if (xpi == pi)
			break;
	}
	if (xpi == (Panel_item *) NULL) {
		(void) fprintf(stderr, "panelitem_draw: item not found within panel\n");
		return;
	}

	(panelitem_proctab[pi->xpi_type].xpi_drw_proc)(p, pi);
	return;
}

void
panelitem_display(p, pi)
/*
   Internal function.
   Plots a Panel_item on the display.
*/
Panel *p;
Panel_item *pi;
{
	Panel_item *xpi;

	if (p == (Panel *) NULL) {
		(void) fprintf(stderr, "panelitem_display: null panel\n");
		return;
	}
	if (pi == (Panel_item *) NULL) {
		(void) fprintf(stderr, "panelitem_display: null panel item\n");
		return;
	}
	for (xpi= p->xp_items; xpi != (Panel_item *) NULL; xpi= xpi->xpi_next) {
		if (xpi == pi)
			break;
	}
	if (xpi == (Panel_item *) NULL) {
		(void) fprintf(stderr, "panelitem_display: item not found within panel\n");
		return;
	}

	(panelitem_proctab[pi->xpi_type].xpi_dsp_proc)(p, pi);
	return;
}

int
panelitem_redisplay(p, pi)
/*
   User-callable function.
   Redraws a Panel_item on the display.
*/
Panel *p;
Panel_item *pi;
{
	Panel_item *xpi;

	if (p == (Panel *) NULL) {
		(void) fprintf(stderr, "panelitem_redisplay: null panel\n");
		return(LX_ERROR);
	}
	if (p->xp_magic != LX_PANEL) {
		(void) fprintf(stderr, "panelitem_redisplay: object is not a panel\n");
		return(LX_ERROR);
	}
	if (pi == (Panel_item *) NULL) {
		(void) fprintf(stderr, "panelitem_redisplay: null panel item\n");
		return(LX_ERROR);
	}
	if (pi->xpi_magic != LX_PANELITEM) {
		(void) fprintf(stderr, "panelitem_redisplay: object is not a panel item\n");
		return(LX_ERROR);
	}

	for (xpi= p->xp_items; xpi != (Panel_item *) NULL; xpi= xpi->xpi_next) {
		if (xpi == pi)
			break;
	}
	if (xpi == (Panel_item *) NULL) {
		(void) fprintf(stderr, "panelitem_redisplay: item not found within panel\n");
		return(LX_ERROR);
	}
	if (p->xp_ppm == (Pixmap) None)
		return(LX_ERROR);

	(panelitem_proctab[pi->xpi_type].xpi_drw_proc)(p, pi);
	(panelitem_proctab[pi->xpi_type].xpi_dsp_proc)(p, pi);

	return(LX_SUCCESS);
}

void
xpi_std_dsp_proc(p, pi)
/*
   Internal function.
   Generic routine to plot a Panel_item on the display.
*/
Panel *p;
Panel_item *pi;
{
	XCopyArea(p->xp_dpy, p->xp_ppm, p->xp_pwin, pi->xpi_gc, pi->xpi_x, pi->xpi_y, pi->xpi_w, pi->xpi_h, pi->xpi_x-p->xp_vx, pi->xpi_y-p->xp_vy);
	XFlush(p->xp_dpy);
	return;
}
