#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <ctype.h>
#include <math.h>

#include "lxt.h"

extern Textsw *xt_textsws;
extern Void *lxt_selsrc;

boolean xt_usrresize= FALSE;
int xt_confx, xt_confy;
int xt_confw, xt_confh;

textsw_event(evt)
/*
   Internal routine.
   Processes events in the text edit window.
*/
XEvent *evt;
{
	Textsw *t;
	XAnyEvent *xa;
	void textsw_win_evt(), textsw_twin_evt();

	xa= (XAnyEvent *) evt;

	for (t = xt_textsws; t != (Textsw *) NULL; t = t->xt_next) {
		if (xa->window == t->xt_win) {
			textsw_win_evt(t, evt);
			return(TRUE);
		}
		if (xa->window == t->xt_twin) {
			textsw_twin_evt(t, evt);
			return(TRUE);
		}
		else if (xa->window == t->xt_vswin) {
			textsw_vswin_evt(t, evt);
			return(TRUE);
		}
		else if (xa->window == t->xt_hswin) {
			textsw_hswin_evt(t, evt);
			return(TRUE);
		}
	}

	return(FALSE);
}

void
textsw_win_evt(t, evt)
Textsw *t;
XEvent *evt;
{
	XConfigureEvent *cf;
	XExposeEvent *ex;
	int textsw_config();
	void textsw_draw(), textsw_display();
	void textsw_drawscrolls();
	void textsw_unmapsubwins();

	switch (evt->type) {
	case MapNotify:
		t->xt_flags|= LXT_FRAMEMAPPED;
		(void) textsw_config(t);
		XMapWindow(t->xt_dpy, t->xt_twin);
		textsw_draw(t);
		textsw_display(t);
		XMapWindow(t->xt_dpy, t->xt_vswin);
		XMapWindow(t->xt_dpy, t->xt_hswin);
		textsw_drawscrolls(t);
		break;

	case ConfigureNotify:
		if (!(t->xt_flags & LXT_FRAMEMAPPED))
			return;

		/* make sure a resize has actually occurred */
		cf= (XConfigureEvent *) evt;
		if ((t->xt_w == cf->width) && (t->xt_h == cf->height)) {
			return;
		}

		xt_usrresize= TRUE;
		xt_confx= cf->x;
		xt_confy= cf->y;
		xt_confw= cf->width;
		xt_confh= cf->height;
#ifdef PRE_R3
		/* this causes a server crash under X.V11R3/SunOS3.4! */
		XUnmapSubwindows(t->xt_dpy, t->xt_win);
#else
		textsw_unmapsubwins(t);
#endif PRE_R3
		(void) textsw_config(t);
		xt_usrresize= FALSE;
		XMapWindow(t->xt_dpy, t->xt_twin);
		textsw_draw(t);
		textsw_display(t);
		XMapWindow(t->xt_dpy, t->xt_vswin);
		XMapWindow(t->xt_dpy, t->xt_hswin);
		textsw_drawscrolls(t);
		break;

	case UnmapNotify:
		t->xt_flags&= ~LXT_FRAMEMAPPED;
		break;

	case Expose:
		if (!(t->xt_flags & LXT_FRAMEMAPPED))
			return;

		ex= (XExposeEvent *) evt;
		textsw_fillrect(t->xt_dpy, t->xt_win, t->xt_gc, ex->x, ex->y, ex->width, ex->height);
		textsw_fillrect(t->xt_dpy, t->xt_win, t->xt_cgc, t->xt_ibw, t->xt_ibw, t->xt_vscroll->xs_barwidth+1, t->xt_hscroll->xs_barwidth+1);
		break;

	default:
		break;
	}
}

void
textsw_twin_evt(t, evt)
Textsw *t;
XEvent *evt;
{
	void textsw_bp_proc();
	void textsw_kp_proc();
	void textsw_pm_proc();
	void textsw_sc_proc();
	void textsw_sr_proc();
	void textsw_sn_proc();

	switch (evt->type) {
	case ButtonPress:
		textsw_bp_proc(t, (XButtonPressedEvent *) evt);
		break;
	case MotionNotify:
		textsw_pm_proc(t, (XPointerMovedEvent *) evt);
		break;
	case KeyPress:
		if (t->xt_drawcursor == FALSE) {
			XBell(t->xt_dpy, 10);
			break;
		}
		textsw_kp_proc(t, (XKeyEvent *) evt);
		break;
	case SelectionClear:
		textsw_sc_proc(t, (XSelectionClearEvent *) evt);
		break;
	case SelectionRequest:
		textsw_sr_proc(t, (XSelectionRequestEvent *) evt);
		break;
	case SelectionNotify:
		textsw_sn_proc(t, (XSelectionEvent *) evt);
		break;
	case Expose:
		textsw_display(t);
		break;
	}
}

void
textsw_pm_proc(t, evt)
Textsw *t;
XPointerMovedEvent *evt;
{
	void textsw_hltproc();

	if (!t->xt_enablehlt)
		return;

	t->xt_hltcursorx= t->xt_cursorpx;
	t->xt_hltcursory= t->xt_cursorpy;

	t->xt_hltstartx= t->xt_cursorpx;
	t->xt_hltstarty= t->xt_cursorpy;
	t->xt_hltendx= evt->x;
	t->xt_hltendy= evt->y;

	/* note the origin of the window when the highlight begins */
	t->xt_hltorigin.x= t->xt_leftpix;
	t->xt_hltorigin.y= t->xt_topline*t->xt_cheight;

	/* process first mouse drag and then call procedure
	   to loop while mouse button is held down */
	if (textsw_sameln(t, t->xt_hltendy, t->xt_hltstarty))
		textsw_hltsameline(t, evt);
	else
		textsw_hltdifflines(t, evt);

	textsw_hltproc(t);
	t->xt_enablehlt= FALSE;
}

void
textsw_kp_proc(t, evt)
Textsw *t;
XKeyEvent *evt;
{
	char buffer[80];
	char c, a;
	int i, l;
	void lxt_clearsel();

	/* check that the user is allowed input */
	if (!t->xt_allowinput) {
		XBell(t->xt_dpy, 10);
		return;
	}

	
	bzero(buffer, 80);
	XLookupString(evt, buffer, 9*sizeof(char), (KeySym *) NULL, (XComposeStatus *) NULL);

	l= strlen(buffer);


	for (i= 0; i < l; i++) {
		c= buffer[i];

		
		/* check that user wants to process this character */
		if (t->xt_charproc != NULL)
			if (!(t->xt_charproc)(t, c))
				continue;

		if (c >= ' ' && c <= '~')
			a= 'a';
		else if (c == '\t')
			a= 'a';
		else if (c == '\r') {
			a= '\n';
			c= '\n';
		}
		else
			a= c;

		switch (a) {
		case 'a':
			if (t->xt_drawhlt) {
				textsw_drawhlt(t, TRUE, (boolean *) NULL);
				t->xt_drawhlt= FALSE;
			}
			if (t == (Textsw *) lxt_selsrc)
				lxt_clearsel(FALSE);
			textsw_insprint(t, c);
			t->xt_textedited= TRUE;
			break;
		case '\b':
		case 0177:
			if (t->xt_drawhlt) {
				textsw_drawhlt(t, TRUE, (boolean *) NULL);
				t->xt_drawhlt= FALSE;
			}
			if (t == (Textsw *) lxt_selsrc)
				lxt_clearsel(FALSE);
			textsw_delchar(t);
			t->xt_textedited= TRUE;
			break;
		case '\n':
			if (t->xt_drawhlt) {
				textsw_drawhlt(t, TRUE, (boolean *) NULL);
				t->xt_drawhlt= FALSE;
			}
			if (t == (Textsw *) lxt_selsrc)
				lxt_clearsel(FALSE);
			textsw_insnewline(t);
			t->xt_textedited= TRUE;
			break;
		default:
			break;
		}

		/* xt_test(t, c); */
	}
}

void
textsw_bp_proc(t, evt)
Textsw *t;
XButtonPressedEvent *evt;
{
	int line_num, char_num;
	int row, col, col2, save_x, save_y, save_flag;
	boolean textsw_compute_cursor_pos();
	void lxt_clearsel();

	switch (evt->button) {
	case Button1:

		/* undraw highlight if necessary */
		if (t->xt_drawhlt) {
			textsw_drawhlt(t, TRUE, (boolean *) NULL);
			t->xt_drawhlt= FALSE;
		}

		/* clear selection if in this window */
		if (t == (Textsw *) lxt_selsrc)
			lxt_clearsel(FALSE);

		if (textsw_get_cursor_pos(evt, t, &line_num, &char_num)) {
			if (t->xt_drawcursor) {
				textsw_drawcaret(t);
				t->xt_drawcursor= FALSE;
			}

			t->xt_irow= line_num;
			t->xt_icol= char_num;
			t->xt_ichar= t->xt_linestarts[t->xt_irow] + t->xt_icol;

			/* draw new caret */
			textsw_drawcaret(t);
			t->xt_drawcursor= TRUE;
			t->xt_enablehlt= TRUE;

		}
		break;

	case Button2:

		/* clear selection if any */
		if (t == (Textsw *) lxt_selsrc)
			lxt_clearsel(FALSE);

		save_x= t->xt_cursorpx;
		save_y= t->xt_cursorpy;
		save_flag= t->xt_drawcursor;
		t->xt_cursorpx= evt->x;
		t->xt_cursorpy= evt->y;

		t->xt_drawcursor= TRUE;
		(void) textsw_compute_cursor_pos(t, &row, &col, &col2);

		t->xt_cursorpx= save_x;
		t->xt_cursorpy= save_y;
		t->xt_drawcursor= save_flag;
		(void) textsw_set(t, LXT_DRAWHIGHLIGHT, FALSE, 
			LXT_HIGHLIGHTENDLINE, row, 
			LXT_HIGHLIGHTENDCOLUMN, col2, 
			LXT_DRAWHIGHLIGHT, TRUE,
			LXT_NULL);
		break;

	case Button3:
		menu_show(t->xt_menu, evt);
		break;
	}
}

int
textsw_destroy(t)
/*
   User-callable routine.
   Destroys a textsw.
*/
Textsw *t;
{
	Textsw *u;

	if (t == (Textsw *) NULL) {
		(void) fprintf(stderr, "textsw_destroy: null textsw\n");
		return(LX_ERROR);
	}
	if (t->xt_magic != LX_TEXTSW) {
		(void) fprintf(stderr, "textsw_destroy: object is not a textsw\n");
		return(LX_ERROR);
	}

	if (xt_textsws == t) {
		xt_textsws= t->xt_next;
		textsw_free(t);
	}
	else {
		for (u= xt_textsws; u != (Textsw *) NULL; u= u->xt_next) {
			if (u->xt_next == t) {
				u->xt_next= t->xt_next;
				textsw_free(t);
			}
		}
	}

	return(LX_SUCCESS);
}

textsw_free(t)
Textsw *t;
{
	Menu_itemptr *mip, *pmip;
	Menu *pm;
	void lxt_clearsel();

	XUnmapWindow(t->xt_dpy, t->xt_win);

	if (t == (Textsw *) lxt_selsrc)
		lxt_clearsel(FALSE);

	XDestroySubwindows(t->xt_dpy, t->xt_win);

	if (t->xt_name != (char *) NULL)
		cfree((char *) t->xt_name);
	if (t->xt_fontnm != (char *) NULL)
		cfree((char *) t->xt_fontnm);
	if (t->xt_text != (char *) NULL)
		free(t->xt_text);
	if (t->xt_linestarts != (int *) NULL)
		free((char *) t->xt_linestarts);

	if (t->xt_vscroll != (Scrollbar *) NULL)
		cfree((char *) t->xt_vscroll);
	if (t->xt_hscroll != (Scrollbar *) NULL)
		cfree((char *) t->xt_hscroll);

	if (t->xt_stipplepm != (Pixmap) None)
		XFreePixmap(t->xt_dpy, t->xt_stipplepm);
	if (t->xt_tpm != (Pixmap) None)
		XFreePixmap(t->xt_dpy, t->xt_tpm);

	XFreeGC(t->xt_dpy, t->xt_gc);
	XFreeGC(t->xt_dpy, t->xt_cgc);
	XFreeGC(t->xt_dpy, t->xt_sgc);
	XFreeGC(t->xt_dpy, t->xt_igc);
	XFreeGC(t->xt_dpy, t->xt_xgc);

	/* destroy menu and its items */
	if (t->xt_menu != (Menu *) NULL) {
		for (mip= t->xt_menu->xm_items; mip != (Menu_itemptr *) NULL; mip= mip->xmip_next) {
			if ((pm= (Menu *) mip->xmip_item->xmi_pullright) != (Menu *) NULL) {
				for (pmip= pm->xm_items; pmip != (Menu_itemptr *) NULL; pmip= pmip->xmip_next)
					(void) menuitem_destroy(pmip->xmip_item);
				(void) menu_destroy(pm);
			}
			(void) menuitem_destroy(mip->xmip_item);
		}
		(void) menu_destroy(t->xt_menu);
	}

	cfree((char *) t);
}

textsw_longestline(t)
/*
   Internal function.
   Find the number and length of the longest line.
*/
Textsw *t;
{
	int i, j, k, l;
	int length, new_length, map[512];
	char s[512];

	t->xt_maxlinenum= 0;
	t->xt_maxlinelen= 0;
	for (i= 0; i < t->xt_nbuflines; i++) {
		j= t->xt_linestarts[i];
		k= t->xt_linestarts[i + 1];
		length= k-j; /*bug fix*/

		textsw_expand_string(&(t->xt_text[j]), s, map, length, &new_length);
		l= XTextWidth(t->xt_font, s, new_length);

		if (t->xt_maxlinelen < l) {
			t->xt_maxlinelen= l;
			t->xt_maxlinenum= i;
		}
			
	}
}

textsw_config(t)
/*
   Internal function.
   Resize of the text window.
*/
Textsw *t;
{
	XWindowAttributes xwa;
	Scrollbar *vs, *hs;
	int depth, scr;
	void lxt_clearsel();
	void textsw_drawscrolls();

	/* make sure that a resize has occurred	*/
	if (xt_usrresize) {
		t->xt_w= xt_confw;
		t->xt_h= xt_confh;
	}
	else {
		XGetWindowAttributes(t->xt_dpy, t->xt_win, &xwa);
		if ((xwa.width == t->xt_w) && (xwa.height == t->xt_h))
			return(LX_SUCCESS);
		t->xt_w= xwa.width;
		t->xt_h= xwa.height;
	}

	vs= t->xt_vscroll;
	hs= t->xt_hscroll;
	t->xt_atw= t->xt_w-(2*t->xt_ibw)-vs->xs_barwidth-1;
	t->xt_ath= t->xt_h-(2*t->xt_ibw)-vs->xs_barwidth-1;
	if (t->xt_atw < 1)
		t->xt_atw= 1;
	if (t->xt_ath < 1)
		t->xt_ath= 1;

	/* if the user resizes the window, the viewable
	   text starts at the uppermost left */
	t->xt_topline= 0;
	t->xt_leftpix= 0;
	t->xt_wcw= (int) t->xt_w/t->xt_cwidth;
	t->xt_wch= (int) t->xt_h/t->xt_cheight;
	textsw_longestline(t);
	t->xt_drawcursor= FALSE;
	t->xt_drawhlt= FALSE;
	if (t == (Textsw *) lxt_selsrc)
		lxt_clearsel(FALSE);
	t->xt_hltorigin.x= 0;
	t->xt_hltorigin.y= 0;


	t->xt_vscrolllen= t->xt_ath+1-(2*vs->xs_buttonlen);
	t->xt_hscrolllen= t->xt_atw+1-(2*hs->xs_buttonlen);

	/* move and resize all subwindows */
	XMoveWindow(t->xt_dpy, t->xt_twin, t->xt_ibw+vs->xs_barwidth+1, t->xt_ibw+hs->xs_barwidth+1);
	XResizeWindow(t->xt_dpy, t->xt_twin, t->xt_atw, t->xt_ath);

	XMoveWindow(t->xt_dpy, t->xt_hswin, t->xt_ibw+vs->xs_barwidth, t->xt_ibw);
	XResizeWindow(t->xt_dpy, t->xt_hswin, t->xt_atw+1, hs->xs_barwidth+1);

	XMoveWindow(t->xt_dpy, t->xt_vswin, t->xt_ibw, t->xt_ibw+hs->xs_barwidth);
	XResizeWindow(t->xt_dpy, t->xt_vswin, vs->xs_barwidth+1, t->xt_ath+1);

	/* free pixmap and reallocate a new one	*/
	scr= DefaultScreen(t->xt_dpy);
	depth= DefaultDepth(t->xt_dpy, scr);
	if (t->xt_tpm != (Pixmap) None)
		XFreePixmap(t->xt_dpy, t->xt_tpm);
	t->xt_tpm= XCreatePixmap(t->xt_dpy, DefaultRootWindow(t->xt_dpy), t->xt_atw, t->xt_ath, depth);
	return(LX_SUCCESS);
}

void
textsw_unmapsubwins(t)
/*
   Internal function.
   Calls to XUnmapSubwindows(t->xt_dpy, t->xt_win) cause
   server crashes under X.V11R3/SunOS3.4, hence this hack.
*/
Textsw *t;
{
	XWindowAttributes xwa;

	XGetWindowAttributes(t->xt_dpy, t->xt_twin, &xwa);
	switch (xwa.map_state) {
	case IsUnviewable:
	case IsViewable:
		XUnmapWindow(t->xt_dpy, t->xt_twin);
		break;
	case IsUnmapped:
		break;
	default:
		break;
	}

	XGetWindowAttributes(t->xt_dpy, t->xt_hswin, &xwa);
	switch (xwa.map_state) {
	case IsUnviewable:
	case IsViewable:
		XUnmapWindow(t->xt_dpy, t->xt_hswin);
		break;
	case IsUnmapped:
		break;
	default:
		break;
	}

	XGetWindowAttributes(t->xt_dpy, t->xt_vswin, &xwa);
	switch (xwa.map_state) {
	case IsUnviewable:
	case IsViewable:
		XUnmapWindow(t->xt_dpy, t->xt_vswin);
		break;
	case IsUnmapped:
		break;
	default:
		break;
	}
}

textsw_insprint(t, c)
Textsw *t;
char c;
{
	int i, r, ch;
	int map[512], length, new_length;
	int test_width;
	char *tx;
	char str[512];

	r= t->xt_irow;
	ch= t->xt_ichar;
	tx= t->xt_text;

	/* undraw the caret */
	textsw_drawcaret(t);
	t->xt_drawcursor= FALSE;

	if (t->xt_textbufsz < t->xt_nbufchars + LXT_NSPHOLDERS) {
		t->xt_text= realloc(t->xt_text, (unsigned) ((t->xt_textbufsz+LXT_TEXTBUFINC)*sizeof(char)));
		t->xt_textbufsz+= LXT_TEXTBUFINC;
	}

	if (ch > t->xt_nbufchars)
		ch= t->xt_nbufchars;
	if (ch < 0)
		ch= 0;

	if ((t->xt_nbufchars != 0) && (ch == t->xt_nbufchars)) {
		t->xt_nbufchars++;
		tx[ch]= LXT_SPHOLDER;
		t->xt_nspholders++;
		t->xt_linestarts[t->xt_nbuflines]= t->xt_nbufchars;
	}

	/* if there is no text */
	if ((ch == 0) && (t->xt_nbufchars == 0)) {
		tx[ch]= c;
		t->xt_nbufchars= 1;
		t->xt_linestarts[0]= 0;
		t->xt_linestarts[1]= 1;
		t->xt_nbuflines= 1;
	}
	else if (tx[ch] == LXT_SPHOLDER) {
		tx[ch]= c;
		--t->xt_nspholders;
	}
	else {
		bcopy(&tx[ch], &tx[ch+LXT_NSPHOLDERS], t->xt_nbufchars-ch);

		for (i= 0; i < LXT_NSPHOLDERS; i++)
			tx[ch+i]= LXT_SPHOLDER;

		t->xt_nspholders+= LXT_NSPHOLDERS;

		tx[ch]= c;
		t->xt_nbufchars+= LXT_NSPHOLDERS;
		--t->xt_nspholders;

		/* adjust the line starts */
		for (i= r+1; i <= t->xt_nbuflines; i++)
			t->xt_linestarts[i]+= LXT_NSPHOLDERS;
	}

	/* update new insertion point */
	++t->xt_icol;
	++t->xt_ichar;


	/* clear the line on the display */
	textsw_cleararea(t->xt_dpy, t->xt_twin, t->xt_cursorpx - t->xt_clbearing, t->xt_cursorpy,
	    t->xt_atw, t->xt_cheight, False);

	textsw_fillrect(t->xt_dpy, t->xt_tpm, t->xt_cgc, t->xt_cursorpx - t->xt_clbearing,
	    t->xt_cursorpy, t->xt_atw, t->xt_cheight);

	length= t->xt_linestarts[r + 1] - t->xt_linestarts[r];
	textsw_expand_string(&(t->xt_text[t->xt_linestarts[r]]), str, map, length, &new_length);

	XDrawString(t->xt_dpy, t->xt_twin, t->xt_gc, -t->xt_leftpix,
	    t->xt_cursorpy + t->xt_cheight - t->xt_cdescent, str, new_length);

	XDrawString(t->xt_dpy, t->xt_tpm, t->xt_gc, -t->xt_leftpix,
	    t->xt_cursorpy + t->xt_cheight - t->xt_cdescent, str, new_length);
	textsw_drawcaret(t);
	t->xt_drawcursor= TRUE;

	/* check if user inserted into the longest line	*/
	if (r == t->xt_maxlinenum) {
		t->xt_maxlinelen= XTextWidth(t->xt_font, str, new_length);
		t->xt_updatescroll= LXT_LONGESTLINEKNOWN;
	}
	else {
		if ((test_width= XTextWidth(t->xt_font, str, new_length)) > t->xt_maxlinelen) {
			t->xt_maxlinenum= r;
			t->xt_maxlinelen= test_width;
			t->xt_updatescroll= LXT_LONGESTLINEKNOWN;
		}
	}
	
}

textsw_insnewline(t)
Textsw *t;
{
	int i, r, ch;
	int old_left_pix_in_window, start_row;
	int map[512], length, new_length;
	char *tx;
	char str[512];
	void textsw_drawscrolls();

	r= t->xt_irow;
	ch= t->xt_ichar;
	tx= t->xt_text;

	/* undraw the caret */
	textsw_drawcaret(t);
	t->xt_drawcursor= FALSE;

	if (t->xt_textbufsz < t->xt_nbufchars + LXT_NSPHOLDERS) {
		t->xt_text= realloc(t->xt_text, (unsigned) ((t->xt_textbufsz+LXT_TEXTBUFINC)*sizeof(char)));
		t->xt_textbufsz+= LXT_TEXTBUFINC;
	}


	if (t->xt_lsbufsz < t->xt_nbuflines + 2) {
		t->xt_linestarts= (int *) realloc((char *) t->xt_linestarts, (unsigned) ((t->xt_lsbufsz+LXT_LSBUFINC)*sizeof(int)));
		t->xt_lsbufsz+= LXT_LSBUFINC;
	}

	if (ch > t->xt_nbufchars)
		ch= t->xt_nbufchars;
	if (ch < 0)
		ch= 0;

	if ((t->xt_nbufchars != 0) && (ch == t->xt_nbufchars)) {
		t->xt_nbufchars++; 
		tx[ch]=LXT_SPHOLDER;
		t->xt_nspholders++;
		t->xt_linestarts[t->xt_nbuflines]= t->xt_nbufchars;
	} 

	/* if there is no text */
	if ((ch == 0) && (t->xt_nbufchars == 0)) {
		tx[ch]= '\0';
		t->xt_nbufchars= 1;
		t->xt_linestarts[0]= 0;
		t->xt_linestarts[1]= 1;
		t->xt_nbuflines= 1;
	}
	else if (tx[ch] == LXT_SPHOLDER) {
		tx[ch]= '\0';
		--t->xt_nspholders;
	}
	else {
		bcopy(&tx[ch], &tx[ch + LXT_NSPHOLDERS], t->xt_nbufchars - ch);
		for (i= 0; i < LXT_NSPHOLDERS; i++)
			tx[ch+i]= LXT_SPHOLDER;

		t->xt_nspholders+= LXT_NSPHOLDERS;

		tx[ch]= '\0';
		t->xt_nbufchars+= LXT_NSPHOLDERS;
		--t->xt_nspholders;

		/* adjust the line starts */
		for (i= r+1; i <= t->xt_nbuflines; i++)
			t->xt_linestarts[i]+= LXT_NSPHOLDERS;
	}

	/* re-adjust line starts beacause of the new line */
	for (i= t->xt_nbuflines; i > r; i--)
		t->xt_linestarts[i+1]= t->xt_linestarts[i];
	t->xt_linestarts[r+1]= ch+1;

	++t->xt_nbuflines;


	/* update new insertion point */
	t->xt_icol= 0;
	t->xt_irow= r+1;
	t->xt_ichar= ch+1;

	/* make sure that new line is visible */
	if (t->xt_cursorpy >= t->xt_ath - t->xt_cheight) {
		t->xt_topline+= 3;
		textsw_drawscrolls(t);
		textsw_draw(t);
		textsw_display(t);
		textsw_drawcaret(t);
		t->xt_drawcursor= TRUE;
		return;
	}

	/* clear the line on the display */
	textsw_cleararea(t->xt_dpy, t->xt_twin, t->xt_cursorpx - t->xt_clbearing, t->xt_cursorpy, t->xt_atw, t->xt_cheight, False);
	textsw_fillrect(t->xt_dpy, t->xt_tpm, t->xt_cgc, t->xt_cursorpx - t->xt_clbearing, t->xt_cursorpy, t->xt_atw, t->xt_cheight);
	textsw_cleararea(t->xt_dpy, t->xt_twin, 0, t->xt_cursorpy + t->xt_cheight, t->xt_atw, t->xt_ath - (t->xt_cursorpy + t->xt_cheight), False);
	textsw_fillrect(t->xt_dpy, t->xt_tpm, t->xt_cgc, 0, t->xt_cursorpy + t->xt_cheight, t->xt_atw, t->xt_ath - (t->xt_cursorpy + t->xt_cheight));


	old_left_pix_in_window= t->xt_leftpix;
	t->xt_leftpix= 0;

	/* check if user hit return when scrolled to right */
	if (old_left_pix_in_window == 0) {
		start_row= r+1;
		textsw_cleararea(t->xt_dpy, t->xt_twin, t->xt_cursorpx - t->xt_clbearing, t->xt_cursorpy, t->xt_atw, t->xt_cheight, False);
		textsw_fillrect(t->xt_dpy, t->xt_tpm, t->xt_cgc, t->xt_cursorpx - t->xt_clbearing, t->xt_cursorpy, t->xt_atw, t->xt_cheight);
		textsw_cleararea(t->xt_dpy, t->xt_twin, 0, t->xt_cursorpy + t->xt_cheight, t->xt_atw, t->xt_ath - (t->xt_cursorpy + t->xt_cheight), False);
		textsw_fillrect(t->xt_dpy, t->xt_tpm, t->xt_cgc, 0, t->xt_cursorpy + t->xt_cheight, t->xt_atw, t->xt_ath - (t->xt_cursorpy + t->xt_cheight));
	}
	else {
		start_row= t->xt_topline;
		XClearWindow(t->xt_dpy, t->xt_twin);
		textsw_fillrect(t->xt_dpy, t->xt_tpm, t->xt_cgc, 0, 0, t->xt_atw, t->xt_ath);
		textsw_longestline(t);
		textsw_drawscrolls(t);
	}

	for (i= start_row; (i < (t->xt_wch+t->xt_topline)) && (i < t->xt_nbuflines); i++) {
		length= t->xt_linestarts[i+1]-t->xt_linestarts[i];
		textsw_expand_string(&(t->xt_text[t->xt_linestarts[i]]), str, map, length, &new_length);
		XDrawString(t->xt_dpy, t->xt_twin, t->xt_gc, -t->xt_leftpix,
		    (1 + i - t->xt_topline) * t->xt_cheight - t->xt_cdescent, str, new_length);
		XDrawString(t->xt_dpy, t->xt_tpm, t->xt_gc, -t->xt_leftpix,
		    (1 + i - t->xt_topline) * t->xt_cheight - t->xt_cdescent, str, new_length);
	}

	textsw_drawcaret(t);
	t->xt_drawcursor= TRUE;

}

textsw_delchar(t)
Textsw *t;
{
	int i, r, ch;
	int map[512], length, new_length;
	char del_char;
	char *tx;
	char str[512];
	void textsw_drawscrolls();

	r= t->xt_irow;
	ch= t->xt_ichar;
	tx= t->xt_text;

	/* if caret is undrawn or in position zero there is nothing to delete */
	if ((ch == 0) || (t->xt_drawcursor == FALSE))
		return;

	/* undraw the caret */
	textsw_drawcaret(t);
	t->xt_drawcursor= FALSE;

	/* t->xt_ichar is where an insertion belongs,
	   decrement by 1 to get the character to delete */
	--ch;

	/* make sure next char to delete is not a space holder	*/
	while (tx[ch] == LXT_SPHOLDER) {
		if (ch == 0)
			break;
		--ch;
	}

	/* save deleted character */
	del_char = tx[ch];

	/* move the text array back one	*/
	bcopy(&tx[ch +1], &tx[ch], t->xt_nbufchars - ch);

	/* if deleted character was a new line, then the
	   line_starts array must be shifted down by 1 */
	if (del_char == '\0') {
		--t->xt_nbuflines;

		/* readjust line starts beacause of the new line */
		for (i= r; i <= t->xt_nbuflines; i++)
			t->xt_linestarts[i]= t->xt_linestarts[i + 1];

		/* the current line start needs to be decremented */
		--t->xt_linestarts[r];
	}


	/* decrement the line starts */
	for (i= r+1; i <= t->xt_nbuflines; i++)
		--t->xt_linestarts[i];

	if (del_char == '\0') {
		--t->xt_irow;
		t->xt_icol= t->xt_linestarts[t->xt_irow + 1] - t->xt_linestarts[t->xt_irow] - 1;
		t->xt_ichar= t->xt_linestarts[t->xt_irow] + t->xt_icol;
	}
	else {
		--t->xt_icol;
		--t->xt_ichar;
	}


	/* if a newline was not deleted only redraw the
	   line otherwise redraw the screen */
	if (del_char != '\0') {
		textsw_cleararea(t->xt_dpy, t->xt_twin, t->xt_cursorpx - t->xt_clbearing - t->xt_cwidth, t->xt_cursorpy, t->xt_atw, t->xt_cheight, False);
		textsw_fillrect(t->xt_dpy, t->xt_tpm, t->xt_cgc, t->xt_cursorpx - t->xt_clbearing - t->xt_cwidth, t->xt_cursorpy, t->xt_atw, t->xt_cheight);

		length= t->xt_linestarts[t->xt_irow + 1] - t->xt_linestarts[t->xt_irow];
 /*bug fix*/	textsw_expand_string(&(t->xt_text[t->xt_linestarts[r]]), str, map, length, &new_length);
		XDrawString(t->xt_dpy, t->xt_twin, t->xt_gc, -t->xt_leftpix, t->xt_cursorpy + t->xt_cheight - t->xt_cdescent, str, new_length);
		XDrawString(t->xt_dpy, t->xt_tpm, t->xt_gc, -t->xt_leftpix, t->xt_cursorpy + t->xt_cheight - t->xt_cdescent, str, new_length);

		textsw_drawcaret(t);
		t->xt_drawcursor= TRUE;
	}
	else {
		textsw_cleararea(t->xt_dpy, t->xt_twin, 0, t->xt_cursorpy, t->xt_atw, t->xt_ath - (t->xt_cursorpy + t->xt_cheight), False);
		textsw_fillrect(t->xt_dpy, t->xt_tpm, t->xt_cgc, 0, t->xt_cursorpy, t->xt_atw, t->xt_ath - (t->xt_cursorpy + t->xt_cheight));

		for (i= r-1; (i < (t->xt_wch + t->xt_topline)) && (i < t->xt_nbuflines); i++) {
			length= t->xt_linestarts[i+1] - t->xt_linestarts[i];
			textsw_expand_string(&(t->xt_text[t->xt_linestarts[i]]), str, map, length, &new_length);
			XDrawString(t->xt_dpy, t->xt_twin, t->xt_gc, -t->xt_leftpix, (1 + i - t->xt_topline) * t->xt_cheight - t->xt_cdescent, str, new_length);
			XDrawString(t->xt_dpy, t->xt_tpm, t->xt_gc, -t->xt_leftpix, (1 + i - t->xt_topline) * t->xt_cheight - t->xt_cdescent, str, new_length);
		}

		textsw_drawcaret(t);
		t->xt_drawcursor= TRUE;
	}

	/* cursor out of view to the right */
	if (t->xt_cursorpx > t->xt_leftpix + t->xt_atw)	{
		t->xt_leftpix= t->xt_cursorpx - t->xt_cwidth;
		textsw_longestline(t);
		textsw_drawscrolls(t);
		textsw_draw(t);

		/* set flag so that backing store routine will redraw the caret	*/
		t->xt_drawcursor= TRUE;
		textsw_display(t);
	}

	/* cursor is off top of page */
	if (t->xt_cursorpy < 0) {
		--t->xt_topline;
		if (t->xt_topline < 0)
			t->xt_topline= 0;

		textsw_longestline(t);
		textsw_drawscrolls(t);

		textsw_draw(t);
		t->xt_drawcursor= TRUE;

		/* set flag so that backing store
		   routine will redraw the caret */
		textsw_display(t);
	}

	/* deleting a character from the longest line or
	   if the user deleted before the first character then
	   they may have created a new longest line */
	if ((r == t->xt_maxlinenum) || (del_char == '\0'))
		t->xt_updatescroll= LXT_LONGESTLINEUNKNOWN;

}

textsw_drawhlt(t, draw, check)
/*
   Inernal function.
   Draw inverse highlighting for selected text if draw is TRUE.
   If draw is FALSE, determine whether highlighting could in fact
   be drawn if requested, and return the answer in *check.
*/
Textsw *t;
boolean draw, *check;
{
	int ydiff, y_offset, x_offset;
	int start_x, start_y, end_x, end_y;	/* start and end points of
						   the highlighted text */
	int x, y, width, height;

	/* draw the highlighted text in one of three ways --
	   (1) only one line must be drawn, (2) exactly two lines
	   must be drawn, or (3) more than two lines must be drawn */
		

	/* first check that the end (the highlight start is really the
	   end of the highlight because it is where the next highlight would
	   begin) is greater than the beginning -- this will not be true
	   if the user scolled upwards so they will be switched  */
	if ((t->xt_hltstarty > t->xt_hltcursory) ||
	    ((t->xt_hltstarty == t->xt_hltcursory) &&
	     (t->xt_hltstartx > t->xt_hltcursorx))) {
		start_x= t->xt_hltcursorx;
		start_y= t->xt_hltcursory;
		end_x= t->xt_hltstartx;
		end_y= t->xt_hltstarty;
	}
	else {
		start_x= t->xt_hltstartx;
		start_y= t->xt_hltstarty;
		end_x= t->xt_hltcursorx;
		end_y= t->xt_hltcursory;
	}

	/* calculate the number of lines in the highlighted portion of text
	   and the offsets which are the difference between the current upper
	   left and the upper left when the highlight was made */
	ydiff= (end_y - start_y)/t->xt_cheight;
	x_offset= t->xt_leftpix - t->xt_hltorigin.x;
	y_offset= (t->xt_topline * t->xt_cheight) - t->xt_hltorigin.y;

	/* highlight one line only */
	if (ydiff == 0) {
		x= start_x - x_offset;
		y= end_y - y_offset;
		width= end_x - start_x;
		height= t->xt_cheight;

		if (draw == FALSE) {
			if ((width > 0) && (height > 0))
				*check= TRUE;
			else
				*check= FALSE;
			return;
		}

		textsw_fillrect(t->xt_dpy, t->xt_twin, t->xt_igc, x, y, width, height);
	}

	/* highlight two lines only */
	else if (ydiff == 1) {
		if (draw == FALSE) {
			*check= TRUE;
			return;
		}

		x= start_x - x_offset;
		y= start_y - y_offset;
		width= t->xt_atw - x;
		height= t->xt_cheight;
		textsw_fillrect(t->xt_dpy, t->xt_twin, t->xt_igc, x, y, width, height);

		x= 0;
		y= y + t->xt_cheight;
		width= end_x - x_offset;
		height= t->xt_cheight;
		textsw_fillrect(t->xt_dpy, t->xt_twin, t->xt_igc, x, y, width, height);
	}

	/* highlight more than two lines */
	else {
		if (draw == FALSE) {
			*check= TRUE;
			return;
		}

		x= start_x - x_offset;
		y= start_y - y_offset;
		width= t->xt_atw - x;
		height= t->xt_cheight;
		textsw_fillrect(t->xt_dpy, t->xt_twin, t->xt_igc, x, y, width, height);

		x= 0;
		y= y + t->xt_cheight;
		width= t->xt_atw;
		height= end_y - start_y - t->xt_cheight;
		textsw_fillrect(t->xt_dpy, t->xt_twin, t->xt_igc, x, y, width, height);

		x= 0;
		y= end_y - y_offset;
		width= end_x - x_offset;
		height= t->xt_cheight;
		textsw_fillrect(t->xt_dpy, t->xt_twin, t->xt_igc, x, y, width, height);
	}
}

textsw_fillrect(display, drawable, gc, x, y, width, height)
Display *display;
Drawable drawable;
GC gc;
int x, y;
int width, height;
{
	int xx, yy;
	unsigned int w, h;

	if (width >= 0) {
		w= (unsigned int) width;
		xx= x;
	}
	else {
		w= (unsigned int) (-width);
		xx= x-w;
	}

	if (height >= 0) {
		h= (unsigned int) height;
		yy= y;
	}
	else {
		h= (unsigned int) (-height);
		yy= y-h;
	}

	if (h > 4096)
		h= 4096;
	if (w > 4096)
		w= 4096;
	XFillRectangle(display, drawable, gc, xx, yy, w, h);
}

textsw_cleararea(display, window, x, y, width, height, exposures)
Display *display;
Window window;
int x, y;
int width, height;
Bool exposures;
{
	int xx, yy;
	unsigned int w, h;

	if (width >= 0) {
		w= (unsigned int) width;
		xx= x;
	}
	else {
		w= (unsigned int) (-width);
		xx= x-w;    
	}
 
	if (height >= 0) {
		h= (unsigned int) height;
		yy= y; 
	}   
	else {  
		h= (unsigned int) (-height); 
		yy= y-h;  
	} 
 
	if (h > 4096)
		h= 4096;
	if (w > 4096)
		w= 4096;
 
	XClearArea(display, window, xx, yy, w, h, exposures);
}
		

textsw_copyarea(display, src, dest, gc, src_x, src_y, width, height, dest_x, dest_y)
Display *display;
Drawable src, dest;
GC gc;
int src_x, src_y;
int width, height;
int dest_x, dest_y;
{
	int sxx, syy, dxx, dyy;
	unsigned int w, h;

	if (width >= 0) {
		w= (unsigned int) width;
		sxx= src_x;
		dxx= dest_x;
	}   
	else {    
		w= (unsigned int) (-width);
		sxx= src_x-w;
		dxx= dest_x-w;    
	}
 
	if (height >= 0) {
		h= (unsigned int) height;
		syy= src_y;
		dyy= dest_y; 
	}   
	else {  
		h= (unsigned int) (-height); 
		syy= src_y-h;
		dyy= dest_y-h;  
	} 
 
	if (h > 4096)
		h= 4096;
	if (w > 4096)
		w= 4096;
 
	XCopyArea(display, src, dest, gc, sxx, syy, w, h, dxx, dyy);
}

xt_test(t, c)
Textsw *t;
char c;
{
	int i, j;

	if ((c == '\n') || (c == '\0'))
		c= '!';
	(void) fprintf(stderr, "********** %c\n", c);
	(void) fprintf(stderr, "ichar=%d nbufchars=%d\n", t->xt_ichar, t->xt_nbufchars);
	for (i= 0; i <= t->xt_nbuflines; i++)
		(void) fprintf(stderr, "%d %d\n", i, t->xt_linestarts[i]);
	j= 0;
	for (i= 0; i < t->xt_nbufchars; i++) {
		if (t->xt_text[i] == LXT_SPHOLDER)
			(void) fprintf(stderr, "#");
		else {
			if (j == 0) {
				(void) fprintf(stderr, "%d   ", i);
				j++;
			}
			(void) fprintf(stderr, "%c", t->xt_text[i]);
			if ((t->xt_text[i] == '\n') || (t->xt_text[i] == '\0')) {
				(void) fprintf(stderr, "\n");
				j= 0;
			}
		}
	}
	(void) fprintf(stderr, "\n\n");
	(void) fflush(stderr);
}
