/*
 *  @(#) obuffer.c 1.6, last edit: 2/21/94 18:10:13
 *  @(#) Copyright (C) 1993, 1994 Tobias Bading (bading@cs.tu-berlin.de)
 *  @(#) Berlin University of Technology
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include "obuffer.h"
#include "header.h"


extern "C" int ioctl (int, int ...);


ShortObuffer::ShortObuffer (uint32 number_of_channels)
{
#ifdef DEBUG
  if (!number_of_channels || number_of_channels > MAXCHANNELS)
  {
    cerr << "ShortObuffer: number of channels has to be in [1, " <<  MAXCHANNELS << "] !\n";
    exit (1);
  }
#endif
  channels = number_of_channels;
  for (int i = 0; i < number_of_channels; ++i)
    bufferp[i] = buffer + i;
}


void ShortObuffer::append (uint32 channel, int16 value)
{
#ifdef DEBUG
  if (channel >= channels)
  {
    cerr << "illegal channelnumber in ShortObuffer::append()!\n";
    exit (1);
  }
  if (bufferp[channel] - buffer >= OBUFFERSIZE)
  {
    cerr << "ShortObuffer: buffer overflow!\n";
    exit (1);
  }
#endif
  *bufferp[channel] = value;
  bufferp[channel] += channels;
}


void ShortObuffer::write_buffer (int fd)
{
  int length = (int)((char *)bufferp[0] - (char *)buffer);
  if (write (fd, buffer, length) != length)
    cerr << "Warning: couldn't write all samples\n";
  for (int i = 0; i < channels; ++i)
    bufferp[i] = buffer + i;
}



#ifdef Indigo
IndigoObuffer::IndigoObuffer (uint32 number_of_channels, Header *header)
{
#ifdef DEBUG
  if (!number_of_channels || number_of_channels > MAXCHANNELS)
  {
    cerr << "IndigoObuffer: number of channels has to be in [1, " <<  MAXCHANNELS << "] !\n";
    exit (1);
  }
#endif
  channels = number_of_channels;
  for (int i = 0; i < number_of_channels; ++i)
    bufferp[i] = buffer + i;

  // open an audio port and configure it:
  ALconfig config;
  if (!(config = ALnewconfig ()))
  {
    cerr << "ALnewconfig failed!\n";
    exit (1);
  }
  ALsetwidth (config, AL_SAMPLE_16);
  if (channels == 1)
    ALsetchannels (config, AL_MONO);
  else
    ALsetchannels (config, AL_STEREO);
  if (!(port = ALopenport ("MPEG audio player", "w", config)))
  {
    cerr << "can't allocate an audio port!\n";
    exit (1);
  }

  // set sample rate:
  long pvbuffer[2] = { AL_OUTPUT_RATE, 0 };
  pvbuffer[1] = header->frequency ();
  ALsetparams (AL_DEFAULT_DEVICE, pvbuffer, 2);
  ALfreeconfig (config);
}


IndigoObuffer::~IndigoObuffer (void)
{
  while (ALgetfilled (port) > 0)
    sleep (1);
  ALcloseport (port);
}


void IndigoObuffer::append (uint32 channel, int16 value)
{
#ifdef DEBUG
  if (channel >= channels)
  {
    cerr << "illegal channelnumber in IndigoObuffer::append()!\n";
    exit (1);
  }
  if (bufferp[channel] - buffer >= OBUFFERSIZE)
  {
    cerr << "IndigoObuffer: buffer overflow!\n";
    exit (1);
  }
#endif
  *bufferp[channel] = value;
  bufferp[channel] += channels;
}


void IndigoObuffer::write_buffer (int)
{
  ALwritesamps (port, buffer, (long)(bufferp[0] - buffer));
  for (int i = 0; i < channels; ++i)
    bufferp[i] = buffer + i;
}
#endif // Indigo


#ifdef SPARC
int SparcObuffer::audio_fd = -1;

SparcObuffer::SparcObuffer (uint32 number_of_channels, Header *header,
			    bool use_speaker, bool use_headphone, bool use_line_out)
{
#ifdef DEBUG
  if (!number_of_channels || number_of_channels > MAXCHANNELS)
  {
    cerr << "SparcObuffer: 0 < number of channels < " << MAXCHANNELS << "!\n";
    exit (1);
  }
#endif
  channels = number_of_channels;
  for (int i = 0; i < number_of_channels; ++i)
    bufferp[i] = buffer + i;

  if (audio_fd < 0)
  {
    cerr << "Internal error, SparcObuffer::audio_fd has to be initialized\n"
	    "by SparcObuffer::class_suitable()!\n";
    exit (1);
  }

  // configure the device:
  audio_info info;
  AUDIO_INITINFO (&info);
  info.output_muted = False;
  info.play.encoding = AUDIO_ENCODING_LINEAR;
  info.play.precision = 16;
  info.play.channels = channels;
  info.play.sample_rate = header->frequency ();
  if (use_speaker)
    info.play.port |= AUDIO_SPEAKER;
  if (use_headphone)
    info.play.port |= AUDIO_HEADPHONE;
  if (use_line_out)
    info.play.port |= AUDIO_LINE_OUT;
  info.play.balance = AUDIO_MID_BALANCE;
  if (ioctl (audio_fd, AUDIO_SETINFO, &info))
  {
    perror ("configuration of /dev/audio failed");
    exit (1);
  }
}


SparcObuffer::~SparcObuffer (void)
{
  sleep (1);
  close (audio_fd);
}


void SparcObuffer::append (uint32 channel, int16 value)
{
#ifdef DEBUG
  if (channel >= channels)
  {
    cerr << "illegal channelnumber in SparcObuffer::append()!\n";
    exit (1);
  }
  if (bufferp[channel] - buffer >= OBUFFERSIZE)
  {
    cerr << "buffer overflow!\n";
    exit (1);
  }
#endif
  *bufferp[channel] = value;
  bufferp[channel] += channels;
}


void SparcObuffer::write_buffer (int)
{
  int length = (int)((char *)bufferp[0] - (char *)buffer);
  if (write (audio_fd, buffer, length) != length)
    cerr << "Warning: couldn't write all samples to /dev/audio\n";
  for (int i = 0; i < channels; ++i)
    bufferp[i] = buffer + i;
}


int SparcObuffer::open_audio_device (void)
{
  int fd;

  if ((fd = open ("/dev/audio", O_WRONLY | O_NDELAY)) < 0)
    if (errno == EBUSY)
    {
      cerr << "Sorry, the audio device is busy!\n";
      exit (1);
    }
    else
    {
      perror ("can't open /dev/audio for writing");
      exit (1);
    }
  return fd;
}


#ifdef Solaris
void SparcObuffer::get_device_type (int fd, audio_device *devtype)
{
  if (ioctl (fd, AUDIO_GETDEV, devtype))
  {
    perror ("ioctl on /dev/audio");
    exit (1);
  }
}
#else
int SparcObuffer::get_device_type (int fd)
{
#ifdef AUDIO_GETDEV
  int devtype;
  if (ioctl (fd, AUDIO_GETDEV, &devtype))
  {
    perror ("ioctl on /dev/audio");
    exit (1);
  }
  return devtype;
#else
  cerr << "Warning: AUDIO_GETDEV ioctl not available!\n";
  return -1;
#endif
}
#endif // !Solaris


bool SparcObuffer::class_suitable (void)
{
  // check for the dbri audio device:
  audio_fd = open_audio_device ();
#ifdef Solaris
  audio_device devtype;
  get_device_type (audio_fd, &devtype);
  if (!strcmp (devtype.name, "SUNW,dbri"))
#else
  if (get_device_type (audio_fd) == AUDIO_DEV_SPEAKERBOX)
#endif
    return True;
  else
    return False;
}
#endif
