// commenteditor.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "commenteditor.h"

#include <InterViews/button.h>
#include <InterViews/sensor.h>
#include <InterViews/shape.h>
#include <InterViews/textbuffer.h>
#include <InterViews/textdisplay.h>
#include <stdlib.h>
#include <ctype.h>

#include "data.h"

/*****************************************************************************/

static const int gap = 2;
static const int rows = 6;
static const int cols = 80;
static const int tabs = 4;

/*****************************************************************************/

CommentEditor::CommentEditor(ButtonState* bs, Data* data)
		: TextEditor(rows, cols, tabs, Reversed), myData(data) {
    Init(bs);
}

CommentEditor::~CommentEditor() {
	if(_state != nil)
		_state->Detach(this);
	Resource::unref(myData);
	delete [] commtext;
	delete myTextBuffer;
}

void CommentEditor::Init (ButtonState* bs) {
    SetClassName("CommentEditor");
    input = new Sensor;
    input->Catch(DownEvent);
    input->Catch(KeyEvent);

	_state = bs;
	if(_state != nil)
		_state->Attach(this);
	myData->ref();
	const char* text = myData->comment();
	commtext = new char[rows*cols];
    strcpy(commtext, text);
	myTextBuffer = new TextBuffer(commtext, strlen(text), rows*cols);
    Edit(myTextBuffer);
}

void
CommentEditor::save() {
	myData->setComment(commtext);
}

void
CommentEditor::revert() {
}
	
void CommentEditor::Handle (Event& e) {
	boolean done = false;
	do {
		if (e.eventType == KeyEvent && e.len > 0) {
            HandleChar(e.keystring[0]);
		}
		else if (e.eventType == DownEvent) {
			if (e.target == this)
				HandleDownEvent(e);
			else {
                UnRead(e);
                done = true;
            }
        }
		if(!done)
			Read(e);
	} while(!done);
}

void CommentEditor::LeftMouse (Event& e) {
    Select(Locate(e.x, e.y));
    do {
        ScrollToView(e.x, e.y);
        SelectMore(Locate(e.x, e.y));
        Poll(e);
    } while (e.leftmouse);
}

void
CommentEditor::HandleDownEvent (Event& e) {
    if (e.target == this) {
        switch (e.button) {
            case LEFTMOUSE:     LeftMouse(e); break;
            case MIDDLEMOUSE:   GrabScroll(e); break;
            case RIGHTMOUSE:    RateScroll(e); break;
        }

    } else {
        UnRead(e);
    }
}

void
CommentEditor::HandleChar (char c) {
    int lines = display->LineNumber(0) - display->LineNumber(ymax) + 1;
    int i;
    char copy = c;

    switch (c) {
    case '\026':  ForwardPage(1); break;
    case '\001':  BeginningOfLine(); break;
    case '\005':  EndOfLine(); break;
    case '\006':  ForwardCharacter(1); break;
    case '\002':  BackwardCharacter(1); break;
    case '\016':  ForwardLine(1); break;
    case '\020':  BackwardLine(1); break;
    case '\004':
        if (Dot() != Mark()) {
            DeleteSelection();
        } else {
            DeleteText(1);
        }
        break;

    case '\010':
    case '\177':
        if (Dot() != Mark()) {
            DeleteSelection();
        } else {
            DeleteText(-1);
        }
        break;
        
    case '\011':
        copy = '\t'; 
        DeleteSelection();
        InsertText(&copy, 1);
        break;

    case '\015':
        copy = '\n'; 
        DeleteSelection();
        InsertText(&copy, 1);
        break;

    default:
        if (!iscntrl(c)) {
            DeleteSelection();
            InsertText(&copy, 1);
        }
        break;
    }
    ScrollToSelection();
}
