// controller.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// The Controller class is one of the three central classes in MiXViews.  It is
// named such because of its role in the Model-View-Controller paradigm on
// which MiXViews is based.  It coordinates and routes messages between various
// other graphic and non-graphic components.  It contains pointers to an Editor
// subclass instance, a DataView subclass instance, and a DataWindow instance.
// It controls the creation and destruction of each display window and all of
// its associated parts.  Each is connected via a doubly-linked circular list,
// each instance being a link in the list.  Operations that pass data between
// windows do so via methods invoked upon the Controller via the DataEditor 
// subclass member, and the entire list can be searched from anywhere on itself.

#ifndef CONTROLLER_H
#ifdef __GNUG__
#pragma interface
#endif
#define CONTROLLER_H

#include <InterViews/resource.h>
#include "localdefs.h"
#include "range.h"
#include <String.h>

class DataFile;
class Data;
class DataWindow;
class DataView;
class ViewInfo;
class World;
class Event;
class DataEditor;
class Request;
class StatusAction;
class EditStatusDelegate;

class Controller : public Resource {
private:
	static Controller* head;
	static int numControllers;
	static int quitting;
	Controller* prev;
	Controller* next;
	enum ViewType { AsChannels, AsFrames };
	static ViewInfo& DefaultViewInfo;
private:
	void addToList();
	void removeFromList();
	void closeChain();
public:
	static Controller* create(DataFile *);
	Controller(Data *, const char *name = nil, ViewInfo &info=DefaultViewInfo);

	virtual void display(World *);
	virtual const char *fileName() const;
	const char* windowName () const;
	Data* model() const;
	DataEditor* editor() const { return myEditor; }

	virtual void busy(boolean);

	// methods used (called by) Graphs	
	virtual void setEditRegion(const Range &region, int chan);
	virtual void setInsertPoint(int point, int chan);

	// methods used by editor
	World *world();	
	virtual boolean handleEvent(Event &);
	virtual boolean handleKeyEvent(Event &);
	virtual int handleRequest(Request &);
	boolean keyCommand(unsigned long);
	Data* findSelection();
	void showInsertPoint(int, const Range &, boolean scroll = false);
	void showEditRegion(const Range &, const Range &, boolean scroll = false);
	void unselectView();
	void resetScaleTimes();
	virtual void alert(const char* m1, const char* m2=nil, const char* m3=nil);
	virtual boolean confirm(const char* m1, const char* m2=nil,
		const char* m3=nil, Response r=Yes);
	virtual Response choice(const char* m1, const char* m2=nil,
		const char* m3=nil, Response r=Yes);

	// methods called via menus or keyboard
	virtual void newViewOfSelection();
	virtual void viewAsFrames();
	virtual void viewAsChannels();
	virtual void zoomToSelection();
	virtual void zoomToFull();
	virtual void close();
	void showVersion();
	virtual void quit();
	virtual void changeName();
	void setProgramOptions();
 	// converter methods
	void record();
	void play();
	void stop(); 					// defined in controller_conv.c
	void configureConverter();
protected:
	Controller(const char *);
	virtual ~Controller();
	Data* getSelection(long &timestamp);
	void setFileName(const char *);
	void viewAs(ViewType type);
 	// converter methods
	StatusAction* getConverterAction() { return converterAction; }
private:
	void earlyInit();
	boolean initialize(ViewInfo &info=DefaultViewInfo);
	DataView* newView(ViewInfo &);
	void createAndAttachView(ViewInfo &info=DefaultViewInfo);
	void detachView();
	int checkForStop();
private:
	String filename;
	DataWindow *window;
	DataView *myView;
	DataEditor *myEditor;
	StatusAction *converterAction;
	EditStatusDelegate* editDelegate;
	friend DataEditor;
	friend class SoundEditor;
	friend class FFTEditor;
	friend class LPCEditor;
	friend class PCHEditor;
	friend class EnvelopeEditor;
	friend class ProcessFunction;
	friend class FileSaver;
};

inline const char *
Controller::fileName() const { return filename; }

#endif
