// converter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <InterViews/action.h>
#include "application.h"
#include "converter.h"
#include "localdefs.h"
#include "sound.h"

// global initializations

Converter* Converter::converterInstance = nil;

// Create converter appropriate for platform, check for successful
// initialization.  NullConverter instance is returned if init failed.

Converter *
Converter::getInstance() {
	if(converterInstance == nil) {
		converterInstance = create();
		if(converterInstance->ready() == false) {
			destroyInstance();
			converterInstance = new NullConverter;
		}
		converterInstance->ref();
	}
	return converterInstance;
}

void
Converter::destroyInstance() {
	Resource::unref(converterInstance);
	converterInstance = nil;
}

void
Converter::useNative(boolean b) {
	if(usingNative != b) {
		usingNative = b;
		destroyInstance();
	}
}

// methods for base class for "real" (associated with hardware) converters

// global initializations

Action* RealConverter::recordDoneAction = nil;
RealConverter::Status RealConverter::myStatus = Stopped;

RealConverter::RealConverter() : mySound(nil), goingTo(Play) {
	initialize();
}

RealConverter::~RealConverter() {
	setSound(nil);
	setRecordDoneAction(nil);
}

void
RealConverter::initialize() {
	setStatus(Stopped);		// static, so reset for each ctor call
}

boolean
RealConverter::hasPlayableFormat(Sound *snd) {
	return isPlayableFormat(snd->dataType());
}

boolean
RealConverter::ready() const { return good(); }

boolean
RealConverter::running() {
	return playing() || recording();
}

void
RealConverter::reset() {
	if(!running())		// dont do reset if currently running
		initialize();
}

void
RealConverter::setSound(Sound* sound) {
	if(sound != mySound) {
		Resource::unref(mySound);
		mySound = sound;
		if(mySound != nil)
			mySound->ref();
	}
}

int
RealConverter::sampleRate() { return mySound->sRate(); }

int
RealConverter::channels() { return mySound->nChans(); }

DataType
RealConverter::dataType() { return mySound->dataType(); }

const void *
RealConverter::pointerToData() { return mySound->getDataPointer(); }

long
RealConverter::dataSize() { return mySound->sizeInBytes(); }

double
RealConverter::peakAmplitude() { return mySound->peakAmp(); }

int
RealConverter::checkLength(int nsamps) {
	return (nsamps == 0) ?
		error("Play/Record error:", "Zero length sound.") : true;
}

void
RealConverter::setRecordDoneAction(Action *a) {
	if(a != recordDoneAction) {
		Resource::unref(recordDoneAction);
		recordDoneAction = a;
		if(recordDoneAction != nil)
			recordDoneAction->ref();
	}
}

int
RealConverter::configure(Sound* sound, GoingTo wantTo) {
	int status = false;	
	if(!running()) {
		goingTo = wantTo;
		reset();
		setSound(sound);
		if(checkLength(sound->nSamps()) &&
		   checkDataType(sound->dataType()) &&
		   checkChannels(sound->nChans()) &&
		   checkSampleRate(sound->sRate()) 
		) {
			status = doConfigure();
		}
		else fail();	
	}
	else
		Application::alert("Converter is currently in use.");
	return status;
}

int
RealConverter::start(StatusAction *askedToStop, Action* whenDone) {
	int status = (good() && soundIsLoaded());
	if(status) {
		ignoreSignals(true);
		status = willPlay() ?
			startPlaying(askedToStop) : startRecording(askedToStop, whenDone);
		ignoreSignals(false);
	}
	return status;
}

int
RealConverter::stop() {
	if(recording())
		doRecordDoneAction();
	setStatus(Stopped);
	return true;
};

void
RealConverter::configureHardware(Controller* controller) {
	Requester* requester = nil;
	if((requester = configRequester()) != nil)
		requester->configure(controller);
}

void
RealConverter::doRecordDoneAction() {
	if(recordDoneAction != nil)
		recordDoneAction->execute();
}

int
RealConverter::startPlaying(StatusAction *askedToStop) {
	setStatus(Playing);
	return doConversion(askedToStop);
}

int
RealConverter::startRecording(StatusAction *askedToStop, Action* whenDone) {
	setRecordDoneAction(whenDone);
	setStatus(Recording);
	return doRecording(askedToStop);
}

int
RealConverter::error(const char* msg1, const char* msg2) {
	Application::error(msg1, msg2);
	fail();
	return false;
}

boolean
RealConverter::notImplemented() {
	error("This command is not implemented for this converter.");
	return false;
}
