// converter.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// The base class for all objects implementing an interface to a D/A A/D 
// converter.  At least one subclass is defined for each architecture, more if
// there are multiple devices available.  The base class makes no assumptions
// about how the sound is delivered to the device.  There is a derived class
// ConverterDevice which implements additional methods for writing and reading
// buffers (see conv_device.h).  Note:  Only one Converter class instance can
// exist at one time -- if switching converters, the old one must be deleted
// first.

#ifndef CONVERTER_H
#ifdef __GNUG__
#pragma interface
#endif
#define CONVERTER_H

#include <InterViews/resource.h>
#include "localdefs.h"
#include "requester.h"

class Action;		// encapsulated function to call when recording done
class StatusAction;	// encapsulated function to call to check for stop
class Sound;

class Converter : virtual public Resource {
public:		// class methods
	enum GoingTo { Record, Play };
	static Converter* getInstance();		// assures single object instance
	static void destroyInstance();
	static boolean useNative() { return usingNative; }
	static void useNative(boolean b);
protected:	// class methods
	static Converter* create();
public:		// object methods
	virtual boolean ready() const = 0;
	virtual void configureHardware(class Controller*) = 0;
	virtual DataType bestPlayableType() = 0;
	virtual boolean hasPlayableFormat(Sound *) = 0;
	virtual int configure(Sound *sound, GoingTo) = 0;
	virtual int start(StatusAction *, Action *) = 0;
	virtual int pause() = 0;
	virtual int resume() = 0;
	virtual int stop() = 0;
protected:
	virtual Requester* configRequester() { return nil; }
private:	// class members
	static boolean usingNative;
	static Converter* converterInstance;
};

class NullConverter : public Converter {
public:		// object methods
	redefined boolean ready() const { return true; }
	redefined void configureHardware(class Controller*) {}
	redefined DataType bestPlayableType() { return NoData; }
	redefined boolean hasPlayableFormat(Sound *) { return true; }
	redefined int configure(Sound *sound, GoingTo) { return false; }
	redefined int start(StatusAction *, Action *) { return false; }
	redefined int pause() { return false; }
	redefined int resume() { return false; }
	redefined int stop() { return false; }
};

class RealConverter : public Converter {
public:		// object methods
	redefined boolean ready() const;			// check for successful constr.
	redefined void configureHardware(class Controller*);
	redefined boolean hasPlayableFormat(Sound *);
	redefined int configure(Sound *sound, GoingTo);
	redefined int start(StatusAction* askedToStop, Action* whenDone);
	redefined int pause();
	redefined int resume();
	redefined int stop();
protected:		// class methods
	enum Status { Error = 0, Stopped, Paused, Recording, Playing };
	virtual int startPlaying(StatusAction* askedToStop);
	virtual int startRecording(StatusAction* askedToStop, Action* whenDone);
	static void setStatus(Status s) { myStatus = s; }
	static void setRecordDoneAction(Action *);
	static void doRecordDoneAction();
	static boolean playing() { return myStatus == Playing; }
	static boolean recording() { return myStatus == Recording; }
	static boolean paused() { return myStatus == Paused; }
	static boolean running();
	static boolean good() { return myStatus != Error; }
protected:		// object methods
	RealConverter();
	virtual ~RealConverter();
	virtual boolean isPlayableFormat(DataType type) = 0;
	virtual int currentPlayLevel();				// scaled between 0 - 100
	virtual int currentRecordLevel();
	virtual int setPlayLevel(int volume);
	virtual int setRecordLevel(int volume);
	virtual int setSpeakerOutput(boolean);
	virtual int doConfigure() = 0;				// implm. by all subclasses
	virtual int doConversion(StatusAction* askedToStop) = 0;
	virtual int doRecording(StatusAction* askedToStop) = 0;
	virtual int waitForStop(StatusAction *) { return true; }
	// status methods
	virtual boolean willPlay() const { return goingTo == Play; }
	virtual boolean willRecord() const { return goingTo == Record; }
	virtual boolean soundIsLoaded() { return mySound != nil; }
	int checkLength(int);
	// these should be redefined for each converter
	virtual int checkSampleRate(int) { return true; }
	virtual int checkChannels(int) { return true; }
	virtual int checkDataType(DataType) { return true; }
	// these return information about current sound chunk
	virtual int sampleRate();
	virtual int channels();
	virtual DataType dataType();
	virtual const void* pointerToData();
	virtual long dataSize();
	virtual double peakAmplitude();
	virtual void fail() { setStatus(Error); }
	virtual void failIf(int condition) { if(condition) fail(); }
	virtual int error(const char* msg1, const char* msg2=nil);
	virtual void catchSignals(boolean flag) {}
	virtual void ignoreSignals(boolean flag) {}
	void initialize();
	void reset();
	boolean notImplemented();		// pops up message and returns false
	// internal base class which queries the user for config. parameters
	class ConfigRequester : public Requester {
	public:
		ConfigRequester(RealConverter* converter);
	protected:
		redefined void configureRequest(Request &);
		redefined boolean setValues(Request &);
		RealConverter* myConverter;
	private:
		redefined const QueryInfo *requestInfo();
	};
	friend ConfigRequester;
private:
	void setSound(Sound *);
private:	// class members
	static Status myStatus;
	static Action* recordDoneAction;
private:
	friend class Controller;
	Sound* mySound;
	GoingTo goingTo;
};

inline int
RealConverter::pause() {
	setStatus(Paused);
	return true;
}

inline int
RealConverter::resume() {
	setStatus(willPlay() ? Playing : Recording);
	return true;
}

// these are defined as not being implemented, so subclasses do not need to
// redefine these unless they will be functional.

inline int
RealConverter::currentPlayLevel() { notImplemented(); return -1; }

inline int
RealConverter::currentRecordLevel() { notImplemented(); return -1; }

inline int
RealConverter::setPlayLevel(int volume) {		// between 0 and 100
	return notImplemented();
}

inline int
RealConverter::setRecordLevel(int volume) {		// between 0 and 100
	return notImplemented();
}

inline int
RealConverter::setSpeakerOutput(boolean) {
	return notImplemented();
}

#endif
