// lpcfilter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "lpcdata.h"
#include "lpcfilter.h"

LPC_Filter::LPC_Filter(LPCData* lpd, int nsamps, int normalize)
		: totalSamps(nsamps), doNormalize(normalize), lpcdata(nil),
		  pastValues(nil), coeffArray(nil), currentArray(nil), nextArray(nil),
		  amplitudes(nil) {
	if(lpd->isA(LPC_Data))
		setLPC(lpd);
	else
		Application::alert("Selected source for filter is not an LPC data file.");
}

LPC_Filter::~LPC_Filter() {
	delete [] pastValues;
	delete [] nextArray;
	delete [] currentArray;
	delete [] coeffArray;
	Resource::unref(lpcdata);
	Resource::unref(amplitudes);
}

void
LPC_Filter::setLPC(LPCData* lpd) {
	if (lpd != lpcdata) {
		Resource::unref(lpcdata);
		lpcdata = lpd;
		if (lpcdata != nil) {
			lpcdata->ref();
			nPoles = lpcdata->nPoles();		// cache these for speed
			sampleRate = lpcdata->sRate();
			lastFrame = lpcdata->length() - 1;
		}
	}
}

void
LPC_Filter::initialize() {
	amplitudes = new Data(FloatData, lpcdata->length() + 1);
	amplitudes->ref();
	lpcdata->getEnvelope(amplitudes, 0, AbsoluteMagnitude);
	if(doNormalize)
		amplitudes->normalize();
	pastValues = new float[nPoles * 2];
	coeffArray = new float[nPoles + 4];
	currentArray = new float[nPoles + 4];
	nextArray = new float[nPoles + 4];
	reset();
}

void
LPC_Filter::reset() {
	counter = 0;
	frameSamps = 0;
	oldFrame = -1;
	sampsToGo = totalSamps;
	framesPerSamp = double(lpcdata->length() - 1) / totalSamps;
	int end = nPoles * 2;
	for(int i=0; i < end; i++)
		pastValues[i] = 0;
}

float *
LPC_Filter::getFrame(float* voicedAmp, float* frameError, float* framePitch) {
	static const int lpcOffset = 4;			// first 4 locs are not coeffs
	float frame = (totalSamps - sampsToGo) * framesPerSamp;
	float* array = coeffArray;
	if(needNewFrame()) {
		int currentFrame = int(frame);
 		if(currentFrame != oldFrame) {
			cacheFrames(currentFrame);
 			oldFrame = currentFrame;
 		}
		// get interpolated coeff values from cached coeffs
		array = interpolate(frame - currentFrame);
		*frameError = *(array+2);
		float pitch = *(array+3);
		*framePitch = pitch > 0.0 ? pitch : 256.0;	// can't be zero!
		frameSamps = int(min(sampleRate / *framePitch, float(sampsToGo)));
	}
	else {							// these must be computed every time
		*frameError = *(array+2);
		float pitch = *(array+3);
		*framePitch = pitch > 0.0 ? pitch : 256.0;	// can't be zero!
	}
	*voicedAmp = amplitudes->get(frame);	// may be normalized
	advance();
	return array + lpcOffset;
}

// frame arrays are cached to avoid doing interp frame lookup every samp

void
LPC_Filter::cacheFrames(int currentFrame) {
	int current = min(lastFrame, currentFrame);
	int next = min(lastFrame, currentFrame + 1);
	int framesize = nPoles + 4;
	lpcdata->getFrame(currentArray, framesize, current);
	lpcdata->getFrame(nextArray, framesize, next);
}

float *
LPC_Filter::interpolate(double fraction) {
	int framesize = nPoles + 4;
	register float* coeffs = coeffArray;
	register float* current = currentArray;
	register float* next = nextArray;
	for(int n = 0; n < framesize; n++)
		*(coeffs+n) = *(current+n) + fraction * (*(next+n) - *(current+n));
	return coeffArray;
}

void
LPC_Filter::advance() {
	sampsToGo--;
	frameSamps--;
}

double
LPC_Filter::filter(float* coeffs, double sig) {
	register float* past = pastValues;
	for(int j = counter, nfint=0;  nfint < nPoles;  nfint++, j++)
		sig += (*(coeffs+nfint) * *(past+j));
	*(past + counter) = *(past + counter + nPoles) = sig;
	counter = (counter + 1) % nPoles;
	return sig;	
}

double
LPC_Filter::warpFilter(float* coeffs, double sig) {
	float d=0, cq=0, oldSig=0;			// JUST TO ALLOW COMPILE
	register float* past = pastValues;
	float temp1 = *(past+nPoles-1);		// start with last past value
	*(past+nPoles-1) = cq * oldSig - d * *(past+nPoles-1);
	for(int n = nPoles - 2; n >= 0; n--) {
		float temp2 = *(past+n);
		*(past+n) = d * (*(past+n+1) - *(past+n)) + temp1;
		temp1 = temp2;
	}
	for(n = 0; n < nPoles; n++)
		sig += *(coeffs+n) * *(past+n);
	oldSig = sig;
	return sig;
}
