// processfun.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// ProcessFunction is a subclass of Modifier that includes all routines
// that alter the contents of a Data subclass via iterative operations on the
// samples.
// ProcessFunction is abstract because its DataModifier methods are not
// redefined.

#ifndef PROCESSFUN_H
#ifdef __GNUG__
#pragma interface
#endif
#define PROCESSFUN_H

#include "datamodifier.h"

class Data;
class DataEditor;

class ProcessFunction : public DataModifier {
public:
	ProcessFunction(Data* d) : DataModifier(d) {}
	virtual void restoreState() {}
protected:
	redefined boolean setValues(Request& r)=0;
};

// a SimpleFunction takes a single double input and returns a double

class SimpleFunction : public ProcessFunction {
	typedef ProcessFunction Super;
public:
	SimpleFunction(Data* d) : ProcessFunction(d) {}
	virtual double operator () (double input) = 0;
protected:
	redefined int doApply(Data *);
};

// abstract base class for simple functions with only one factor

class SingleFactorFunction : public SimpleFunction {
	typedef SimpleFunction Super;
protected:
	SingleFactorFunction(Data* d, double val) 
		: SimpleFunction(d), factor(val) {}
	SingleFactorFunction(Data* d) : SimpleFunction(d), factor(0) {}
	void setFactor(double f) { factor = f; }
protected:
	redefined boolean setValues(Request& r);

	double factor;
private:
	redefined const QueryInfo *requestInfo() = 0;
};

// this does f(x) = x + n

class Offset : public SingleFactorFunction {
public:
	Offset(Data* d, double val) : SingleFactorFunction(d, val) {
		initialize();
	}
	Offset(Data* d) : SingleFactorFunction(d) {}
	static Modifier* create(DataEditor*);
	virtual ~Offset() {}
	redefined double operator () (double input) { return input + factor; }
	redefined const char* message() { return "Adding offset..."; }
	redefined Modifier* createUnDo();
private:
	redefined const QueryInfo *requestInfo();
};

// this does f(x) = rx

class Scaler : public SingleFactorFunction {
public:
	Scaler(Data* d, double val) : SingleFactorFunction(d, val) {
		initialize();
	}
	Scaler(Data* d) : SingleFactorFunction(d) {}
	static Modifier* create(DataEditor*);
	virtual ~Scaler() {}
	redefined double operator () (double input) { return input * factor; }
	redefined const char* message() {
		return (factor == 0.0) ? "Erasing..." : "Scaling...";
	}
	redefined Modifier* createUnDo();
private:
	redefined const QueryInfo *requestInfo();
};

class Eraser : public Scaler {
public:
	Eraser(Data* d) : Scaler(d, 0.0) {}		// initialize() called in Scaler
	static Modifier* create(DataEditor*);
	redefined Modifier* createUnDo();
};

#endif
